/*
 * Copyright 2004-2007 J. Dahl and L. Vandenberghe.
 *
 * This file is part of CVXOPT version 0.9.
 *
 * CVXOPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * CVXOPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define BASE_MODULE

#include "Python.h"
#include "cvxopt.h"
#include "misc.h" 

#include <complexobject.h>

PyDoc_STRVAR(base__doc__,"Convex optimization package");

PyTypeObject matrix_tp ;
matrix * Matrix_New(int, int, int) ;
matrix * Matrix_NewFromMatrix(matrix *, int) ;
matrix * Matrix_NewFromSequence(PyObject *, int) ; 
matrix * Matrix_NewFromArrayStruct(PyObject *, int, int *) ;

PyTypeObject spmatrix_tp ;
spmatrix * SpMatrix_New(int_t, int_t, int, int ) ;
spmatrix * SpMatrix_NewFromMatrix(matrix *, int) ;
spmatrix * SpMatrix_NewFromSpMatrix(spmatrix *, int, int) ; 
spmatrix * SpMatrix_NewFromIJV(matrix *, matrix *, matrix *, 
    int_t, int_t, int, int) ;
void free_ccs(ccs *) ;

extern int (*sp_axpy[])(number, void *, void *, int, int, int, void **) ;

extern int (*sp_gemm[])(char, char, number, void *, void *, number, void *, 
    int, int, int, int, void **, int, int, int);

extern int (*sp_gemv[])(char, int, int, number, void *, int, void *, int,  
    number, void *, int) ;

extern int (*sp_symv[])(char, int, number, ccs *, int, void *, int,
    number, void *, int) ;

extern int (*sp_syrk[])(char, char, number, void *, number, 
    void *, int, int, int, int, void **) ;

const int  E_SIZE[] = { sizeof(int_t), sizeof(double), sizeof(complex) };
const char TC_CHAR[][2] = {"i","d","z"} ;
const char PRINTOPT[][15] = {"iformat","dformat","zformat"} ;

PyObject *base_mod;

/* 
 *  Helper routines and definitions to implement type transparency.
 */

number One[3], MinusOne[3], Zero[3];

static void write_inum(void *dest, int i, void *src, int j) {
  ((int_t *)dest)[i]  = ((int_t *)src)[j];
}

static void write_dnum(void *dest, int i, void *src, int j) {
  ((double *)dest)[i]  = ((double *)src)[j];
}

static void write_znum(void *dest, int i, void *src, int j) {
  ((complex *)dest)[i]  = ((complex *)src)[j];
}

void (*write_num[])(void *, int, void *, int) = {
  write_inum, write_dnum, write_znum };

static PyObject * inum2PyObject(void *src, int i) {
  return Py_BuildValue("l", ((int_t *)src)[i]);
}

static PyObject * dnum2PyObject(void *src, int i) {
  return Py_BuildValue("d", ((double *)src)[i]);
}

static PyObject * znum2PyObject(void *src, int i) {
  Py_complex z;
  z.real = creal (((complex *)src)[i]);
  z.imag = cimag (((complex *)src)[i]);
  return Py_BuildValue("D", &z);
}

PyObject * (*num2PyObject[])(void *, int) = {
  inum2PyObject, dnum2PyObject, znum2PyObject };

/* val_id: 0 = matrix, 1 = PyNumber */
static int 
convert_inum(void *dest, void *val, int val_id, int offset) 
{
  if (val_id==0) { /* 1x1 matrix */
    switch (MAT_ID(val)) {
    case INT: 
      *(int_t *)dest = MAT_BUFI(val)[offset]; return 0; 
      //case DOUBLE: 
      //*(int_t *)dest = (int_t)round(MAT_BUFD(val)[offset]); return 0; 
    default: PY_ERR_INT(PyExc_TypeError,"cannot cast argument as integer");
    }
  } else { /* PyNumber */
    if (PyInt_Check((PyObject *)val)) {
      *(int_t *)dest = PyInt_AS_LONG((PyObject *)val); return 0; 
    } 
    //else if (PyFloat_Check((PyObject *)val)) {
    //  *(int_t *)dest = roundl(PyFloat_AS_DOUBLE((PyObject *)val)); return 0; 
    //}
    else PY_ERR_INT(PyExc_TypeError,"cannot cast argument as integer");
  }    
}

static int 
convert_dnum(void *dest, void *val, int val_id, int offset) 
{
  if (val_id==0) { /* matrix */
    switch (MAT_ID(val)) {
    case INT:    *(double *)dest = MAT_BUFI(val)[offset]; return 0;
    case DOUBLE: *(double *)dest = MAT_BUFD(val)[offset]; return 0;
    default: PY_ERR_INT(PyExc_TypeError, "cannot cast argument as double");
    }
  } else { /* PyNumber */
    if (PyInt_Check((PyObject *)val) || PyFloat_Check((PyObject *)val)) {
      *(double *)dest = PyFloat_AsDouble((PyObject *)val); 
      return 0;
    }
    else PY_ERR_INT(PyExc_TypeError,"cannot cast argument as double");
  } 
}

static int 
convert_znum(void *dest, void *val, int val_id, int offset) 
{  
  if (val_id==0) { /* 1x1 matrix */
    switch (MAT_ID(val)) {
    case INT:     
      *(complex *)dest = MAT_BUFI(val)[offset]; return 0;
    case DOUBLE:  
      *(complex *)dest = MAT_BUFD(val)[offset]; return 0;
    case COMPLEX:
      *(complex *)dest = MAT_BUFZ(val)[offset]; return 0;
    default: return -1; 
    }
  } else { /* PyNumber */
    Py_complex c = PyComplex_AsCComplex((PyObject *)val);
    *(complex *)dest = c.real + I*c.imag;
    return 0;
  }    
}

int (*convert_num[])(void *, void *, int, int) = { 
  convert_inum, convert_dnum, convert_znum };

extern void daxpy_(int *, void *, void *, int *, void *, int *) ;
extern void zaxpy_(int *, void *, void *, int *, void *, int *) ;

static void i_axpy(int *n, void *a, void *x, int *incx, void *y, int *incy) {
  int i;
  for (i=0; i < *n; i++) {
    ((int_t *)y)[i*(*incy)] += *((int_t *)a)*((int_t *)x)[i*(*incx)];
  }
}

void (*axpy[])(int *, void *, void *, int *, void *, int *) = { 
  i_axpy, daxpy_, zaxpy_ };

extern void dscal_(int *, void *, void *, int *) ;
extern void zscal_(int *, void *, void *, int *) ;

/* we dont implement a BLAS iscal */
static void i_scal(int *n, void *a, void *x, int *incx) {
  int i;
  for (i=0; i < *n; i++) {
    ((int_t *)x)[i*(*incx)] *= *((int_t *)a);
  }
}

void (*scal[])(int *, void *, void *, int *) = { i_scal, dscal_, zscal_ };

extern void dgemm_(char *, char *, int *, int *, int *, void *, void *,
    int *, void *, int *, void *, void *, int *) ;
extern void zgemm_(char *, char *, int *, int *, int *, void *, void *,
    int *, void *, int *, void *, void *, int *) ;

/* we dont implement a BLAS igemm */
static void i_gemm(char *transA, char *transB, int *m, int *n, int *k, 
    void *alpha, void *A, int *ldA, void *B, int *ldB, void *beta, 
    void *C, int *ldC) 
{
  int i, j, l;
  for (j=0; j<*n; j++) {   
    for (i=0; i<*m; i++) {      
      ((int_t *)C)[i+j*(*m)] = 0;
      for (l=0; l<*k; l++)
	((int_t *)C)[i+j*(*m)]+=((int_t *)A)[i+l*(*m)]*((int_t *)B)[j*(*k)+l];
    }    
  }  
}

void (*gemm[])(char *, char *, int *, int *, int *, void *, void *, int *, 
    void *, int *, void *, void *, int *) = { i_gemm, dgemm_, zgemm_ };

extern void dgemv_(char *, int *, int *, void *, void *, int *, void *, 
    int *, void *, void *, int *);
extern void zgemv_(char *, int *, int *, void *, void *, int *, void *, 
    int *, void *, void *, int *);
static void (*gemv[])(char *, int *, int *, void *, void *, int *, void *, 
    int *, void *, void *, int *) = { NULL, dgemv_, zgemv_ };

extern void dsyrk_(char *, char *, int *, int *, void *, void *,
    int *, void *, void *, int *);
extern void zsyrk_(char *, char *, int *, int *, void *, void *,
    int *, void *, void *, int *);
void (*syrk[])(char *, char *, int *, int *, void *, void *,
    int *, void *, void *, int *) = { NULL, dsyrk_, zsyrk_ };

extern void dsymv_(char *, int *, void *, void *, int *, void *, int *, 
    void *, void *, int *);
extern void zsymv_(char *, int *, void *, void *, int *, void *, int *, 
    void *, void *, int *);
void (*symv[])(char *, int *, void *, void *, int *, void *, int *, 
    void *, void *, int *) = { NULL, dsymv_, zsymv_ };

static void mtx_iabs(void *src, void *dest, int n) {
  int i;
  for (i=0; i<n; i++)
    ((int_t *)dest)[i] = labs(((int_t *)src)[i]);
}

static void mtx_dabs(void *src, void *dest, int n) {
  int i;
  for (i=0; i<n; i++)
    ((double *)dest)[i] = fabs(((double *)src)[i]);
}

static void mtx_zabs(void *src, void *dest, int n) {
  int i;
  for (i=0; i<n; i++) 
    ((double *)dest)[i] = cabs(((complex *)src)[i]);
}

void (*mtx_abs[])(void *, void *, int) = { mtx_iabs, mtx_dabs, mtx_zabs };

static int idiv(void *dest, number a, int n) {
  if (a.i==0) PY_ERR_INT(PyExc_ZeroDivisionError, "division by zero");
  int i;
  for (i=0; i<n; i++)
    ((int_t *)dest)[i] /= a.i;

  return 0;
}

static int ddiv(void *dest, number a, int n) {
  if (a.d==0.0) PY_ERR_INT(PyExc_ZeroDivisionError, "division by zero");
  int _n = n, int1 = 1;
  double _a = 1/a.d;
  dscal_(&_n, (void *)&_a, dest, &int1);
  return 0;
}

static int zdiv(void *dest, number a, int n) {
  if (cabs(a.z) == 0.0)
    PY_ERR_INT(PyExc_ZeroDivisionError, "division by zero");

  int _n = n, int1 = 1;
  complex _a = 1.0/a.z; 
  zscal_(&_n, (void *)&_a, dest, &int1);
  return 0;  
}

int (*div_array[])(void *, number, int) = { idiv, ddiv, zdiv };

static int mtx_irem(void *dest, number a, int n) {
  if (a.i==0) PY_ERR_INT(PyExc_ZeroDivisionError, "division by zero");
  int i;
  for (i=0; i<n; i++)
    ((int_t *)dest)[i] %= a.i;

  return 0;
}

static int mtx_drem(void *dest, number a, int n) {
  if (a.d==0.0) PY_ERR_INT(PyExc_ZeroDivisionError, "division by zero");
  int i;
  for (i=0; i<n; i++) 
    ((double *)dest)[i] -= floor(((double *)dest)[i]/a.d)*a.d;

  return 0;
}

int (*mtx_rem[])(void *, number, int) = { mtx_irem, mtx_drem };

/* val_type = 0: (sp)matrix if type = 0, PY_NUMBER if type = 1 */
int get_id(void *val, int val_type) {
  if (!val_type) {
    if Matrix_Check((PyObject *)val)
      return MAT_ID((matrix *)val);
    else 
      return SP_ID((spmatrix *)val);
  }
  else if (PyInt_Check((PyObject *)val)) 
    return INT;
  else if (PyFloat_Check((PyObject *)val))
    return DOUBLE;
  else 
    return COMPLEX;
}



static int Matrix_Check_func(void *o) {
  return Matrix_Check((PyObject*)o);
}

static int SpMatrix_Check_func(void *o) {
  return SpMatrix_Check((PyObject *)o);
}


static char doc_axpy[] =
    "Constant times a vector plus a vector (y := alpha*x+y).\n\n"
    "axpy(x, y, n=None, alpha=1.0)\n\n"
    "ARGUMENTS\n"
    "x         'd' or 'z' (sp)matrix\n\n"
    "y         'd' or 'z' (sp)matrix.  Must have the same type as x.\n\n"
    "n         integer.  If n<0, the default value of n is used.\n"
    "          The default value is equal to\n"
    "          (len(x)>=offsetx+1) ? 1+(len(x)-offsetx-1)/incx : 0.\n\n"
    "alpha     number (int, float or complex).  Complex alpha is only\n"
    "          allowed if x is complex";

PyObject * base_axpy(PyObject *self, PyObject *args, PyObject *kwrds)
{
  PyObject *x, *y, *partial = NULL; 
  PyObject *ao=NULL;
  number a;
  char *kwlist[] = {"x", "y", "alpha", "partial", NULL};
  
  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "OO|OO", kwlist, 
	  &x, &y, &ao, &partial)) return NULL;
  
  if (!Matrix_Check(x) && !SpMatrix_Check(x)) err_mtrx("x");
  if (!Matrix_Check(y) && !SpMatrix_Check(y)) err_mtrx("y");
  if (partial && !PyBool_Check(partial)) err_bool("partial");

  if (X_ID(x) != X_ID(y)) err_conflicting_ids;
  int id = X_ID(x);

  if (X_NROWS(x) != X_NROWS(y) || X_NCOLS(x) != X_NCOLS(y))
    PY_ERR_TYPE("dimensions of x and y do not match");
  
  if (ao && convert_num[id](&a, ao, 1, 0)) err_type("alpha");
  
  if (Matrix_Check(x) && Matrix_Check(y)) {
    int n = X_NROWS(x)*X_NCOLS(x);
    axpy[id](&n, (ao ? &a : &One[id]), MAT_BUF(x), (int *)&One[INT], 
	MAT_BUF(y), (int *)&One[INT]);
  }
  else {
    
    void *z = NULL;
    if (sp_axpy[id]((ao ? a : One[id]), 
	    Matrix_Check(x) ? MAT_BUF(x): ((spmatrix *)x)->obj,  
	    Matrix_Check(y) ? MAT_BUF(y): ((spmatrix *)y)->obj,  
	    SpMatrix_Check(x), SpMatrix_Check(y), 
	    partial ? PyInt_AS_LONG(partial) : 0, &z))
      return PyErr_NoMemory(); 

    if (z) { 
      free_ccs( ((spmatrix *)y)->obj );
      ((spmatrix *)y)->obj = z;
    }
  } 

  return Py_BuildValue("");
}

static char doc_gemm[] =
    "General matrix-matrix product.\n\n"
    "gemm(A, B, C, transA='N', transB='N', alpha=1.0, beta=0.0, \n"
    "     partial=False) \n\n"
    "PURPOSE\n"   
    "Computes \n"
    "C := alpha*A*B + beta*C     if transA = 'N' and transB = 'N'.\n"
    "C := alpha*A^T*B + beta*C   if transA = 'T' and transB = 'N'.\n"
    "C := alpha*A^H*B + beta*C   if transA = 'C' and transB = 'N'.\n"
    "C := alpha*A*B^T + beta*C   if transA = 'N' and transB = 'T'.\n"
    "C := alpha*A^T*B^T + beta*C if transA = 'T' and transB = 'T'.\n"
    "C := alpha*A^H*B^T + beta*C if transA = 'C' and transB = 'T'.\n"
    "C := alpha*A*B^H + beta*C   if transA = 'N' and transB = 'C'.\n"
    "C := alpha*A^T*B^H + beta*C if transA = 'T' and transB = 'C'.\n"
    "C := alpha*A^H*B^H + beta*C if transA = 'C' and transB = 'C'.\n"
    "If k=0, this reduces to C := beta*C.\n\n"
    "ARGUMENTS\n\n" 
    "A         'd' or 'z' matrix\n\n"
    "B         'd' or 'z' matrix.  Must have the same type as A.\n\n"
    "C         'd' or 'z' matrix.  Must have the same type as A.\n\n"
    "transA    'N', 'T' or 'C'\n\n"
    "transB    'N', 'T' or 'C'\n\n"
    "alpha     number (int, float or complex).  Complex alpha is only\n"
    "          allowed if A is complex.\n\n"
    "beta      number (int, float or complex).  Complex beta is only\n"
    "          allowed if A is complex.\n\n"
    "partial   boolean. If C is sparse and partial is True, then only the\n"
    "          nonzero elements of C are updated irrespective of the\n"
    "          sparsity patterns of A and B.";

PyObject* base_gemm(PyObject *self, PyObject *args, PyObject *kwrds)
{
  PyObject *A, *B, *C, *partial=NULL;
  PyObject *ao=NULL, *bo=NULL;
  number a, b;
  int m, n, k;
  char transA='N', transB='N';
  char *kwlist[] = {"A", "B", "C", "transA", "transB", "alpha", "beta",
		    "partial", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "OOO|ccOOO", 
	  kwlist, &A, &B, &C, &transA, &transB, &ao, &bo, &partial))
    return NULL;

  if (!(Matrix_Check(A) || SpMatrix_Check(A)))
    PY_ERR_TYPE("A must a matrix or spmatrix");
  if (!(Matrix_Check(B) || SpMatrix_Check(B)))
    PY_ERR_TYPE("B must a matrix or spmatrix");
  if (!(Matrix_Check(C) || SpMatrix_Check(C)))
    PY_ERR_TYPE("C must a matrix or spmatrix");
  if (partial && !PyBool_Check(partial)) err_bool("partial");

  if (X_ID(A) != X_ID(B) || X_ID(A) != X_ID(C) ||
      X_ID(B) != X_ID(C)) err_conflicting_ids;

  if (transA != 'N' && transA != 'T' && transA != 'C') 
    err_char("transA", "'N', 'T', 'C'");
  if (transB != 'N' && transB != 'T' && transB != 'C')
    err_char("transB", "'N', 'T', 'C'");

  m = (transA == 'N') ? X_NROWS(A) : X_NCOLS(A);
  n = (transB == 'N') ? X_NCOLS(B) : X_NROWS(B);
  k = (transA == 'N') ? X_NCOLS(A) : X_NROWS(A);
  if (k != ((transB == 'N') ? X_NROWS(B) : X_NCOLS(B)))  
    PY_ERR_TYPE("dimensions of A and B do not match");

  if (m == 0 || n == 0) return Py_BuildValue("");

  if (ao && convert_num[X_ID(A)](&a, ao, 1, 0)) err_type("alpha");
  if (bo && convert_num[X_ID(A)](&b, bo, 1, 0)) err_type("beta");

  int id = X_ID(A);
  if (Matrix_Check(A) && Matrix_Check(B) && Matrix_Check(C)) {

    int ldA = MAX(1,MAT_NROWS(A));
    int ldB = MAX(1,MAT_NROWS(B));
    int ldC = MAX(1,MAT_NROWS(C));  
    if (id == INT) err_invalid_id;
    gemm[id](&transA, &transB, &m, &n, &k, (ao ? &a : &One[id]),
	MAT_BUF(A), &ldA, MAT_BUF(B), &ldB, (bo ? &b : &Zero[id]), 
	MAT_BUF(C), &ldC);
  } else {

    void *z = NULL;
    if (sp_gemm[id](transA, transB, (ao ? a : One[id]), 
	    Matrix_Check(A) ? MAT_BUF(A) : ((spmatrix *)A)->obj, 
	    Matrix_Check(B) ? MAT_BUF(B) : ((spmatrix *)B)->obj,  
	    (bo ? b : Zero[id]),
	    Matrix_Check(C) ? MAT_BUF(C) : ((spmatrix *)C)->obj, 
	    SpMatrix_Check(A), SpMatrix_Check(B), SpMatrix_Check(C), 
	    partial ? PyInt_AS_LONG(partial) : 0, &z, m, n, k))
      return PyErr_NoMemory(); 

    if (z) { 
      free_ccs( ((spmatrix *)C)->obj );
      ((spmatrix *)C)->obj = z;
    }    
  }

  return Py_BuildValue("");
}

static char doc_gemv[] =
    "General matrix-vector product for sparse and dense matrices. \n\n"
    "gemv(A, x, y, trans='N', alpha=1.0, beta=0.0, m=A.size[0],\n"
    "     n=A.size[1], incx=1, incy=1, offsetA=0, offsetx=0, offsety=0)\n\n"
    "PURPOSE\n"
    "If trans is 'N', computes y := alpha*A*x + beta*y.\n"
    "If trans is 'T', computes y := alpha*A^T*x + beta*y.\n"
    "If trans is 'C', computes y := alpha*A^H*x + beta*y.\n"
    "The matrix A is m by n.\n"
    "Returns immediately if n=0 and trans is 'T' or 'C', or if m=0 \n"
    "and trans is 'N'.\n"
    "Computes y := beta*y if n=0, m>0 and trans is 'N', or if m=0, \n"
    "n>0 and trans is 'T' or 'C'.\n\n"
    "ARGUMENTS\n"
    "A         'd' or 'z' (sp)matrix\n\n"
    "x         'd' or 'z' matrix.  Must have the same type as A.\n\n"
    "y         'd' or 'z' matrix.  Must have the same type as A.\n\n"
    "trans     'N', 'T' or 'C'\n\n"
    "alpha     number (int, float or complex).  Complex alpha is only\n"
    "          allowed if A is complex.\n\n"
    "beta      number (int, float or complex).  Complex beta is only\n"
    "          allowed if A is complex.\n\n"
    "m         integer.  If negative, the default value is used.\n\n"
    "n         integer.  If negative, the default value is used.\n\n"
    "          If zero, the default value is used.\n\n"
    "incx      nonzero integer\n\n"
    "incy      nonzero integer\n\n"
    "offsetA   nonnegative integer\n\n"
    "offsetx   nonnegative integer\n\n"
    "offsety   nonnegative integer\n\n"
    "This sparse version of GEMV requires that\n"
    "  m <= A.size[0] - (offsetA % A.size[0])";

static PyObject* base_gemv(PyObject *self, PyObject *args, PyObject *kwrds)
{
  matrix *x, *y;
  PyObject *A;
  PyObject *ao=NULL, *bo=NULL;
  number a, b;
  int m=-1, n=-1, ix=1, iy=1, oA=0, ox=0, oy=0; 
  char trans='N'; 
  char *kwlist[] = {"A", "x", "y", "trans", "alpha", "beta", "m", "n", 
		    "incx", "incy", "offsetA", "offsetx",
		    "offsety", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "OOO|cOOiiiiiii", 
	  kwlist, &A, &x, &y, &trans, &ao, &bo, &m, &n, &ix, &iy, 
	  &oA, &ox, &oy))
    return NULL;
  
  if (!Matrix_Check(A) && !SpMatrix_Check(A))
    PY_ERR(PyExc_TypeError, "A must be a dense or sparse matrix");
  if (!Matrix_Check(x)) err_mtrx("x");
  if (!Matrix_Check(y)) err_mtrx("y");

  if (MAT_ID(x) == INT || MAT_ID(y) == INT || X_ID(A) == INT) 
    PY_ERR_TYPE("invalid matrix types");

  if (X_ID(A) != MAT_ID(x) || X_ID(A) != MAT_ID(y))
    err_conflicting_ids;
  
  if (trans != 'N' && trans != 'T' && trans != 'C') 
    err_char("trans", "'N','T','C'");

  if (ix == 0) err_nz_int("incx");
  if (iy == 0) err_nz_int("incy");
  
  int id = MAT_ID(x);
  if (m < 0) m = X_NROWS(A);
  if (n < 0) n = X_NCOLS(A);
  if ((!m && trans == 'N') || (!n && (trans == 'T' || trans == 'C')))
    return Py_BuildValue("");

  if (oA < 0) err_nn_int("offsetA");
  if (n > 0 && m > 0 && oA + (n-1)*MAX(1,X_NROWS(A)) + m > 
      X_NROWS(A)*X_NCOLS(A))
    err_buf_len("A");
  
  if (ox < 0) err_nn_int("offsetx");
  if ((trans == 'N' && n > 0 && ox + (n-1)*abs(ix) + 1 > MAT_LGT(x)) || 
      ((trans == 'T' || trans == 'C') && m > 0 && 
	  ox + (m-1)*abs(ix) + 1 > MAT_LGT(x))) err_buf_len("x");
  
  if (oy < 0) err_nn_int("offsety");
  if ((trans == 'N' && oy + (m-1)*abs(iy) + 1 > MAT_LGT(y)) || 
      ((trans == 'T' || trans == 'C') && 
	  oy + (n-1)*abs(iy) + 1 > MAT_LGT(y))) err_buf_len("y");

  if (ao && convert_num[MAT_ID(x)](&a, ao, 1, 0)) err_type("alpha");
  if (bo && convert_num[MAT_ID(x)](&b, bo, 1, 0)) err_type("beta");
  
  if (Matrix_Check(A)) {
    int ldA = MAX(1,X_NROWS(A));
    if (trans == 'N' && n == 0)
      scal[id](&m, (bo ? &b : &Zero[id]), MAT_BUF(y)+oy*E_SIZE[id], &iy);
    else if ((trans == 'T' || trans == 'C') && m == 0)
      scal[id](&n, (bo ? &b : &Zero[id]), MAT_BUF(y)+oy*E_SIZE[id], &iy);
    else 
      gemv[id](&trans, &m, &n, (ao ? &a : &One[id]), 
	  MAT_BUF(A) + oA*E_SIZE[id], &ldA, 
	  MAT_BUF(x) + ox*E_SIZE[id], &ix, (bo ? &b : &Zero[id]), 
	  MAT_BUF(y) + oy*E_SIZE[id], &iy);
  } else {
    if (sp_gemv[id](trans, m, n, (ao ? a : One[id]), ((spmatrix *)A)->obj, 
	    oA, MAT_BUF(x) + ox*E_SIZE[id], ix, (bo ? b : Zero[id]), 
	    MAT_BUF(y) + oy*E_SIZE[id], iy))
      return PyErr_NoMemory();      
  }
  
  return Py_BuildValue("");
}

static char doc_syrk[] =
    "Rank-k update of symmetric sparse or dense matrix.\n\n"
    "syrk(A, C, uplo='L', trans='N', alpha=1.0, beta=0.0, partial=False)\n\n"
    "PURPOSE   \n"
    "If trans is 'N', computes C := alpha*A*A^T + beta*C.\n"
    "If trans is 'T', computes C := alpha*A^T*A + beta*C.\n"
    "C is symmetric (real or complex) of order n. \n"
    "ARGUMENTS\n"
    "A         'd' or 'z' (sp)matrix\n\n"
    "C         'd' or 'z' (sp)matrix.  Must have the same type as A.\n\n"
    "uplo      'L' or 'U'\n\n"
    "trans     'N' or 'T'\n\n"
    "alpha     number (int, float or complex).  Complex alpha is only\n"
    "          allowed if A is complex.\n\n"
    "beta      number (int, float or complex).  Complex beta is only\n"
    "          allowed if A is complex.\n\n"
    "partial   boolean. If C is sparse and partial is True, then only the\n"
    "          nonzero elements of C are updated irrespective of the\n"
    "          sparsity patterns of A.";

static PyObject* base_syrk(PyObject *self, PyObject *args, PyObject *kwrds)
{
  PyObject *A, *C, *partial=NULL, *ao=NULL, *bo=NULL;
  number a, b; 
  char trans='N', uplo='L';
  char *kwlist[] = {"A", "C", "uplo", "trans", "alpha", "beta", "partial",
		    NULL};
  
  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "OO|ccOOO", kwlist, 
	  &A, &C, &uplo, &trans, &ao, &bo, &partial))
    return NULL;
  
  if (!(Matrix_Check(A) || SpMatrix_Check(A)))
    PY_ERR_TYPE("A must be a dense or sparse matrix");
  if (!(Matrix_Check(C) || SpMatrix_Check(C)))
    PY_ERR_TYPE("C must be a dense or sparse matrix");

  int id = X_ID(A);
  if (id == INT) PY_ERR_TYPE("invalid matrix types");
  if (id != X_ID(C)) err_conflicting_ids;
  
  if (uplo != 'L' && uplo != 'U') err_char("uplo", "'L', 'U'");
  if (id == DOUBLE && trans != 'N' && trans != 'T' &&
      trans != 'C') err_char("trans", "'N', 'T', 'C'");
  if (id == COMPLEX && trans != 'N' && trans != 'T') 
    err_char("trans", "'N', 'T'");

  if (partial && !PyBool_Check(partial)) err_bool("partial");
  
  int n = (trans == 'N') ? X_NROWS(A) : X_NCOLS(A);
  int k = (trans == 'N') ? X_NCOLS(A) : X_NROWS(A);
  if (n == 0) return Py_BuildValue("");
  
  if (ao && convert_num[id](&a, ao, 1, 0)) err_type("alpha");
  if (bo && convert_num[id](&b, bo, 1, 0)) err_type("beta");

  if (Matrix_Check(A) && Matrix_Check(C)) {

    int ldA = MAX(1,MAT_NROWS(A));
    int ldC = MAX(1,MAT_NROWS(C));

    syrk[id](&uplo, &trans, &n, &k, (ao ? &a : &One[id]), 
	MAT_BUF(A), &ldA, (bo ? &b : &Zero[id]), MAT_BUF(C), &ldC);
  } else {

    void *z = NULL;
    if (sp_syrk[id](uplo, trans, 
	    (ao ? a : One[id]), 
	    Matrix_Check(A) ? MAT_BUF(A) : ((spmatrix *)A)->obj, 
	    (bo ? b : Zero[id]),
	    Matrix_Check(C) ? MAT_BUF(C) : ((spmatrix *)C)->obj, 
	    SpMatrix_Check(A), SpMatrix_Check(C), 
	    partial ? PyInt_AS_LONG(partial) : 0, 
	    (trans == 'N' ? X_NCOLS(A) : X_NROWS(A)), &z))
      return PyErr_NoMemory(); 

    if (z) { 
      free_ccs( ((spmatrix *)C)->obj );
      ((spmatrix *)C)->obj = z;
    }    
  }
  
  return Py_BuildValue("");
}

static char doc_symv[] =
    "Matrix-vector product with a real symmetric dense or sparse matrix.\n\n"
    "symv(A, x, y, uplo='L', alpha=1.0, beta=0.0, n=A.size[0], \n"
    "     ldA=max(1,A.size[0]), incx=1, incy=1, offsetA=0, offsetx=0,\n"
    "     offsety=0)\n\n"
    "PURPOSE\n"
    "Computes y := alpha*A*x + beta*y with A real symmetric of order n."
    "n\n"
    "ARGUMENTS\n"
    "A         'd' (sp)matrix\n\n"
    "x         'd' matrix\n\n"
    "y         'd' matrix\n\n"
    "uplo      'L' or 'U'\n\n"
    "alpha     real number (int or float)\n\n"
    "beta      real number (int or float)\n\n"
    "n         integer.  If negative, the default value is used.\n"
    "          If the default value is used, we require that\n"
    "          A.size[0]=A.size[1].\n\n"
    "incx      nonzero integer\n\n"
    "incy      nonzero integer\n\n"
    "offsetA   nonnegative integer\n\n"
    "offsetx   nonnegative integer\n\n"
    "offsety   nonnegative integer\n\n"
    "This sparse version of SYMV requires that\n"
    "  m <= A.size[0] - (offsetA % A.size[0])";

static PyObject* base_symv(PyObject *self, PyObject *args, PyObject *kwrds)
{
  PyObject *A, *ao=NULL, *bo=NULL;
  matrix *x, *y;
  number a, b;
  int n=-1, ix=1, iy=1, oA=0, ox=0, oy=0, ldA; 
  char uplo='L'; 
  char *kwlist[] = {"A", "x", "y", "uplo", "alpha", "beta", "n",
		    "incx", "incy", "offsetA", "offsetx", "offsety", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "OOO|cOOiiiiii", kwlist, &A, 
	  &x, &y, &uplo, &ao, &bo, &n, &ix, &iy, &oA, &ox, &oy))
    return NULL;

  if (!Matrix_Check(A) && !SpMatrix_Check(A)) 
    PY_ERR_TYPE("A must be a dense or sparse matrix");

  ldA = MAX(1,X_NROWS(A));

  if (!Matrix_Check(x)) err_mtrx("x");
  if (!Matrix_Check(y)) err_mtrx("y");
  if (X_ID(A) != MAT_ID(x) || X_ID(A) != MAT_ID(y) ||
      MAT_ID(x) != MAT_ID(y)) err_conflicting_ids;

  int id = X_ID(A);
  if (id == INT) PY_ERR_TYPE("invalid matrix types");
  
  if (uplo != 'L' && uplo != 'U') err_char("uplo", "'L', 'U'");

  if (ix == 0) err_nz_int("incx");
  if (iy == 0) err_nz_int("incy");

  if (n < 0) { 
    if (X_NROWS(A) != X_NCOLS(A)) {
      PyErr_SetString(PyExc_ValueError, "A is not square");
      return NULL;
    }
    n = X_NROWS(A);
  }
  if (n == 0) return Py_BuildValue("");

  if (oA < 0) err_nn_int("offsetA");
  if (oA + (n-1)*ldA + n > len(A)) err_buf_len("A");
  if (ox < 0) err_nn_int("offsetx");
  if (ox + (n-1)*abs(ix) + 1 > len(x)) err_buf_len("x");
  if (oy < 0) err_nn_int("offsety");
  if (oy + (n-1)*abs(iy) + 1 > len(y)) err_buf_len("y");
    
  if (ao && convert_num[id](&a, ao, 1, 0)) err_type("alpha");
  if (bo && convert_num[id](&b, bo, 1, 0)) err_type("beta");

  if (Matrix_Check(A)) {
    
    symv[id](&uplo, &n, (ao ? &a : &One[id]), 
	MAT_BUF(A) + oA*E_SIZE[id], &ldA, MAT_BUF(x) + ox*E_SIZE[id], 
	&ix, (bo ? &b : &Zero[id]), MAT_BUF(y) + oy*E_SIZE[id], &iy);    
  }
  else {
    
    if (sp_symv[id](uplo, n, (ao ? a : One[id]), ((spmatrix *)A)->obj, 
	    oA, MAT_BUF(x) + ox*E_SIZE[id], ix, 
	    (bo ? b : Zero[id]), MAT_BUF(y) + oy*E_SIZE[id], iy))
      return PyErr_NoMemory();      
  }
  
  return Py_BuildValue("");
}

spmatrix * sparse_concat(PyObject *L, int id_arg) ;

static PyObject *
sparse(PyTypeObject *type, PyObject *args, PyObject *kwds)
{
  PyObject *Objx = NULL;
  static char *kwlist[] = { "x", "tc", NULL};
   char tc = 0;

  if (!PyArg_ParseTupleAndKeywords(args, kwds, "O|c:matrix", kwlist, 
	  &Objx, &tc))
    return NULL;

  if (tc && !(VALID_TC_SP(tc))) PY_ERR_TYPE("tc must be 'd' or 'z'");  
  int id = (tc ? TC2ID(tc) : -1);

  spmatrix *ret = NULL;
  /* a matrix */
  if (Matrix_Check(Objx)) {

    int m = MAT_NROWS(Objx), n = MAT_NCOLS(Objx);    
    ret = SpMatrix_NewFromMatrix((matrix *)Objx, 
	(id == -1 ? MAX(DOUBLE,MAT_ID(Objx)) : id)); 

    MAT_NROWS(Objx) = m; MAT_NCOLS(Objx) = n;    
  }

  /* sparse matrix */
  else if (SpMatrix_Check(Objx)) {

    int nnz=0, ik, jk;

    for (jk=0; jk<SP_NCOLS(Objx); jk++) {
      for (ik=SP_COL(Objx)[jk]; ik<SP_COL(Objx)[jk+1]; ik++) {
	if (((SP_ID(Objx) == DOUBLE) && (SP_VALD(Objx)[ik] != 0.0)) ||
	    ((SP_ID(Objx) == COMPLEX) && (SP_VALZ(Objx)[ik] != 0.0)))
	  nnz++;	
      }
    }

    ret = SpMatrix_New(SP_NROWS(Objx), SP_NCOLS(Objx), nnz, SP_ID(Objx));
    if (!ret) return PyErr_NoMemory();

    nnz = 0;
    for (jk=0; jk<SP_NCOLS(Objx); jk++) {
      for (ik=SP_COL(Objx)[jk]; ik<SP_COL(Objx)[jk+1]; ik++) {
	if ((SP_ID(Objx) == DOUBLE) && (SP_VALD(Objx)[ik] != 0.0)) {
	  SP_VALD(ret)[nnz] = SP_VALD(Objx)[ik];
	  SP_ROW(ret)[nnz++] = SP_ROW(Objx)[ik];
	  SP_COL(ret)[jk+1]++;	
	}
	else if ((SP_ID(Objx) == COMPLEX) && (SP_VALZ(Objx)[ik] != 0.0)) {
	  SP_VALZ(ret)[nnz] = SP_VALZ(Objx)[ik];
	  SP_ROW(ret)[nnz++] = SP_ROW(Objx)[ik];
	  SP_COL(ret)[jk+1]++;	
	}
      }
    }
    
    for (jk=0; jk<SP_NCOLS(Objx); jk++) 
      SP_COL(ret)[jk+1] += SP_COL(ret)[jk];
    
    /*
    ret = SpMatrix_NewFromSpMatrix((spmatrix *)Objx, 0, 
	(id == -1 ? SP_ID(Objx) : id));
    */
  }

  /* x is a list of lists */
  else if (PyList_Check(Objx)) 
    ret = sparse_concat(Objx, id);
  
  else PY_ERR_TYPE("invalid matrix initialization");

  return (PyObject *)ret;
}

extern PyObject * matrix_exp(matrix *, PyObject *, PyObject *) ;
extern PyObject * matrix_log(matrix *, PyObject *, PyObject *) ;
extern PyObject * matrix_sqrt(matrix *, PyObject *, PyObject *) ;
extern PyObject * matrix_cos(matrix *, PyObject *, PyObject *) ;
extern PyObject * matrix_sin(matrix *, PyObject *, PyObject *) ;
extern PyObject * matrix_elem_mul(matrix *, PyObject *, PyObject *) ;
extern PyObject * matrix_elem_div(matrix *, PyObject *, PyObject *) ;

static PyMethodDef base_functions[] = {
  {"exp", (PyCFunction)matrix_exp, METH_VARARGS|METH_KEYWORDS, 
  "Computes the element-wise expontial of a matrix"},
  {"log", (PyCFunction)matrix_log, METH_VARARGS|METH_KEYWORDS, 
  "Computes the element-wise logarithm of a matrix"},
  {"sqrt", (PyCFunction)matrix_sqrt, METH_VARARGS|METH_KEYWORDS, 
  "Computes the element-wise square-root of a matrix"},
  {"cos", (PyCFunction)matrix_cos, METH_VARARGS|METH_KEYWORDS, 
  "Computes the element-wise cosine of a matrix"},
  {"sin", (PyCFunction)matrix_sin, METH_VARARGS|METH_KEYWORDS, 
  "Computes the element-wise sine of a matrix"},
  {"axpy", (PyCFunction)base_axpy, METH_VARARGS|METH_KEYWORDS, doc_axpy},
  {"gemm", (PyCFunction)base_gemm, METH_VARARGS|METH_KEYWORDS, doc_gemm},
  {"gemv", (PyCFunction)base_gemv, METH_VARARGS|METH_KEYWORDS, doc_gemv},
  {"syrk", (PyCFunction)base_syrk, METH_VARARGS|METH_KEYWORDS, doc_syrk},
  {"symv", (PyCFunction)base_symv, METH_VARARGS|METH_KEYWORDS, doc_symv},
  {"mul", (PyCFunction)matrix_elem_mul, METH_VARARGS|METH_KEYWORDS, 
   "elementwise product of two matrices"},
  {"div", (PyCFunction)matrix_elem_div, METH_VARARGS|METH_KEYWORDS, 
   "elementwise division between two matrices"},
  {"sparse", (PyCFunction)sparse, METH_VARARGS|METH_KEYWORDS, 
   "convenience function for creating sparse matrices"},
  {NULL}		/* sentinel */
};

PyMODINIT_FUNC
initbase(void)
{  
  static void *base_API[8];
  PyObject *c_api_object;
  
  if (!(base_mod = Py_InitModule3("base", base_functions, base__doc__)))
    return;

  /* for MS VC++ compatibility */
  matrix_tp.tp_alloc = PyType_GenericAlloc;
  matrix_tp.tp_free = PyObject_Del;
  if (PyType_Ready(&matrix_tp) < 0)
    return;
  
  if (PyType_Ready(&matrix_tp) < 0)
    return;
    
  Py_INCREF(&matrix_tp);
  if (PyModule_AddObject(base_mod, "matrix", (PyObject *) &matrix_tp) < 0)
    return;   

  spmatrix_tp.tp_alloc = PyType_GenericAlloc;
  spmatrix_tp.tp_free = PyObject_Del;
  if (PyType_Ready(&spmatrix_tp) < 0)
    return;
    
  Py_INCREF(&spmatrix_tp);
  if (PyModule_AddObject(base_mod, "spmatrix", (PyObject *) &spmatrix_tp) < 0)
    return;   

  /* Create the print_options dictionary */
  PyObject *printopt = PyDict_New();
  PyObject *dstr = PyString_FromString("5.4e");
  PyObject *istr = PyString_FromString("5li");
  PyObject *zstr = PyString_FromString("5.4e");
  if (PyDict_SetItemString(printopt, "dformat", dstr) ||
      PyDict_SetItemString(printopt, "iformat", istr) ||
      PyDict_SetItemString(printopt, "zformat", zstr)) 
  {
    Py_XDECREF(printopt); 
    Py_XDECREF(dstr); Py_XDECREF(istr); Py_XDECREF(zstr);
    return;
  }
  Py_DECREF(dstr); Py_DECREF(istr); Py_DECREF(zstr);

  if (PyModule_AddObject(base_mod, "print_options", printopt) < 0)
    return;   

  One[INT].i = 1; One[DOUBLE].d = 1.0; One[COMPLEX].z = 1.0;

  MinusOne[INT].i = -1; MinusOne[DOUBLE].d = -1.0; MinusOne[COMPLEX].z = -1.0;

  Zero[INT].i = 0; Zero[DOUBLE].d = 0.0; Zero[COMPLEX].z = 0.0;

  /* initialize the C API object */
  base_API[0] = (void *)Matrix_New;
  base_API[1] = (void *)Matrix_NewFromMatrix;
  base_API[2] = (void *)Matrix_NewFromSequence;
  base_API[3] = (void *)Matrix_Check_func;
  base_API[4] = (void *)SpMatrix_New;
  base_API[5] = (void *)SpMatrix_NewFromSpMatrix;
  base_API[6] = (void *)SpMatrix_NewFromIJV;
  base_API[7] = (void *)SpMatrix_Check_func;

  /* Create a CObject containing the API pointer array's address */
  c_api_object = PyCObject_FromVoidPtr((void *)base_API, NULL);

  if (c_api_object != NULL)
    PyModule_AddObject(base_mod, "_C_API", c_api_object);
}
