"""
Solver for linear, second-order cone and semidefinite programming.
"""

# Copyright 2004-2007 J. Dahl and L. Vandenberghe.
# 
# This file is part of CVXOPT version 0.9.
#
# CVXOPT is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# CVXOPT is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import math
from cvxopt import base, blas, lapack, cholmod, misc
from cvxopt.base import matrix, spmatrix

__all__ = []
options = {}


def scale(x, W, trans = 'N', inverse = 'N'):  

    # Computes 
    #
    #     x := W*x        (trans is 'N', inverse = 'N')  
    #     x := W^T*x      (trans is 'T', inverse = 'N')  
    #     x := W^{-1}*x   (trans is 'N', inverse = 'I')  
    #     x := W^{-T}*x   (trans is 'T', inverse = 'I'). 
    #
    # x is a dense 'd' matrix.
    #
    # W is a dictionary with entries:
    #
    # - W['d']: positive vector
    # - W['di']: componentwise inverse of W['d']
    # - W['v']: lists of second order cone vectors with unit hyperbolic 
    #   norms
    # - W['beta']: list of positive numbers
    # - W['r']: list of square matrices 
    # - W['rti']: list of square matrices.  rti[k] is the inverse  
    #   transpose of r[k].


    # Scaling for 'l' component xk is xk := d .* xk; inverse scaling is 
    # xk ./ d = di .* xk, where d = W['d'], di = W['di'].

    if inverse == 'N': w = W['d']
    else: w = W['di']
    m = w.size[0]
    for k in xrange(x.size[1]):
        blas.tbmv(w, x, n = m, k = 0, ldA = 1, offsetx = k*x.size[0])


    # Scaling for 'q' component is 
    #
    #     xk := beta * (2*v*v' - J) * xk
    #         = beta * (2*v*(xk'*v)' - J*xk)
    #
    # where beta = W['beta'][k], v = W['v'][k], J = [1, 0; 0, -I].
    #
    # Inverse scaling is
    #
    #     xk := 1/beta * (2*J*v*v'*J - J) * xk
    #         = 1/beta * (-J) * (2*v*((-J*xk)'*v)' + xk). 

    w = matrix(0.0, (x.size[1], 1))
    ind = m
    for k in xrange(len(W['v'])):
        v = W['v'][k]
        m = v.size[0]
        if inverse == 'I':  
            blas.scal(-1.0, x, offset = ind, inc = x.size[0])
        blas.gemv(x, v, w, trans = 'T', m = m, n = x.size[1], offsetA = 
            ind, ldA = x.size[0])
        blas.scal(-1.0, x, offset = ind, inc = x.size[0])
        blas.ger(v, w, x, alpha = 2.0, m = m, n = x.size[1], ldA = 
            x.size[0], offsetA = ind)
        if inverse == 'I': 
            blas.scal(-1.0, x, offset = ind, inc = x.size[0])
            a = 1.0 / W['beta'][k] 
        else:
            a = W['beta'][k] 
        for i in xrange(x.size[1]):
            blas.scal(a, x, n = m, offset = ind + i*x.size[0])
        ind += m


    # Scaling for 's' component xk is
    #
    #     xk := vec( r' * mat(xk) * r )  if trans = 'N'
    #     xk := vec( r * mat(xk) * r' )  if trans = 'T'.
    #
    # r is kth element of W['r'].
    #
    # Inverse scaling is
    #
    #     xk := vec( rti * mat(xk) * rti' )  if trans = 'N'
    #     xk := vec( rti' * mat(xk) * rti )  if trans = 'T'.
    #
    # rti is kth element of W['rti'].

    maxn = max( [0] + [ r.size[0] for r in W['r'] ] )
    a = matrix(0.0, (maxn, maxn))
    for k in xrange(len(W['r'])):

        if inverse == 'N':
            r = W['r'][k]
            if trans == 'N': t = 'T'
            else: t = 'N'
        else:
            r = W['rti'][k]
            t = trans

        n = r.size[0]
        for i in xrange(x.size[1]):

            # scale diagonal of xk by 0.5
            blas.scal(0.5, x, offset = ind + i*x.size[0], inc = n+1, n = n)

            # a = r*tril(x) (t is 'N') or a = tril(x)*r  (t is 'T')
            blas.copy(r, a)
            if t == 'N':   
                blas.trmm(x, a, side = 'R', m = n, n = n, ldA = n, ldB = n,
                    offsetA = ind + i*x.size[0])
            else:    
                blas.trmm(x, a, side = 'L', m = n, n = n, ldA = n, ldB = n,
                    offsetA = ind + i*x.size[0])
 
            # x := (r*a' + a*r')  if t is 'N'
            # x := (r'*a + a'*r)  if t is 'T'
            blas.syr2k(r, a, x, trans = t, n = n, k = n, ldB = n, ldC = n,
                offsetC = ind + i*x.size[0])
 
        ind += n**2


def scale2(lmbda, x, dims, inverse = 'N'):

    # x := H(lambda^{1/2}) * x   (inverse is 'N')
    # x := H(lambda^{-1/2}) * x  (inverse is 'I')
    #
    # H is the Hessian of the logarithmic barrier.
      

    # For the 'l' block, 
    #
    #     xk := xk ./ l   (inverse is 'N')
    #     xk := xk .* l   (inverse is 'I')
    #
    # where l is lmbda[:dims['l']].

    if inverse == 'N':
        blas.tbsv(lmbda, x, n = dims['l'], k = 0, ldA = 1)
    else:
        blas.tbmv(lmbda, x, n = dims['l'], k = 0, ldA = 1)
   
  
    # For 'q' blocks, if inverse is 'N',
    #
    #     xk := 1/a * [ l'*J*xk;  
    #         xk[1:] - (xk[0] + l'*J*xk) / (l[0] + 1) * l[1:] ].
    #
    # If inverse is 'I',
    #
    #     xk := a * [ l'*xk; 
    #         xk[1:] + (xk[0] + l'*xk) / (l[0] + 1) * l[1:] ].
    #
    # a = sqrt(lambda_k' * J * lambda_k), l = lambda_k / a.

    ind = dims['l']
    for m in dims['q']:
        a = jnrm2(lmbda, n = m, offset = ind)
        if inverse == 'N':
            lx = jdot(lmbda, x, n = m, offsetx = ind, offsety = ind)/a
        else:
            lx = blas.dot(lmbda, x, n = m, offsetx = ind, offsety = ind)/a
        x0 = x[ind]
        x[ind] = lx
        c = (lx + x0) / (lmbda[ind]/a + 1) / a 
        if inverse == 'N':  c *= -1.0
        blas.axpy(lmbda, x, alpha = c, n = m-1, offsetx = ind+1, offsety =
            ind+1)
        if inverse == 'N': a = 1.0/a 
        blas.scal(a, x, offset = ind, n = m)
        ind += m
        

    # For the 's' blocks, if inverse is 'N',
    #
    #     xk := vec( diag(l)^{-1/2} * mat(xk) * diag(k)^{-1/2}).
    #
    # If inverse is 'I',
    #
    #     xk := vec( diag(l)^{1/2} * mat(xk) * diag(k)^{1/2}).
    #
    # where l is kth block of lambda.
    # 
    # We scale upper and lower triangular part of mat(xk) because the
    # inverse operation will be applied to nonsymmetric matrices.

    ind2 = ind
    for k in xrange(len(dims['s'])):
        m = dims['s'][k]
        for j in xrange(m):
            c = math.sqrt(lmbda[ind2+j]) * base.sqrt(lmbda[ind2:ind2+m])
            if inverse == 'N':  
                blas.tbsv(c, x, n = m, k = 0, ldA = 1, offsetx = ind + j*m)
            else:
                blas.tbmv(c, x, n = m, k = 0, ldA = 1, offsetx = ind + j*m)
        ind += m*m
        ind2 += m


def pack(x, y, dims, offsetx = 0, offsety = 0):

     # The vector x is an element of S, with the 's' components stored 
     # in unpacked storage.  On return, x is copied to y with the 's' 
     # components matrices stored in packed storage and the off-diagonal 
     # entries scaled by sqrt(2).

     nlq = dims['l'] + sum(dims['q'])
     np = sum([ n*(n+1)/2 for n in dims['s'] ])
     blas.copy(x, y, n = nlq, offsetx = offsetx, offsety = offsety)
     iu, ip = offsetx + nlq, offsety + nlq
     for n in dims['s']:
         for k in xrange(n):
             blas.copy(x, y, n = n-k, offsetx = iu + k*(n+1), offsety = ip)
             y[ip] /= math.sqrt(2)
             ip += n-k
         iu += n**2 
     blas.scal(math.sqrt(2.0), y, n = np, offset = offsety+nlq)
     

def unpack(x, y, dims, offsetx = 0, offsety = 0):

     # The vector x is an element of S, with the 's' components stored
     # in unpacked storage and off-diagonal entries scaled by sqrt(2).
     # On return, x is copied to y with the 's' components stored in 
     # unpacked storage and off-diagonal entries scaled by sqrt(2).

     nlq = dims['l'] + sum(dims['q'])
     nu = sum([ n**2 for n in dims['s'] ])
     blas.copy(x, y, n = nlq, offsetx = offsetx, offsety = offsety)
     iu, ip = offsety+nlq, offsetx+nlq
     for n in dims['s']:
         for k in xrange(n):
             blas.copy(x, y, n = n-k, offsetx = ip, offsety = iu+k*(n+1))
             y[iu+k*(n+1)] *= math.sqrt(2)
             ip += n-k
         iu += n**2 
     blas.scal(1.0/math.sqrt(2.0), y, n = nu, offset = offsety+nlq)


def sdot(x, y, dims):

    # Returns the inner product of two vectors in S
    
    ind = dims['l'] + sum(dims['q'])
    a = blas.dot(x, y, n = ind)
    for m in dims['s']:
        a += blas.dot(x, y, offsetx = ind, offsety = ind, incx = m+1, 
            incy = m+1, n = m)
        for j in xrange(1, m):
            a += 2.0 * blas.dot(x, y, incx = m+1, incy = m+1, 
                offsetx = ind+j, offsety = ind+j, n = m-j)
        ind += m**2
    return a


def snrm2(x, dims): 

    # Returns the norm of a vector in S

    return math.sqrt(sdot(x, x, dims))


def sgemv(A, x, y, dims, trans = 'N', alpha = 1.0, beta = 0.0, m = None, 
    n = None, offsetA = 0, offsety = 0): 

    # A is a matrix or spmatrix of size (N, n) where 
    #
    #     N = dims['l'] + sum(dims['q']) + sum( k**2 for k in dims['s'] ). 
    #
    # If trans is 'N': 
    #
    #     y := alpha*A*x + beta * y   (trans = 'N').
    #
    # x is a vector of length n.  y is a vector of length N.
    #
    # If trans is 'T':
    #
    #     y := alpha*A'*x + beta * y  (trans = 'T').
    #
    # x is a vector of length N.  y is a vector of length n.
    #
    # The 's' components in S are stored in unpacked 'L' storage.

    if m is None: m = A.size[0]
    if n is None: n = A.size[1]

    if trans == 'T' and alpha:
        ind = dims['l'] + sum(dims['q'])
        for mk in dims['s']:
            # Set upper triangular part of x to zero and scale strict 
            # lower triangular part by 2.
            for j in xrange(1, mk):  
                blas.scal(0.0, x, n = mk-j, inc = mk, offset = 
                    ind + j*(mk + 1) - 1) 
                blas.scal(2.0, x, offset = ind + mk*(j-1) + j, n = mk-j) 
            ind += mk**2

    base.gemv(A, x, y, trans = trans, alpha = alpha, beta = beta, m = m,
        n = n, offsetA = offsetA, offsety = offsety)

    if trans == 'T' and alpha:
        ind = dims['l'] + sum(dims['q'])
        for mk in dims['s']:
            # Scale strict lower triangular part of x by 0.5.
            for j in xrange(1, mk):  
                blas.scal(0.5, x, offset = ind + mk*(j-1) + j, n = mk-j) 
            ind += mk**2


def jdot(x, y, n = None, offsetx = 0, offsety = 0):

    # Returns x' * J * y, where J = [1, 0; 0, -I].

    if n is None: 
         if len(x) != len(y): raise ValueError, "x and y must have the "\
             "same length"
         n = len(x)
    return x[offsetx] * y[offsety] - blas.dot(x, y, n = n-1, 
        offsetx = offsetx + 1, offsety = offsety + 1) 


def jnrm2(x, n = None, offset = 0):

    # Returns sqrt(x' * J * x) where J = [1, 0; 0, -I], for a vector
    # x in a second order cone. 

    if n is None:  n = len(x)
    a = blas.nrm2(x, n = n-1, offset = offset+1)
    return math.sqrt(x[offset] - a) * math.sqrt(x[offset] + a)


def symm(x, n, offset = 0):

    # Fills in the upper triangular part of the symmetric matrix stored in
    # x[offset : offset+n*n] using 'L' storage.

    if n <= 1:  pass
    for i in xrange(n-1):
        blas.copy(x, x, offsetx = offset + i*(n+1) + 1, offsety = 
            offset + (i+1)*(n+1) - 1, incy = n, n = n-i-1)


def sprod(x, y, dims, diag = 'N'):   

    # The product x := (y o x).  If diag is 'D', the 's' part of y is 
    # diagonal and only the diagonal is stored.


    # For the 'l' block:  
    #
    #     yk o xk = yk .* xk.

    blas.tbmv(y, x, n = dims['l'], k = 0, ldA = 1) 


    # For 'q' blocks: 
    #
    #               [ lo   l1'  ]
    #     yk o xk = [           ] * xk
    #               [ lo   l0*I ] 
    #
    # where yk = (l0, l1).
    
    ind = dims['l']
    for m in dims['q']:
        dd = blas.dot(x, y, offsetx = ind, offsety = ind, n = m)
        blas.scal(y[ind], x, offset = ind+1, n = m-1)
        blas.axpy(y, x, alpha = x[ind], n = m-1, offsetx = ind+1, offsety 
            = ind+1)
        x[ind] = dd
        ind += m


    # For the 's' blocks:
    #
    #    yk o sk = .5 * ( Yk * mat(xk) + mat(xk) * Yk )
    # 
    # where Yk = mat(yk) if diag is 'N' and Yk = diag(yk) if diag is 'D'.

    if diag is 'N':
        maxm = max([0] + dims['s'])
        A = matrix(0.0, (maxm, maxm))

        for m in dims['s']:
            blas.copy(x, A, offsetx = ind, n = m*m)

            # Write upper triangular part of A and yk.
            for i in xrange(m-1):
                symm(A, m)
                symm(y, m, offset = ind)

            # xk = 0.5 * (A*yk + yk*A)
            blas.syr2k(A, y, x, alpha = 0.5, n = m, k = m, ldA = m,  ldB = 
                m, ldC = m, offsetB = ind, offsetC = ind)

            ind += m*m

    else:
        ind2 = ind
        for m in dims['s']:
            for j in xrange(m):
                u = 0.5 * ( y[ind2+j:ind2+m] + y[ind2+j] )
                blas.tbmv(u, x, n = m-j, k = 0, ldA = 1, offsetx = 
                    ind + j*(m+1))  
            ind += m*m
            ind2 += m


def sinv(x, y, dims):   

    # The inverse product x := (y o\ x), when the 's' components of y are 
    # diagonal.
    
    # For the 'l' block:  
    # 
    #     yk o\ xk = yk .\ xk.

    blas.tbsv(y, x, n = dims['l'], k = 0, ldA = 1)


    # For the 'q' blocks: 
    #
    #                        [ l0   -l1'              ]  
    #     yk o\ xk = 1/a^2 * [                        ] * xk
    #                        [ -l1  (a*I + l1*l1')/l0 ]
    #
    # where yk = (l0, l1) and a = l0^2 - l1'*l1.

    ind = dims['l']
    for m in dims['q']:
        aa = jnrm2(y, n = m, offset = ind)**2
        cc = x[ind]
        dd = blas.dot(y, x, offsetx = ind+1, offsety = ind+1, n = m-1)
        x[ind] = cc * y[ind] - dd
        blas.scal(aa / y[ind], x, n = m-1, offset = ind+1)
        blas.axpy(y, x, alpha = dd/y[ind] - cc, n = m-1, offsetx = ind+1, 
            offsety = ind+1)
        blas.scal(1.0/aa, x, n = m, offset = ind)
        ind += m


    # For the 's' blocks:
    #
    #     yk o\ xk =  xk ./ gamma
    #
    # where gammaij = .5 * (yk_i + yk_j).

    ind2 = ind
    for m in dims['s']:
        for j in xrange(m):
            u = 0.5 * ( y[ind2+j:ind2+m] + y[ind2+j] )
            blas.tbsv(u, x, n = m-j, k = 0, ldA = 1, offsetx = ind + 
                j*(m+1))  
        ind += m*m
        ind2 += m
        

def max_step(x, dims, sigma = None):

    # Returns min {t | x + t*e >= 0}.
    # When called with the argument sigma, also returns the eigenvalues 
    # (in sigma) and the eigenvectors (in x) of the 's' components of x.

    t = []
    ind = dims['l']
    if ind: t += [ -min(x[:ind]) ] 
    for m in dims['q']:
        if m: t += [ blas.nrm2(x, offset = ind+1, n = m-1) - x[ind] ]
        ind += m
    if sigma is None and dims['s']:  
        Q = matrix(0.0, (max(dims['s']), max(dims['s'])))
        w = matrix(0.0, (max(dims['s']),1))
    ind2 = 0
    for m in dims['s']:
        if sigma is None:
            blas.copy(x, Q, offsetx = ind, n = m**2)
            lapack.syevr(Q, w, range = 'I', il = 1, iu = 1, n = m, ldA = m)
            if m:  t += [ -w[0] ]
        else:            
            lapack.syevd(x, sigma, jobz = 'V', n = m, ldA = m, offsetA = 
                ind, offsetW = ind2)
            if m:  t += [ -sigma[ind2] ] 
        ind += m*m
        ind2 += m
    if t: return max(t)
    else: return 0.0



def conelp(c, G, h, dims, A = None, b = None, primalstart = None, 
    dualstart = None, kktsolver = None, xnewcopy = matrix, xdot = 
    blas.dot,  xaxpy = blas.axpy, xscal = blas.scal, ynewcopy = matrix, 
    ydot = blas.dot, yaxpy = blas.axpy, yscal = blas.scal):

    """
    Solves a pair of primal and dual cone programs

        minimize    c'*x              maximize    -h'*z - b'*y 
        subject to  G*x + s = h       subject to  G'*z + A'*y + c = 0
                    A*x = b                       z >= 0.
                    s >= 0

    The inequalities are with respect to a cone C defined as the Cartesian
    product of N + M + 1 cones:
    
        C = C_0 x C_1 x .... x C_N x C_{N+1} x ... x C_{N+M}.

    The first cone C_0 is the nonnegative orthant of dimension ml.
    The next N cones are second order cones of dimension mq[0], ..., 
    mq[N-1].  The second order cone of dimension m is defined as
    
        { (u0, u1) in R x R^{m-1} | u0 >= ||u1||_2 }.

    The next M cones are positive semidefinite cones of order ms[0], ...,
    ms[M-1] >= 0.  


    Input arguments (basic usage).
   
        c is a dense 'd' matrix of size (n,1), where n is the dimension of
        the primal variable x.

        dims is a dictionary with the dimensions of the components of C.  
        It has three fields.
        - dims['l'] = ml, the dimension of the nonnegative orthant C_0.
          (ml >= 0.)
        - dims['q'] = mq = [ mq[0], mq[1], ..., mq[N-1] ], a list of N 
          integers with the dimensions of the second order cones C_1, ..., 
          C_N.  (N >= 0 and mq[k] >= 1.)
        - dims['s'] = ms = [ ms[0], ms[1], ..., ms[M-1] ], a list of M  
          integers with the orders of the semidefinite cones C_{N+1}, ...,
          C_{N+M}.  (M >= 0 and ms[k] >= 0.)

        G is a dense or sparse 'd' matrix of size (K,n), where

            K = ml + mq[0] + ... + mq[N-1] + ms[0]**2 + ... + ms[M-1]**2.

        Each column of G describes a vector 

            v = ( v_0, v_1, ..., v_N, vec(v_{N+1}), ..., vec(v_{N+M}) ) 

        in V = R^ml x R^mq[0] x ... x R^mq[N-1] x S^ms[0] x ... x S^ms[M-1]
        stored as a column vector

            [ v_0; v_1; ...; v_N; vec(v_{N+1}); ...; vec(v_{N+M}) ].

        Here, if u is a symmetric matrix of order m, then vec(u) is the 
        matrix u stored in column major order as a vector of length m**2.
        We use BLAS unpacked 'L' storage, i.e., the entries in vec(u) 
        corresponding to the strictly upper triangular entries of u are 
        not referenced.

        h is a dense 'd' matrix of size (K,1), representing a vector in V,
        in the same format as the columns of G.
    
        A is a dense or sparse 'd' matrix of size (p,n).   The default
        value is a sparse 'd' matrix of size (0,n).

        b is a dense 'd' matrix of size (p,1).   The default value is a 
        dense 'd' matrix of size (0,1).

        The argument primalstart is a dictionary with keys 'x', 's'.  It
        specifies an optional primal starting point.  
        - primalstart['x'] is a dense 'd' matrix of size (n,1).   
        - primalstart['s'] is a dense 'd' matrix of size (K,1), 
          representing a vector that is strictly positive with respect 
          to the cone C.  

        The argument dualstart is a dictionary with keys 'y', 'z'.  It
        specifies an optional dual starting point.   
        - dualstart['y'] is a dense 'd' matrix of size (p,1).  
        - dualstart['z'] is a dense 'd' matrix of size (K,1), representing
          a vector that is strictly positive with respect to the cone C.
 
        The other arguments are normally not needed.  They allow one to 
        exploit certain types of structure in cone LPs, as described below.


    Output.

        conelp() returns a dictionary with keys 'status', 'x', 's', 'z', 
        'y'.

        If status is 'optimal', x, s, y, z are approximate primal and 
        dual optimal solutions.

        If status is 'primal infeasible', x = s = None, and z, y are an
        approximate proof of infeasibility: 

            h'*z + b'*y = -1,  G'*z + A'*y = 0,  z >= 0.

        If status is 'dual infeasible', z = y = None, and x, s are an
        approximate proof of dual infeasibility: 

            c'*x = -1,  G*x + s = 0,  A*x = 0,  s >= 0.

        If status is 'unknown', x, y, s, z are None.



    Exploiting problem structure.

        Three mechanisms are provided to express problem structure in 
        cone LPs.  

        First, instead of matrices, G and A are allowed to be Python 
        functions that evaluate the linear mappings G*x, A*x and their 
        adjoints.  If G is a function, the call G(x, y, alpha, beta, trans)
        should evaluate the matrix-vector products

            y := alpha * G * x + beta * y  if trans is 'N' 
            y := alpha * G' * x + beta * y  if trans is 'T'.

        The arguments x and y are required.  The other arguments have 
        default values alpha = 1.0, beta = 0.0, trans = 'N'.

        If A is a function, the call A(x, y, alpha, beta, trans) should 
        evaluate the matrix-vectors products

            y := alpha * A * x + beta * y if trans is 'N'
            y := alpha * A' * x + beta * y if trans is 'T'.

        The arguments x and y are required.  The other arguments 
        have default values alpha = 1.0, beta = 0.0, trans = 'N'.

        If G and/or A are functions, then the argument kktsolver is 
        required. 

  
        Second, the user can provide a customized routine for solving the 
        linear equations (`KKT systems')
        
            [ 0  A'  G'   ] [ x ]   [ bx ]
            [ A  0   0    ] [ y ] = [ by ]
            [ G  0  -W'*W ] [ z ]   [ bz ]

        that form the most expensive step of the algorithm.  Here W is a 
        scaling matrix, a block diagonal mapping 

           W*z = ( W0*z_0, ..., W_{N+M}*z_{N+M} ) 

        from V to V, defined as follows.  

        - For the 'l' block (W_0): 

              W_0 = diag(d), 

          with d a positive vector of length ml. 

        - For the 'q' blocks (W_{k+1}, k = 0, ..., N-1): 
                           
              W_{k+1} = beta_k * ( 2 * v_k * v_k' - J )

          where beta_k is a positive scalar, v_k is a vector in R^mq[k] 
          with v_k[0] > 0 and v_k'*J*v_k = 1, and J = [1, 0; 0, -I].

        - For the 's' blocks (W_{k+N}, k = 0, ..., M-1):

              W_k * x = vec(r_k' * mat(x) * r_k) 

          where r_k is a nonsingular matrix of order ms[k], and mat(x) is 
          the inverse of the vec operation.
 
        The optional argument kktsolver is a Python function that will be 
        called as f = kktsolver(W), where W is a dictionary that contains 
        the parameters of the scaling:

        - W['d'] is a positive 'd' matrix of size (ml, 1).
        - W['di'] is a positive 'd' matrix with the elementwise inverse of
          W['d'].
        - W['beta'] is a list [ beta_0, ..., beta_{N-1} ]
        - W['v'] is a list [ v_0, ..., v_{N-1} ] 
        - W['r'] is a list [ r_0, ..., r_{M-1} ] 
        - W['rti'] is a list [ rti_0, ..., rti_{M-1} ], with rti_k the
          inverse of the transpose of r_k.

        The call f = kktsolver(W) should return a function f for solving 
        the KKT system.  The KKT system is solved by f(x, y, z).  On 
        entry, x, y, z contain the righthand side bx, by, bz.  On exit, 
        they contain the solution, with z scaled: W*z is returned instead 
        of z.


        Finally, instead of using the default representation of the primal 
        variable x and the dual variable y as one-column 'd' matrices, 
        we can represent these variables (and the corresponding parameters
        c and b) by arbitrary Python objects (matrices, lists, 
        dictionaries, etc), provided the user supplies the functions 
        xnewcopy, xdot, xscal, xaxpy, ynewcopy, ydot, yscal, yaxpy.  

        If X is the vector space of primal variables x, then:
        - xnewcopy(u) creates a new copy of the vector u in X.
        - xdot(u, v) returns the inner product of two vectors u and v in X.
        - xscal(alpha, u) computes u := alpha*u, where alpha is a scalar 
          and u is a vector in X.
        - xaxpy(u, v, alpha = 1.0, beta = 0.0) computes v := alpha*u + v 
          for a scalar alpha and two vectors u and v in X.
        If this option is used, the argument c must be in the same format
        as x, the arguments G and A must be Python functions, and the
        argument kktsolver is required.

        If Y is the vector space of primal variables y:
        - ynewcopy(u) creates a new copy of the vector u in Y.
        - ydot(u, v) returns the inner product of two vectors u and v in Y.
        - yscal(alpha, u) computes u := alpha*u, where alpha is a scalar 
          and u is a vector in Y.
        - yaxpy(u, v, alpha = 1.0, beta = 0.0) computes v := alpha*u + v 
          for a scalar alpha and two vectors u and v in Y.
        If this option is used, the argument b must be in the same format
        as y, the argument A must be a Python function, and the argument 
        kktsolver is required.


    Control parameters.

       The following control parameters can be modified by adding an 
       entry to the dictionary options.  

       options['show_progress'] True/False (default: True)
       options['maxiters'] positive integer (default: 100)
       options['abstol'] scalar (default: 1e-7)
       options['reltol'] scalar (default: 1e-6)
       options['feastol'] scalar (default: 1e-7).
    """

    EXPON = 3
    STEP = 0.99

    try: MAXITERS = options['maxiters']
    except KeyError: MAXITERS = 100
    else:
        if type(MAXITERS) is not int or MAXITERS < 1:
           raise ValueError, "options['maxiters'] must be a positive "\
               "integer"

    try: ABSTOL = options['abstol']
    except KeyError: ABSTOL = 1e-7
    else:
        if type(ABSTOL) is not float and type(ABSTOL) is not int:
            raise ValueError, "options['abstol'] must be a scalar"

    try: RELTOL = options['reltol']
    except KeyError: RELTOL = 1e-6
    else:
        if type(RELTOL) is not float and type(RELTOL) is not int:
            raise ValueError, "options['reltol'] must be a scalar"

    try: FEASTOL = options['feastol']
    except KeyError: FEASTOL = 1e-7
    else:
        if type(FEASTOL) is not float and type(FEASTOL) is not int:
            raise ValueError, "options['feastol'] must be a scalar"

    try: show_progress = options['show_progress']
    except KeyError: show_progress = True

    if type(dims['l']) is not int or dims['l'] < 0: 
        raise TypeError, "'dims['l']' must be a nonnegative integer"
    if [ k for k in dims['q'] if type(k) is not int or k < 1 ]:
        raise TypeError, "'dims['q']' must be a list of positive integers"
    if [ k for k in dims['s'] if type(k) is not int or k < 0 ]:
        raise TypeError, "'dims['s']' must be a list of nonnegative " \
            "integers"

    try: refinement = options['refinement']
    except KeyError: refinement = True

    # Number of second-order and positive semidefinite cones.
    Nq, Ns = len(dims['q']), len(dims['s'])

    # Logarithmic degree of the product cone.
    cdeg = dims['l'] + Nq + sum(dims['s'])  

    # Dimension of the product cone, with 's' components unpacked.  
    cdim = dims['l'] + sum(dims['q']) + sum([k**2 for k in dims['s']])

    # Dimension of the product cone, with 's' components packed.  
    cdim_pckd = dims['l'] + sum(dims['q']) + sum([k*(k+1)/2 for k 
        in dims['s']])

    # Dimension of the product cone, with diagonal 's' components.
    cdim_diag = dims['l'] + sum(dims['q']) + sum(dims['s'])

    # Data for kth 'q' constraint are found in rows indq[k]:indq[k+1] of G.
    indq = [ dims['l'] ]  
    for k in dims['q']:  indq = indq + [ indq[-1] + k ] 

    # Data for kth 's' constraint are found in rows inds[k]:inds[k+1] of G.
    inds = [ indq[-1] ]
    for k in dims['s']:  inds = inds + [ inds[-1] + k**2 ] 

    if type(h) is not matrix or h.typecode != 'd' or h.size[1] != 1:
        raise TypeError, "'h' must be a 'd' matrix with 1 column" 
    if type(G) is matrix or type(G) is spmatrix:
        n = G.size[1]
        if G.typecode != 'd' or G.size[0] != cdim:
            raise TypeError, "'G' must be a 'd' matrix with %d rows " %cdim
        if h.size[0] != cdim:
            raise TypeError, "'h' must have %d rows" %cdim 
        def Gf(x, y, trans = 'N', alpha = 1.0, beta = 0.0): 
            sgemv(G, x, y, dims, trans = trans, alpha = alpha, beta = beta)
    else: 
        if kktsolver is None:
            raise ValueError, "argument 'kktsolver' must be provided if "\
                "'G' is a function"
        Gf = G

    if b is None: b = matrix(0.0, (0,1))
    if type(b) is not matrix or b.typecode != 'd' or b.size[1] != 1:
        raise TypeError, "'b' must be a 'd' matrix with 1 column" 
    if A is None: 
        if type(G) is matrix or type(G) is spmatrix:
            A = spmatrix([], [], [], (0,n))
        else:
            def A(x, y, trans = 'N', alpha = 1.0, beta = 0.0):
                if trans == 'N': pass
                else: xscal(beta, y)
    if type(A) is matrix or type(A) is spmatrix:
        p = A.size[0] 
        if A.typecode != 'd' or A.size[1] != n:
            raise TypeError, "'A' must be a 'd' matrix with %d columns " %n
        if b.size[0] != p:
            raise TypeError, "'b' must have %d rows" %p 
        def Af(x, y, trans = 'N', alpha = 1.0, beta = 0.0): 
            base.gemv(A, x, y, trans = trans, alpha = alpha, beta = beta)
    else: 
        if kktsolver is None:
            raise ValueError, "argument 'kktsolver' must be provided if "\
                "'A' is a function"
        Af = A

    def xcopy(x, y): 
        xscal(0.0, y) 
        xaxpy(x, y)

    def ycopy(x, y): 
        yscal(0.0, y) 
        yaxpy(x, y)

    if kktsolver is None: 
        if dims['q'] or dims['s']: 
            kktsolver = 'qr'
        else: 
            kktsolver = 'chol'
    if kktsolver in ('qr', 'ldl', 'ldl2', 'chol'):
        if type(A) is not matrix and type(A) is not spmatrix:
            raise TypeError, "A must be a matrix or spmatrix if " \
                "kktsolver is '" + kktsolver + "'"
        if type(G) is not matrix and type(G) is not spmatrix:
            raise TypeError, "G must be a matrix or spmatrix if " \
                "kktsolver is '" + kktsolver + "'"
        if p > n or cdim_pckd + p < n:
            raise ValueError, "Rank(A) < p or Rank([G; A]) < n"

    if kktsolver == 'qr':

        # The default kktsolver, except for LPs.
        #
        # Two QR factorizations
        #
        #     A' = [Q1, Q2] * [R1; 0],   W^{-T} * G * Q1 = Q3*R3
        # 
        # (with columns of W^{-T}*G in packed storage).

        # A' = [Q1, Q2] * [R1; 0]
        if type(A) is matrix:
            QA = +A.T
        else:
            QA = matrix(A.T)
        tauA = matrix(0.0, (p,1))
        lapack.geqrf(QA, tauA)

        Gs = matrix(0.0, (cdim, n))
        tauG = matrix(0.0, (n-p,1))
        g = matrix(0.0, (cdim, 1))
        u = matrix(0.0, (cdim_pckd, 1))
        vv = matrix(0.0, (n,1))
        w = matrix(0.0, (cdim_pckd, 1))

        def kktsolver(W):

            # Gs = W^{-T}*G, in packed storage.
            Gs[:,:] = G
            scale(Gs, W, trans = 'T', inverse = 'I')
            for k in xrange(n):
                g[:] = Gs[:, k]
                pack(g, Gs, dims, offsety = k*Gs.size[0])
 
            # Gs := [ Gs1, Gs2 ] 
            #     = Gs * [ Q1, Q2 ]
            lapack.ormqr(QA, tauA, Gs, side = 'R', m = cdim_pckd)

            # QR factorization Gs2 := [ Q3, Q4 ] * [ R3; 0 ] 
            lapack.geqrf(Gs, tauG, n = n-p, m = cdim_pckd, offsetA = 
                Gs.size[0]*p)

            def solve_kkt(x, y, z):

                # On entry, x, y, z contain bx, by, bz.  On exit, they 
                # contain the solution x, y, W*z of
                #
                #     [ 0         A'  G'*W^{-1} ]   [ x   ]   [bx       ]
                #     [ A         0   0         ] * [ y   ] = [by       ].
                #     [ W^{-T}*G  0   -I        ]   [ W*z ]   [W^{-T}*bz]
                #
                # The system is solved in five steps:
                #
                #       w := W^{-T}*bz - Gs1*R1^{-T}*by 
                #       u := R3^{-T}*Q2'*bx + Q3'*w
                #     W*z := Q3*u - w
                #       y := R1^{-1} * (Q1'*bx - Gs1'*(W*z))
                #       x := [ Q1, Q2 ] * [ R1^{-T}*by;  R3^{-1}*u ]

                # w := W^{-T} * bz in packed storage 
                scale(z, W, trans = 'T', inverse = 'I')
                pack(z, w, dims)

                # vv := [ Q1'*bx;  R3^{-T}*Q2'*bx ]
                blas.copy(x, vv)
                lapack.ormqr(QA, tauA, vv, trans='T') 
                lapack.trtrs(Gs, vv, uplo = 'U', trans = 'T', n = n-p,
                    offsetA = Gs.size[0]*p, offsetB = p)

                # x[:p] := R1^{-T} * by 
                blas.copy(y, x)
                lapack.trtrs(QA, x, uplo = 'U', trans = 'T', n = p)

                # w := w - Gs1 * x[:p] 
                #    = W^{-T}*bz - Gs1*by 
                blas.gemv(Gs, x, w, alpha = -1.0, beta = 1.0, n = p,
                    m = cdim_pckd)

                # u := [ Q3'*w + v[p:];  0 ]
                #    = [ Q3'*w + R3^{-T}*Q2'*bx; 0 ]
                blas.copy(w, u)
                lapack.ormqr(Gs, tauG, u, trans = 'T', k = n-p, offsetA = 
                    Gs.size[0]*p, m = cdim_pckd)
                blas.axpy(vv, u, offsetx = p, n = n-p)
                blas.scal(0.0, u, offset = n-p)

                # x[p:] := R3^{-1} * u[:n-p]  
                blas.copy(u, x, offsety = p, n = n-p)
                lapack.trtrs(Gs, x, uplo='U', n = n-p, offsetA = 
                    Gs.size[0]*p, offsetB = p)

                # x is now [ R1^{-T}*by;  R3^{-1}*u[:n-p] ]
                # x := [Q1 Q2]*x
                lapack.ormqr(QA, tauA, x) 
 
                # u := [Q3, Q4] * u - w 
                #    = Q3 * u[:n-p] - w
                lapack.ormqr(Gs, tauG, u, k = n-p, m = cdim_pckd,
                    offsetA = Gs.size[0]*p)
                blas.axpy(w, u, alpha = -1.0)  

                # y := R1^{-1} * ( v[:p] - Gs1'*u )
                #    = R1^{-1} * ( Q1'*bx - Gs1'*u )
                blas.copy(vv, y, n = p)
                blas.gemv(Gs, u, y, m = cdim_pckd, n = p, trans = 'T', 
                    alpha = -1.0, beta = 1.0)
                lapack.trtrs(QA, y, uplo = 'U', n=p) 

                unpack(u, z, dims)

            return solve_kkt


    elif kktsolver == 'ldl':   

        # LDL factorization of 
        #
        #         [ 0         A'   G'*W^{-1} ]
        #     K = [ A         0    0         ].
        #         [ W^{-T}*G  0   -I         ]

        ldK = n + p + cdim_pckd 
        K = matrix(0.0, (ldK, ldK))
        ipiv = matrix(0, (ldK, 1))
        u = matrix(0.0, (ldK, 1))
        g = matrix(0.0, (G.size[0], 1))

        def kktsolver(W):

            blas.scal(0.0, K)
            K[(ldK+1)*(p+n) :: ldK+1]  = -1.0
            K[n:n+p, :n] = A
            for k in xrange(n):
                g[:] = G[:,k]
                scale(g, W, trans = 'T', inverse = 'I')
                pack(g, K, dims, offsety = k*ldK+n+p)
            lapack.sytrf(K, ipiv)

            def solve_kkt(x, y, z):

                # Solve
                #
                #         [ x   ]   [ bx        ]
                #     K * [ y   ] = [ by        ]
                #         [ W*z ]   [ W^{-T}*bz ] 
                #
                # and return x, y, W*z.
                #
                # On entry, x, y, z contain bx, by, bz.  On exit, they 
                # contain the solution.

                blas.copy(x, u)
                blas.copy(y, u, offsety=n)
                scale(z, W, trans='T', inverse='I') 
                pack(z, u, dims, offsety = n+p)
                lapack.sytrs(K, ipiv, u)
                blas.copy(u, x, n=n)
                blas.copy(u, y, offsetx = n, n = p)
                unpack(u, z, dims, offsetx = n+p)
	    
            return solve_kkt


    elif kktsolver == 'ldl2':

        # LDL or Cholesky factorization of 
        #
        #         [ G' * W^{-1} * W^{-T} * G   A' ]
        #     K = [                               ]
        #         [ A                          0  ].

        ldK = n + p 
        K = matrix(0.0, (ldK, ldK))
        if p: ipiv = matrix(0, (ldK, 1))
        g = matrix(0.0, (G.size[0], 1))
        u = matrix(0.0, (ldK, 1))

        def kktsolver(W):

            blas.scal(0.0, K)
            K[n:,:n] = A
            for k in xrange(n):
                g[:] = G[:,k]
                scale(g, W, trans = 'T', inverse = 'I')
                scale(g, W, inverse = 'I')
                sgemv(G, g, K, dims, trans = 'T', beta = 1.0, n = n-k, 
                    offsetA = G.size[0]*k, offsety = (ldK + 1)*k)
            if p: lapack.sytrf(K, ipiv)
            else: lapack.potrf(K)

            def solve_kkt(x, y, z):

                # Solve
                #
                #         [ x   ]   [ bx + G' * W^{-1} * W^{-T} * bz ]
                #     K * [     ] = [                                ] 
                #         [ y   ]   [ by                             ]
                #
                # and return x, y, W*z = W^{-T} * (G*x - bz).

                blas.copy(z, g)
                scale(g, W, trans = 'T', inverse = 'I')
                scale(g, W, inverse = 'I')
                sgemv(G, g, u, dims, trans = 'T')
                blas.axpy(x, u)
                blas.copy(y, u, offsety = n)
                if p: lapack.sytrs(K, ipiv, u)
                else: lapack.potrs(K, u)
                blas.copy(u, x, n = n)
                blas.copy(u, y, offsetx = n, n = p)
                sgemv(G, x, z, dims, alpha = 1.0, beta = -1.0)
                scale(z, W, trans = 'T', inverse = 'I')
	    
            return solve_kkt

    elif kktsolver == 'chol' and (dims['q'] or dims['s']):

        # Dense Cholesky factorizations of 
        #
        #     S = G' * W^{-1} * W^{-T} * G  +  A' * A 
        #     K = A * S^{-1} * A'.

        S, K = matrix(0.0, (n,n)), matrix(0.0, (p,p))
        g = matrix(0.0, (G.size[0], 1))

        def kktsolver(W):

            # S = G' * W^{-1} * W^{-T} * G  +  A' * A 
            blas.scal(0.0, S)
            for k in xrange(n):
                g[:] = G[:,k]
                scale(g, W, trans = 'T', inverse = 'I')
                scale(g, W, inverse = 'I')
                sgemv(G, g, S, dims, trans = 'T', beta = 1.0, n = n-k, 
                    offsetA = G.size[0]*k, offsety = (n+1)*k)
            base.syrk(A, S, trans='T', beta=1.0)
            lapack.potrf(S) 

            # Asct := L^{-1}*A'.  Factor K = Asct'*Asct.
            if type(A) is matrix:
                Asct = A.T
            else:
                Asct = matrix(A.T)
            blas.trsm(S, Asct)
            blas.syrk(Asct, K, trans='T')
            lapack.potrf(K)

            def solve_kkt(x, y, z):
        
                # Solve for y, x:
                #
                #     K * y = A * S^{-1} * ( bx + G'*W^{-1}*W^{-T}*bz + 
                #         A'*by ) - by
                #     S*x = bx + G'*W^{-1}*W^{-T}*bz + A'*by - A'*y.
                #     
                #     Wz = W^{-T} * ( G*x - bz ).

                # x := L^{-1} * ( bx + G'*W^{-1}*W^{-T}*bz + A' * by ).
                blas.copy(z, g)
                scale(g, W, trans = 'T', inverse = 'I')
                scale(g, W, inverse = 'I')
                sgemv(G, g, x, dims, beta = 1.0, trans = 'T')
                base.gemv(A, y, x, trans='T', beta=1.0)
                blas.trsv(S, x)

                # y := K^{-1} * (A * L^{-T} * x - by)
                base.gemv(Asct, x, y, trans = 'T', beta = -1.0)
                lapack.potrs(K, y)

                # x := L^{-T} * (x - Asc'*y)
                base.gemv(Asct, y, x, alpha = -1.0, beta = 1.0)
                blas.trsv(S, x, trans='T')

                #     Wz = W^{-T} * ( G*x - bz ).
                sgemv(G, x, z, dims, alpha = 1.0, beta = -1.0)
                scale(z, W, trans = 'T', inverse = 'I')

            return solve_kkt 


    elif kktsolver == 'chol' and not dims['q'] and not dims['s']:

        # This is the default kktsolver for LPs.  It exploits sparsity to 
        # some extent.
        #
        # Factor
        #
        #     S = G' * W^{-2} * G  where W = diag( W['di'] )^{-1} 
        #     K = A * S^{-1} * A',
        #
        # using dense (L*L') or sparse (P'*L*L'*P) Cholesky factorizations.
        # If S turns out to be singular in the first factorization, then 
        # switch to factoring 
        # 
        #     S = G' * W^{-2} * G  +  A' * A 
        #     K = A * S^{-1} * A'.

        F = {'firstcall': True, 'singular': False}
        if type(G) is matrix: 
            Gs = matrix(0.0, G.size) 
            F['S'] = matrix(0.0, (n,n))
            K = matrix(0.0, (p,p))
        else:
            Gs = spmatrix(0.0, G.I, G.J, G.size) 
            F['S'] = spmatrix([], [], [], (n,n), 'd')
            F['Sf'] = None
            if type(A) is matrix:
                K = matrix(0.0, (p,p))
            else:
                K = spmatrix([], [], [], (p,p), 'd')
        m = dims['l']

        def kktsolver(W):

            # Gs = W^{-1} * G
            base.gemm( spmatrix(W['di'], range(m), range(m)), G, Gs, 
                partial = True)

            if F['firstcall']:
                F['firstcall'] = False
                base.syrk(Gs, F['S'], trans = 'T') 
                try:
                    if type(F['S']) is matrix: 
                        lapack.potrf(F['S']) 
                    else:
                        F['Sf'] = cholmod.symbolic(F['S'])
                        cholmod.numeric(F['S'], F['Sf'])
                except ArithmeticError:
                    F['singular'] = True 
                    if type(A) is matrix and type(F['S']) is spmatrix:
                        F['S'] = matrix(0.0, (n,n))
                    base.syrk(Gs, F['S'], trans = 'T') 
                    base.syrk(A, F['S'], trans = 'T', beta = 1.0) 
                    if type(F['S']) is matrix: 
                        lapack.potrf(F['S']) 
                    else:
                        F['Sf'] = cholmod.symbolic(F['S'])
                        cholmod.numeric(F['S'], F['Sf'])

            else:
                base.syrk(Gs, F['S'], trans = 'T', partial = True)
                if F['singular']:
                    base.syrk(A, F['S'], trans = 'T', beta = 1.0, partial 
                        = True) 
                if type(F['S']) is matrix: 
                    lapack.potrf(F['S']) 
                else:
                    cholmod.numeric(F['S'], F['Sf'])

            if type(F['S']) is matrix: 
                # Asct := L^{-1}*A'.  Factor K = Asct'*Asct.
                if type(A) is matrix: 
                    Asct = A.T
                else: 
                    Asct = matrix(A.T)
                blas.trsm(F['S'], Asct)
                blas.syrk(Asct, K, trans='T')
                lapack.potrf(K)

            else:
                # Asct := L^{-1}*P*A'.  Factor K = Asct'*Asct.
                if type(A) is matrix:
                    Asct = A.T
                    cholmod.solve(F['Sf'], Asct, sys = 7)
                    cholmod.solve(F['Sf'], Asct, sys = 4)
                    blas.syrk(Asct, K, trans = 'T')
                    lapack.potrf(K) 
                else:
                    Asct = cholmod.spsolve(F['Sf'], A.T, sys = 7)
                    Asct = cholmod.spsolve(F['Sf'], Asct, sys = 4)
                    base.syrk(Asct, K, trans = 'T')
                    Kf = cholmod.symbolic(K)
                    cholmod.numeric(K, Kf)

            def solve_kkt(x, y, z):

                # If not F['singular']:
                #
                #     K*y = A * S^{-1} * ( bx + G'*W^{-2}*bz ) - by
                #     S*x = bx + G'*W^{-2}*bz - A'*y
                #     W*z = W^{-1} * ( G*x - bz ).
                #    
                # If F['singular']:
                #
                #     K*y = A * S^{-1} * ( bx + G'*W^{-1}*W^{-T}*bz + 
                #         A'*by ) - by
                #     S*x = bx + G'*W^{-1}*W^{-T}*bz + A'*by - A'*y.
                #     W*z = W^{-T} * ( G*x - bz ).


                # z := W^{-1} * z = W^{-1} * bz
                blas.tbmv(W['di'], z, n = m, k = 0, ldA = 1)

                # If not F['singular']:
                #     x := L^{-1} * P * (x + Gs'*z)
                #        = L^{-1} * P * (x + G'*W^{-2}*bz)
                #
                # If F['singular']:
                #     x := L^{-1} * P * (x + Gs'*z + A'*y))
                #        = L^{-1} * P * (x + G'*W^{-2}*bz + A'*y)

                base.gemv(Gs, z, x, trans = 'T', beta = 1.0)
                if F['singular']:
                    base.gemv(A, y, x, trans = 'T', beta = 1.0)
                if type(F['S']) is matrix:
                    blas.trsv(F['S'], x)
                else:
                    cholmod.solve(F['Sf'], x, sys = 7)
                    cholmod.solve(F['Sf'], x, sys = 4)

                # y := K^{-1} * (Asc*x - y)
                #    = K^{-1} * (A * S^{-1} * (bx + G'*W^{-2}*bz) - by)  
                #      (if not F['singular'])
                #    = K^{-1} * (A * S^{-1} * (bx + G'*W^{-2}*bz + A'*by) 
                #      - by)  
                #      (if F['singular']).

                base.gemv(Asct, x, y, trans = 'T', beta = -1.0)
                if type(K) is matrix:
                    lapack.potrs(K, y)
                else:
                    cholmod.solve(Kf, y)

                # x := P' * L^{-T} * (x - Asc'*y)
                #    = S^{-1} * (bx + G'*W^{-2}*bz - A'*y) 
                #      (if not F['singular'])  
                #    = S^{-1} * (bx + G'*W^{-2}*bz + A'*by - A'*y) 
                #      (if F['singular'])

                base.gemv(Asct, y, x, alpha = -1.0, beta = 1.0)
                if type(F['S']) is matrix:
                    blas.trsv(F['S'], x, trans='T')
                else:
                    cholmod.solve(F['Sf'], x, sys = 5)
                    cholmod.solve(F['Sf'], x, sys = 8)

                # W*z := Gs*x - z = W^{-1} * (G*x - bz)
                base.gemv(Gs, x, z, beta = -1.0)

            return solve_kkt

    else: 
        # User provided kktsolver.

        pass

    x = xnewcopy(c);  xscal(0.0, x)
    y = ynewcopy(b);  yscal(0.0, y)
    s, z = matrix(0.0, (cdim,1)), matrix(0.0, (cdim,1))

    resx, hresx = xnewcopy(c), xnewcopy(c)
    resy, hresy = ynewcopy(b), ynewcopy(b)
    resz, hresz = matrix(0.0, (cdim,1)), matrix(0.0, (cdim,1))
    dx, dy = xnewcopy(c), ynewcopy(b)
    ds, dz = matrix(0.0, (cdim,1)), matrix(0.0, (cdim,1))
    dx1, dy1 = xnewcopy(c), ynewcopy(b)
    ds1, dz1 = matrix(0.0, (cdim,1)), matrix(0.0, (cdim,1))
    dkappa, dtau = matrix(0.0, (1,1)), matrix(0.0, (1,1))
    dx2, dy2 = xnewcopy(c), ynewcopy(b)
    ds2, dz2 = matrix(0.0, (cdim,1)), matrix(0.0, (cdim,1))
    dkappa2, dtau2 = matrix(0.0, (1,1)), matrix(0.0, (1,1))
    th = matrix(0.0, (cdim,1))
    sigs = matrix(0.0, (sum(dims['s']), 1))
    sigz = matrix(0.0, (sum(dims['s']), 1))
    lmbda = matrix(0.0, (cdim_diag + 1, 1))
    lmbdasq = matrix(0.0, (cdim_diag + 1, 1))
    work = matrix(0.0, (max( [0] + dims['s'] )**2, 1))


    # Select initial points.

    if primalstart is None or dualstart is None:

        # Factor
        #
        #     [ 0   A'  G' ] 
        #     [ A   0   0  ].
        #     [ G   0  -I  ]
    
        W = {}
        W['d'] = matrix(1.0, (dims['l'], 1)) 
        W['di'] = matrix(1.0, (dims['l'], 1)) 
        W['v'] = [ matrix(0.0, (m,1)) for m in dims['q'] ]
        W['beta'] = Nq * [ 1.0 ] 
        for v in W['v']: v[0] = 1.0
        W['r'] = [ matrix(0.0, (m,m)) for m in dims['s'] ]
        W['rti'] = [ matrix(0.0, (m,m)) for m in dims['s'] ]
        for r in W['r']: r[::r.size[0]+1 ] = 1.0
        for rti in W['rti']: rti[::rti.size[0]+1 ] = 1.0
        try: f = kktsolver(W)
        except ArithmeticError:  
            raise ValueError, "Rank(A) < p or Rank([G; A]) < n"

    if primalstart is None:

	# minimize    || G * x - h ||^2
	# subject to  A * x = b
	#
	# by solving
	#
	#     [ 0   A'  G' ]   [ x  ]   [ 0 ]
	#     [ A   0   0  ] * [ dy ] = [ b ].
	#     [ G   0  -I  ]   [ -s ]   [ h ]

        xscal(0.0, x)
        ycopy(b, dy)  
        blas.copy(h, s)
        try: f(x, dy, s) 
        except ArithmeticError:  
            raise ValueError, "Rank(A) < p or Rank([G; A]) < n"
        blas.scal(-1.0, s)  

    else:
        xcopy(primalstart['x'], x)
        blas.copy(primalstart['s'], s)

    # ts = min{ t | s + t*e >= 0 }
    ts = max_step(s, dims)
    if ts >= 0 and primalstart: 
        raise ValueError, "initial s is not positive"


    if dualstart is None:

        # minimize   || z ||^2
        # subject to G'*z + A'*y + c = 0
        #
        # by solving
        #
        #     [ 0   A'  G' ] [ dx ]   [ -c ]
        #     [ A   0   0  ] [ y  ] = [  0 ].
        #     [ G   0  -I  ] [ z  ]   [  0 ]

        xcopy(c, dx); 
        xscal(-1.0, dx)
        yscal(0.0, y)
        blas.scal(0.0, z)
        try: f(dx, y, z)
        except ArithmeticError:  
            raise ValueError, "Rank(A) < p or Rank([G; A]) < n"

    else:
        if 'y' in dualstart: ycopy(dualstart['y'], y)
        blas.copy(dualstart['z'], z)

    # tz = min{ t | z + t*e >= 0 }
    tz = max_step(z, dims)
    if tz >= 0 and dualstart: 
        raise ValueError, "initial z is not positive"

    trz = sum(z[:dims['l']]) + sum(z[indq[:-1]]) + sum([ 
        sum(z[inds[k] : inds[k+1] : dims['s'][k]+1]) for k in 
        xrange(len(dims['s'])) ])
    trs = sum(s[:dims['l']]) + sum(s[indq[:-1]]) + sum([ 
        sum(s[inds[k] : inds[k+1] : dims['s'][k]+1]) for k in 
        xrange(len(dims['s'])) ])
    nrms = snrm2(s, dims)
    nrmz = snrm2(z, dims)

    if primalstart is None and dualstart is None: 

        gap = sdot(s, z, dims) 
        pcost = xdot(c,x)
        dcost = -ydot(b,y) - sdot(h,z,dims) 

        if ts <= 0 and tz <= 0 and (gap <= ABSTOL or ( pcost < 0.0 and 
            gap / -pcost <= RELTOL ) or (dcost > 0.0 and gap / dcost 
            <= RELTOL)):

            # The initial points we constructed happen to be feasible and 
            # optimal.  
            ind = dims['l'] + sum(dims['q'])
            for m in dims['s']:
                symm(s, m, ind)
                symm(z, m, ind)
                ind += m**2
            return {'status': 'optimal', 'x': x, 'y': y, 's': s, 'z': z}

        if ts >= -1e-8 * max(nrms, 1.0):  
            a = 1.0 + ts  
            s[:dims['l']] += a
            s[indq[:-1]] += a
            ind = dims['l'] + sum(dims['q'])
            for m in dims['s']:
                s[ind : ind+m*m : m+1] += a
                ind += m**2

        if tz >= -1e-8 * max(nrmz, 1.0):
            a = 1.0 + tz  
            z[:dims['l']] += a
            z[indq[:-1]] += a
            ind = dims['l'] + sum(dims['q'])
            for m in dims['s']:
                z[ind : ind+m*m : m+1] += a
                ind += m**2

    elif primalstart is None and dualstart is not None:

        if ts >= -1e-8 * max(nrms, 1.0):  
            a = 1.0 + ts  
            s[:dims['l']] += a
            s[indq[:-1]] += a
            ind = dims['l'] + sum(dims['q'])
            for m in dims['s']:
                s[ind : ind+m*m : m+1] += a
                ind += m**2

    elif primalstart is not None and dualstart is None:

        if tz >= -1e-8 * max(nrmz, 1.0):
            a = 1.0 + tz  
            z[:dims['l']] += a
            z[indq[:-1]] += a
            ind = dims['l'] + sum(dims['q'])
            for m in dims['s']:
                z[ind : ind+m*m : m+1] += a
                ind += m**2


    if 0:  # random starting points for debugging

        from cvxopt import random
        n = len(c)
        x = random.normal(n,1)
        y = random.normal(p,1)
	s, z = matrix(0.0, (cdim,1)), matrix(0.0, (cdim,1))
	s[:dims['l']] = random.uniform(dims['l'], 1) 
	z[:dims['l']] = random.uniform(dims['l'], 1) 
        for k in xrange(Nq):
            mk = dims['q'][k]
            sk = random.normal(mk, 1)
            sk[0] = blas.nrm2(sk[1:]) + random.uniform(1)
            s[indq[k]:indq[k+1]] = sk
            zk = random.normal(mk, 1)
            zk[0] = blas.nrm2(zk[1:]) + random.uniform(1)
            z[indq[k]:indq[k+1]] = zk
        for k in xrange(Ns):
            mk = dims['s'][k]
            sk = random.normal(mk, mk)
            sk = sk*sk.T
            s[inds[k]:inds[k+1]] = sk[:]
            zk = random.normal(mk, mk)
            zk = zk*zk.T
            z[inds[k]:inds[k+1]] = zk[:]

    if 0:  # starting point = e for debugging

        from cvxopt import random
        n = len(c)
        x = matrix(0.0, (n,1))
        y = matrix(0.0, (p,1))
	s, z = matrix(0.0, (cdim,1)), matrix(0.0, (cdim,1))
	s[:dims['l']] = 1.0
        s[indq[:-1]] = 1.0
        ind = dims['l'] + sum(dims['q'])
        for m in dims['s']:
            s[ind : ind+m*m : m+1] = 1.0
            ind += m**2
	z[:dims['l']] = 1.0
        z[indq[:-1]] = 1.0
        ind = dims['l'] + sum(dims['q'])
        for m in dims['s']:
            z[ind : ind+m*m : m+1] = 1.0
            ind += m**2

    tau, kappa = 1.0, 1.0
    gap = sdot(s, z, dims) 

    for iters in xrange(MAXITERS):

        # hresx = -A'(y) - G'(z) 
        Af(y, hresx, alpha = -1.0, trans = 'T') 
        Gf(z, hresx, alpha = -1.0, beta = 1.0, trans = 'T') 

        # resx = hresx - c*tau 
        #      = -A'(y) - G'(z) - c*tau
        xcopy(hresx, resx)
        xaxpy(c, resx, alpha = -tau)

        # hresy = A(x)  
        Af(x, hresy)

        # resy = hresy - b*tau 
        #      = A(x) - b*tau
        ycopy(hresy, resy)
        yaxpy(b, resy, alpha = -tau)

        # hresz = s + G(x)  
        Gf(x, hresz)
        blas.axpy(s, hresz)

        # resz = hresz - h*tau 
        #      = s + G(x) - h*tau
        blas.scal(0, resz)
        blas.axpy(hresz, resz)
        blas.axpy(h, resz, alpha = -tau)

        # rest = kappa + <c,x> + <b,y> + <h,z> 
        cx, by, hz = xdot(c,x), ydot(b,y), sdot(h, z, dims) 
        rest = kappa + cx + by + hz 

        # stopping criteria
        pcost, dcost = cx/tau, -(by + hz) / tau        
        nrmhresx = math.sqrt( xdot(hresx, hresx) ) 
        nrmresx = math.sqrt( xdot(resx, resx) ) / tau
        nrmhresy = math.sqrt( ydot(hresy, hresy) )
        nrmresy = math.sqrt( ydot(resy, resy) ) / tau
        nrmhresz = snrm2(hresz, dims) 
        nrmresz = snrm2(resz, dims) / tau 
        if pcost < 0.0:
            relgap = gap / -pcost
        elif dcost > 0.0:
            relgap = gap / dcost
        else: 
            relgap = None

        if iters == 0: 
            nrmresx0 = max(1.0, math.sqrt(xdot(c,c)))
            nrmresy0 = max(1.0, math.sqrt(ydot(b,b)))
            nrmresz0 = max(1.0, snrm2(h, dims))

        if show_progress:
            if iters==0:
                print "% 10s% 12s% 10s% 8s% 7s % 5s" %("pcost", "dcost",
                    "gap", "pres", "dres", "k/t")
            print "%2d: % 8.4e % 8.4e % 4.0e% 7.0e% 7.0e% 7.0e" \
                %(iters, pcost, dcost, gap, max(nrmresz/nrmresz0,
                nrmresy/nrmresy0), nrmresx/nrmresx0, kappa/tau)

        if max(nrmresz/nrmresz0, nrmresy/nrmresy0) <= FEASTOL and \
            nrmresx/nrmresx0 <= FEASTOL and ( gap <= ABSTOL or 
            (relgap is not None and relgap <= RELTOL) ):
            xscal(1.0/tau, x)
            yscal(1.0/tau, y)
            blas.scal(1.0/tau, s)
            blas.scal(1.0/tau, z)
            ind = dims['l'] + sum(dims['q'])
            for m in dims['s']:
                symm(s, m, ind)
                symm(z, m, ind)
                ind += m**2
            return {'status': 'optimal', 'x': x, 'y': y, 's': s, 'z': z}

        elif hz + by < 0.0 and nrmhresx/nrmresx0 / (-hz - by) <= FEASTOL:
            yscal(1.0/(-hz - by), y)
            blas.scal(1.0/(-hz - by), z)
            ind = dims['l'] + sum(dims['q'])
            for m in dims['s']:
                symm(z, m, ind)
                ind += m**2
            return {'status': 'primal infeasible', 'x': None, 's': None, 
                'y': y, 'z': z }

        elif cx < 0.0 and max(nrmhresy/nrmresy0, nrmhresz/nrmresz0) \
            / (-cx) <= FEASTOL:
            xscal(1.0/(-cx), x)
            blas.scal(1.0/(-cx), s)
            ind = dims['l'] + sum(dims['q'])
            for m in dims['s']:
                symm(s, m, ind)
                ind += m**2
            return {'status': 'dual infeasible', 'x': x, 's': s, 'y': None,
                'z': None}


        if iters == 0:

            # Compute initial scaling W:
            # 
            #     W * z = W^{-T} * s = lambda
            #     dg * tau = 1/dg * kappa = lambdag.

            W = {}


            # For kappa, tau block: 
            #
            #     dg = sqrt( kappa / tau )
            #     dgi = sqrt( tau / kappa )
            #     lambda_g = sqrt( tau * kappa )  
            # 
            # lambda_g is stored in the last position of lmbda.
    
            dg = math.sqrt( kappa / tau )
            dgi = math.sqrt( tau / kappa )
            lmbda[-1] = math.sqrt( tau * kappa )


            # For the 'l' block: 
            #
            #     W['d'] = sqrt( sk ./ zk )
            #     W['di'] = sqrt( zk ./ sk )
            #     lambdak = sqrt( sk .* zk )
            #
            # where sk and zk are the first dims['l'] entries of s and z.
            # lambda_k is stored in the first dims['l'] positions of lmbda.
             
            m = dims['l']
            W['d'] = base.sqrt( base.div( s[:m], z[:m] ))
            W['di'] = W['d']**-1
            lmbda[:m] = base.sqrt( base.mul( s[:m], z[:m] ) ) 


            # For the 'q' blocks, compute lists 'v', 'beta' of length Nq. 
            #
            # The vector v[k] has unit hyperbolic norm: 
            # 
            #     (sqrt( v[k]' * J * v[k] ) = 1 with J = [1, 0; 0, -I]).
            # 
            # beta[k] is a positive scalar.
            #
            # The hyperbolic Householder matrix H = 2*v[k]*v[k]' - J
            # defined by v[k] satisfies 
            # 
            #     (beta[k] * H) * zk  = (beta[k] * H) \ sk = lambda_k
            #
            # where sk = s[indq[k]:indq[k+1]], zk = z[indq[k]:indq[k+1]].
            #
            # lambda_k is stored in lmbda[indq[k]:indq[k+1]].
           
            ind = dims['l']
            W['v'] = [ matrix(0.0, (k,1)) for k in dims['q'] ]
            W['beta'] = Nq * [ 0.0 ] 

            for k in xrange(Nq):
                m = dims['q'][k]
                v = W['v'][k]

                # a = sqrt( sk' * J * sk )  where J = [1, 0; 0, -I]
                aa = jnrm2(s, offset = ind, n = m)

                # b = sqrt( zk' * J * zk )
                bb = jnrm2(z, offset = ind, n = m) 

                # beta[k] = ( a / b )**1/2
                W['beta'][k] = math.sqrt( aa / bb )

                # c = sqrt( (sk/a)' * (zk/b) + 1 ) / sqrt(2)    
                cc = math.sqrt( ( blas.dot(s, z, n = m, offsetx = ind, 
                    offsety = ind) / aa / bb + 1.0 ) / 2.0 )

                # vk = 1/(2*c) * ( (sk/a) + J * (zk/b) )
                blas.copy(z, v, offsetx = ind, n = m)
                blas.scal(-1.0/bb, v)
                v[0] *= -1.0 
                blas.axpy(s, v, 1.0/aa, offsetx = ind, n = m)
                blas.scal(1.0/2.0/cc, v)

                # v[k] = 1/sqrt(2*(vk0 + 1)) * ( vk + e ),  e = [1; 0]
                v[0] += 1.0
                blas.scal(1.0/math.sqrt(2.0 * v[0]), v)
            

                # To get the scaled variable lambda_k
                # 
                #     d =  sk0/a + zk0/b + 2*c
                #     lambda_k = [ c;  (c + zk0/b)/d * sk1/a + 
                #         (c + sk0/a)/d * zk1/b ]
                #     lambda_k *= sqrt(a * b)

                lmbda[ind] = cc
                dd = 2*cc + s[ind]/aa + z[ind]/bb
                blas.copy(s, lmbda, offsetx = ind+1, offsety = ind+1,
                    n = m-1) 
                blas.scal((cc + z[ind]/bb)/dd/aa, lmbda, n = m-1, offset 
                    = ind+1)
                blas.axpy(z, lmbda, (cc + s[ind]/aa)/dd/bb, n = m-1, 
                    offsetx = ind+1, offsety = ind+1)
                blas.scal(math.sqrt(aa*bb), lmbda, offset = ind, n = m)

                ind += m


            # For the 's' blocks: compute two lists 'r' and 'rti' of 
            # length Ns.
            #
            #     r[k]' * sk^{-1} * r[k] = diag(lambda_k)^{-1}
            #     r[k]' * zk * r[k] = diag(lambda_k)
            #
            # where sk and zk are the entries inds[k] : inds[k+1] of
            # s and z, reshaped into symmetric matrices.
            #
            # rti[k] is the inverse of r[k]', so 
            #
            #     rti[k]' * sk * rti[k] = diag(lambda_k)^{-1}
            #     rti[k]' * zk^{-1} * rti[k] = diag(lambda_k).
            #
            # The vectors lambda_k are stored in 
            # 
            #     lmbda[ dims['l'] + sum(dims['q']) : -1 ]
            
            W['r'] = [ matrix(0.0, (m,m)) for m in dims['s'] ]
            W['rti'] = [ matrix(0.0, (m,m)) for m in dims['s'] ]

            ind2 = ind
            for k in xrange(Ns):
                m = dims['s'][k]
                r, rti = W['r'][k], W['rti'][k]

                # Factor sk = L1*L1'; store L1 in ds[inds[k]:inds[k+1]].
                blas.copy(s, ds, offsetx = ind2, offsety = ind2, n = m**2) 
                lapack.potrf(ds, n = m, ldA = m, offsetA = ind2)

                # Factor zs[k] = L2*L2'; store L2 in dz[inds[k]:inds[k+1]].
                blas.copy(z, dz, offsetx = ind2, offsety = ind2, n = m**2) 
                lapack.potrf(dz, n = m, ldA = m, offsetA = ind2)
	 
                # SVD L2'*L1 = U*diag(lambda_k)*V'.  Keep U in work. 
                for i in xrange(m): 
                    blas.scal(0.0, ds, offset = ind2 + i*m, n = i)
                blas.copy(ds, work, offsetx = ind2, n = m**2)
                blas.trmm(dz, work, transA = 'T', ldA = m, ldB = m, n = m,
                    m = m, offsetA = ind2)
                lapack.gesvd(work, lmbda, jobu = 'O', ldA = m, m = m, 
                    n = m, offsetS = ind)
	       
                # r = L2^{-T} * U 
                blas.copy(work, r, n = m*m)
                blas.trsm(dz, r, transA = 'T', m = m, n = m, ldA = m,
                    offsetA = ind2)

                # rti = L2 * U 
                blas.copy(work, rti, n = m*m)
                blas.trmm(dz, rti, m = m, n = m, ldA = m, offsetA = ind2)

                # r := r * diag(sqrt(lambda_k))
                # rti := rti * diag(1 ./ sqrt(lambda_k))
                for i in xrange(m):
                    a = math.sqrt( lmbda[ind+i] )
                    blas.scal(a, r, offset = m*i, n = m)
                    blas.scal(1.0/a, rti, offset = m*i, n = m)

                ind += m
                ind2 += m*m


        # Define a function 
        #
        # solve_newton(dx, dy, dz, dtau, ds, dkappa)
        #
        # for solving 
        #
        #     [ 0      ]   [  0   A'  G'  c ] [ dx   ]    [ rhsx   ]
        #     [ 0      ]   [ -A   0   0   b ] [ dy   ]    [ rhsy   ]
        #     [ ds     ] - [ -G   0   0   h ] [ dz   ] = -[ rhsz   ]
        #     [ dkappa ]   [ -c' -b' -h'  0 ] [ dtau ]    [ rhstau ]
        # 
        #     s o dz + z o dz = -rhss
        #     kappa*dtau + tau*dkappa = -rhskappa.

        try: f = kktsolver(W)
        except ArithmeticError:
            if iters == 0 and primalstart and dualstart: 
                raise ValueError, "Rank(A) < p or Rank([G; A]) < n"
            else:
                raise ArithmeticError, "singular KKT matrix"

        # th = W^{-T} * h
        blas.copy(h, th)
        scale(th, W, trans = 'T', inverse = 'I')

        # Solve
        #
        #     [ 0   A'  G'    ] [ dx1 ]          [ c ]
        #     [-A   0   0     ]*[ dy1 ] = -dgi * [ b ].
        #     [-G   0   W'*W  ] [ dzl ]          [ h ]
         
        xcopy(c, dx1);  xscal(-1, dx1)
        ycopy(b, dy1)
        blas.copy(h, dz1)
        try: f(dx1, dy1, dz1)
        except ArithmeticError:
            if iters == 0 and primalstart and dualstart: 
                raise ValueError, "Rank(A) < p or Rank([G; A]) < n"
            else:
                raise ArithmeticError, "singular KKT matrix"
        xscal(dgi, dx1)
        yscal(dgi, dy1)
        blas.scal(dgi, dz1)

        def solve_newton1(dx, dy, dz, dtau, ds, dkappa):

            # Solve without refinement
            #
            #     [0     ]   [ 0   A'  G'  0 ] [dx  ]    [rhsx  ]
            #     [0     ]   [-A   0   0   b ] [dy  ]    [rhsy  ]
            #     [ds    ] - [-G   0   0   h ] [dz  ] = -[rhsz  ]
            #     [dkappa]   [-c' -b' -h'  0 ] [dtau]    [rhstau]
            #
            #     s o dz + z o dz = -rhss
            #     kappa*dtau + tau*dkappa = -rhskappa.
            #
            # Last two equations in scaled variables: 
            #
            #     lmbda o (W*dz + W^{-T}*ds) = -rhss
            #     lmbdg * (w*dtau + dkappa/w) = -rhskappa.
            #
            # On entry, the righthand sides are stored in dx, dy, dz, dtau,
            # ds, dkappa.  On exit, scaled quantities are returned for ds,
            # dz, dtau, dkappa.

            # dy := -dy = -rhsy
            yscal(-1.0, dy) 

            # ds := -lambda o\ ds (\o is inverse of o)
            sinv(ds, lmbda, dims)
            blas.scal(-1.0, ds)

            # dz := -(dz + W'*ds) = -rhsz + W'*(lambda o\ rhss)
            blas.copy(ds, ds1)
            scale(ds1, W, trans = 'T')
            blas.axpy(ds1, dz)
            blas.scal(-1.0, dz)

            # dkappa[0] := -dkappa[0]/lmbd[-1] 
            #            = -rhskappa / sqrt(kappa*tau)
            dkappa[0] = -dkappa[0] / lmbda[-1]

            # dtau[0] = dtau[0] + dkappa[0] / dgi
            #         = rhstau[0] - rhskappa / tau
            dtau[0] += dkappa[0] / dgi
 

            # Solve 
            #
            #  [  0   A'  G'    0   ] [dx  ]   [ rhsx                  ]
            #  [ -A   0   0     b   ] [dy  ]   [ rhsy                  ]
            #  [ -G   0   W'*W  h   ] [dz  ] = [ rhsz - lambda o\ rhss ]
            #  [ -c'  -b' -h'   k/t ] [dtau]   [ rhst - rhsk/tau       ].

            f(dx, dy, dz)

            dtau[0] = dgi * ( dtau[0] + xdot(c, dx) + ydot(b, dy) + 
                sdot(th, dz, dims) ) / ( 1.0 + sdot(dz1, dz1, dims) )
            xaxpy(dx1, dx, alpha = dtau[0])
            yaxpy(dy1, dy, alpha = dtau[0])
            blas.axpy(dz1, dz, alpha = dtau[0])
 
            # ds := ds - dz = - lambda o\ rhs - dz 
            blas.axpy(dz, ds, alpha = -1)

            dkappa[0] -= dtau[0]


        if refinement:  
            def solve_newton(dx, dy, dz, dtau, ds, dkappa):
            
                # copy righthand sides to dx2 etc
                xcopy(dx, dx2)
                ycopy(dy, dy2)
                blas.copy(dz, dz2)
                dtau2[0] = dtau[0]
                blas.copy(ds, ds2)
                dkappa2[0] = dkappa[0]
 
                solve_newton1(dx, dy, dz, dtau, ds, dkappa)

                # store residuals in dx2, dy2, etc
                #
                #    [0     ]   [ 0  A'  G'  c ] [dx  ]   [rhsx  ]
                #    [0     ]   [-A  0   0   b ] [dy  ]   [rhsy  ]
                #    [ds    ] - [-Gl 0   0   h ] [dz  ] + [rhsz  ]
                #    [dkappa]   [-c' -b' -h' 0 ] [dtau]   [rhstau]
                # 
                #    s o dz + z o dz = -rhss
                #    kappa*dtau + tau*dkappa = -rhskappa.
                #
                # Last two equations in scaled variables: 
                #
                #     lmbda o (W*dz + W^{-T}*ds) = -rhss
                #     lmbdg * (w*dtau + dkappa/w) = -rhskappa.
 
                # Store unscaled steps in s, z.
                blas.copy(dz, z)
                scale(z, W, inverse = 'I')
                blas.copy(ds, s)
                scale(s, W, trans = 'T')
                dutau = dtau[0] * dgi
                dukappa = dkappa[0] / dgi

                Af(dy, dx2, alpha = -1.0, beta = 1.0, trans = 'T')
                Gf(z, dx2, alpha = -1.0, beta = 1.0, trans = 'T')
                xaxpy(c, dx2, alpha =- dutau)

                Af(dx, dy2, alpha = 1.0, beta = 1.0)
                yaxpy(b, dy2, alpha = -dutau)

                Gf(dx, dz2, alpha = 1.0, beta = 1.0)
                blas.axpy(h, dz2, alpha = -dutau)
                blas.axpy(s, dz2)

                dtau2[0] += dukappa + xdot(c,dx) + ydot(b,dy) + sdot(h, z, 
                    dims) 

                # s := lmbda o (W*dz + W^{-T}*ds) 
                blas.copy(ds, s)
                blas.axpy(dz, s)
                sprod(s, lmbda, dims, diag = 'D')
 
                blas.axpy(s, ds2)

                dkappa2[0] += lmbda[-1] * (dtau[0] + dkappa[0])

                solve_newton1(dx2, dy2, dz2, dtau2, ds2, dkappa2)

                xaxpy(dx2, dx)
                yaxpy(dy2, dy)
                blas.axpy(dz2, dz)
                dtau[0] += dtau2[0]
                blas.axpy(ds2, ds)
                dkappa[0] += dkappa2[0]

        else:  # no iterative refinement

            solve_newton = solve_newton1

        mu = blas.nrm2(lmbda)**2 / (cdeg + 1)

        
        # Affine scaling step is solution with right hand side
        #
        #     rhsx = resx  
        #     rhsy = resy
        #     rhsz = resz 
        #     rhstau = rest
        #     rhss = lambda o lambda
        #     rhskappa = kappa * tau.

        # dx := resx, dy := resy, dz := resz, dtau := rest
        xcopy(resx, dx)
        ycopy(resy, dy)
        blas.copy(resz, dz)
        dtau[0] = rest

        # lmbdasq := lmbda o lmbda 
        blas.copy(lmbda, lmbdasq)
        blas.tbmv(lmbda, lmbdasq, n = dims['l'], k = 0, ldA = 1) 
        ind = dims['l']
        for m in dims['q']:
            lmbdasq[ind] = blas.nrm2(lmbda, offset = ind, n = m)**2
            blas.scal(2.0*lmbda[ind], lmbdasq, n = m-1, offset = ind+1)
            ind += m
        # Diagonal symmetric matrices are stored as vectors in lmbdasq.
        blas.tbmv(lmbda, lmbdasq, n = sum(dims['s']) + 1, k = 0, ldA = 1, 
            offsetA = ind, offsetx = ind) 

        # ds := lambda o lambda 
        blas.copy(lmbdasq, ds, n = dims['l'] + sum(dims['q']))
        ind = dims['l'] + sum(dims['q'])
        ind2 = ind
        blas.scal(0.0, ds, offset = ind) 
        for m in dims['s']:
            blas.copy(lmbdasq, ds, n = m, offsetx = ind2, offsety = ind,
                incy = m+1)
            ind += m*m
            ind2 += m

        # dkappa = kappa*tau
        dkappa[0] = lmbdasq[-1]

        solve_newton(dx, dy, dz, dtau, ds, dkappa)

        # Maximum step to boundary
        blas.copy(ds, ds2)
        scale2(lmbda, ds2, dims)
        ts = max_step(ds2, dims)
        blas.copy(dz, dz2)
        scale2(lmbda, dz2, dims)
        tz = max_step(dz2, dims)
        tt = -dtau[0]/lmbda[-1]
        tk = -dkappa[0]/lmbda[-1]
        step = min(1.0, 1.0 / max( [ 0.0, ts, tz, tt, tk ] ))
        sigma = (1.0 - step)**EXPON


        # Centering-corrector step is solution with right hand side
        #
        #     rhsx = (1 - sigma) * resx 
        #     rhsy = (1 - sigma) * resy
        #     rhsz = (1 - sigma) * resz 
        #     rhstau = (1 - sigma) * rest
        #     rhss = lambda o lambda + ds o dz - sigma*mu*e
        #     rhskappa = lambdag**2 + dkappa * dtau - sigma*mu.

        # ds := dz o ds  + lambda o lambda - sigma*mu*e 
        sprod(ds, dz, dims)
        blas.axpy(lmbdasq, ds, n = dims['l'] + sum(dims['q']))
        ds[:dims['l']] -= sigma*mu 
        ds[indq[:-1]] -= sigma*mu
        ind = dims['l'] + sum(dims['q'])
        ind2 = ind
        for m in dims['s']:
            blas.axpy(lmbdasq, ds, n = m, offsetx = ind2, offsety = ind,
                incy = m+1)
            ds[ind : ind+m*m : m+1] -= sigma*mu
            ind += m*m
            ind2 += m
        dkappa[0] = lmbdasq[-1] + dkappa[0]*dtau[0] - sigma*mu
        xcopy(resx, dx);  xscal(1.0 - sigma, dx)
        ycopy(resy, dy);  yscal(1.0 - sigma, dy)
        blas.copy(resz, dz);  blas.scal(1.0 - sigma, dz)
        dtau[0] = (1.0 - sigma) * rest 

        solve_newton(dx, dy, dz, dtau, ds, dkappa)

        # Maximum step to boundary.
        # Compute eigenvalue decomposition of symmetric matrices in ds, dz.
        # The eigenvectors of dsk, dzk are stored in dsk, dzk.  
        # The eigenvalues are stored in sigs, sigz. 

        scale2(lmbda, ds, dims)
        ts = max_step(ds, dims, sigs)
        scale2(lmbda, dz, dims)
        tz = max_step(dz, dims, sigz)
        tt = -dtau[0]/lmbda[-1]
        tk = -dkappa[0]/lmbda[-1]
        step = min(1.0, STEP / max([ 0.0, ts, tz, tt, tk ]))

        # ds := e + step*ds for 'l' and 'q' blocks.
        blas.scal(step, ds, n = dims['l'] + sum(dims['q']))
        ds[:dims['l']] += 1.0
        ds[indq[:-1]] += 1.0

        # sigs := e + step*sigs for 's' blocks.
        blas.scal(step, sigs)
        sigs += 1.0

        # dz := e + step*dz for 'l' and 'q' blocks.
        blas.scal(step, dz, n = dims['l'] + sum(dims['q']))
        dz[:dims['l']] += 1.0
        dz[indq[:-1]] += 1.0

        # sigz := e + step*sigz for 's' blocks.
        blas.scal(step, sigz)
        sigz += 1.0

        # ds := H(lambda)^{-1/2} * ds and dz := H(lambda)^{-1/2} * dz.
        scale2(lmbda, ds, dims, inverse = 'I')
        scale2(lmbda, dz, dims, inverse = 'I')

        # The 'l' and 'q' components of ds and dz now contain the updated
        # variables in the current scaling.  The 's' components of ds 
        # and dz contain 
        #
        #     Lambda^1/2 * Qs * Lambda^1/2
        #     Lambda^1/2 * Qz * Lambda^1/2
        #
        # where Lambda^1/2 * (Qs * diag(sigs) * Qs') * Lambda^1/2 and 
        # Lambda^1/2 * (Qz * diag(sigs) * Qz') * Lambda^1/2 are the 
        # updated variablaes in the current scaling.


        # Update lambda and scaling.

        # For kappa, tau block: 
        #
        #     dg := sqrt( (kappa + step*dkappa) / (tau + step*dtau) ) 
        #         = dg * sqrt( (1 - step*tk) / (1 - step*tt) )
        #
        #     lmbda[-1] := sqrt((tau + step*dtau) * (kappa + step*dkappa))
        #                = lmbda[-1] * sqrt(( 1 - step*tt) * (1 - step*tk))

        dg *= math.sqrt(1.0 - step*tk) / math.sqrt(1.0 - step*tt) 
        dgi = 1.0 / dg
        lmbda[-1] *= math.sqrt(1.0 - step*tt) * math.sqrt(1.0 - step*tk) 


        # 'l' blocks
        #
        #    d :=  d .* sqrt( ds ./ dz )
        #    lmbda := lmbda .* sqrt(ds) .* sqrt(dz)

        m = dims['l']
        ds[:m] = base.sqrt( ds[:m] )
        dz[:m] = base.sqrt( dz[:m] )
 
        # d := d .* ds .* dz 
        blas.tbmv(ds, W['d'], n = m, k = 0, ldA = 1)
        blas.tbsv(dz, W['d'], n = m, k = 0, ldA = 1)
        W['di'][:m] = W['d'][:m] ** -1
         
        # lmbda := ds .* dz
        blas.copy(ds, lmbda, n = m)
        blas.tbmv(dz, lmbda, n = m, k = 0, ldA = 1)


        # 'q' blocks.
        # 
        # Let st and zt be the new variables in the old scaling:
        #
        #     st = ds_k,   zt = dz_k
        #
        # and a = sqrt(st' * J * st),  b = sqrt(zt' * J * zt).
        #
        # 1. Compute the hyperbolic Householder transformation 2*q*q' - J 
        #    that maps st/a to zt/b.
        # 
        #        c = sqrt( (1 + st'*zt/(a*b)) / 2 ) 
        #        q = (st/a + J*zt/b) / (2*c). 
        #
        #    The new scaling point is 
        #
        #        wk := betak * sqrt(a/b) * (2*v[k]*v[k]' - J) * q 
        #
        #    with betak = W['beta'][k].
        # 
        # 3. The scaled variable:
        #
        #        lambda_k0 = sqrt(a*b) * c
        #        lambda_k1 = sqrt(a*b) * ( (2vk*vk' - J) * (-d*q + u/2) )_1
        #
        #    where 
        #
        #        u = st/a - J*zt/b 
        #        d = ( vk0 * (vk'*u) + u0/2 ) / (2*vk0 *(vk'*q) - q0 + 1).
        #
        # 4. Update scaling
        #   
        #        v[k] := wk^1/2 
        #              = 1 / sqrt(2*(wk0 + 1)) * (wk + e).
        #        beta[k] *=  sqrt(a/b)


        ind = dims['l']
        for k in xrange(Nq):

            m = dims['q'][k]
            v = W['v'][k]

            # ln = sqrt( lambda_k' * J * lambda_k )
            ln = jnrm2(lmbda, n = m, offset = ind) 

            # a = sqrt( dsk' * J * dsk ) = sqrt( st' * J * st ) 
            # ds := ds / a = st / a
            aa = jnrm2(ds, offset = ind, n = m)
            blas.scal(1.0/aa, ds, offset = ind, n = m)

            # b = sqrt( dzk' * J * dzk ) = sqrt( zt' * J * zt )
            # dz := dz / a = zt / b
            bb = jnrm2(dz, offset = ind, n = m) 
            blas.scal(1.0/bb, dz, offset = ind, n = m)

            # c = sqrt( ( 1 + (st'*zt) / (a*b) ) / 2 )
            cc = math.sqrt( ( 1.0 + blas.dot(ds, dz, offsetx = ind,
                offsety = ind, n = m) ) / 2.0 )

            # vs = v' * st / a 
            vs = blas.dot(v, ds, offsety = ind, n = m) 

            # vz = v' * J *zt / b
            vz = jdot(v, dz, offsety = ind, n = m) 

            # vq = v' * q where q = (st/a + J * zt/b) / (2 * c)
            vq = (vs + vz ) / 2.0 / cc

            # vu = v' * u  where u =  st/a - J * zt/b 
            vu = vs - vz  

            # lambda_k0 = c
            lmbda[ind] = cc

            # wk0 = 2 * vk0 * (vk' * q) - q0 
            wk0 = 2 * v[0] * vq - ( ds[ind] + dz[ind] ) / 2.0 / cc 

            # d = (v[0] * (vk' * u) - u0/2) / (wk0 + 1)
            dd = (v[0] * vu - ds[ind]/2.0 + dz[ind]/2.0) / (wk0 + 1.0)

            # lambda_k1 = 2 * v_k1 * vk' * (-d*q + u/2) - d*q1 + u1/2
            blas.copy(v, lmbda, offsetx = 1, offsety = ind+1, n = m-1)
            blas.scal(2.0 * (-dd * vq + 0.5 * vu), lmbda, offset = ind+1, 
                n = m-1)
            blas.axpy(ds, lmbda, 0.5 * (1.0 - dd/cc), offsetx = ind+1,
                offsety = ind+1, n = m-1)
            blas.axpy(dz, lmbda, 0.5 * (1.0 + dd/cc), offsetx = ind+1,
                offsety = ind+1, n = m-1)

            # Scale so that sqrt(lambda_k' * J * lambda_k) = sqrt(aa*bb).
            blas.scal(math.sqrt(aa*bb), lmbda, offset = ind, n = m)
            
            # v := (2*v*v' - J) * q 
            #    = 2 * (v'*q) * v' - (J* st/a + zt/b) / (2*c)
            blas.scal(2.0 * vq, v)
            v[0] -= ds[ind] / 2.0 / cc
            blas.axpy(ds, v,  0.5/cc, offsetx = ind+1, offsety = 1,
                n = m-1)
            blas.axpy(dz, v, -0.5/cc, offsetx = ind, n = m)

            # v := v^{1/2} = 1/sqrt(2 * (v0 + 1)) * (v + e)
            v[0] += 1.0
            blas.scal(1.0 / math.sqrt(2.0 * v[0]), v)

            # beta[k] *= ( aa / bb )**1/2
            W['beta'][k] *= math.sqrt( aa / bb )
            
            ind += m


        # 's' blocks
        # 
        # Let st, zt be the updated variables in the old scaling:
        # 
        #     st = ds * diag(sigs ./ lambda) * ds'
        #     zt = dz * diag(sigs ./ lambda) * dz'.
        #
        # 1.  Compute 
        #
        #         L1 = dsk * diag(sigs_k ./ lambda_k)^{1/2}
        #         L2 = dzk * diag(sigz_k ./ lambda_k)^{1/2}.
        #
        #     We have 
        #
        #         L1 * L1' = st,  L2 * L2' = zt.
        #
        # 2.  SVD L2'*L1 = Uk * lambda_k^+ * Vk'.
        #
        # 3.  New scaling is 
        #
        #         r[k] := r[k] * L1 * Vk * diag(lambda_k^+)^{-1/2}
        #         rti[k] := r[k] * L2 * Uk * diag(lambda_k^+)^{-1/2}.

        ind = dims['l'] + sum(dims['q'])
        ind2, ind3 = ind, 0

        # sigs := sigs./lambda.  sigz := sigz./lambda.
        blas.tbsv(lmbda, sigs, n = sum(dims['s']), k = 0, ldA = 1, 
            offsetA = ind)
        blas.tbsv(lmbda, sigz, n = sum(dims['s']), k = 0, ldA = 1, 
            offsetA = ind)

        for k in xrange(Ns):
            m = dims['s'][k]
            r, rti = W['r'][k], W['rti'][k]

            # dsk := L1 = dsk * sqrt(sigs).  dzk := L2 = dzk * sqrt(sigz).
            for i in xrange(m):
                blas.scal(math.sqrt(sigs[ind3+i]), ds, offset = ind2 + m*i,
                    n = m)
                blas.scal(math.sqrt(sigz[ind3+i]), dz, offset = ind2 + m*i,
                    n = m)

            # r := r*dsk = r*L1
            blas.gemm(r, ds, work, m = m, n = m, k = m, ldB = m, ldC = m,
                offsetB = ind2)
            blas.copy(work, r, n = m**2)

            # rti := rti*dzk = rti*L2
            blas.gemm(rti, dz, work, m = m, n = m, k = m, ldB = m, ldC = m,
                offsetB = ind2)
            blas.copy(work, rti, n = m**2)

            # SVD L2'*L1 = U * lmbds^+ * V'; store U in dsk and V' in dzk.
            blas.gemm(dz, ds, work, transA = 'T', m = m, n = m, k = m,
                ldA = m, ldB = m, ldC = m, offsetA = ind2, offsetB = ind2)
            lapack.gesvd(work, lmbda, jobu = 'A', jobvt = 'A', m = m,
                n = m, ldA = m, U = ds, Vt = dz, ldU = m, ldVt = m,
                offsetS = ind, offsetU = ind2, offsetVt = ind2)

            # r := r*V
            blas.gemm(r, dz, work, transB = 'T', m = m, n = m, k = m, 
                ldB = m, ldC = m, offsetB = ind2)
            blas.copy(work, r, n = m**2)

            # rti := rti*U
            blas.gemm(rti, ds, work, n = m, m = m, k = m, ldB = m, ldC = m,
                offsetB = ind2)
            blas.copy(work, rti, n = m**2)

            # r := r*lambda^{-1/2}; rti := rti*lambda^{-1/2}
            for i in xrange(m):    
                a = 1.0 / math.sqrt(lmbda[ind+i])
                blas.scal(a, r, offset = m*i, n = m)
                blas.scal(a, rti, offset = m*i, n = m)

            ind += m
            ind2 += m*m
            ind3 += m

        xaxpy(dx, x, alpha = step)
        yaxpy(dy, y, alpha = step)


        # Unscale s, z, tau, kappa (unscaled variables are used only to 
        # compute feasibility residuals).

        blas.copy(lmbda, s, n = dims['l'] + sum(dims['q']))
        ind = dims['l'] + sum(dims['q'])
        ind2 = ind
        for m in dims['s']:
            blas.scal(0.0, s, offset = ind2)
            blas.copy(lmbda, s, offsetx = ind, offsety = ind2, n = m, 
                incy = m+1)
            ind += m
            ind2 += m*m
        scale(s, W, trans = 'T')

        blas.copy(lmbda, z, n = dims['l'] + sum(dims['q']))
        ind = dims['l'] + sum(dims['q'])
        ind2 = ind
        for m in dims['s']:
            blas.scal(0.0, z, offset = ind2)
            blas.copy(lmbda, z, offsetx = ind, offsety = ind2, n = m, 
                    incy = m+1)
            ind += m
            ind2 += m*m
        scale(z, W, inverse = 'I')

        kappa, tau = lmbda[-1]/dgi, lmbda[-1]*dgi
        gap = blas.dot(lmbda, lmbda, n = cdeg) / tau**2


    return {'status': 'unknown', 'x': None, 'y': None, 's': None, 
        'z': None}



def lp(c, G, h, A = None, b = None, solver = None, primalstart = None,
    dualstart = None):
    """

    Solves a pair of primal and dual LPs

        minimize    c'*x             maximize    -h'*z - b'*y 
        subject to  G*x + s = h      subject to  G'*z + A'*y + c = 0
                    A*x = b                      z >= 0.
                    s >= 0


    Input arguments 

        G is m x n, h is m x 1, A is p x n, b is p x 1.  G and A must be 
        dense or sparse 'd' matrices.   h and b are dense 'd' matrices 
        with one column.  The default values for A and b are empty 
        matrices with zero rows.

        solver is None, 'glpk' or 'mosek'.  The default solver (None)
        uses the cvxopt conelp() function.  The 'glpk' solver is the 
        simplex LP solver from GLPK.  The 'mosek' solver is the LP solver 
        from MOSEK.

        The arguments primalstart and dualstart are ignored when solver
        is 'glpk' or 'mosek' and are optional when solver is None.  
        The argument primalstart is a dictionary with keys 'x' and 's'
        and specifies a primal starting point.  primalstart['x'] must 
        be a dense 'd' matrix of length n;  primalstart['s'] must be a 
        positive dense 'd' matrix of length m.
        The argument dualstart is a dictionary with keys 'z' and 'y' 
        and specifies a dual starting point.   dualstart['y'] must 
        be a dense 'd' matrix of length p;  dualstart['z'] must be a 
        positive dense 'd' matrix of length m.

        When solver is None, we require n >= 1, Rank(A) = p and 
        Rank([G; A]) = n


    Returns a dictionary with keys 'status', 'x', 's', 'y', 'z'.

        If solver is None or 'mosek':
        
            If status is 'optimal', x, s, y, z are the primal and dual 
            optimal solutions.

            If status is 'primal infeasible', x = s = None, and z, y 
            are a proof of infeasibility: 

                h'*z + b'*y = -1,  G'*z + A'*y = 0,  z >= 0.

            If status is 'dual infeasible', z = y = None, and x, s are 
            a proof of dual infeasibility: 

                c'*x = -1,  G*x + s = 0,  A*x = 0,  s >= 0.

            If status is 'unknown', x, y, s, z are None.


        If solver is 'glpk':
        
            If status is 'optimal', x, s, y, z are the primal and dual 
            optimal solutions.

            If status is 'primal infeasible', 'dual infeasible',
            or 'unknown', then x = s = z = y = None.


    The control parameters for the different solvers can be modified by 
    adding an entry to the dictionary cvxopt.solvers.options.  The 
    following parameters control the execution of the default solver.

        options['show_progress'] True/False (default: True)
        options['maxiters'] positive integer (default: 100)
        options['abstol'] scalar (default: 1e-7)
        options['reltol'] scalar (default: 1e-6)
        options['feastol'] scalar (default: 1e-7).

    The control parameter names for GLPK and MOSEK can be found in the
    GLPK and MOSEK documentation.  Options that are not recognized are 
    replaced by their default values.
    """

    if type(c) is not matrix or c.typecode != 'd' or c.size[1] != 1: 
        raise TypeError, "'c' must be a dense column matrix"
    n = c.size[0]
    if n < 1: raise ValueError, "number of variables must be at least 1"

    if (type(G) is not matrix and type(G) is not spmatrix) or \
        G.typecode != 'd' or G.size[1] != n:
        raise TypeError, "'G' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n 
    m = G.size[0]
    if type(h) is not matrix or h.typecode != 'd' or h.size != (m,1):
        raise TypeError, "'h' must be a 'd' matrix of size (%d,1)" %m

    if A is None:  A = spmatrix([], [], [], (0,n), 'd')
    if (type(A) is not matrix and type(A) is not spmatrix) or \
        A.typecode != 'd' or A.size[1] != n:
        raise TypeError, "'A' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n
    p = A.size[0]
    if b is None: b = matrix(0.0, (0,1))
    if type(b) is not matrix or b.typecode != 'd' or b.size != (p,1): 
        raise TypeError, "'b' must be a dense matrix of size (%d,1)" %p

    if solver == 'glpk':
        try: from cvxopt import glpk
        except ImportError: raise ValueError, "invalid option "\
            "(solver='glpk'): cvxopt.glpk is not installed" 
        glpk.options = options
        status, x, z, y = glpk.solvelp(c,G,h,A,b)
        if status == 'optimal':
            s = matrix(h)
            base.gemv(G, x, s, alpha=-1.0, beta=1.0)
        else:
            s = None
        return {'status': status, 'x': x, 's': s, 'y': y, 'z': z}

    if solver == 'mosek':
        try: from cvxopt import mosek
        except ImportError: raise ValueError, "invalid option "\
            "(solver='mosek'): cvxopt.mosek is not installed" 
        mosek.options = options
        prosta, solsta, x, z, y  = mosek.solvelp(c,G,h,A,b)
        if solsta == 'OPTIMAL':
            s = matrix(h)
            base.gemv(G, x, s, alpha=-1.0, beta=1.0)
            status = 'optimal'
        elif solsta == 'PRIMAL_INFEASIBLE_CER':
            status = 'primal infeasible'
            ducost = -blas.dot(h,z) - blas.dot(b,y)
            blas.scal(1.0/ducost, y)
            blas.scal(1.0/ducost, z)
            x, s = None, None
        elif solsta == 'DUAL_INFEASIBLE_CER':
            status = 'dual infeasible'
            pcost = blas.dot(c,x)
            blas.scal(-1.0/pcost, x)
            s = matrix(0.0, (m,1))
            base.gemv(G, x, s, alpha=-1.0)
            z, y = None, None
        else: 
            status = 'unknown'
            x, s, y, z = None, None, None, None
        return {'status': status, 'x': x, 's': s, 'y': y, 'z': z}

    return conelp(c, G, h, {'l': m, 'q': [], 's': []}, A,  b, primalstart 
        = primalstart, dualstart = dualstart)



def socp(c, Gl = None, hl = None, Gq = None, hq = None, A = None, b = None,
    primalstart = None, dualstart = None):

    """
    Solves a pair of primal and dual SOCPs

        minimize    c'*x             
        subject to  Gl*x + sl = hl      
                    Gq[k]*x + sq[k] = hq[k],  k = 0, ..., N-1
                    A*x = b                      
                    sl >= 0,  
                    sq[k] >= 0, k = 0, ..., N-1

        maximize    -hl'*z - sum_k hq[k]'*zq[k] - b'*y
        subject to  Gl'*zl + sum_k Gq[k]'*zq[k] + A'*y + c = 0
                    zl >= 0,  zq[k] >= 0, k = 0, ..., N-1.

    The inequalities sl >= 0 and zl >= 0 are elementwise vector 
    inequalities.  The inequalities sq[k] >= 0, zq[k] >= 0 are second 
    order cone inequalities, i.e., equivalent to 
    
        sq[k][0] >= || sq[k][1:] ||_2,  zq[k][0] >= || zq[k][1:] ||_2.


    Input arguments 

        Gl is a dense or sparse 'd' matrix of size (ml, n).  hl is a 
        dense 'd' matrix of size (ml, 1). The default values of Gl and hl 
        are matrices with zero rows.
       
        The argument Gq is a list of N dense or sparse 'd' matrices of 
        size (m[k] n), k = 0, ..., N-1, where m[k] >= 1.   hq is a list 
        of N dense 'd' matrices of size (m[k], 1), k = 0, ..., N-1.  
        The default values of Gq and hq are empty lists.

        A is a dense or sparse 'd' matrix of size (p,1).  b is a dense 'd'
        matrix of size (p,1).  The default values of A and b are matrices 
        with zero rows.

        The argument primalstart is a dictionary with keys 'x', 'sl', 'sq',
        and specifies an optional primal starting point.  
        primalstart['x'] is a dense 'd' matrix of size (n,1).  
        primalstart['sl'] is a positive dense 'd' matrix of size (ml,1).
        primalstart['sq'] is a list of matrices of size (m[k],1), positive
        with respect to the second order cone of order m[k].

        The argument dualstart is a dictionary with keys 'y', 'zl', 'zq', 
        and specifies an optional dual starting point.  
        dualstart['y'] is a dense 'd' matrix of size (p,1).  
        dualstart['zl'] is a positive dense 'd' matrix of size (ml,1).  
        dualstart['sq'] is a list of matrices of size (m[k],1), positive 
        with respect to the second order cone of order m[k].


    Returns a dictionary with keys 'status', 'x', 'sl', 'sq', 'y', 'zl',
        'zq'.

        If status is 'optimal', x, sl, sq, y, zl, zq are approximate 
        primal and dual optimal solutions.

        If status is 'primal infeasible', x = sl = sq = None, and y, zl, 
        zq are a proof of infeasibility: 

            hl'*zl + sum_k hq[k]' * zq[k] + b'*y = -1,  
            Gl'*zl + sum_k Gq[k]' * zq[k] + A'*y = 0,  
            zl >= 0,   zq[k] >= 0, k = 0, ..., N-1.

        If status is 'dual infeasible', zl = zq = y = None, and x, sl, sq 
        are a proof of dual infeasibility: 

            c'*x = -1,  Gl*x + sl = 0,  Gq[k]*x + sq[k] = 0,  A*x = 0,  
            sl >= 0,  sq[k] >= 0, k = 0, ..., N-1.

        If status is 'unknown', x, y, sl, sq, zl, zq are None.


    The following parameters control the execution of the default 
    solver.

        options['show_progress'] True/False (default: True)
        options['maxiters'] positive integer (default: 100)
        options['abstol'] scalar (default: 1e-7)
        options['reltol'] scalar (default: 1e-6)
        options['feastol'] scalar (default: 1e-7).
    """

    if type(c) is not matrix or c.typecode != 'd' or c.size[1] != 1: 
        raise TypeError, "'c' must be a dense column matrix"
    n = c.size[0]
    if n < 1: raise ValueError, "number of variables must be at least 1"

    if Gl is None:  Gl = spmatrix([], [], [], (0,n), tc='d')
    if (type(Gl) is not matrix and type(Gl) is not spmatrix) or \
        Gl.typecode != 'd' or Gl.size[1] != n:
        raise TypeError, "'Gl' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n
    ml = Gl.size[0]
    if hl is None: hl = matrix(0.0, (0,1))
    if type(hl) is not matrix or hl.typecode != 'd' or \
        hl.size != (ml,1):
        raise TypeError, "'hl' must be a dense 'd' matrix of " \
            "size (%d,1)" %ml

    if Gq is None: Gq = []
    if type(Gq) is not list or [ G for G in Gq if (type(G) is not matrix 
        and type(G) is not spmatrix) or G.typecode != 'd' or 
        G.size[1] != n ]:
        raise TypeError, "'Gq' must be a list of sparse or dense 'd' "\
            "matrices with %d columns" %n 
    mq = [ G.size[0] for G in Gq ]
    a = [ k for k in xrange(len(mq)) if mq[k] == 0 ] 
    if a: raise TypeError, "the number of rows of Gq[%d] is zero" %a[0]
    if hq is None: hq = []
    if type(hq) is not list or len(hq) != len(mq) or [ h for h in hq if
        (type(h) is not matrix and type(h) is not spmatrix) or 
        h.typecode != 'd' ]: 
        raise TypeError, "'hq' must be a list of %d dense or sparse "\
            "'d' matrices" %len(mq)
    a = [ k for k in xrange(len(mq)) if hq[k].size != (mq[k], 1) ]
    if a:
        k = a[0]
        raise TypeError, "'hq[%d]' has size (%d,%d).  Expected size "\
            "is (%d,1)." %(k, hq[k].size[0], hq[k].size[1], mq[k]) 

    if A is None: A = spmatrix([], [], [], (0,n), 'd')
    if (type(A) is not matrix and type(A) is not spmatrix) or \
        A.typecode != 'd' or A.size[1] != n:
        raise TypeError, "'A' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n
    p = A.size[0]
    if b is None: b = matrix(0.0, (0,1))
    if type(b) is not matrix or b.typecode != 'd' or b.size != (p,1): 
        raise TypeError, "'b' must be a dense matrix of size (%d,1)" %p

    dims = {'l': ml, 'q': mq, 's': []}
    N = ml + sum(mq)
    h = matrix(0.0, (N,1))
    if type(Gl) is matrix or [ Gk for Gk in Gq if type(Gk) is matrix ]:
        G = matrix(0.0, (N, n))
    else:
        G = spmatrix([], [], [], (N, n), 'd')
    h[:ml] = hl
    G[:ml,:] = Gl
    ind = ml
    for k in xrange(len(mq)):
        h[ind : ind + mq[k]] = hq[k]
        G[ind : ind + mq[k], :] = Gq[k]
        ind += mq[k]

    if primalstart:
        ps = {}
        ps['x'] = primalstart['x']
        ps['s'] = matrix(0.0, (N,1))
        if ml: ps['s'][:ml] = primalstart['sl']
        if mq:
            ind = ml
            for k in xrange(len(mq)): 
                ps['s'][ind : ind + mq[k]] = primalstart['sq'][k][:]
                ind += mq[k]
    else: 
        ps = None

    if dualstart:
        ds = {}
        if p:  ds['y'] = dualstart['y']
        ds['z'] = matrix(0.0, (N,1))
        if ml: ds['z'][:ml] = dualstart['zl']
        if mq: 
            ind = ml
            for k in xrange(len(mq)):
                ds['z'][ind : ind + mq[k]] = dualstart['zq'][k][:]
                ind += mq[k]
    else: 
        ds = None

    sol = conelp(c, G, h, dims, A = A, b = b, primalstart = ps, dualstart
        = ds)
    val = {'status': sol['status'],  'x': sol['x'], 'y': sol['y']}
    if sol['s'] is None:  
        val['sl'] = None
        val['sq'] = None
    else: 
        val['sl'] = sol['s'][:ml]  
        val['sq'] = [ matrix(0.0, (m,1)) for m in mq ] 
        ind = ml
        for k in xrange(len(mq)):
            val['sq'][k][:] = sol['s'][ind : ind+mq[k]]
            ind += mq[k]

    if sol['z'] is None: 
        val['zl'] = None
        val['zq'] = None
    else: 
        val['zl'] = sol['z'][:ml]
        val['zq'] = [ matrix(0.0, (m,1)) for m in mq] 
        ind = ml
        for k in xrange(len(mq)):
            val['zq'][k][:] = sol['z'][ind : ind+mq[k]]
            ind += mq[k]

    return val

    
def sdp(c, Gl = None, hl = None, Gs = None, hs = None, A = None, b = None, 
    solver = None, primalstart = None, dualstart = None):
    """

    Solves a pair of primal and dual SDPs

        minimize    c'*x             
        subject to  Gl*x + sl = hl      
                    mat(Gs[k]*x) + ss[k] = hs[k], k = 0, ..., N-1
                    A*x = b                      
                    sl >= 0,  ss[k] >= 0, k = 0, ..., N-1

        maximize    -hl'*z - sum_k trace(hs[k]*zs[k]) - b'*y
        subject to  Gl'*zl + sum_k Gs[k]'*vec(zs[k]) + A'*y + c = 0
                    zl >= 0,  zs[k] >= 0, k = 0, ..., N-1.

    The inequalities sl >= 0 and zl >= 0 are elementwise vector 
    inequalities.  The inequalities ss[k] >= 0, zs[k] >= 0 are matrix 
    inequalities, i.e., the symmetric matrices ss[k] and zs[k] must be
    positive semidefinite.  mat(Gs[k]*x) is the symmetric matrix X with 
    X[:] = Gs[k]*x.  For a symmetric matrix, zs[k], vec(zs[k]) is the 
    vector zs[k][:].
    

    Input arguments 

        Gl is a dense or sparse 'd' matrix of size (ml, n).  hl is a 
        dense 'd' matrix of size (ml, 1). The default values of Gl and hl 
        are matrices with zero rows.

        The argument Gs is a list of N dense or sparse 'd' matrices of 
        size (m[k]**2, n), k = 0, ..., N-1.   The columns of Gs[k] 
        represent symmetric matrices stored as vectors in column major 
        order.  hs is a list of N dense 'd' matrices of size (m[k], m[k]),
        k = 0, ..., N-1.  The columns of Gs[k] and the matrices hs[k]
        represent symmetric matrices in 'L' storage, i.e., only the lower 
        triangular elements are accessed.  The default values of Gs and 
        hs are empty lists.

        A is a dense or sparse 'd' matrix of size (p,n).  b is a dense 'd'
        matrix of size (p,1).  The default values of A and b are matrices 
        with zero rows.
 
        solver is None or 'dsdp'.  The default solver (None) calls 
        cvxopt.conelp().  The 'dsdp' solver uses an interface to DSDP5.  
        The 'dsdp' solver does not accept problems with equality 
        constraints (A and b must have zero rows, or be absent).

        The argument primalstart is a dictionary with keys 'x', 'sl',
        'ss', and specifies an optional primal starting point.  
        primalstart['x'] is a dense 'd' matrix of length n;   
        primalstart['sl'] is a  positive dense 'd' matrix of length ml;  
        primalstart['ss'] is a list of positive definite matrices of 
        size (ms[k], ms[k]).  Only the lower triangular parts of these 
        matrices will be accessed.

        The argument dualstart is a dictionary with keys 'zl', 'zs', 'y' 
        and specifies an optional dual starting point.   
        dualstart['y'] is a dense 'd' matrix of length p;  
        dualstart['zl'] must be a positive dense 'd' matrix of length ml;
        dualstart['zs'] is a list of positive definite matrices of 
        size (ms[k], ms[k]).  Only the lower triangular parts of these 
        matrices will be accessed.

        The arguments primalstart and dualstart are ignored when solver
        is 'dsdp'.


    Returns a dictionary with keys 'status', 'x', 'sl', 'ss', 'y', 'zl',
        'zs'.

        If status is 'optimal', x, sl, ss, y, zl, zs are approximate 
        primal and dual optimal solutions.

        If status is 'primal infeasible', x = sl = ss = None, and zl, 
        zs, y are a proof of infeasibility: 

            hl'*zl + sum_k tr(hs[k]*zs[k]) + b'*y = -1,  
            Gl'*zl + sum_k Gs[k]'*vec(zs[k]) + A'*y = 0,  
            zl >= 0,  zs[k] >= 0, k = 0, ..., N-1.

        If status is 'dual infeasible', zl = zs = y = None, and x, sl, 
        ss are a proof of dual infeasibility: 

            c'*x = -1,  
            Gl*x + sl = 0,  mat(Gs[k]*x] + ss[k] = 0,  k = 0, ..., N-1
            A*x = 0,  sl >= 0, ss[k] >= 0, k = 0, ..., N-1.

        If status is 'unknown', x, y, sl, ss, zl, zs are None.


    The following parameters control the execution of the default 
    solver.

        options['show_progress'] True/False (default: True)
        options['maxiters'] positive integer (default: 100)
        options['abstol'] scalar (default: 1e-7)
        options['reltol'] scalar (default: 1e-6)
        options['feastol'] scalar (default: 1e-7).

    The execution of the 'dsdp' solver is controlled by: 

        options['show_progress'] integer (default: 0)
        options['maxiters'] positive integer 
        options['rgaptol'] scalar (default: 1e-5).
    """

    if type(c) is not matrix or c.typecode != 'd' or c.size[1] != 1: 
        raise TypeError, "'c' must be a dense column matrix"
    n = c.size[0]
    if n < 1: raise ValueError, "number of variables must be at least 1"

    if Gl is None: Gl = spmatrix([], [], [], (0,n), tc='d')
    if (type(Gl) is not matrix and type(Gl) is not spmatrix) or \
        Gl.typecode != 'd' or Gl.size[1] != n:
        raise TypeError, "'Gl' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n
    ml = Gl.size[0]
    if hl is None: hl = matrix(0.0, (0,1))
    if type(hl) is not matrix or hl.typecode != 'd' or \
        hl.size != (ml,1):
        raise TypeError, "'hl' must be a 'd' matrix of size (%d,1)" %ml

    if Gs is None: Gs = []
    if type(Gs) is not list or [ G for G in Gs if (type(G) is not matrix 
        and type(G) is not spmatrix) or G.typecode != 'd' or 
        G.size[1] != n ]:
        raise TypeError, "'Gs' must be a list of sparse or dense 'd' "\
            "matrices with %d columns" %n 
    ms = [ int(math.sqrt(G.size[0])) for G in Gs ]
    a = [ k for k in xrange(len(ms)) if ms[k]**2 != Gs[k].size[0] ]
    if a: raise TypeError, "the squareroot of the number of rows in "\
        "'Gs[%d]' is not an integer" %k
    if hs is None: hs = []
    if type(hs) is not list or len(hs) != len(ms) or [ h for h in hs if
        (type(h) is not matrix and type(h) is not spmatrix) or
        h.typecode != 'd' ]:
        raise TypeError, "'hs' must be a list of %d dense or sparse "\
            "'d' matrices" %len(ms)
    a = [ k for k in xrange(len(ms)) if hs[k].size != (ms[k],ms[k]) ]
    if a:
        k = a[0]
        raise TypeError, "hs[%d] has size (%d,%d).  Expected size is "\
            "(%d,%d)." %(k,hs[k].size[0], hs[k].size[1], ms[k], ms[k])

    if A is None: A = spmatrix([], [], [], (0,n), 'd')
    if (type(A) is not matrix and type(A) is not spmatrix) or \
        A.typecode != 'd' or A.size[1] != n:
        raise TypeError, "'A' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n
    p = A.size[0]
    if b is None: b = matrix(0.0, (0,1))
    if type(b) is not matrix or b.typecode != 'd' or b.size != (p,1): 
        raise TypeError, "'b' must be a dense matrix of size (%d,1)" %p

    if solver == 'dsdp':
        try: from cvxopt import dsdp
        except ImportError: raise ValueError, "invalid option "\
            "(solver = 'dsdp'): cvxopt.dsdp is not installed"
        dsdp.options = options
        if p: raise ValueError, "sdp() with the solver = 'dsdp' option "\
            "does not handle problems with equality constraints"
        dsdpstatus, x, r, zl, zs = dsdp.sdp(c, Gl, hl, Gs, hs)
        sl = hl - Gl*x
        ss = [ hs[k] - matrix(Gs[k]*x, (ms[k], ms[k])) for k in 
            xrange(len(ms)) ]
        for k in xrange(len(ms)):  
            symm(ss[k], ms[k])
            symm(zs[k], ms[k])
        if dsdpstatus == 'DSDP_PDFEASIBLE':
            y = matrix(0.0, (0,1))
            status = 'optimal'
        elif dsdpstatus == 'DSDP_UNBOUNDED':
            pcost = blas.dot(c,x)
            x /= -pcost
            sl /= -pcost
            ss = [sk / -pcost for sk in ss]
            zl, zs = None, None
            status = 'dual infeasible'
        elif dsdpstatus == 'DSDP_INFEASIBLE':
            dcost = -blas.dot(hl,zl) - misc.sdot(hs, zs)
            zl /= -dcost 
            zs = [zs / -docst for zk in zs]
            y = matrix(0.0, (0,1))
            x, sl, ss = None, None, None
            status = 'primal infeasible'
        else:
            status = 'unknown'
            x, sl, ss, y, zl, zs, = None, None, None, None, None, None

        return {'status': status, 'x': x, 'y': y, 'sl': sl, 'ss': ss, 
            'zl': zl, 'zs': zs}
         
    dims = {'l': ml, 'q': [], 's': ms}
    N = ml + sum([ m**2 for m in ms ])
    h = matrix(0.0, (N,1))
    if type(Gl) is matrix or [ Gk for Gk in Gs if type(Gk) is matrix ]:
        G = matrix(0.0, (N, n))
    else:
        G = spmatrix([], [], [], (N, n), 'd')
    h[:ml] = hl
    G[:ml,:] = Gl
    ind = ml
    for k in xrange(len(ms)):
        m = ms[k]
        h[ind : ind + m*m] = hs[k][:]
        G[ind : ind + m*m, :] = Gs[k]
        ind += m**2

    if primalstart:
        ps = {}
        ps['x'] = primalstart['x']
        ps['s'] = matrix(0.0, (N,1))
        if ml: ps['s'][:ml] = primalstart['sl']
        if ms:
            ind = ml
            for k in xrange(len(ms)):
                m = ms[k]
                ps['s'][ind : ind + m*m] = primalstart['ss'][k][:]
                ind += m**2
    else: 
        ps = None

    if dualstart:
        ds = {}
        if p:  ds['y'] = dualstart['y']
        ds['z'] = matrix(0.0, (N,1))
        if ml: ds['z'][:ml] = dualstart['zl']
        if ms: 
            ind = ml
            for k in xrange(len(ms)):
                m = ms[k]
                ds['z'][ind : ind + m*m] = dualstart['zs'][k][:]
                ind += m**2
    else: 
        ds = None

    sol = conelp(c, G, h, dims, A = A, b = b, primalstart = ps, dualstart
        = ds)
    val = {'status': sol['status'],  'x': sol['x'], 'y': sol['y']}
    if sol['s'] is None:
        val['sl'] = None
        val['ss'] = None
    else:
        val['sl'] = sol['s'][:ml]
        val['ss'] = [ matrix(0.0, (mk, mk)) for mk in ms ]
        ind = ml
        for k in xrange(len(ms)):
             m = ms[k]
             val['ss'][k][:] = sol['s'][ind:ind+m*m]
             ind += m**2
    if sol['z'] is None:
        val['zl'] = None
        val['zs'] = None
    else:
        val['zl'] = sol['z'][:ml]
        val['zs'] = [ matrix(0.0, (mk, mk)) for mk in ms ]
        ind = ml
        for k in xrange(len(ms)):
             m = ms[k]
             val['zs'][k][:] = sol['z'][ind:ind+m*m]
             ind += m**2
    return val
