"""
Miscellaneous functions used by the CVXOPT solvers.
"""

# Copyright 2004-2007 J. Dahl and L. Vandenberghe.
# 
# This file is part of CVXOPT version 0.9.
#
# CVXOPT is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# CVXOPT is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import math
from cvxopt import base, blas, lapack, cholmod
from cvxopt.base import matrix, spmatrix
cholmod.options['supernodal'] = 2

__all__ = []

def had(x,y):  
    """
    Returns Hadamard product x.*y of dense 'd' matrices x and y.

    x and y are 'd' matrices of equal size.

    Returns x .* y as a new matrix.
    """ 

    if x.size != y.size:
        raise ValueError, "x and y must have the same size"
    z = +x 
    blas.tbmv(y, z, n=len(y), k=0, ldA=1) 
    return z


def had2(x, y, n=None):  
    """
    In-place Hadamard product x.*y of dense 'd' matrices x and y.

    x and y are 'd' matrices of equal size.
    
    n is a nonnegative integer, with default value len(x).  If the
    default value is used, len(x) must be equal to len(y).

    On exit x[:n] is replaced with  x[:n].*y[:n].
    """ 

    if n is None: 
        if x.size != y.size:
            raise ValueError, "x and y must have the same size"
        else:
            n = len(x)
    if len(x) < n or len(y) < n:
        raise ValueError, "x and y must have length at least n"
    blas.tbmv(y, x, n=n, k=0, ldA=1) 


def ihad(x,y):  
    """
    Returns componentwise division x./y of dense 'd' matrices x and y.

    y is a 'd' matrix.   x is a scalar or a 'd' matrix of the same
    size as y.

    Returns x ./ y as a new matrix.
    """ 

    if type(x) is float or type(x) is int:
        z = matrix(x, y.size, tc='d')
    else:
        z = +x 
    if z.size != y.size:
        raise ValueError, "x and y must have the same size"
    blas.tbsv(y, z, n=len(y), k=0, ldA=1) 
    return z


def ihad2(x, y, n=None):  
    """
    In-place Hadamard division x./y of dense matrices x and y.

    x and y are dense 'd' matrices of equal size.
    
    n is a nonnegative integer, with default value len(x).  If the
    default value is used, len(x) must be equal to len(y).

    On exit x[:n] is replaced with  x[:n]./y[:n].
    """ 

    if n is None: 
        if x.size != y.size:
            raise ValueError, "x and y must have the same size"
        else:
            n = len(x)
    if len(x) < n or len(y) < n:
        raise ValueError, "x and y must have length at least n"
    blas.tbsv(y, x, n=n, k=0, ldA=1) 


def scopy(x, y):
    """
    y := x for symmetric or block-diagonal symmetric dense matrices.
    """

    if type(x) is matrix: 
	blas.copy(x, y)
    else:
	for k in xrange(len(x)): blas.copy(x[k], y[k])

	
def sdot(x, y):
    """
    Inner product of two block-diagonal symmetric dense 'd' matrices.

    x and y are square dense 'd' matrices, or lists of N square dense
    'd' matrices.
    """

    a = 0.0
    if type(x) is matrix:
	n = x.size[0]
	a += blas.dot(x, y, incx=n+1, incy=n+1, n=n)
	for j in xrange(1,n):
	    a += 2.0 * blas.dot(x, y, incx=n+1, incy=n+1, offsetx=j,
		offsety=j, n=n-j)

    else:
	for k in xrange(len(x)):
	    n = x[k].size[0]
	    a += blas.dot(x[k], y[k], incx=n+1, incy=n+1, n=n)
	    for j in xrange(1,n):
		a += 2.0 * blas.dot(x[k], y[k], incx=n+1, incy=n+1, 
		    offsetx=j, offsety=j, n=n-j)
    return a


def sscal(alpha, x):
    """
    x := alpha*x for symmetric or block-diagonal symmetric dense 'd'
    matrices.
    """

    if type(x) is matrix: 
	blas.scal(alpha, x)
    else:
	for xk in x: blas.scal(alpha, xk)


def saxpy(x, y, alpha=1.0):
    """
    y := alpha*x + y for symmetric or block-diagonal symmetric dense 
    'd' matrices.
    """

    if type(x) is matrix: 
	blas.axpy(x, y, alpha=alpha)
    else:
	for k in xrange(len(x)): blas.axpy(x[k], y[k], alpha=alpha)


def snrm2(x):
    """
    Frobenius norm of symmetric matrix or block-diagonal symmetric 
    dense 'd' matrix.
    """

    return math.sqrt(sdot(x,x))


def cngrnc(R, X, Y, trans='N', alpha=1.0, beta=0.0):
    """
    Congruence transformation

	Y := alpha * R*X*R^T + beta*Y  if trans is 'N' 
	Y := alpha * R^T*X*R + beta*Y  if trans is 'T'

    R, X and Y are lists of square dense 'd' matrices.  
    X may be equal to Y.
    Note that trans='T' is slightly faster than trans='N'.
    """

    maxn = max([0] + [R_k.size[0] for R_k in R])
    W = matrix(0.0, (maxn,maxn))
    for k in xrange(len(R)):
        n = R[k].size[0]

        # scale diagonal of X by 1/2
        blas.scal(0.5, X[k], inc=n+1)
    
        # W := R*tril(X) (trans is 'N') or W := tril(X)*R (trans is 'T')
        blas.copy(R[k], W)
        if trans == 'N': 
	    blas.trmm(X[k], W, side='R', m=n, n=n, ldB=n)
        else: 
	    blas.trmm(X[k], W, side='L', m=n, n=n, ldB=n)

        # scale diagonal of X by 2
        blas.scal(2.0, X[k], inc=n+1)

        # Y := alpha*(W*R' + R*W') + beta*Y 
        blas.syr2k(R[k], W, Y[k], trans=trans, n=n, k=n, ldB=n, 
            alpha=alpha, beta=beta)


def kkt(H, A, Df, G, method=3):
    """
    Generates a solver for the KKT system

        [ H   *   *     *  ] [ x   ]   [ bx   ]
        [ A   0   *     *  ] [ y   ] = [ by   ].
        [ Df  0  -Dnl   *  ] [ znl ]   [ bznl ] 
        [ G   0   0    -Dl ] [ zl  ]   [ bzl  ]

    kkt() does a symbolic factorization and returns a function for the 
    numeric factorization.

    If f = kkt(H, A, Df, G, method) then f(H, Df, dnl, dl) does the 
    numeric factorization of the KKT matrix, with  
    
        Dnl = diag(dnl)^-2,  Dl = diag(dl)^-2, 

    and returns a function for solving the system.   The arguments H 
    and Df in f() can be different from the arguments in the call to 
    kkt() but they must have the same sparsity pattern.

    If g = f(H, Df, dnl, dl) then g(x, y, znl, zl) solves the system. 
    On entry x, y, znl, zl are the righthand side.  On exit they are 
    overwritten with the solution.
    """

    n, p, mnl, ml = H.size[0], A.size[0], Df.size[0], G.size[0]

    if method == 1:

        # Factor entire matrix via dense LDL.

        ldK = n + p + mnl + ml
        K = matrix(0.0, (ldK,ldK))
        ipiv = matrix(0, (ldK,1))
        u = matrix(0.0, (ldK,1))
        def f(H, Df, dnl, dl):
            blas.scal(0.0, K)
            K[:n, :n] = H
            K[n:n+p, :n] = A
            K[n+p:n+p+mnl, :n] = Df
            K[n+p+mnl:, :n] = G
            K[(ldK+1)*(n+p) : (ldK+1)*(n+p+mnl) : ldK+1] = -dnl**-2
            K[(ldK+1)*(n+p+mnl) :: ldK+1] = -dl**-2
            lapack.sytrf(K, ipiv)
            def g(x, y, znl, zl):
                u[:n] = x
                u[n:n+p] = y
                u[n+p:n+p+mnl] = znl 
                u[n+p+mnl:] = zl 
                lapack.sytrs(K, ipiv, u)
                x[:] = u[:n]
                y[:] = u[n:n+p]
                znl[:] = u[n+p:n+p+mnl]
                zl[:] = u[n+p+mnl:]
            return g 
        return f

    elif method == 2:
 
        # Factor 
        #
        #        [ H + Df'*Dnl^-2*Df + G'*Dl^-2*G    * ]
        #    S = [                                     ]
        #        [ A                                 0 ] 
        #
        # via dense LDL (if p > 0) or Cholesky (if p = 0).
        # To solve the KKT system, first solve 
        #        
        #        [ x ]   [ bx + Df'*Dnl^-2*bznl + G'*Dl^-2*bzl ]
        #    S * [   ] = [                                     ]
        #        [ y ]   [ by                                  ]
        #
        #    Dnl^2 * znl = Df*x - bznl
        #    Dl^2 * zl = Dl.^-2 * (G*x - bzl).

        ldK = n + p
        K = matrix(0.0, (ldK,ldK))
        if p: 
            ipiv = matrix(0, (ldK,1))
        if type(G) is matrix: 
            Gsc = matrix(0.0, G.size) 
        else:
            Gsc = spmatrix(0.0, G.I, G.J, G.size) 
        if type(Df) is matrix: 
            Dfsc = matrix(0.0, Df.size) 
        else:
            Dfsc = spmatrix(0.0, Df.I, Df.J, Df.size) 
        u = matrix(0.0, (ldK,1))
        def f(H, Df, dnl, dl):
            blas.scal(0.0, K)
            K[:n, :n] = H
            base.gemm(spmatrix(dnl, range(mnl), range(mnl), tc='d'), 
                Df, Dfsc, partial=True)
            base.syrk(Dfsc, K, trans='T', beta=1.0)
            base.gemm(spmatrix(dl, range(ml), range(ml), tc='d'), G, 
                Gsc, partial=True)
            base.syrk(Gsc, K, trans='T', beta=1.0)
            K[n:,:n] = A
            if p: 
                lapack.sytrf(K, ipiv)
            else: 
                lapack.potrf(K)
            def g(x, y, znl, zl):
                u[:n] = x
                had2(znl, dnl)
                base.gemv(Dfsc, znl, u, trans='T', beta=1.0)
                had2(zl, dl)
                base.gemv(Gsc, zl, u, trans='T', beta=1.0)
                u[n:n+p] = y
                if p: 
                    lapack.sytrs(K, ipiv, u)
                else: 
                    lapack.potrs(K, u)
                x[:] = u[:n]
                y[:] = u[n:n+p]
                base.gemv(Dfsc, x, znl, beta=-1.0)
                had2(znl, dnl)
                base.gemv(Gsc, x, zl, beta=-1.0)
                had2(zl, dl)
            return g 
        return f

    else: 

        # Factor 
        #
        #     S = H + Df'*Dnl^-2*Df + G'*Dl^-2*G  
        #     K = A*S^-1*A'
        #
        # using dense (L*L') or sparse (P'*L*L'*P) Cholesky 
        # factorizations.  To solve the system, solve
        #
        #     K*y = A * S^{-1} * (bx + Df'*Dnl^-2*bznl + G'*Dl^-2*bzl) 
        #          - by
        #     S*x = bx + Df'*Dnl.^-2* bznl + G'*Dl^-2*bzl - A'*y
        #     Dnl^2 * znl = Df*x - bznl
        #     Dl^2 * zl  = G*x - bzl
        #    
        # If S turns out to be singular in the first factorization,
        # then switch to the following method.
        #
        # Factor 
        #
        #     S = H + Df'*Dnl^-2*Df + G'*Dl^-2*G  + A'*A
        #     K = A*S^-1*A'
        #
        # using dense (L*L') or sparse (P'*L*L'*P) Cholesky 
        # factorizations.  To solve the system, solve
        #
        #    K*y = A*S^{-1} * (bx + Df'*Dnl^-2*bznl + G'*Dl^-2*bzl + 
        #          A'*by) - by
        #    S*x = bx + Df'*Dnl.^-2*bznl + G'*Dl^-2*bzl + A'*by - A'*y
        #    Dnl^2*znl = Df*x - bznl
        #    Dl^2*zl = G*x - bzl.

        if type(G) is matrix: 
            Gsc = matrix(0.0, G.size) 
        else:
            Gsc = spmatrix(0.0, G.I, G.J, G.size) 
        if type(Df) is matrix: 
            Dfsc = matrix(0.0, Df.size) 
        else:
            Dfsc = spmatrix(0.0, Df.I, Df.J, Df.size) 
        F = {'firstcall': True, 'singular': False}
        if matrix in (type(H), type(G), type(Df)):
            F['S'] = matrix(0.0, (n,n))
            F['K'] = matrix(0.0, (p,p))
        else: 
            F['S'] = spmatrix([], [], [], (n,n), 'd')
            F['Sf'] = None
            if type(A) is matrix:
                F['K'] = matrix(0.0, (p,p))
            else:
                F['K'] = spmatrix([], [], [], (p,p), 'd')

        def f(H, Df, dnl, dl):
            base.gemm(spmatrix(dnl, range(mnl), range(mnl), tc='d'), Df,
                Dfsc, partial=True)
            base.gemm(spmatrix(dl, range(ml), range(ml), tc='d'), G, 
                Gsc, partial=True)
            if F['firstcall']:
                F['firstcall'] = False
                F['S'][:,:] = H
                base.syrk(Dfsc, F['S'], trans='T', beta=1.0) 
                base.syrk(Gsc, F['S'], trans='T', beta=1.0) 
                try:
                    if type(F['S']) is matrix: 
                        lapack.potrf(F['S']) 
                    else:
                        F['Sf'] = cholmod.symbolic(F['S'])
                        cholmod.numeric(F['S'], F['Sf'])
                except ArithmeticError:
                    F['singular'] = True 
                    if type(A) is matrix and type(F['S']) is spmatrix:
                        F['S'] = matrix(0.0, (n,n))
                    F['S'][:,:] = H 
                    base.syrk(Dfsc, F['S'], trans='T', beta=1.0) 
                    base.syrk(Gsc, F['S'], trans='T', beta=1.0) 
                    base.syrk(A, F['S'], trans='T', beta=1.0) 
                    if type(F['S']) is matrix: 
                        lapack.potrf(F['S']) 
                    else:
                        F['Sf'] = cholmod.symbolic(F['S'])
                        cholmod.numeric(F['S'], F['Sf'])

            else:
                # S := H, but do not remove nonzeros from the zero 
                # pattern of S if S is sparse.
                if type(F['S']) is spmatrix and type(H) is spmatrix:
                    F['S'] *= 0.0
                    F['S'] += H
                else:
                    F['S'][:,:] = H 

                base.syrk(Dfsc, F['S'], trans='T', beta=1.0, 
                    partial=True)
                base.syrk(Gsc, F['S'], trans='T', beta=1.0, 
                    partial=True)
                if F['singular']:
                    base.syrk(A, F['S'], trans='T', beta=1.0, 
                        partial=True) 
                if type(F['S']) is matrix: 
                    lapack.potrf(F['S']) 
                else:
                    cholmod.numeric(F['S'], F['Sf'])

            if type(F['S']) is matrix: 
                # Asct := L^-1*A', factor K = Asct'*Asct.
                if type(A) is matrix: 
                    Asct = A.T
                else: 
                    Asct = matrix(A.T)
                blas.trsm(F['S'], Asct)
                blas.syrk(Asct, F['K'], trans='T')
                lapack.potrf(F['K'])

            else:
                # Asct := L^{-1}*P*A' and factor K = Asct'*Asct.
                if type(A) is matrix:
                    Asct = A.T
                    cholmod.solve(F['Sf'], Asct, sys=7)
                    cholmod.solve(F['Sf'], Asct, sys=4)
                    blas.syrk(Asct, F['K'], trans='T')
                    lapack.potrf(F['K']) 
                else:
                    Asct = cholmod.spsolve(F['Sf'], A.T, sys=7)
                    Asct = cholmod.spsolve(F['Sf'], Asct, sys=4)
                    base.syrk(Asct, F['K'], trans='T')
                    Kf = cholmod.symbolic(F['K'])
                    cholmod.numeric(F['K'], Kf)

            def g(x, y, znl, zl):

                # If not F['singular']:
                # x := L^{-1} * P * (x + Df'*(dnl.^2.*znl) + 
                #      G'*dl.^2.*zl)
                #    = L^{-1} * P * (bx + Df'*Dnl^-2*bznl + 
                #      G'*Dl^-2*bzl)
                #
                # If F['singular']:
                # x := L^{-1} * P * (x + Df'*(dnl.^2.*znl) + 
                #      G'*(dl.^2.*zl) + A'*y)
                #    = L^{-1} * P * (bx + Df'*Dnl^-2*bznl + 
                #      G'*Dl^-2*bzl + A'*by)

                had2(znl, dnl)
                base.gemv(Dfsc, znl, x, trans='T', beta=1.0)
                had2(zl, dl)
                base.gemv(Gsc, zl, x, trans='T', beta=1.0)
                if F['singular']:
                    base.gemv(A, y, x, trans='T', beta=1.0)
                if type(F['S']) is matrix:
                    blas.trsv(F['S'], x)
                else:
                    cholmod.solve(F['Sf'], x, sys=7)
                    cholmod.solve(F['Sf'], x, sys=4)

                # y := K^{-1} * (Asc*x - y)
                #    = K^{-1} * (A * S^{-1} * (bx + Df'*Dnl^-2*bznl +
                #      G'*Dl^-2*bzl) - by)  (if not F['singular'])
                #    = K^{-1} * (A * S^{-1} * (bx + Df'*Dnl^-2*bznl +
                #      G'*Dl^-2*bzl + A'*by) - by)  (if F['singular'])

                base.gemv(Asct, x, y, trans='T', beta=-1.0)
                if type(F['K']) is matrix:
                    lapack.potrs(F['K'], y)
                else:
                    cholmod.solve(Kf, y)

                # x := P' * L^{-T} * (x - Asc'*y)
                #    = S^{-1} * (bx + Df'*Dnl^-2*bznl + G'*Dl^-2*bzl 
                #      - A'*y) (if not F['singular'])  
                #    = S^{-1} * (bx + Df'*Dnl^-2*bznl + G'*Dl^-2*bzl
                #      + A'*by - A'*y) (if F['singular'])

                base.gemv(Asct, y, x, alpha=-1.0, beta=1.0)
                if type(F['S']) is matrix:
                    blas.trsv(F['S'], x, trans='T')
                else:
                    cholmod.solve(F['Sf'], x, sys=5)
                    cholmod.solve(F['Sf'], x, sys=8)

                # znl := dnl .* (Dfsc*x - znl) 
                #      = Dnl^-2 * (Df*x - bznl)
                base.gemv(Dfsc, x, znl, beta=-1.0)
                had2(znl, dnl)

                # zl := dl .* (Gsc*x - zl) 
                #     = Dl^-2 * (G*x - bzl)
                base.gemv(Gsc, x, zl, beta=-1.0)
                had2(zl, dl)

            return g

        return f
