/* CynthiuneController.m - this file is part of Cynthiune
 *
 * Copyright (C) 2003, 2004  Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <AppKit/AppKit.h>

#import "CynthiuneBundle.h"
#import "Format.h"
#import "FormatTester.h"
#import "GeneralPreference.h"
#import "CynthiuneController.h"
#import "MailService.h"
#import "Song.h"
#import "PlaylistController.h"
#import "Preference.h"
#import "PreferencesController.h"

#ifdef __MACOSX__ /* NSStandardLibraryPaths */
#import "utils.h"
#endif /* __MACOSX */

#define LOCALIZED(X) NSLocalizedString (X, nil)

static void
localizeMenu (NSMenu *menu)
{
  id <NSMenuItem> menuItem;
  unsigned int count, max;

  [menu setTitle: LOCALIZED ([menu title])];

  max = [menu numberOfItems];
  for (count = 0; count < max; count++)
    {
      menuItem = [menu itemAtIndex: count];
      if (menuItem && ![menuItem isSeparatorItem])
        {
          [menuItem setTitle: LOCALIZED ([menuItem title])];
          if ([menuItem hasSubmenu])
            localizeMenu ([menuItem submenu]);
        }
    }
}

@implementation CynthiuneController : NSObject

+ (void) initialize
{
  NSArray *sendTypes;

  sendTypes = [NSArray arrayWithObject: NSFilenamesPboardType];
  [NSApp registerServicesMenuSendTypes: sendTypes
         returnTypes: nil];
}

- (void) awakeFromNib
{
  [appWindow setMiniwindowImage: [NSImage imageNamed: @"Cynthiune"]];

  localizeMenu ([NSApp mainMenu]);

//   if (![[MailService instance] isAvailable])
//     [bugReportMenuItem setAction: NULL];

  [self _parseArguments];
  [playlistController initializeWidgets];
  [self _initWindowsPosition];

  [playlistWindow setTitle: LOCALIZED (@"Playlist")];
  [playlistWindow setExcludedFromWindowsMenu: YES];

  [playlistSwitch setImage: [NSImage imageNamed: @"playlist"]];
  [playlistSwitch setAlternateImage: [NSImage imageNamed: @"playlist-pushed"]];
  [playlistSwitch setState: playlistWindowIsVisible];
}

- (unsigned int) _computeDeltaXOfFrame: (NSRect) mainFrame
                             withFrame: (NSRect) aFrame
{
  unsigned int cond;
  float a1, a2, x1, x2;

  cond = 0;
  x1 = aFrame.origin.x;
  x2 = aFrame.origin.x + aFrame.size.width;
  a1 = mainFrame.origin.x;
  a2 = mainFrame.origin.x + mainFrame.size.width;

  if (x1 <= (a1 + STICK_DELTA)
      && x1 >= (a1 - STICK_DELTA))
    {
      cond = 2;
      deltaX = 0.0;
    }
  else if (x2 <= (a2 + STICK_DELTA)
           && x2 >= (a2 - STICK_DELTA))
    {
      cond = 3;
      deltaX = x1 - x2 + a2 - a1;
    }
  else if (x2 <= (a1 + STICK_DELTA)
           && x2 >= (a1 - STICK_DELTA))
    {
      cond = 4;
      deltaX = x1 - x2 - 2.0;
    }
  else if (x1 <= (a2 + STICK_DELTA)
           && x1 >= (a2 - STICK_DELTA))
    {
      cond = 5;
      deltaX = a2 - a1 + 2.0;
    }
  else if ((x1 > a1 && x1 < a2)
           || (x2 > a1 && x2 < a2)
           || (x1 < a1 && x2 > a2))
    {
      cond = 1;
      deltaX = x1 - a1;
    }

  return cond;
}

- (unsigned int) _computeDeltaYOfFrame: (NSRect) mainFrame
                             withFrame: (NSRect) aFrame 
{
  unsigned int cond;
  float b1, b2, y1, y2;

  cond = 0;
  y1 = aFrame.origin.y;
  y2 = aFrame.origin.y + aFrame.size.height;
  b1 = mainFrame.origin.y;
  b2 = mainFrame.origin.y + mainFrame.size.height;

  if (y1 <= (b1 + STICK_DELTA)
      && y1 >= (b1 - STICK_DELTA))
    {
      cond = 2;
// #ifdef __MACOSX__
      deltaY = 0.0;
// #else
//       deltaY = -1.0;
// #endif
    }
  else if (y2 <= (b2 + STICK_DELTA)
           && y2 >= (b2 - STICK_DELTA))
    {
      cond = 3;
      deltaY = y1 - y2 + b2 - b1;
    }
  else if (y2 <= (b1 + STICK_DELTA)
           && y2 >= (b1 - STICK_DELTA))
    {
      cond = 4;
// #ifdef __MACOSX__
      deltaY = y1 - y2 - 1.0;
// #else
//       deltaY = y1 - y2 - 2.0;
// #endif
    }
  else if (y1 <= (b2 + STICK_DELTA)
           && y1 >= (b2 - STICK_DELTA))
    {
      cond = 5;
      deltaY = b2 - b1 + 1.0;
    }
  else if ((y1 > b1 && y1 < b2)
           || (y2 > b1 && y2 < b2)
           || (y1 < b1 && y2 > b2))
    {
      cond = 1;
      deltaY = y1 - b1;
    }

  return cond;
}

- (void) _recheckIfIsStuck
{
  unsigned int condX, condY;
  NSRect mainFrame, playlistFrame;

  mainFrame = [appWindow frame];
  playlistFrame = [playlistWindow frame];

  condX = [self _computeDeltaXOfFrame: mainFrame
                withFrame: playlistFrame];
  condY = [self _computeDeltaYOfFrame: mainFrame
                withFrame: playlistFrame];
  isStuck = (condX && condY
             && (condX > 3 || condY > 3));
}

- (void) _initWindowsPosition
{
  GeneralPreference *generalPreference;

  generalPreference = [GeneralPreference instance];
  if ([generalPreference saveWindowsInformation])
    {
      [generalPreference restoreInformation: appWindow
                         forWindow: @"MainWindow"];
      [generalPreference restoreInformation: playlistWindow
                         forWindow: @"Playlist"];
      [appWindow makeKeyAndOrderFront: self];
      [playlistWindow makeKeyAndOrderFront: self];
      playlistWindowIsVisible = YES;
      [self _recheckIfIsStuck];
    }
  else
    {
      [appWindow center];
      [appWindow makeKeyAndOrderFront: self];
      playlistWindowIsVisible = NO;
    }
}

- (id) init
{
  if ((self = [super init]))
    {
      [self loadBundles];
    }

  return self;
}

- (void) _parseArguments
{
  NSArray *arguments;
  NSString *currArg;
  Song *song;
  unsigned int count, max;

  arguments = [[NSProcessInfo processInfo] arguments];
  max = [arguments count];

  for (count = 1; count < max; count++)
    {
      currArg = [arguments objectAtIndex: count];
      if (![currArg hasPrefix: @"-"])
        {
          song = [Song songWithFilename: currArg];
          [playlistController openSongFromNSApp: song];
        }
    }
}

/* this seems to crash for the moment... */
// - (void) updateMiniwindowTitle: (NSString *) aTitle
// {
//   [appWindow setMiniwindowTitle: aTitle];
// }

- (void) _registerClass: (Class) class
{
  PreferencesController *preferencesController;
  FormatTester *formatTester;
  GeneralPreference *generalPreference;

  preferencesController = [PreferencesController preferencesController];
  formatTester = [FormatTester formatTester];
  generalPreference = [GeneralPreference instance];

  if ([class conformsToProtocol: @protocol(Preference)])
    [preferencesController registerPreferenceClass: class];
  if ([class conformsToProtocol: @protocol(Format)])
    [formatTester registerFormatClass: class];
  if ([class conformsToProtocol: @protocol(Output)])
    [generalPreference registerOutputClass: class];
}

- (void) _registerArrayOfClasses: (NSArray *) classes
{
  Class class;
  unsigned int count, max;

  max = [classes count];
  for (count = 0; count < max; count++)
    {
      class = [classes objectAtIndex: count];
      [self _registerClass: class];
    }
}

- (void) _registerBundle: (NSBundle *) bundle
{
  Class mainClass;
  NSMutableArray *classes;

  mainClass = [bundle principalClass];
  if (mainClass
      && [mainClass conformsToProtocol: @protocol (CynthiuneBundle)])
    {
      classes = [NSMutableArray arrayWithArray: [mainClass bundleClasses]];
      [self _registerArrayOfClasses: classes];
    }
}

- (void) loadBundlesForPath: (NSString *) aPath
            withFileManager: (NSFileManager *) aFileManager
{
  int count, max;
  NSString *aString, *path;
  NSBundle *aBundle;
  NSArray *allFiles;

  allFiles = [aFileManager directoryContentsAtPath: aPath];
  max = [allFiles count];

  for (count = 0; count < max; count++)
    {
      aString = [allFiles objectAtIndex: count];

      path = [NSString stringWithFormat: @"%@/%@", aPath, aString];
      aBundle = [NSBundle bundleWithPath: path];
      if (aBundle)
        [self _registerBundle: aBundle];
    }
}

- (void) loadBundlesInSystemDirectories: (NSFileManager *) aFileManager
{
  int count, max;
  NSArray *systemPaths;
  NSString *aPath;

  systemPaths = NSStandardLibraryPaths ();
  max = [systemPaths count];
  for (count = 0; count < max; count++)
    {
      aPath = [NSString stringWithFormat: @"%@/Cynthiune",
                        [systemPaths objectAtIndex: count]];
      [self loadBundlesForPath: aPath withFileManager: aFileManager];
    }
}

#ifdef GNUSTEP
/* GNUstep */
- (void) loadBundlesInLocalDirectory: (NSFileManager *) aFileManager
{
  int count, max;
  NSString *sourceDir, *extBundlesDir, *aDir, *aFile;
  NSArray *allFiles;

  sourceDir = [[[NSBundle mainBundle] bundlePath]
                stringByDeletingLastPathComponent];

  extBundlesDir = [sourceDir stringByAppendingPathComponent: @"Bundles"];
  allFiles = [aFileManager directoryContentsAtPath: extBundlesDir];

  max = [allFiles count];
  for (count = 0; count < max; count++)
    {
      aFile = [allFiles objectAtIndex: count];
      aDir = [NSString stringWithFormat: @"%@/%@", extBundlesDir, aFile];
      [self loadBundlesForPath: aDir withFileManager: aFileManager];
    }
}
#else /* GNUSTEP */
/* MacOSX */
- (void) loadBundlesInLocalDirectory: (NSFileManager *) aFileManager
{
  NSString *bundlePath;

  bundlePath = [[NSBundle mainBundle] bundlePath];
  [self
    loadBundlesForPath: [bundlePath
                          stringByAppendingPathComponent: @"Contents/PlugIns"]
    withFileManager: aFileManager];
}
#endif /* GNUSTEP */

- (void) loadBundles
{
  NSFileManager *aFileManager;

  aFileManager = [NSFileManager defaultManager];

  [self loadBundlesInLocalDirectory: aFileManager];
  [self loadBundlesInSystemDirectories: aFileManager];
}

- (void) openFile: (id) anObject
{
  [playlistController addSongFromNSApp: self];
}

- (void) preferencesWindow: (id) anObject
{
  [[PreferencesController preferencesController] loadPreferencesWindow];
}

- (void) sendABugReport: (id) sender
{
  [[MailService instance] composeBugReport];
}

/* as delegate */

- (BOOL) application: (NSApplication *) anApp
	    openFile: (NSString *) aFilename
{
  Song *song;

  song = [Song songWithFilename: aFilename];

  return [playlistController openSongFromNSApp: song];
}

- (void) togglePlaylistWindow: (id) sender
{
  [self setPlaylistWindowVisible: [sender state]];
}

- (void) setPlaylistWindowVisible: (BOOL) isVisible
{
  if (isVisible)
    {
      if (!playlistWindowIsVisible)
        {
          [playlistWindow makeKeyAndOrderFront: self];
          playlistWindowIsVisible = YES;
        }
    }
  else
    {
      if (playlistWindowIsVisible)
        {
          [playlistWindow orderOut: self];
          playlistWindowIsVisible = NO;
        }
    }
}

// notifications
- (void) windowWillClose: (NSNotification *) aNotification
{
  GeneralPreference *generalPreference;
  static BOOL closing = NO;

  if (!closing)
    {
      closing = YES;

      generalPreference = [GeneralPreference instance];
      if ([generalPreference saveWindowsInformation])
        {
          [generalPreference saveInformation: appWindow
                             forWindow: @"MainWindow"];
          [generalPreference saveInformation: playlistWindow
                             forWindow: @"Playlist"];
        }

      [NSApp terminate: [aNotification object]];
    }
}

- (void) windowDidDeminiaturize: (NSNotification *) aNotification
{
  if (playlistWindowIsVisible)
    [playlistWindow orderFront: self];
}

- (void) windowDidMiniaturize: (NSNotification *) aNotification
{
  if (playlistWindowIsVisible)
    [playlistWindow orderOut: self];
}

- (void) repositionPlaylistWindow
{
  NSPoint newOrigin;
  NSRect mainFrame;

  mainFrame = [appWindow frame];

  newOrigin = mainFrame.origin;
  newOrigin.x += deltaX;
  newOrigin.y += deltaY;

  [playlistWindow setFrameOrigin: newOrigin];
}

- (void) _appWindowDidMove
{
  NSRect frame;
  GeneralPreference *generalPreference;
  static BOOL _inited = NO;

  generalPreference = [GeneralPreference instance];
  if (!_inited)
    {
      _inited = YES;

      if (![generalPreference saveWindowsInformation])
        {
          frame = [playlistWindow frame];
          isStuck = YES;
          deltaX = 0.0;
          deltaY = -frame.size.height - 2.0;
          [self repositionPlaylistWindow];
        }
    }
  else
    if ([generalPreference windowsAreSticky] && isStuck)
      [self repositionPlaylistWindow];
}

- (void) _playlistWindowDidMove
{
  GeneralPreference *generalPreference;

  generalPreference = [GeneralPreference instance];
  if ([generalPreference windowsAreSticky])
    {
      [self _recheckIfIsStuck];
      if (isStuck)
        [self repositionPlaylistWindow];
    }
}

- (void) windowDidMove: (NSNotification *) aNotification
{
  id object;

  object = [aNotification object];
  if (object == appWindow)
    [self _appWindowDidMove];
  else if (object == playlistWindow)
    [self _playlistWindowDidMove];
  else
    NSLog (@"%s(%s): unexpected notification object:\n  %@",
           __FILE__, __LINE__, [object description]);
}

- (void) windowDidResize: (NSNotification *) aNotification
{
  GeneralPreference *generalPreference;

  generalPreference = [GeneralPreference instance];
  if ([aNotification object] == playlistWindow
      && [generalPreference windowsAreSticky]
      && isStuck)
    [self _recheckIfIsStuck];
}

// /* Services */
- (id) validRequestorForSendType: (NSString *)sendType
                      returnType: (NSString *)returnType
{
  return [playlistController validRequestorForSendType: sendType
                             returnType: returnType];
}

@end
