/* PlaylistFile.m - this file is part of Cynthiune
 *
 * Copyright (C) 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <Wolfgang@Contre.COM>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import "Playlist.h"
#import "PlaylistFile.h"
#import "Song.h"
#import "utils.h"

@implementation PlaylistFile : NSObject

- (PlaylistFile *) init
{
  if ((self = [super init]))
    {
      _filename = nil;
      _playlist = nil;
    }

  return self;
}

- (void) setFilename: (NSString *) aFilename
{
  SET (_filename, aFilename);
}

- (NSString *) filename
{
  return _filename;
}

- (void) setPlaylist: (Playlist *) playlist
{
  SET (_playlist, playlist);
}

- (Playlist *) playlist
{
  return _playlist;
}

- (NSArray *) _parseM3UFile: (NSArray *) fileLines
{
  NSMutableArray *playlist;
  NSString *currLine;
  int count, max;

  playlist = [NSMutableArray new];
  max = [fileLines count];
  for (count = 0; count < max; count++)
    {
      currLine = [fileLines objectAtIndex: count];
      if (![currLine hasPrefix: @"#"])
        [playlist addObject: currLine];
    }

  return playlist;
}

- (NSArray *) _parsePLSFile: (NSArray *) fileLines
{
  NSMutableArray *playlist;
  NSString *currLine, *newString;
  int count, max;
  const char *filePtr, *newStr;

  playlist = [NSMutableArray new];
  max = [fileLines count];
  for (count = 0; count < max; count++)
    {
      currLine = [fileLines objectAtIndex: count];
      newStr = [currLine cString];
      if (!strncasecmp (newStr, "file", 4))
        {
          filePtr = strchr (newStr, '=') + 1;
          newString = [NSString stringWithCString: filePtr];
          [playlist addObject: newString];
        }
    }

  return playlist;
}

- (NSString *) _M3UContentWithArray: (NSArray *) anArray
{
  unsigned int count, max;
  Song *aSong;
  NSString *ctString;

  max = [anArray count];
  ctString = @"#EXTM3U\r\n";
  for (count = 0; count < max; count++)
    {
      aSong = [anArray objectAtIndex: count];
      ctString = [ctString
                   stringByAppendingFormat: @"#EXTINF:%@,%@\r\n%@\r\n",
                   [aSong duration], [aSong playlistRepresentation],
                   [aSong filename]];
    }

  return ctString;
}

- (NSString *) _PLSContentWithArray: (NSArray *) anArray
{
  unsigned int count, max;
  Song *aSong;
  NSString *ctString;

  max = [anArray count];
  ctString = @"[playlist]\r\n";
  ctString = [ctString
               stringByAppendingFormat: @"NumberOfEntries=%d\r\n", max];
  for (count = 0; count < max; count++)
    {
      aSong = [anArray objectAtIndex: count];
      ctString = [ctString stringByAppendingFormat: @"File%d=%@\r\n",
                           count + 1, [aSong filename]];
    }

  return ctString;
}

- (void) _fillPlaylistWithArray: (NSArray *) anArray
{
  Song *newSong;
  unsigned int count, max;

  max = [anArray count];
  for (count = 0; count < max; count++)
    {
      newSong = [Song songWithFilename: [anArray objectAtIndex: count]];
      [_playlist appendSong: newSong];
    }
}

- (void) load
{
  NSArray *playlistArray, *fileLines;
  NSString *extension;

  if (!_filename)
    NSLog (@"No filename defined.");

  fileLines = linesOfFile (_filename);

  if (fileLines)
    {
      extension = [_filename pathExtension];

      if ([extension isEqualToString: @"m3u"])
        playlistArray = [self _parseM3UFile: fileLines];
      else if ([extension isEqualToString: @"pls"])
        playlistArray = [self _parsePLSFile: fileLines];
      else
        playlistArray = nil;
    }
  else
    playlistArray = nil;

  if (playlistArray)
    {
      if (!_playlist)
        _playlist = [Playlist new];
      [self _fillPlaylistWithArray: playlistArray];
      [playlistArray release];
    }
}

- (NSString *) _playlistFileContents
{
  NSArray *anArray;
  NSString *contents, *extension;

  anArray = [_playlist getSongs];
  contents = nil;
  extension = [_filename pathExtension];

  if ([extension isEqualToString: @"m3u"])
    contents = [self _M3UContentWithArray: anArray];
  else if ([extension isEqualToString: @"pls"])
    contents = [self _PLSContentWithArray: anArray];
  else
    NSLog (@"unknown extension for playlists: '%@'", extension);

  return contents;
}

- (void) save
{
  NSData *fileContents;
  NSString *contents;

  if (!_filename)
    NSLog (@"No filename defined.");
  if (!_playlist)
    NSLog (@"No playlist defined.");

  contents = [self _playlistFileContents];
  if (contents)
    {
      fileContents = [NSData dataWithBytes: [contents cString]
                             length: [contents length]];
      if (![[NSFileManager defaultManager]
             createFileAtPath: _filename
             contents: fileContents
             attributes: nil])
        NSLog (@"PlaylistFile: unable to write '%@'...", _filename);
    }
}

- (void) dealloc
{
  RELEASEIFSET (_filename);
  RELEASEIFSET (_playlist);

  [super dealloc];
}

@end
