/* Esound.m - this file is part of Cynthiune
 *
 * Copyright (C) 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _REENTRANT
#define _REENTRANT 1
#endif

#import <AppKit/AppKit.h>
#import <unistd.h>
#import <esd.h>

#import <CynthiuneBundle.h>
#import <Format.h>
#import <Player.h>
#import <Preference.h>
#import <Song.h>

#import "Esound.h"

#define NSLocalLocalizedString(key, comment) \
[[NSBundle bundleForClass: [Esound class]] localizedStringForKey:(key) \
                                           value: @"" \
                                           table: nil]
#define _b(X) NSLocalLocalizedString(X, nil)

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

static NSLock *lock;

@implementation EsoundPlayerThread : NSObject

- (id) init
{
  if ((self = [super init]))
    {
      totalBytes = 0;
      bytesPerSec = 0;

      stream = nil;
      streamToRelease = nil;
    }

  return self;
}

- (void) setStream: (id <Format>) aStream
{
  if (stream)
    {
      while (streamToRelease);
      streamToRelease = stream;
    }

  stream = aStream;
  totalBytes = 0;
  /* 2 = 16 bits / 8 */
  if (stream)
    bytesPerSec = [stream readChannels] * [stream readRate] * 2;
}

- (void) seek: (unsigned int) aPos
{
  if (stream)
    {
      [lock lock];
      [stream seek: aPos];
      totalBytes = aPos * bytesPerSec;
      [lock unlock];
    }
  else
    NSLog (@"seeking within an inactive stream?");
}

- (void) handleReadProblem: (int) bytesRead
{
}

- (int) getSeconds
{
  return (totalBytes / bytesPerSec);
}

- (void) threadWillExit: (NSDictionary *) aDict
{
  id parent = [aDict valueForKey: @"parent"];

  NSDebugLog (@"player thread exiting...\n");
  [parent threadWillExit];
}

- (void) songEnded: (NSDictionary *) aDict
{
  [[aDict valueForKey: @"parent"] postSongEndedNotification];
}

#if (BYTE_ORDER != __LITTLE_ENDIAN)
- (void) _invertBufferBytes
{
  unsigned int count;
  char tmpChar;
  char *ptr;

  count = 0;
  while (count < bytesRead)
    {
      ptr = buffer + count;
      tmpChar = *ptr;
      *ptr = *(ptr + 1);
      *(ptr + 1) = tmpChar;
      count += 2;
    }
}
#endif

- (void) flushOutputBuffer: (NSDictionary *) dict;
{
  id parent;
  unsigned int errorCount;
  int esdSock;

  errorCount = 0;
  esdSock = [[dict valueForKey: @"dsp"] intValue];
#if (BYTE_ORDER != __LITTLE_ENDIAN)
  [self _invertBufferBytes];
#endif

  while ((esdSock < 0
          || write (esdSock, buffer, bytesRead) == -1)
         && errorCount < 3)
    {
      NSLog (@"write error on esdSock, trying again...");
      parent = [dict valueForKey: @"parent"];
      [parent _esdReinit];
      [NSThread sleepUntilDate: [NSDate dateWithTimeIntervalSinceNow: 1.0]];
      esdSock = [[dict valueForKey: @"dsp"] intValue];
      errorCount++;

      if (errorCount == 3)
        [parent stopPlayLoop];
    }

  totalBytes += bytesRead;
}

- (void) playerThread: (id) dict
{
  NSAutoreleasePool *arp;

  arp = [NSAutoreleasePool new];

  if ([lock lockBeforeDate: [NSDate dateWithTimeIntervalSinceNow: 5.0]])
    {
      [lock unlock];

      while (stream
             && [[dict valueForKey: @"running"] isEqualToString: @"yes"])
        {
          [lock lock];
          bytesRead = [stream readNextChunk: buffer withSize: BUF_SIZE];
          [lock unlock];

          if ([[dict valueForKey:@"muted"] isEqualToString:@"yes"])
            memset (buffer, 0, bytesRead);

          if (bytesRead > 0)
            [self flushOutputBuffer: dict];
          else if (bytesRead == 0)
            {
              [self songEnded: dict];
              while (totalBytes
                     && [[dict valueForKey:@"running"] isEqualToString:@"yes"])
                {};
            }
          else
            break;

          while ([[dict valueForKey:@"paused"] isEqualToString:@"yes"])
            [NSThread sleepUntilDate:
                        [NSDate dateWithTimeIntervalSinceNow: 1.0]];

          if (streamToRelease)
            {
              [streamToRelease streamClose];
              [streamToRelease release];
              streamToRelease = nil;
            }
        }
    }
  else
    NSLog(@"Failed to obtain lock");

  if (!stream)
    [self songEnded: dict];

  [self performSelectorOnMainThread: @selector (threadWillExit:)
        withObject: dict
        waitUntilDone: NO];

  [arp release];
  [NSThread exit];
}

- (void) dealloc
{
  if (stream)
    {
      [stream streamClose];
      [stream autorelease];
    }

  [super dealloc];
}

@end

@implementation Esound : PlayerBase

+ (NSArray *) bundleClasses
{
  return [NSArray arrayWithObjects: [self class],[EsoundPreference class],nil];
}

- (id) init
{
  if ((self = [super init]))
    {
      _thread = nil;
      esdSock = 0;
      dict = [[NSMutableDictionary alloc] initWithCapacity: 5];
      [dict setObject: @"no" forKey: @"running"];
      [dict setObject: @"no" forKey: @"paused"];
      [dict setObject: @"no" forKey: @"muted"];
      [dict setObject: self forKey: @"parent"];
    }

  return self;
}

- (id) initWithSong: (Song *) aSong
{
  self = [self init];
  [self setSong: aSong];

  return self;
}

- (void) setFormat: (int) newFormat
{
}

- (void) setSpeed: (int) newSpeed
{
  speed = newSpeed;
}

- (void) setChannels: (int) newChannels
{
  if (newChannels == 1)
    channels = ESD_MONO;
  else if (newChannels == 2)
    channels = ESD_STEREO;
  else
    NSLog (@"Invalid number of channels specified!\n");
}

- (void) audioInit
{
}

- (void) _esdReinit
{
  NSString *hostString;
  EsoundPreference *esoundPreference;

  if (esdSock)
    close (esdSock);

  esoundPreference = [EsoundPreference instance];

  if ([esoundPreference socketIsTCP])
    {
      hostString = [esoundPreference tcpHostConnectString];
      esdSock = esd_play_stream ((ESD_BITS16 | ESD_STREAM
                                  | ESD_PLAY | channels),
                                 speed, [hostString cString],
                                 "Cynthiune stream");
    }
  else
    esdSock = esd_play_stream_fallback ((ESD_BITS16 | ESD_STREAM
                                         | ESD_PLAY | channels),
                                        speed, NULL,
                                        "Cynthiune stream");

  [dict setObject: [NSNumber numberWithInt: esdSock] forKey: @"dsp"];
}

- (Song *) song
{
  return song;
}

- (void) setSong: (Song *) aSong
{
  id stream;

  song = aSong;

  if (aSong)
    {
      stream = [aSong openStreamForSong];

      if (stream)
        {
          [self setSpeed: [stream readRate]];
          [self setChannels: [stream readChannels]];
          [self _esdReinit];
          if (esdSock > -1)
            {
              if (!_thread)
                _thread = [[EsoundPlayerThread new] init];
              [_thread setStream: stream];

              if ([[dict valueForKey: @"paused"] isEqualToString: @"yes"])
                {
                  [dict setObject: @"no" forKey: @"paused"];
                  [self postResumedNotification];
                }
            }
          else
            {
              [stream autorelease];
              NSLog (@"Invalid file descriptor. We won't start.");
            }
        }
      else
        {
          song = nil;
          [self postSongEndedNotification];
        }
    }
  else
    {
      if (_thread)
        [_thread setStream: nil];
      if (esdSock)
        close (esdSock);
    }
}

/* Launch play loop */
- (void) startPlayerThread
{
  if (![self isRunning])
    {
      [dict setObject: @"yes" forKey: @"running"];
      [dict setObject: @"no" forKey: @"paused"];

      lock = [NSLock new];
      [lock lock];
      [NSThread detachNewThreadSelector: @selector (playerThread:)
                toTarget: _thread
                withObject: dict];
      [NSThread sleepUntilDate: [NSDate dateWithTimeIntervalSinceNow: 1.0]];
      [lock unlock];
    }
}

- (void) setPaused: (BOOL) paused
{
  if (paused)
    {
      [dict setObject: @"yes" forKey: @"paused"];
      [self postPausedNotification];
    }
  else
    {
      [dict setObject: @"no" forKey: @"paused"];
      [self postResumedNotification];
    }
}

- (BOOL) paused
{
  return [[dict valueForKey:@"paused"] isEqualToString:@"yes"];
}

- (void) setMuted: (BOOL) muted
{
  if (muted)
    [dict setObject: @"yes" forKey: @"muted"];
  else
    [dict setObject: @"no" forKey: @"muted"];
}

- (BOOL) muted
{
  return [[dict valueForKey:@"muted"] isEqualToString:@"yes"];
}

- (void) startPlayLoop
{
  if (!esdSock)
    [self _esdReinit];

  if (esdSock > -1)
    {
      if ([[dict valueForKey:@"running"] isEqualToString:@"no"])
        {
          [self startPlayerThread];
          [self postPlayingNotification];
        }
    }
  else
    NSLog (@"Invalid file descriptor. We won't start.");
}

- (void) stopPlayLoop
{
  [dict setObject:@"no" forKey:@"running"];
  close (esdSock);
  esdSock = 0;
  [self postStoppedNotification];
}

- (void) threadWillExit
{
  [_thread autorelease];
  _thread = nil;
}

- (BOOL) isRunning
{
  return ([[dict valueForKey:@"running"] isEqualToString:@"yes"]);
}

- (int) timer
{
  return [_thread getSeconds];
}

- (void) seek: (unsigned int) aPos
{
  if (_thread)
    [_thread seek: aPos];
  else
    NSLog (@"seeking within an inactive thread?");
}

- (void) dealloc
{
  if (host)
    [host autorelease];
  [super dealloc];
}

@end

@implementation EsoundPreference : NSObject

// Preference protocol
+ (EsoundPreference *) instance
{
  static EsoundPreference *singleton = nil;

  if (!singleton)
    singleton = [[self alloc] _init];

  return singleton;
}

- (EsoundPreference *) _init
{
  NSDictionary *tmpDict;

  if ((self = [super init]))
    {
      tmpDict = [[NSUserDefaults standardUserDefaults]
                  dictionaryForKey: @"Esound"];
      preference = [NSMutableDictionary dictionaryWithDictionary: tmpDict];
      [preference retain];
      defaultsInited = NO;
    }

  return self;
}

- (id) preferenceSheet
{
  NSView *aView;

  if (!defaultsInited)
    {
      defaultsInited = YES;
      [self _initDefaults];
    }

  [NSBundle loadNibNamed: @"EsoundPreferences"
            owner: self];
  aView = [prefsWindow contentView];
  [aView retain];
  [aView removeFromSuperview];
  [prefsWindow release];

  return aView;
}

- (void) awakeFromNib
{
  [connectionTypeBox setTitle: _b(@"Connection type")];
  [tcpOptionsBox setTitle: _b(@"TCP options")];
  [unixBtn setTitle: _b(@"UNIX socket")];
  [tcpBtn setTitle: _b(@"TCP socket")];
  [hostLabel setStringValue: _b(@"Hostname (or IP)")];
  [portLabel setStringValue: _b(@"Port")];

  if ([[preference objectForKey: @"socketType"] isEqualToString: @"UNIX"])
    [self selectUnixBtn: self];
  else    
    [self selectTcpBtn: self];

  [hostField setStringValue: [preference objectForKey: @"tcpHostname"]];
  [portField setStringValue: [preference objectForKey: @"tcpPort"]];
}

- (void) _initDefaults
{
  NSString *socketType, *tcpHost, *tcpPort;

  socketType = [preference objectForKey: @"socketType"];
  if (!socketType)
    {
      socketType = @"UNIX";
      [preference setObject: socketType
                  forKey: @"socketType"];
    }

  tcpHost = [preference objectForKey: @"tcpHostname"];
  if (!tcpHost)
    {
      tcpHost = @"localhost";
      [preference setObject: tcpHost
                  forKey: @"tcpHostname"];
    }
  tcpPort = [preference objectForKey: @"tcpPort"];
  if (!tcpPort)
    {
      tcpPort = @"16001";
      [preference setObject: tcpPort
                  forKey: @"tcpPort"];
    }
}

- (id) preferenceTitle
{
  return @"Esound"; 
}

- (void) save
{
  NSUserDefaults *defaults;

  defaults = [NSUserDefaults standardUserDefaults];
  [defaults setObject: preference
            forKey: @"Esound"];
  [defaults synchronize];
}

- (BOOL) socketIsTCP
{
  NSString *socketType;

  socketType = [preference objectForKey: @"socketType"];
  return [socketType isEqualToString: @"TCP"];
}

- (NSString *) tcpHostConnectString
{
  NSString *connectString, *tcpHost, *tcpPort;

  tcpHost = [preference objectForKey: @"tcpHostname"];
  tcpPort = [preference objectForKey: @"tcpPort"];
  connectString = [NSString stringWithFormat: @"%@:%@", tcpHost, tcpPort];
  
  return connectString;
}

- (void) selectUnixBtn: (id) sender
{
  NSColor *disabledColor;

  disabledColor = [NSColor controlBackgroundColor];
  [unixBtn setState: 1];
  [tcpBtn setState: 0];
  [hostField setEditable: NO];
  [portField setEditable: NO];
  [hostField setBackgroundColor: disabledColor];
  [portField setBackgroundColor: disabledColor];
  [portField display];
  [hostField display];

  [preference setObject: @"UNIX" forKey: @"socketType"];
}

- (void) selectTcpBtn: (id) sender
{
  NSColor *enabledColor;

  enabledColor = [NSColor textBackgroundColor];
  [unixBtn setState: 0];
  [tcpBtn setState: 1];
  [hostField setEditable: YES];
  [portField setEditable: YES];
  [hostField setBackgroundColor: enabledColor];
  [portField setBackgroundColor: enabledColor];
  [portField display];
  [hostField display];

  [preference setObject: @"TCP"
              forKey: @"socketType"];
}

- (void) hostAction: (id) sender
{
  [preference setObject: [sender stringValue]
              forKey: @"tcpHostname"];
}

- (void) portAction: (id) sender
{
  [preference setObject: [sender stringValue]
              forKey: @"tcpPort"];
}

@end
