/* MacOSXPlayer.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003, 2004  Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <CoreAudio/CoreAudio.h>
#import <CoreAudio/AudioHardware.h>
#import <AudioUnit/AudioUnit.h>
#import <AudioUnit/AudioUnitProperties.h>
#import <AudioToolbox/DefaultAudioOutput.h>
#import <AudioToolbox/AudioConverter.h>

#import <stdio.h>

#import <CynthiuneBundle.h>
#import <Format.h>
#import <Player.h>
#import <Song.h>

#import "MacOSXPlayer.h"

#define BUF_SIZE 4096

static void
updateSeconds (playerref *ref)
{
  int newSeconds;

  newSeconds = (ref->totalBytes / ref->bytesPerSec);

  if (ref->timer < newSeconds)
    ref->timer = newSeconds;
}

static OSStatus
inputCallback (AudioConverterRef inAudioConverter,
	       UInt32* outDataSize,
	       void** outData,
	       playerref* ref)
{
  int bytes_read;
  static char buffer[BUF_SIZE];
  MacOSXPlayer *self;
  NSAutoreleasePool *pool;
  id <Format> stream;

  *outDataSize = 0; 
  *outData = NULL;

  stream = (id) ref->stream;
  if (ref->running && stream)
    {
      pool = [NSAutoreleasePool new];
      bytes_read = [stream readNextChunk: buffer withSize: BUF_SIZE];

      if (bytes_read > 0)
	{
	  if (ref->muted)
	    memset (buffer, 0, bytes_read);
	  ref->totalBytes += bytes_read;
	  updateSeconds (ref);
	  *outDataSize = bytes_read;
	  *outData = buffer;
	}
      else
        {
          self = (MacOSXPlayer *) ref;
          [self postSongEndedNotification];
        }

      [pool release];
    }

  return noErr;
}

static OSStatus
converterRenderer (playerref *player,
                   AudioUnitRenderActionFlags inActionFlags, 
                   const AudioTimeStamp *inTimeStamp, UInt32 inBusNumber, 
                   AudioBuffer *ioData)
{
  UInt32 nFrames;

  UInt32 size = ioData->mDataByteSize;
  AudioConverterFillBuffer (player->converter, inputCallback, player,
			    &size, ioData->mData);

  nFrames = (ioData->mDataByteSize
             / (sizeof (Float32) * ioData->mNumberChannels));

  return noErr;
}

static void
converterPropertyCallback (playerref *player,
                           AudioUnit ci, 
                           AudioUnitPropertyID inID, 
                           AudioUnitScope inScope, 
                           AudioUnitElement inElement)
{
  UInt32 aStreamSize;

  if (inScope == kAudioUnitScope_Output)
    {
      aStreamSize = sizeof (AudioStreamBasicDescription);
      AudioOutputUnitStop (player->outputUnit);
      AudioUnitGetProperty (player->outputUnit,
			    kAudioUnitProperty_StreamFormat,
			    kAudioUnitScope_Output,
			    0,
			    &(player->outputFormat),
			    &aStreamSize);
      AudioConverterDispose (player->converter);
      AudioConverterNew (&(player->inputFormat),
			 &(player->outputFormat),
			 &(player->converter));
      AudioOutputUnitStart (player->outputUnit);
    }
}

@implementation MacOSXPlayer : PlayerBase

+ (NSArray *) bundleClasses
{
  return [NSArray arrayWithObject: [self class]];
}

- (id) init
{
  if ((self = [super init]))
    {
      dsp_opened = NO;
      running = NO;
      dsp_opened = NO;
      muted = NO;
      paused = NO;
      stream = nil;
      timer = 0;
      totalBytes = 0;
      bytesPerSec = 0;
    }

  return self;
}

- (void) dealloc
{
  CloseComponent (outputUnit);

  if (stream)
    {
      [stream streamClose];
      [stream release];
      stream = nil;
    }

  [super dealloc];
}

- (id) initWithSong: (Song *) aSong
{
  self = [self init];

  [self setSong: aSong];

  return self;
}

- (void) setSong: (Song *) aSong
{
  id oldStream;

  if (running)
    {
      [stream streamClose];
      oldStream = stream;
      stream = nil;
      [oldStream autorelease];
    }

  if (aSong)
    {
      totalBytes = 0;
      timer = 0;
      stream = [aSong openStreamForSong];
      if (stream)
        bytesPerSec = [stream readChannels] * [stream readRate] * 2;
    }

  if (song && !stream)
    {
      AudioOutputUnitStop (outputUnit);	
      [self postSongEndedNotification];
      song = nil;
    }
  else
    song = aSong;
}

- (void) audioInit
{
  OSStatus err;
  UInt32 aStreamSize;
  struct AudioUnitInputCallback input;

  if (!initted)
    {
      inputFormat.mSampleRate = 44100;
      inputFormat.mFormatID = kAudioFormatLinearPCM;
      inputFormat.mFormatFlags = (kLinearPCMFormatFlagIsSignedInteger 
				  | kLinearPCMFormatFlagIsPacked);
      inputFormat.mBytesPerPacket = 4;
      inputFormat.mFramesPerPacket = 1;
      inputFormat.mBytesPerFrame = 4;
      inputFormat.mChannelsPerFrame = 2;
      inputFormat.mBitsPerChannel = 16;

      err = OpenDefaultAudioOutput(&outputUnit);
      if (err) return;

      err = AudioUnitInitialize (outputUnit);
      if (err) return;

      input.inputProc = converterRenderer;
      input.inputProcRefCon = self;

      err = AudioUnitSetProperty (outputUnit, 
				  kAudioUnitProperty_SetInputCallback, 
				  kAudioUnitScope_Input,
				  0,
				  &input, 
				  sizeof (input));
      if (err) return;

      aStreamSize = sizeof (AudioStreamBasicDescription);

      err = AudioUnitGetProperty (outputUnit,
				  kAudioUnitProperty_StreamFormat,
				  kAudioUnitScope_Output,
				  0,
				  &outputFormat,
				  &aStreamSize);
      if (err) return;

      err = AudioConverterNew (&inputFormat, 
			       &outputFormat, 
			       &converter);
      if (err) return;

      err = AudioUnitAddPropertyListener(outputUnit,
					 kAudioUnitProperty_StreamFormat,
					 converterPropertyCallback, 0);
      if (err) return;

      dsp_opened = YES;
    }
}

- (void) startPlayLoop
{
  OSStatus err;

  if (!running)
    {
      err = kAudioHardwareNoError;
      [self audioInit];
      err = AudioOutputUnitStart(outputUnit);
      if (err)
	return;
      running = YES;
      paused = NO;
      [self postPlayingNotification];
    }
}

- (void) setPaused: (BOOL) isPaused
{
  if (isPaused && !paused)
    {
      AudioOutputUnitStop (outputUnit);	
      paused = YES;
      [self postPausedNotification];
    }
  else if (!isPaused && paused)
    {
      AudioOutputUnitStart (outputUnit);	
      paused = NO;
      [self postResumedNotification];
    }
}

- (BOOL) paused
{
  return paused;
}

- (void) setMuted: (BOOL) isMuted
{
  muted = isMuted;
}

- (BOOL) muted
{
  return muted;
}

- (void) stopPlayLoop
{
  if (running)
    {
      AudioOutputUnitStop (outputUnit);	
      [stream streamClose];
      [stream autorelease];
      stream = nil;
      running = NO;
      dsp_opened = NO;
      paused = NO;
      [self postStoppedNotification];
    }
}

- (BOOL) isRunning
{
  return running;
}

- (int) timer
{
  return timer;
}

- (void) seek: (unsigned int) aPos
{
  if (stream)
    {
      [stream seek: aPos];
      totalBytes = aPos * bytesPerSec;
    }
  else
    NSLog (@"seeking within an inactive stream?");
}

- (Song *) song
{
  return song;
}

@end
