/* OSS.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _REENTRANT
#define _REENTRANT 1
#endif

#include <AppKit/AppKit.h>

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>

#include <CynthiuneBundle.h>
#include <Format.h>
#include <Player.h>
#include <Song.h>
#include <Preference.h>

#import "OSS.h"

#define NSLocalLocalizedString(key, comment) \
[[NSBundle bundleForClass: [OSS class]] localizedStringForKey: (key) \
                                                        value: @"" \
                                                        table: nil]
#define _b(X) NSLocalLocalizedString(X, nil)

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

static NSLock *lock;

@implementation OSSPlayerThread : NSObject

- (id) init
{
  if ((self = [super init]))
    {
      totalBytes = 0;
      bytesPerSec = 0;

      stream = nil;
      streamToRelease = nil;
    }

  return self;
}

- (void) setStream: (id <Format>) aStream
{
  if (stream)
    {
      while (streamToRelease) {}
      streamToRelease = stream;
    }

  stream = aStream;
  totalBytes = 0;
  /* 2 = 16 bits / 8 */
  if (stream)
    bytesPerSec = [stream readChannels] * [stream readRate] * 2;
}

- (void) seek: (unsigned int) aPos
{
  if (stream)
    {
      [lock lock];
      [stream seek: aPos];
      totalBytes = aPos * bytesPerSec;
      [lock unlock];
    }
  else
    NSLog (@"seeking within an inactive stream?");
}

- (void) handleReadProblem: (int) bytesRead
{
}

- (int) getSeconds
{
  return (totalBytes / bytesPerSec);
}

- (void) threadWillExit: (NSMutableDictionary *) aDict
{
  id parent = [aDict valueForKey: @"parent"];

  NSDebugLog (@"player thread exiting...\n");
  [parent threadWillExit];
}

- (void) songEnded: (NSMutableDictionary *) aDict
{
  [[aDict valueForKey: @"parent"] postSongEndedNotification];
}

- (void) flushOutputBuffer: (int) dspFd;
{
  if (write (dspFd, buffer, bytesRead) == -1)
    {
      fprintf (stderr, "errno: %d\ndspFd: %d\n", errno, dspFd);
      perror ("play:write");
    }
  else
    totalBytes += bytesRead;
}

- (void) playerThread: (id) dict
{
  int dspFd;

  CREATE_AUTORELEASE_POOL (arp);

  if ([lock lockBeforeDate: [NSDate dateWithTimeIntervalSinceNow: 5.0]])
    {
      dspFd = [[dict valueForKey: @"dsp"] fileDescriptor];

      [lock unlock];
 
      while (stream
             && [[dict valueForKey: @"running"] isEqualToString: @"yes"])
        {
          bytesRead = [stream readNextChunk: buffer withSize: BUF_SIZE];

          if ([[dict valueForKey:@"muted"] isEqualToString:@"yes"])
            memset (buffer, 0, bytesRead);

          if (bytesRead > 0)
            [self flushOutputBuffer: dspFd];
          else if (bytesRead == 0)
            {
              [self songEnded: dict];
              while (totalBytes
                     && [[dict valueForKey:@"running"] isEqualToString:@"yes"])
                {};
            }
          else
            break;

          while ([[dict valueForKey:@"paused"] isEqualToString:@"yes"])
            [NSThread sleepUntilDate:
                        [NSDate dateWithTimeIntervalSinceNow: 1.0]];

          if (streamToRelease)
            {
              [streamToRelease streamClose];
              [streamToRelease release];
              streamToRelease = nil;
            }
       }
    }
  else
    NSLog(@"Failed to obtain lock");

  if (!stream)
    [self songEnded: dict];

  [self performSelectorOnMainThread: @selector (threadWillExit:)
        withObject: dict
        waitUntilDone: NO];

  RELEASE (arp);
  [NSThread exit];
}

@end

@implementation OSS : PlayerBase

+ (NSArray *) bundleClasses
{
  return [NSArray arrayWithObjects: [self class],[OSSPreference class],nil];
}

- (id) init
{
  if ((self = [super init]))
    {
      _thread = nil;
      fileHandle = nil;
      dict = [[NSMutableDictionary alloc] initWithCapacity: 5];
      [dict setObject: @"no" forKey:@"running"];
      [dict setObject: @"no" forKey:@"paused"];
      [dict setObject: @"no" forKey:@"muted"];
      [dict setObject: self forKey:@"parent"];
    }

  return self;
}

- (id) initWithSong: (Song *) aSong
{
  self = [self init];
  [self setSong: aSong];

  return self;
}

- (void) setFormat: (int) newFormat
{
  int orig_format, dspFd;

  orig_format = newFormat;
  dspFd = [fileHandle fileDescriptor];

  if (ioctl (dspFd, SNDCTL_DSP_SETFMT, &newFormat) == -1)
    {
      perror("SNDCTL_DSP_SETFMT");
      exit (EXIT_FAILURE);
    }

  if (newFormat != orig_format)
    {
      perror ("format unsupported");
      exit (EXIT_FAILURE);
    }
}

- (void) setSpeed: (int) newSpeed
{  
  int orig_speed, dspFd;

  orig_speed = newSpeed;
  dspFd = [fileHandle fileDescriptor];

  if (ioctl (dspFd, SNDCTL_DSP_SPEED, &newSpeed) == -1)
    {
      perror ("SNDCTL_DSP_SPEED");
      exit (EXIT_FAILURE);
    }

  if (newSpeed < (orig_speed - SPEED_LIMIT)
      || newSpeed > (orig_speed + SPEED_LIMIT))
    {
      fprintf (stderr, "speed return is %d\n", newSpeed);
      perror ("speed unsupported");
      exit (EXIT_FAILURE);
    }
}

- (void) setChannels: (int) newChannels
{
  int orig_channels, dspFd;

  orig_channels = newChannels;
  dspFd = [fileHandle fileDescriptor];

  if (ioctl (dspFd, SNDCTL_DSP_CHANNELS, &newChannels) == -1)
    {
      perror ("SNDCTL_DSP_CHANNELS");
      exit (1);
    }

  if (newChannels != orig_channels)
    {
      fprintf (stderr,
               __FILE__ ": number of channels unsupported (%d != %d)\n",
               newChannels, orig_channels);
      exit (EXIT_FAILURE);
    }
}

- (void) audioInit
{
  int dspFd;

  dspFd = [fileHandle fileDescriptor];

  if (ioctl (dspFd, SNDCTL_DSP_RESET) == -1)
    {
      perror ("SNDCTL_DSP_RESET");
      exit (EXIT_FAILURE);
    }

  [self setFormat: AFMT_S16_LE];
}

- (Song *) song
{
  return song;
}

- (void) setSong: (Song *) aSong
{
  song = aSong;

  if (aSong)
    {
      id stream = [aSong openStreamForSong];

      if (stream)
        {
          speed = [stream readRate];
          channels = [stream readChannels];

          if (!_thread)
            _thread = [[OSSPlayerThread new] init];
          [_thread setStream: stream];

          if ([[dict valueForKey: @"paused"] isEqualToString: @"yes"])
            {
              [dict setObject: @"no" forKey: @"paused"];
              [self postResumedNotification];
            }
        }
      else
        {
          song = nil;
          [self postSongEndedNotification];
        }
    }
  else
    if (_thread)
      [_thread setStream: nil];
}

/* Launch play loop */
- (void) startPlayerThread
{
  if (![self isRunning])
    {
      [self setChannels: channels];
      [self setSpeed: speed];

      [dict setObject: @"yes" forKey: @"running"];
      [dict setObject: @"no" forKey: @"paused"];
      [dict setObject: fileHandle forKey: @"dsp"];

      lock = [NSLock new];
      [lock lock];
      [NSThread detachNewThreadSelector: @selector (playerThread:)
                toTarget: _thread
                withObject: dict];
      [NSThread sleepUntilDate:
                  [NSDate dateWithTimeIntervalSinceNow: 1.0]];
      [lock unlock];
    }
}

- (void) setPaused: (BOOL) paused
{
  if (paused)
    {
      [dict setObject: @"yes" forKey: @"paused"];
      [self postPausedNotification];
    }
  else
    {
      [dict setObject: @"no" forKey: @"paused"];
      [self postResumedNotification];
    }
}

- (BOOL) paused
{
  return [[dict valueForKey:@"paused"] isEqualToString:@"yes"];
}

- (void) setMuted: (BOOL) muted
{
  if (muted)
    [dict setObject: @"yes" forKey: @"muted"];
  else
    [dict setObject: @"no" forKey: @"muted"];
}

- (BOOL) muted
{
  return [[dict valueForKey:@"muted"] isEqualToString:@"yes"];
}

- (void) startPlayLoop
{
  OSSPreference *preference;

  if (!fileHandle)
    {
      preference = [OSSPreference instance];
      fileHandle = [NSFileHandle
                     fileHandleForWritingAtPath: [preference dspDevice]];
      [self audioInit];
    }

  if ([[dict valueForKey:@"running"] isEqualToString:@"no"])
    {
      [self startPlayerThread];
      [self postPlayingNotification];
    }
}

- (void) stopPlayLoop
{
  [dict setObject:@"no" forKey:@"running"];
  [fileHandle closeFile];
  fileHandle = nil;
  [self postStoppedNotification];
}

- (void) threadWillExit
{
  [_thread release];
  _thread = nil;
}

- (BOOL) isRunning
{
  return ([[dict valueForKey:@"running"] isEqualToString:@"yes"]);
}

- (int) timer
{
  return [_thread getSeconds];
}

- (void) seek: (unsigned int) aPos
{
  if (_thread)
    [_thread seek: aPos];
  else
    NSLog (@"seeking within an inactive thread?");
}

@end

@implementation OSSPreference : NSObject

+ (OSSPreference *) instance
{
  static OSSPreference *singleton = nil;

  if (!singleton)
    singleton = [[OSSPreference alloc] _init];

  return singleton;
}

- (OSSPreference *) _init
{
  NSDictionary *tmpDict;

  if ((self = [super init]))
    {
      tmpDict = [[NSUserDefaults standardUserDefaults]
                  dictionaryForKey: @"OSS"];
      preference = [NSMutableDictionary dictionaryWithDictionary: tmpDict];
      [preference retain];
      defaultsInited = NO;
    }

  return self;
}

- (NSView *) preferenceSheet
{
  NSView *aView;

  if (!defaultsInited)
    {
      defaultsInited = YES;
      [self _initDefaults];
    }

  [NSBundle loadNibNamed: @"OSSPreferences"
            owner: self];
  aView = [prefsWindow contentView];
  [aView retain];
  [aView removeFromSuperview];
  [prefsWindow release];
  prefsWindow = nil;

  return aView;
}

- (void) awakeFromNib
{
  [dspDeviceLabel setStringValue: _b(@"DSP device")];
  [dspDeviceField setStringValue: [preference objectForKey: @"dspDevice"]];
}

- (void) _initDefaults
{
  NSString *dspDevice;

  dspDevice = [preference objectForKey: @"dspDevice"];
  if (!dspDevice)
    {
      dspDevice = @"/dev/dsp";
      [preference setObject: dspDevice
                  forKey: @"dspDevice"];
    }
}

- (id) preferenceTitle
{
  return @"OSS";
}

- (void) save
{
  NSUserDefaults *defaults;

  defaults = [NSUserDefaults standardUserDefaults];
  [defaults setObject: preference
            forKey: @"OSS"];
  [defaults synchronize];
}

- (NSString *) dspDevice
{
  return [preference objectForKey: @"dspDevice"];
}

- (void) dspDeviceAction: (id) sender
{
  [preference setObject: [sender stringValue]
              forKey: @"dspDevice"];
}

@end
