/* dummy.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import <Format.h>
#import <Player.h>
#import <Song.h>

#import "dummy.h"

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

static NSLock *lock;

@implementation dummyPlayerThread : NSObject

- (id) init
{
  if ((self = [super init]))
    {
      totalBytes = 0;
      bytesPerSec = 0;

      stream = nil;
      streamToRelease = nil;
    }

  return self;
}

- (void) setStream: (id <Format>) aStream
{
  if (stream)
    {
      while (streamToRelease) {}
      streamToRelease = stream;
    }

  stream = aStream;
  totalBytes = 0;
  /* 2 = 16 bits / 8 */
  if (stream)
    bytesPerSec = [stream readChannels] * [stream readRate] * 2;
}

- (void) seek: (unsigned int) aPos
{
  if (stream)
    {
      [lock lock];
      [stream seek: aPos];
      totalBytes = aPos * bytesPerSec;
      [lock unlock];
    }
  else
    NSLog (@"seeking within an inactive stream?");
}

- (void) handleReadProblem: (int) bytesRead
{
}

- (int) getSeconds
{
  return (totalBytes / bytesPerSec);
}

- (void) threadWillExit: (NSMutableDictionary *) aDict
{
  id parent = [aDict objectForKey: @"parent"];

  NSDebugLog (@"player thread exiting...\n");
  [parent threadWillExit];
}

- (void) songEnded: (NSMutableDictionary *) aDict
{
  [[aDict objectForKey: @"parent"] postSongEndedNotification];
}

- (void) playerThread: (id) dict
{
  NSAutoreleasePool *arp =[[NSAutoreleasePool alloc] init];

  if ([lock lockBeforeDate: [NSDate dateWithTimeIntervalSinceNow: 5.0]])
    {
      [lock unlock];
 
      while (stream
             && [[dict objectForKey: @"running"] isEqualToString: @"yes"])
        {
          bytesRead = [stream readNextChunk: buffer withSize: 176400];

          if ([[dict objectForKey:@"muted"] isEqualToString:@"yes"])
            memset (buffer, 0, bytesRead);

          if (bytesRead > 0)
            {
              /* output code here */
              totalBytes += bytesRead;
            }
          else if (bytesRead == 0)
            {
              [self songEnded: dict];
              while (totalBytes
                     && [[dict objectForKey:@"running"] isEqualToString:@"yes"])
                {};
            }
          else
            break;

          while ([[dict objectForKey:@"paused"] isEqualToString:@"yes"])
            [NSThread sleepUntilDate:
                        [NSDate dateWithTimeIntervalSinceNow: 1.0]];

          if (streamToRelease)
            {
              [streamToRelease streamClose];
              [streamToRelease release];
              streamToRelease = nil;
            }
       }
    }
  else
    NSLog(@"Failed to obtain lock");

  if (!stream)
    [self songEnded: dict];

  [self performSelectorOnMainThread: @selector (threadWillExit:)
        withObject: dict
        waitUntilDone: NO];

  [arp release];
  [NSThread exit];
}

@end

@implementation dummy : PlayerBase

- (id) init
{
  if ((self = [super init]))
    {
      _thread = nil;
      dict = [[NSMutableDictionary alloc] initWithCapacity: 5];
      [dict setObject: @"no" forKey:@"running"];
      [dict setObject: @"no" forKey:@"paused"];
      [dict setObject: @"no" forKey:@"muted"];
      [dict setObject: self forKey:@"parent"];
    }

  return self;
}

- (id) initWithSong: (Song *) aSong
{
  self = [self init];
  [self setSong: aSong];

  return self;
}

- (void) setFormat: (int) newFormat
{
}

- (void) setSpeed: (int) newSpeed
{  
}

- (void) setChannels: (int) newChannels
{
}

- (void) audioInit
{
}

- (Song *) song
{
  return song;
}

- (void) setSong: (Song *) aSong
{
  song = aSong;

  if (aSong)
    {
      id stream = [aSong openStreamForSong];

      if (stream)
        {
          speed = [stream readRate];
          channels = [stream readChannels];

          if (!_thread)
            _thread = [[dummyPlayerThread new] init];
          [_thread setStream: stream];

          if ([[dict objectForKey: @"paused"] isEqualToString: @"yes"])
            {
              [dict setObject: @"no" forKey: @"paused"];
              [self postResumedNotification];
            }
        }
      else
        {
          song = nil;
          [self postSongEndedNotification];
        }
    }
  else
    if (_thread)
      [_thread setStream: nil];
}

/* Launch play loop */
- (void) startPlayerThread
{
  if (![self isRunning])
    {
      [self setChannels: channels];
      [self setSpeed: speed];

      [dict setObject: @"yes" forKey: @"running"];
      [dict setObject: @"no" forKey: @"paused"];

      lock = [NSLock new];
      [lock lock];
      [NSThread detachNewThreadSelector: @selector (playerThread:)
                toTarget: _thread
                withObject: dict];
      [NSThread sleepUntilDate:
                  [NSDate dateWithTimeIntervalSinceNow: 1.0]];
      [lock unlock];
    }
}

- (void) setPaused: (BOOL) paused
{
  if (paused)
    {
      [dict setObject: @"yes" forKey: @"paused"];
      [self postPausedNotification];
    }
  else
    {
      [dict setObject: @"no" forKey: @"paused"];
      [self postResumedNotification];
    }
}

- (BOOL) paused
{
  return [[dict objectForKey:@"paused"] isEqualToString:@"yes"];
}

- (void) setMuted: (BOOL) muted
{
  if (muted)
    [dict setObject: @"yes" forKey: @"muted"];
  else
    [dict setObject: @"no" forKey: @"muted"];
}

- (BOOL) muted
{
  return [[dict objectForKey:@"muted"] isEqualToString:@"yes"];
}

- (void) startPlayLoop
{
  [self audioInit];

  if ([[dict objectForKey:@"running"] isEqualToString:@"no"])
    {
      [self startPlayerThread];
      [self postPlayingNotification];
    }
}

- (void) stopPlayLoop
{
  [dict setObject:@"no" forKey:@"running"];
  [self postStoppedNotification];
}

- (void) threadWillExit
{
  [_thread release];
  _thread = nil;
}

- (BOOL) isRunning
{
  return ([[dict objectForKey:@"running"] isEqualToString:@"yes"]);
}

- (int) timer
{
  return [_thread getSeconds];
}

- (void) seek: (unsigned int) aPos
{
  if (_thread)
    [_thread seek: aPos];
  else
    NSLog (@"seeking within an inactive thread?");
}

@end
