/* GeneralPreference.m - this file is part of Cynthiune
 *
 * Copyright (C) 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <Wolfgang@Contre.COM>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <AppKit/AppKit.h>

#import "GeneralPreference.h"

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

@implementation GeneralPreference : NSObject

- (GeneralPreference *) _init
{
  NSDictionary *tmpDict;

  if ((self = [super init]))
    {
      playersList = [[NSMutableArray alloc] init];
      tmpDict = [[NSUserDefaults standardUserDefaults]
                     dictionaryForKey: @"GeneralPreference"];
      preference = [NSMutableDictionary dictionaryWithDictionary: tmpDict];
      [preference retain];
      defaultsInited = NO;
    }

  return self;
}

+ (GeneralPreference *) instance
{
  static GeneralPreference *singleton = nil;

  if (!singleton)
    singleton = [[self alloc] _init];

  return singleton;
}

- (void) save
{
  NSUserDefaults *defaults;

  defaults = [NSUserDefaults standardUserDefaults];
  [defaults setObject: preference
            forKey: @"GeneralPreference"];
  [defaults synchronize];
}

- (void) registerPlayerClass: (Class) aClass
{
  if ([aClass conformsToProtocol: @protocol(Player)])
    {
      if (![playersList containsObject: aClass])
        [playersList addObject: aClass];
    }
  else
    NSLog (@"Class '%@' not conform to the 'Player' protocol...\n",
           NSStringFromClass (aClass));
}

- (Class) preferredPlayerClass
{
  Class PlayerClass;
  NSString *preferredPlayer;

  preferredPlayer = [preference objectForKey: @"OutputBundle"];
  PlayerClass = NSClassFromString (preferredPlayer);

  return PlayerClass;
}

- (NSString *) preferredPlaylistFormat
{
  NSString *playlistFormat;

  playlistFormat = [preference objectForKey: @"PlaylistFormat"];

  return playlistFormat;
}

- (BOOL) windowsAreSticky
{
  NSNumber *windowsAreSticky;

  windowsAreSticky = [preference objectForKey: @"StickyWindows"];

  return [windowsAreSticky boolValue];
}

- (BOOL) saveWindowsInformation
{
  NSNumber *saveWindowsInformation;

  saveWindowsInformation =
    [preference objectForKey: @"SaveWindowsInformation"];

  return [saveWindowsInformation boolValue];
}

- (void) saveInformation: (NSWindow *) aWindow
               forWindow: (NSString *) windowName
{
  NSMutableArray *information;
  NSMutableDictionary *windowsInformation;
  NSString *frameString;

  windowsInformation = [preference objectForKey: @"WindowsInformation"];
  information = [NSMutableArray arrayWithCapacity: 3];
  frameString = [aWindow stringWithSavedFrame];
  [information addObject: frameString];
  [windowsInformation setObject: information forKey: windowName];
  [self save];
}

- (void) restoreInformation: (NSWindow *) aWindow
                  forWindow: (NSString *) windowName
{
  NSMutableDictionary *windowsInformation;
  NSString *frameString;
  NSArray *information;

  windowsInformation = [preference objectForKey: @"WindowsInformation"];
  information = [windowsInformation objectForKey: windowName];
  if (information)
    {
      frameString = [information objectAtIndex: 0];
      [aWindow setFrameFromString: frameString];
    }
}

// Preference protocol
- (NSString *) preferenceTitle
{
  return _(@"General");
}

- (void) _initDefaults
{
  NSString *outputBundle, *playlistFormat;
  NSNumber *windowsAreSticky, *saveWindowsInformation;
  NSMutableDictionary *windowsInformation;

  outputBundle = [preference objectForKey: @"OutputBundle"];
  if (!outputBundle
      || !([playersList containsObject: NSClassFromString (outputBundle)]))
    {
      outputBundle = NSStringFromClass ([playersList objectAtIndex: 0]);
      [preference setObject: outputBundle
                  forKey: @"OutputBundle"];
    }

  playlistFormat = [preference objectForKey: @"OutputBundle"];
  if (!playlistFormat
      || !([playlistFormat isEqualToString: @"m3u"]
           || [playlistFormat isEqualToString: @"pls"]))
    {
      playlistFormat = @"m3u";
      [preference setObject: playlistFormat
                  forKey: @"PlaylistFormat"];
    }

  windowsAreSticky = [preference objectForKey: @"StickyWindows"];
  if (!windowsAreSticky)
    {
      windowsAreSticky = [NSNumber numberWithBool: YES];
      [preference setObject: windowsAreSticky
                  forKey: @"StickyWindows"];
    }

  saveWindowsInformation =
    [preference objectForKey: @"SaveWindowsInformation"];
  if (!saveWindowsInformation)
    {
      saveWindowsInformation = [NSNumber numberWithBool: YES];
      [preference setObject: saveWindowsInformation
                  forKey: @"SaveWindowsInformation"];
    }

  windowsInformation = [preference objectForKey: @"WindowsInformation"];
  if (!windowsInformation)
    {
      windowsInformation = [NSMutableDictionary dictionaryWithCapacity: 1];
      [preference setObject: windowsInformation
                  forKey: @"WindowsInformation"];
    }
}

- (NSView *) preferenceSheet
{
  NSView *aView;

  if (!defaultsInited)
    {
      defaultsInited = YES;
      [self _initDefaults];
    }

  [NSBundle loadNibNamed: @"GeneralPreference"
            owner: self];
  aView = [prefsWindow contentView];
  [aView retain];
  [aView removeFromSuperview];
  [prefsWindow release];

  return aView;
}

- (void) _initializeSelector
{
  int count, max;
  Class currentClass;

  [outputBundleSelector removeAllItems];

  max = [playersList count];
  if (max > 0)
    {
      for (count = 0; count < max; count++)
        {
          currentClass = [playersList objectAtIndex: count];
          [outputBundleSelector
            addItemWithTitle: NSStringFromClass (currentClass)];
        }
    }
  else
    {
      [outputBundleSelector addItemWithTitle: _(@"None")];
      [outputBundleSelector setEnabled: NO];
    }
}

- (void) awakeFromNib
{
  NSString *outputBundle;
  NSString *playlistFormat;
  NSNumber *stickyNum, *saveWindowsInformationNum;

  [stickyWinToggle setTitle: _(@"Sticky")];
  [saveWindowsInformationToggle setTitle: _(@"Remember location")];
  [windowsBox setTitle: _(@"Windows")];
  [playlistsBox setTitle: _(@"Playlists")];
  [outputModuleBox setTitle: _(@"Output module")];
  [playlistsFormatLabel setStringValue: _(@"Format")];

  [self _initializeSelector];

  outputBundle = [preference objectForKey: @"OutputBundle"];
  if (outputBundle)
    [outputBundleSelector selectItemWithTitle: outputBundle];

  playlistFormat = [self preferredPlaylistFormat];
  [playlistFormatSelector selectItemWithTitle:
                            [playlistFormat uppercaseString]];

  stickyNum = [preference objectForKey: @"StickyWindows"];
  [stickyWinToggle setState: [stickyNum boolValue]];

  saveWindowsInformationNum =
    [preference objectForKey: @"SaveWindowsInformation"];
  [saveWindowsInformationToggle
    setState: [saveWindowsInformationNum boolValue]];
}

- (void) dealloc
{
  [playersList release];
  [preference release];
  [super dealloc];
}

/* as a target */
- (void) outputBundleChanged: (id) sender
{
  NSString *newTitle;

  newTitle = [sender titleOfSelectedItem];
  [sender setTitle: newTitle];
  [sender synchronizeTitleAndSelectedItem];
  [preference setObject: newTitle forKey: @"OutputBundle"];
}

- (void) playlistFormatChanged: (id) sender
{
  NSString *newTitle;

  newTitle = [sender titleOfSelectedItem];
  [sender setTitle: newTitle];
  [sender synchronizeTitleAndSelectedItem];
  [preference setObject: [newTitle lowercaseString]
              forKey: @"PlaylistFormat"];
}

- (void) stickyWindowsChanged: (id) sender
{
  [preference setObject: [NSNumber numberWithBool: [sender state]]
              forKey: @"StickyWindows"];
}

- (void) saveWindowsInformationChanged: (id) sender
{
  [preference setObject: [NSNumber numberWithBool: [sender state]]
              forKey: @"SaveWindowsInformation"];
}

@end
