%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs check}
\begin{code}
module Check ( check ) where
import Monad ( when )
import Directory ( setCurrentDirectory )
import Workaround ( getCurrentDirectory )
import System ( ExitCode(..), exitWith )

import DarcsCommands ( DarcsCommand(..), nodefaults )
import DarcsArguments ( DarcsFlag( Quiet, Verbose, NoTest, LeaveTestDir ),
                        partial_check, any_verbosity, notest,
                        leave_test_dir,
                      )
import Repository ( am_in_repo, read_repo, get_checkpoint_by_default,
                    apply_patches,
                  )
import Patch ( patch2patchinfo )
import SlurpDirectory ( slurp )
import Diff ( cmp, smart_diff )
import Test ( run_test )
import Lock ( withTempDir, withPermDir )
import RepoPrefs ( filetype_function )
import Depends ( get_patches_beyond_tag )
#include "impossible.h"
\end{code}

\options{check}

\haskell{check_description}
\begin{code}
check_description :: String
check_description = "Check the repository for consistency.\n"
\end{code}
Check verifies that the patches stored in the repository, when successively
applied to an empty tree, properly recreate the stored current tree.

If you have a checkpoint of the repository (as is the case if you got the
repo originally using \verb!darcs get --partial!), by default darcs check
will only verify the contents since the most recent checkpoint.  You can
change this behavior using the \verb!--complete! flag.

\begin{code}
check_help :: String
check_help =
 "Check verifies that the patches stored in the repository, when successively\n"++
 "applied to an empty tree, properly recreate the stored current tree.\n"
\end{code}

\begin{code}
check :: DarcsCommand
check = DarcsCommand {command_name = "check",
                      command_help = check_help,
                      command_description = check_description,
                      command_extra_args = 0,
                      command_extra_arg_help = [],
                      command_command = check_cmd,
                      command_prereq = am_in_repo,
                      command_get_arg_possibilities = return [],
                      command_argdefaults = nodefaults,
                      command_darcsoptions = [partial_check,
                                              any_verbosity,notest,
                                              leave_test_dir
                                             ]}
\end{code}

\begin{code}
check_cmd :: [DarcsFlag] -> [String] -> IO ()
check_cmd opts _ =
  let putVerbose s = when (Verbose `elem` opts) $ putStr s
      putInfo s = when (not $ Quiet `elem` opts) $ putStr s
  in do
  patches <- read_repo "."
  maybe_chk <- get_checkpoint_by_default opts "."
  ftf <- filetype_function
  cwd <- getCurrentDirectory
  wd "checking" $ \chd -> do
    putVerbose "Applying patches...\n"
    case maybe_chk of
        Just chk ->
            case patch2patchinfo chk of
            Just chtg -> do
                putVerbose "I am checking from a checkpoint.\n"
                apply_patches putVerbose putInfo $ (chtg, Just chk)
                    : reverse (concat $ get_patches_beyond_tag chtg patches)
            Nothing -> impossible
        Nothing -> apply_patches putVerbose putInfo $ reverse $ concat patches
    is_same <- cmp (cwd++"/_darcs/current") chd
    if is_same
      then do
        putInfo "The repo is consistent!\n"
        if NoTest `elem` opts
           then exitWith ExitSuccess
           else do setCurrentDirectory cwd
                   ec <- run_test opts chd
                   exitWith ec
      else do putInfo "Looks like we have a difference...\n"
              c <- slurp (cwd++"/_darcs/current")
              p <- slurp chd
              putInfo $ show $ smart_diff opts ftf p c
              putInfo $ "\nInconsistent repo!\n"
              exitWith $ ExitFailure 1
    where wd = if LeaveTestDir `elem` opts then withPermDir else withTempDir
\end{code}

\input{Test.lhs}

If you just want to check the consistency of your repository without
running the test, you can call darcs check with the \verb!--no-test!
option.

