%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs unrecord}
\begin{code}
module Unrecord ( unrecord, unpull ) where
import DarcsUtils ( withCurrentDirectory )
import Control.Exception ( block )
import Monad ( when )

import DarcsCommands ( DarcsCommand(..), nodefaults )
import DarcsArguments ( DarcsFlag( Verbose ),
                        working_repo_dir, nocompress, verbose,
                        match_one,
                        ignoretimes,
                      )
import Repository ( PatchSet, read_repo, read_pending,
                    get_unrecorded,
                    write_inventory, write_pending, write_patch,
                    slurp_recorded, sift_for_pending,
                    am_in_repo, sync_repo,
                  )
import Patch ( Patch, invert, apply_to_slurpy, patch2patchinfo,
               join_patches, commute, flatten,
             )
import SlurpDirectory ( slurp_write_dirty, slurp )
import Depends ( deep_optimize_patchset )
import Lock ( withLock )
import SelectChanges ( with_selected_patch_from_repo )
#include "impossible.h"
\end{code}
\begin{code}
unrecord_description :: String
unrecord_description =
 "Unrecord a named patch."
\end{code}

\options{unrecord}

\haskell{unrecord_help} Unrecord actually removes the unrecorded patch from
your repository, so there is no way to ``rerecord'' an unrecorded
patch\footnote{The patch file itself is not actually deleted, but its
context is lost, so it cannot be reliably read---your only choice would be
to go in by hand and read its contents.}.  Note that unrecord doesn't
affect your working copy of the tree at all, so if you want to completely
undo the change, you'll also need to \verb!darcs revert!, or do an unpull.

If you don't revert after unrecording, then the changes made by the
unrecorded patches are left in your working tree.  If these patches are
actually from another repository, interaction (either pushes or pulls) with
that repository may be massively slowed down, as darcs tries to cope with
the fact that you appear to have made a large number of changes that
conflict with those present on the other repository.  So if you really want
to undo the result of a \emph{pull} operation, use unpull! Unrecord is
primarily intended for when you record a patch, realize it needs just one
more change, but would rather not have a separate patch for just that one
change.

\begin{code}
unrecord_help :: String
unrecord_help =
 "Unrecord is used to undo a single recorded patch.  It will prompt you\n"++
 "for which patch to unrecord, and then will undo that patch.\n"
\end{code}
\begin{code}
unrecord :: DarcsCommand
unrecord = DarcsCommand {command_name = "unrecord",
                         command_help = unrecord_help,
                         command_description = unrecord_description,
                         command_extra_args = 0,
                         command_extra_arg_help = [],
                         command_command = unrecord_cmd,
                         command_prereq = am_in_repo,
                         command_get_arg_possibilities = return [],
                         command_argdefaults = nodefaults,
                         command_darcsoptions = [match_one,
                                                 verbose,nocompress,
                                                 working_repo_dir]}
\end{code}
\begin{code}
unrecord_cmd :: [DarcsFlag] -> [String] -> IO ()
unrecord_cmd opts _ = withLock "./_darcs/lock" $ do
  recorded <- slurp_recorded "."
  pend <- do aack <- read_pending
             return $ case aack of Nothing -> []
                                   Just p -> flatten p
  with_selected_patch_from_repo "unrecord" opts True $
    \ (p, skipped) ->
    case apply_to_slurpy (invert p) recorded of
    Nothing -> putStr "Unable to apply inverse patch!\n"
    Just recorded' -> block $ do
          when (Verbose `elem` opts) $
               putStr "About to write out (potentially) modified patches...\n"
          sequence_ $ map (write_patch opts) skipped
          patches' <- read_repo "."
          when (Verbose `elem` opts) $
               putStr "About to write inventory...\n"
          write_inventory "." $ rempatch p patches'
          when (Verbose `elem` opts) $ putStr "Updating _darcs/current...\n"
          withCurrentDirectory "_darcs/current" $
              slurp_write_dirty recorded'
          when (Verbose `elem` opts) $ putStr "Updating pending...\n"
          write_pending $ sift_for_pending $ join_patches $ p : pend
          sync_repo
          putStr $ "Finished unrecording.\n"

rempatch :: Patch -> PatchSet -> PatchSet
rempatch p (pps:ppss) =
    case patch2patchinfo p of
    Nothing -> bug "Weird problem in unrecord or unpull - in rempatch."
    Just pinfo -> if pinfo `elem` simple_infos
                  then (filter ((/= pinfo).fst) pps) : ppss
                  else deep_optimize_patchset $
                       map (filter ((/= pinfo).fst)) (pps:ppss)
    where simple_infos = tail $ reverse $ map fst pps
rempatch _ [] = impossible
\end{code}

\section{darcs unpull}

\begin{code}
unpull_description :: String
unpull_description =
 "Unpull a named patch."
\end{code}

\options{unpull}

\haskell{unpull_help}

Like unrecord, unpull does not just apply an inverse patch to the
repository, it actually deletes the patch from the repository.  This makes
unpull a particularly dangerous command, as it not only deletes the patch
from the repo, but also removes the changes from the working directory.  It
is equivalent to an unrecord followed by a revert, except that revert can
be unreverted.

Contrary to what its name suggests, there is nothing in unpull that
requires that the ``unpulled'' patch originate from different repository.
The name was chosen simply to suggest a situation in which it is ``safe''
to used unpull.  If the patch was originally from another repo, then
unpulling is safe, because you can always pull the patch again if you
decide you want it after all.  If you unpull a locally recorded patch, all
record of that change is lost, which is what makes this a ``dangerous''
command, and thus deserving of a obscure name which is more suggestive of
when it is safe to use than precisely what it does.

\begin{code}
unpull_help :: String
unpull_help =
 "Unpull is used to undo a single patch that has been pulled from another\n"++
 "repository.  It will prompt you for which patch to unpull, and then will\n"++
 "undo that patch.  Beware that unpull undoes the patch both from the repo\n"++
 "records AND from the current working directory, and does NOT check that\n"++
 "the patch originated with a pull.  In otherwords, you could lose precious\n"++
 "code by unpulling!\n"
\end{code}
\begin{code}
unpull :: DarcsCommand
unpull = DarcsCommand {command_name = "unpull",
                       command_help = unpull_help,
                       command_description = unpull_description,
                       command_extra_args = 0,
                       command_extra_arg_help = [],
                       command_command = unpull_cmd,
                       command_prereq = am_in_repo,
                       command_get_arg_possibilities = return [],
                       command_argdefaults = nodefaults,
                       command_darcsoptions = [match_one,
                                               verbose,nocompress,
                                               ignoretimes,
                                               working_repo_dir]}
\end{code}
\begin{code}
unpull_cmd :: [DarcsFlag] -> [String] -> IO ()
unpull_cmd opts _ = withLock "./_darcs/lock" $ do
  recorded <- slurp_recorded "."
  work <- slurp "."
  mpend <- get_unrecorded opts
  let pend = case mpend of
             Nothing -> join_patches []
             Just p -> p
  with_selected_patch_from_repo "unpull" opts False $
    \ (p, skipped) ->
    case commute (pend, p) of
    Nothing -> fail $ "Couldn't commute patch past pending."
    Just (p_after_pending, pend') ->
      case apply_to_slurpy (invert p) recorded of
      Nothing -> fail "Unable to apply inverse patch!"
      Just rec' ->
          case apply_to_slurpy (invert p_after_pending) work of
          Nothing -> fail "Couldn't undo patch in working dir."
          Just work' -> block $ do
                sequence_ $ map (write_patch opts) skipped
                patches' <- read_repo "."
                write_inventory "." $ rempatch p patches'
                withCurrentDirectory "_darcs/current" $
                    slurp_write_dirty rec'
                slurp_write_dirty work'
                write_pending pend'
                sync_repo
                putStr $ "Finished unpulling.\n"
\end{code}

