%  Copyright (C) 2002-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs whatsnew}
\label{whatsnew}
\begin{code}
module WhatsNew ( whatsnew ) where
import System ( ExitCode(..), exitWith )
import List ( sort )
import Monad ( when )

import DarcsCommands ( DarcsCommand(..), nodefaults )
import DarcsArguments ( DarcsFlag(..), working_repo_dir, lookforadds,
                        ignoretimes, verbose, noskip_boring,
#ifdef HAVEWX
                        gui_interactive,
#endif
                        summary, unified,
                        fix_filepath,
                        list_registered_files,
                      )
import TouchesFiles ( choose_touching )
import Repository ( slurp_recorded, get_unrecorded, am_in_repo )
import Patch ( patch_summary, join_patches )
import PrintPatch ( printPatch, contextualPrintPatch )
import Patch ( invert, apply_to_filepath )
#ifdef HAVEWX
import Patch ( Patch, flatten )
import SlurpDirectory ( Slurpy )
import Graphics.UI.WX hiding ( when )
import Graphics.UI.WXCore ( windowSetSizeHints )
#endif
\end{code}

\options{whatsnew}

\haskell{whatsnew_description}
\begin{code}
whatsnew_description :: String
whatsnew_description = "Display unrecorded changes in the working directory.\n"
\end{code}
\haskell{whatsnew_help} Darcs whatsnew will return a non-zero value if
there are no changes, which can be useful if you just want to see in a
script if anything has been modified.  If you want to see some context
around your changes, you can use the \verb!-u! option, to get output
similar to the unidiff format.

\begin{code}
whatsnew_help :: String
whatsnew_help =
 "What's-new gives you a view of what changes you've made in your working\n"++
 "directory that haven't yet been recorded.  The changes are displayed in\n"++
 "darcs patch format.\n"
\end{code}

\begin{code}
whatsnew :: DarcsCommand
whatsnew = DarcsCommand {command_name = "whatsnew",
                         command_help = whatsnew_help,
                         command_description = whatsnew_description,
                         command_extra_args = -1,
                         command_extra_arg_help = ["[FILE or DIRECTORY]..."],
                         command_command = whatsnew_cmd,
                         command_prereq = am_in_repo,
                         command_get_arg_possibilities = list_registered_files,
                         command_argdefaults = nodefaults,
                         command_darcsoptions = [verbose,
#ifdef HAVEWX
                                                 gui_interactive,
#endif
                                                 summary, unified, ignoretimes,
                                                 lookforadds, noskip_boring,
                                                 working_repo_dir]}
\end{code}

\begin{code}
whatsnew_cmd :: [DarcsFlag] -> [String] -> IO ()
whatsnew_cmd opts args =
    let files = sort $ map (fix_filepath opts) args in do
    when (concat files /= "") $
         putStr $ "What's new in "++unwords (map show files)++":\n\n"
    maybe_changes <- get_unrecorded opts
    case maybe_changes of
      Nothing -> do putStr "No changes!\n"
                    exitWith $ ExitFailure 1
      Just changes ->
          let pre_changed_files = map (apply_to_filepath (invert changes)) files
              ch =  choose_touching pre_changed_files changes
                    in
              if ch == join_patches []
              then do putStr "No changes!\n"
                      exitWith $ ExitFailure 1
              else 
#ifdef HAVEWX
                 if Gui `elem` opts
                 then do s <- slurp_recorded "."
                         guiShowPatch ch s
                 else
#endif
                       if Summary `elem` opts
                       then putStr $ patch_summary ch
                       else if Unified `elem` opts
                            then do s <- slurp_recorded "."
                                    contextualPrintPatch s ch
                            else printPatch ch

#ifdef HAVEWX
guiShowPatch :: Patch -> Slurpy -> IO ()
guiShowPatch p _ = start $ hello p
hello p = do f <- frame [text := "Whats New!"]
             scrolled <- scrolledWindow f [scrollRate := size 10 10]
             guips <- sequence $ map (guipatch scrolled) $ flatten p
             set scrolled [layout :=  rigid $ column 0 guips]
             windowSetSizeHints scrolled (-1) (-1) (-1) (-1) (-1) (-1)
             set f [clientSize := size 400 200] -- this is window minimum size
             (ms,mclose) <- default_menubar
             quit <- button f [text := "Quit", on command := close f]
             set f [layout := column 0 [fill $ widget scrolled,
                                        centre $ hstretch $ margin 5 $ widget quit],
                    menuBar := ms, on (menu mclose) := close f,
                    clientSize := size 600 400 -- this is window actual size
                   ]

guipatch :: Window a -> Patch -> IO Layout
guipatch w p = do st <- staticText w [text := show p]
                  return $ floatLeft $ widget st

default_menubar = do
  file <- menuPane [text := "&File"]
  mclose <- menuItem file [text := "&Quit\tCtrl+Q", help := "Quit darcs"]
  return ([file],mclose)
#endif
\end{code}

If you give one or more file or directory names as an argument to whatsnew,
darcs whatsnew will output only changes to those files or to files in those
directories.
