
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>

int stdout_is_a_pipe() {
  struct stat s;
  fstat(1, &s);
  return !S_ISFIFO(s.st_mode);
}

#ifdef _WIN32
#include <windows.h>
#include <io.h>
#endif

static int sloppy_atomic_create(const char *p)
{
    int fd;
    fd = open(p, O_WRONLY | O_EXCL | O_CREAT, 0666);
    if(fd < 0)
        return -1;
    close(fd);
    return 1;
}

#ifdef _WIN32

int atomic_create(const char *p)
{
    return sloppy_atomic_create(p);
}

#else

static int careful_atomic_create(const char *p)
{
    /* O_EXCL is not available over NFSv2, and even under NFSv3, it is
       broken on many systems.  The following protocol is provably
       safe assuming that:
       - creation of hard links is atomic;
       - stat hits the server rather than working from the cache.
    */

    static char hostname[65] = {'\0'};
    int fd, rc, saved_errno;
#define FILENAME_SIZE 11 + 15 + 8 + 1
    char filename[FILENAME_SIZE];
    struct timeval now;
    struct stat sb;

    if(hostname[0] == '\0') {
        char *c;
        rc = gethostname(hostname, 65);
        if(rc < 0 || rc >= 65) {
            fprintf(stderr, "Error reading hostname on locking.\n");
            return -1;
        }
        c = strchr(hostname, '.');
        if(c != NULL)
            *c = '\0';
        hostname[15] = '\0';
    }

    gettimeofday(&now, NULL);

    rc = snprintf(filename, FILENAME_SIZE, "darcs_lock_%s%04x%04x",
                  hostname, ((unsigned)getpid()) & 0xFFFF,
                  ((unsigned int)(now.tv_usec ^ (now.tv_usec << 4))) & 0xFFFF);
    if(rc < 0 || rc >= FILENAME_SIZE) {
        fprintf(stderr, "Error writing to lock filename. (%d)\n", rc);
        return -1;
    }

    fd = open(filename, O_WRONLY | O_EXCL | O_CREAT, 0666);
    if(fd < 0)
        return -1;

    /* This should cause a correctly implemented NFS client to flush
       its metadata cache. */
    rc = close(fd);
    if(rc < 0) {
        fprintf(stderr, "Error closing file %s. (%d)\n", filename, rc);
        goto fail;
    }

    rc = link(filename, p);
    if(rc >= 0)
        goto success;
    else if(errno == EPERM) {
        /* Linux returns EPERM when making hard links on filesystems
           that don't support them. */
        unlink(filename);
        return sloppy_atomic_create(p);
    } else if(errno != EEXIST && errno != EIO)
        goto fail;

    /* The link may still have been successful if we're running over
       UDP and got EEXIST or EIO.  Check the file's link count. */

    rc = stat(filename, &sb);
    if(rc < 0) {
        fprintf(stderr, "Couldn't stat %s! (%d)\n", filename, rc);
        goto fail;
    }

    if(sb.st_nlink != 2) {
        errno = EEXIST;
        fprintf(stderr, "Wrong number of links to %s! (%d != 2)\n",
                filename, sb.st_nlink);
        goto fail;
    }

 success:
    unlink(filename);
    return 1;

 fail:
    saved_errno = errno;
    unlink(filename);
    errno = saved_errno;
    return -1;
}

int atomic_create(const char *p)
{
    static int sloppy = -1;

    if(sloppy < 0) {
        char *s = getenv("DARCS_SLOPPY_LOCKS");
        sloppy = (s != NULL);
    }

    if(sloppy)
        return sloppy_atomic_create(p);
    else
        return careful_atomic_create(p);
}

#endif

#ifdef _WIN32
static int inited_rand = 0;
int mkstemp(char *p)
{
    char* p2;
    int fd;
    int len = strlen(p);
    if (len < 6)
        return -1;
    if (!inited_rand) {
      srand(time(0));
      inited_rand = 1;
    }
    snprintf(p+len-6, 6, "%06d", rand()+rand()<<16);
    fd = open(p, O_CREAT | O_EXCL | O_RDWR, 0666);
    return fd;
}

int pipe( int fildes[2] ) {
return _pipe( fildes, 8 * (2<<10), O_BINARY );
}

int isnt_symlink(const char *file) {
  return 1; /* FIXME: should ignore windows shortcuts */
}

static DWORD get_mode()
{
    DWORD console_mode;
    HANDLE stdin_handle = GetStdHandle(STD_INPUT_HANDLE);
    if (!GetConsoleMode(stdin_handle, &console_mode))
        fprintf(stderr, "GetConsoleMode error: %x\n", GetLastError());
    return console_mode;
}

int get_raw_mode()
{
    return (get_mode() & ENABLE_LINE_INPUT) == 0;
}

void set_raw_mode(int raw)
{
    HANDLE stdin_handle = GetStdHandle(STD_INPUT_HANDLE);
    DWORD console_mode = get_mode();
    if (raw)
        console_mode &= ~(ENABLE_LINE_INPUT|ENABLE_ECHO_INPUT);
    else
        console_mode |= ENABLE_LINE_INPUT|ENABLE_ECHO_INPUT;
    if (!SetConsoleMode(stdin_handle, console_mode))
        fprintf(stderr, "SetConsoleMode error: %x\n", GetLastError());
}


#else

int get_raw_mode()
{
    return 0;
}

void set_raw_mode(int raw)
{
}

int open_read(const char *fname) {
  return open(fname, O_RDONLY);
}

int open_write(const char *fname) {
  return open(fname, O_WRONLY | O_TRUNC | O_CREAT,
              S_IWUSR | S_IRUSR | S_IROTH | S_IRGRP);
}

#include <sys/wait.h>
#include <signal.h>

/* This waits and then returns the exit status of the process that was
   waited for. */
int smart_wait(int pid) {
  int stat;
  waitpid(pid, &stat, 0);
  if (WIFEXITED(stat)) {
    return WEXITSTATUS(stat);
  } else if (WIFSIGNALED(stat)) {
    psignal(WTERMSIG(stat), "Error in subprocess");
    return - WTERMSIG(stat);
  } else {
    return -137;
  }
}

#include <unistd.h>
#include <sys/time.h>
#include <errno.h>
#include <stdio.h>

int execvp_no_vtalarm(const char *file, char *const argv[]) {
  /* Reset the itimers in the child, so it doesn't get plagued
   * by SIGVTALRM interrupts.
   */
  struct timeval tv_null = { 0, 0 };
  struct itimerval itv;
  itv.it_interval = tv_null;
  itv.it_value = tv_null;
  setitimer(ITIMER_REAL, &itv, NULL);
  setitimer(ITIMER_VIRTUAL, &itv, NULL);
  setitimer(ITIMER_PROF, &itv, NULL);
  execvp(file, argv);
  perror("Error in execvp");
  return errno;
}

#include <sys/types.h>
#include <sys/stat.h>

int isnt_symlink(const char *file) {
  struct stat buf;
  if (lstat(file, &buf)) return 0; /* treat error as symlink */
  return !S_ISLNK(buf.st_mode);
}

#endif
