%  Copyright (C) 2002-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs replace}
\begin{code}
module Replace ( replace ) where

import Maybe ( catMaybes )
import Monad (unless, liftM)

import DarcsCommands
import DarcsArguments
import Repository ( am_in_repo, read_pending, write_pending,
                    slurp_recorded_and_unrecorded, slurp_pending,
                  )
import Patch ( apply_to_slurpy, tokreplace, force_replace_slurpy,
               join_patches, flatten, Patch,
             )
import SlurpDirectory ( slurp_write_dirty, slurp_hasfile, Slurpy )
import FileName ( fp2fn )
import RegChars ( regChars )
import Lock ( withRepoLock )
import Diff ( smart_diff )
import RepoPrefs ( FileType(TextFile) )
#include "impossible.h"
\end{code}

\begin{code}
replace_description :: String
replace_description =
 "Replace a token with a new value for that token."
\end{code}

\options{replace}

\haskell{replace_help}

The default regexp is \verb![A-Za-z_0-9]!), and if one of your tokens
contains a `-' or `.', you will then (by default) get the ``filename''
regexp, which is \verb![A-Za-z_0-9\-\.]!.

\begin{options}
--token-chars
\end{options}

If you prefer to choose a different set of characters to define your token
(perhaps because you are programming in some other language), you may do so
with the \verb!--token-chars! option.  You may prefer to define tokens in terms
of delimiting characters instead of allowed characters using a flag such as
\verb!--token-chars '[^ \n\t]'!, which would define a token as being
white-space delimited.

If you do choose a non-default token definition, I recommend using
\verb!_darcs/prefs/defaults! to always specify the same
\verb!--token-chars!, since your replace patches will be better behaved (in
terms of commutation and merges) if they have tokens defined in the same
way.

When using darcs replace, the ``new'' token may not already appear in the
file---if that is the case, the replace change would not be invertible.
This limitation holds only on the already-recorded version of the file.

There is a potentially confusing difference, however, when a replace is
used to make another replace possible:
\begin{verbatim}
% darcs replace newtoken aaack ./foo.c
% darcs replace oldtoken newtoken ./foo.c
% darcs record
\end{verbatim}
will be valid, even if \verb!newtoken! and \verb!oldtoken! are both present
in the recorded version of foo.c, while the sequence
\begin{verbatim}
% [manually edit foo.c replacing newtoken with aaack]
% darcs replace oldtoken newtoken ./foo.c
\end{verbatim}
will fail because ``newtoken'' still exists in the recorded version of
\verb!foo.c!.  The reason for the difference is that when recording, a
``replace'' patch always is recorded \emph{before} any manual changes,
which is usually what you want, since often you will introduce new
occurrences of the ``newtoken'' in your manual changes.  In contrast,
``replace'' changes are recorded in the order in which they were made.

\begin{code}
replace_help :: String
replace_help =
 "Replace allows you to change a specified token wherever it\n"++
 "occurs in the specified files.  The replace is encoded in a\n"++
 "special patch and will merge as expected with other patches.\n"++
 "Tokens here are defined by a regexp specifying the characters\n"++
 "which are allowed.  By default a token corresponds to a C identifier.\n"
\end{code}

\begin{code}
replace :: DarcsCommand
replace = DarcsCommand {command_name = "replace",
                        command_help = replace_help,
                        command_description = replace_description,
                        command_extra_args = -1,
                        command_extra_arg_help = ["<OLD>","<NEW>",
                                                  "<FILE> ..."],
                        command_command = replace_cmd,
                        command_prereq = am_in_repo,
                        command_get_arg_possibilities = list_registered_files,
                        command_argdefaults = nodefaults,
                        command_darcsoptions = [tokens, force_replace, verbose]}
\end{code}

\begin{code}
replace_cmd :: [DarcsFlag] -> [String] -> IO ()
replace_cmd opts (old:new:relfs) = withRepoLock $ do
  toks <- choose_toks opts old new
  let checkToken tok =
        unless (is_tok toks tok) $ fail $ "'"++tok++"' is not a valid token!"
  checkToken old
  checkToken new
  (_, work) <- slurp_recorded_and_unrecorded "."
  cur <- slurp_pending "."
  ps_and_pswork <- catMaybes `liftM` sequence (map (repl toks cur work) fs)
  case apply_to_slurpy (join_patches $ snd $ unzip ps_and_pswork) work of
    Nothing -> fail $ "Can't do replace on working!\n"++
                      "Perhaps one of the files already contains '"++
                      new++"'?"
    Just w' -> slurp_write_dirty [] w'
  pend <- read_pending
  write_pending $ join_patches $
      maybe [] flatten pend ++ fst (unzip ps_and_pswork)
  where fs = map (fix_filepath opts) relfs
        ftf _ = TextFile

        repl :: String -> Slurpy -> Slurpy -> FilePath -> IO (Maybe (Patch, Patch))
        repl toks cur work f =
          if not $ slurp_hasfile (fp2fn f) cur
          then do putStrLn $ "Skipping file '"++f++"' which isn't in the repo."
                  return Nothing
          else case apply_to_slurpy (tokreplace f toks old new) cur of
            Nothing ->
                if ForceReplace `elem` opts
                then return $ Just (get_force_replace f toks cur,
                                    get_force_replace f toks work)
                else do putStrLn $ "Skipping file '"++f++"'"
                        putStrLn $ "Perhaps the recorded version of this " ++
                                   "file already contains '" ++new++"'?"
                        return Nothing
            _ -> return $ Just (tokreplace f toks old new,
                                get_force_replace f toks work)

        get_force_replace :: FilePath -> String -> Slurpy -> Patch
        get_force_replace f toks s =
            case force_replace_slurpy (tokreplace f toks new old) s of
            Nothing -> bug "weird forcing bug in replace."
            Just s' -> case smart_diff [] ftf s s' of
                       Nothing -> tokreplace f toks old new
                       Just pfix -> join_patches
                                    [pfix, tokreplace f toks old new]

replace_cmd _ _ = fail "Usage: darcs replace OLD NEW [FILES]"
\end{code}

\begin{code}
default_toks :: String
default_toks = "A-Za-z_0-9"
filename_toks :: String
filename_toks = "A-Za-z_0-9\\-\\."
is_tok :: String -> String -> Bool
is_tok _ "" = False
is_tok toks s = and $ map (regChars toks) s

choose_toks :: [DarcsFlag] -> String -> String -> IO String
choose_toks (Toks t:_) a b
    | length t > 2 = if is_tok tok a && is_tok tok b
                     then return tok
                     else fail $ "Bad token spec:  "++t
    | otherwise = fail $ "Bad token spec:  "++t
    where tok = init $ tail t :: String
choose_toks (_:fs) a b = choose_toks fs a b
choose_toks [] a b = if is_tok default_toks a && is_tok default_toks b
                     then return default_toks
                     else return filename_toks
\end{code}

