/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "PostProcessor.h"

#include <cmath>
#include <cstring>

#include "ColorManager.h"

struct PostProcessor::Private {
    double exposure;
    bool convertToSRGB;
};

PostProcessor::PostProcessor( double _exposure, bool _convertToSRGB ) : d(new Private)
{
  d->exposure = pow(2, _exposure + 2.47393) * 0.0883883;
  setConvertToSRGB( _convertToSRGB );
}

PostProcessor::PostProcessor( const PostProcessor& _rhs ) : d(new Private(*_rhs.d))
{
  
}

PostProcessor::~PostProcessor()
{
  delete d;
}

void PostProcessor::setConvertToSRGB( bool _v )
{
  d->convertToSRGB = _v;
}

bool PostProcessor::convertToSRGB() const
{
  return d->convertToSRGB;
}

template< typename T>
inline T clamp( T v, T min, T max )
{
  if ( v > max ) return max;
  else if( v < min ) return min;
  else return v;
}

void PostProcessor::apply16( quint16* _rgb )
{
  _rgb[0] = clamp( (int)(_rgb[0] * d->exposure), 0, 0xFFFF);
  _rgb[1] = clamp( (int)(_rgb[1] * d->exposure), 0, 0xFFFF);
  _rgb[2] = clamp( (int)(_rgb[2] * d->exposure), 0, 0xFFFF);
  if( d->convertToSRGB )
  {
    ColorManager::instance()->RGB16linearToSRGB16( _rgb, _rgb );
  }
}

void PostProcessor::apply8( quint8* _rgb )
{
  _rgb[0] = clamp( (int)(_rgb[0] * d->exposure), 0, 0xFF);
  _rgb[1] = clamp( (int)(_rgb[1] * d->exposure), 0, 0xFF);
  _rgb[2] = clamp( (int)(_rgb[2] * d->exposure), 0, 0xFF);  
}
