/***************************************************************************
 *   Copyright (C) 2005 by Adam Treat                                      *
 *   treat@kde.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "reportwizardbaseimpl.h"

#include "project.h"
#include "datareport.h"
#include "datatable.h"
#include "asciivalidator.h"
#include "databaseconnection.h"

#include <kconfig.h>
#include <kmessagebox.h>
#include <kurldrag.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <kaction.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <kdebug.h>
#include <kinputdialog.h>
#include <kicondialog.h>
#include <kurlrequester.h>

#include <qlistbox.h>
#include <qwidget.h>
#include <qcheckbox.h>
#include <qlineedit.h>

#include <qlabel.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qpushbutton.h>
#include <qmultilineedit.h>
#include <qlistview.h>
#include <qfeatures.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <limits.h>

#include <qdatatable.h>
#include <qdatabrowser.h>
#include <qdataview.h>
#include <qsqleditorfactory.h>
#include <qsqlindex.h>
#include <qsqlcursor.h>
#include <qsqldatabase.h>

ReportWizard::ReportWizard( Project *project, DataReport *report, DataTable *parentTable, bool configure )
        : ReportWizardBase( report ),
        m_project( project ),
        m_report( report ),
        m_parentTable( parentTable ),
        m_configure( configure )
{

    setMinimumSize( 800, 600 );
    setCaption( i18n("Report Wizard") );
    setFinishEnabled( sortPage, true );

    wizard1->setPixmap( UserIcon( "wizard_table_1" ) );
    wizard2->setPixmap( UserIcon( "wizard_table_2" ) );

    buttonIcon->setPixmap( SmallIcon( "kugar", 32 ) );
    buttonAddSortField->setPixmap( UserIcon( "right" ) );
    buttonRemoveSortField->setPixmap( UserIcon( "left" ) );
    buttonUpSortField->setPixmap( UserIcon( "up" ) );
    buttonDownSortField->setPixmap( UserIcon( "down" ) );
    buttonReSortField->setPixmap( UserIcon( "re-sort" ) );

    listBoxTables->setSelectionMode( QListBox::Multi );

    connect( nextButton(), SIGNAL( clicked() ), SLOT( nextPageClicked() ) );

    setupFirstPage();
}

ReportWizard::~ReportWizard()
{}

/****************reportPage methods****************/
void ReportWizard::reportTemplateChanged( const QString &temp )
{
    m_report->setTemplateURL( temp );
}

void ReportWizard::reportLabelChanged( const QString &label )
{
    m_report->setName( m_project->uniqueDataTableName( label ) );
}

void ReportWizard::reportIconChanged()
{
    QString iconName = KIconDialog::getIcon( KIcon::NoGroup, KIcon::Any, false, 32, true, this, i18n("Select Icon for This Report") );
    buttonIcon->setPixmap( SmallIcon( iconName, 32 ) );
    m_report->setIconName( iconName );
}

void ReportWizard::enableAllReportPage( bool b )
{
    listBoxTables->setEnabled( b );
    kURLRequesterTemplate->setEnabled( b );
    editReport->setEnabled( b );
    buttonIcon->setEnabled( b );
}

/****************sortPage methods****************/
void ReportWizard::addSortField()
{
    int i = listBoxSortField->currentItem();
    if ( i == -1 )
        return ;
    QString f = listBoxSortField->currentText();
    if ( !f.isEmpty() )
        listBoxSortedField->insertItem( f + " ASC" );
}

void ReportWizard::reSortSortField()
{
    int i = listBoxSortedField->currentItem();
    if ( i != -1 )
    {
        QString text = listBoxSortedField->currentText();
        if ( text.mid( text.length() - 3 ) == "ASC" )
            text = text.mid( 0, text.length() - 3 ) + "DESC";
        else if ( text.mid( text.length() - 4 ) == "DESC" )
            text = text.mid( 0, text.length() - 4 ) + "ASC";
        listBoxSortedField->removeItem( i );
        listBoxSortedField->insertItem( text, i );
        listBoxSortedField->setCurrentItem( i );
    }
}

void ReportWizard::removeSortField()
{
    int i = listBoxSortedField->currentItem();
    if ( i != -1 )
    {
        listBoxSortedField->removeItem( i );
    }
}

void ReportWizard::sortFieldUp()
{
    if ( listBoxSortedField->currentItem() <= 0 ||
         listBoxSortedField->count() < 2 )
        return ;
    int index = listBoxSortedField->currentItem() - 1;
    QListBoxItem *i = listBoxSortedField->item( listBoxSortedField->currentItem() );
    listBoxSortedField->takeItem( i );
    listBoxSortedField->insertItem( i, index );
    listBoxSortedField->setCurrentItem( i );
}

void ReportWizard::sortFieldDown()
{
    if ( listBoxSortedField->currentItem() == -1 ||
         listBoxSortedField->currentItem() == ( int ) listBoxSortedField->count() - 1 ||
         listBoxSortedField->count() < 2 )
        return ;
    int index = listBoxSortedField->currentItem() + 1;
    QListBoxItem *i = listBoxSortedField->item( listBoxSortedField->currentItem() );
    listBoxSortedField->takeItem( i );
    listBoxSortedField->insertItem( i, index );
    listBoxSortedField->setCurrentItem( i );
}

void ReportWizard::enableAllSortPage( bool b )
{
    listBoxSortField->setEnabled( b );
    listBoxSortedField->setEnabled( b );
    buttonAddSortField->setEnabled( b );
    buttonRemoveSortField->setEnabled( b );
    buttonUpSortField->setEnabled( b );
    buttonDownSortField->setEnabled( b );
    buttonReSortField->setEnabled( b );
}

void ReportWizard::nextPageClicked()
{
    if ( currentPage() == reportPage )
    {
        setupReportPage();
        enableAllReportPage( true );
    }
    else if ( currentPage() == sortPage )
    {
        setupSortPage();
        enableAllSortPage( true );
    }
}

void ReportWizard::setupFirstPage()
{
    setNextEnabled( reportPage, true );
    setupReportPage();
    enableAllReportPage( true );
}

void ReportWizard::setupReportPage()
{
    listBoxTables->clear();

    QStringList str;
    DataTableList list = m_project->dataTablesInDataTableTree( m_parentTable );
    DataTableList::Iterator it = list.begin();
    for ( ; it != list.end(); ++it )
    {
        str.append( ( *it ) ->name() );
    }

    listBoxTables->insertStringList( str );

    editReport->blockSignals( true );
    editReport->setText( m_report->name() );
    editReport->blockSignals( false );

    comboSearch->insertItem( i18n("<Default Search>") );
    DataSearchList searchList = m_project->searchList();
    DataSearchList::Iterator it2 = searchList.begin();
    for ( ; it2 != searchList.end(); ++it2 )
    {
        comboSearch->insertItem( ( *it2 ).name() );
    }

    if ( m_configure )
    {
        kURLRequesterTemplate->setURL( m_report->templateURL() );
        buttonIcon->setPixmap( SmallIcon( m_report->iconName(), 32 ) );

        QStringList tables = m_report->tables();
        QStringList::Iterator names = tables.begin();
        for ( ; names != tables.end(); ++names )
        {
            QListBoxItem *item = listBoxTables->findItem( ( *names ) );
            listBoxTables->setSelected( item, true );
        }

        if ( m_report->dataSearch().isEmpty() )
            comboSearch->setCurrentText( i18n("<Default Search>") );
        else
            comboSearch->setCurrentText( m_report->dataSearch() );
    }
}

void ReportWizard::setupSortPage()
{
    listBoxSortField->clear();
    listBoxSortedField->clear();

    DataTableList list = m_project->dataTablesInDataTableTree( m_parentTable );
    DataTableList::Iterator it = list.begin();
    for ( ; it != list.end(); ++it )
    {
        QListBoxItem *item = listBoxTables->findItem( ( *it ) ->name() );
        if ( item->isSelected() )
        {
            DataFieldList fields = ( *it ) ->fieldList();
            DataFieldList::Iterator it2 = fields.begin();
            for ( ; it2 != fields.end(); ++it2 )
            {
                if ( !( *it2 ) ->hidden() && !( *it2 ) ->isVirtual() )
                    listBoxSortField->insertItem( ( *it )->alias() + "." + ( *it2 ) ->name() );
            }
        }
    }

    if ( m_configure )
    {
        listBoxSortedField->insertStringList( m_report->sort() );
    }
}

void ReportWizard::accept()
{
    m_report->clearTables();
    DataTableList list = m_project->dataTablesInDataTableTree( m_parentTable );
    DataTableList::Iterator it = list.begin();
    for ( ; it != list.end(); ++it )
    {
        QListBoxItem *item = listBoxTables->findItem( ( *it ) ->name() );
        if ( item->isSelected() )
            m_report->addTable( ( *it ) ->name() );
    }

    QStringList sort;
    QListBoxItem *item = listBoxSortedField->item( 0 );
    while ( item )
    {
        sort.append( item->text() );
        item = item->next();
    }
    m_report->setSort( sort );

    if ( comboSearch->currentText() == i18n("<Default Search>")  )
        m_report->setDataSearch( QString::null );
    else
        m_report->setDataSearch( comboSearch->currentText() );

    if ( !m_configure )
    {
        m_report->initialize();
        m_project->addDataReport( m_report );
    }
    else
        m_report->refreshReport();

    ReportWizardBase::accept();
}

#include "reportwizardbaseimpl.moc"
