/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include "kcompletiontable.h"

#include <qstyle.h>
#include <qpainter.h>
#include <kdebug.h>

KCompletionTable::KCompletionTable( QWidget *parent, const char *name )
        : KCompletionBox( parent, name ),
        m_boxWidth( 0 ),
        m_boxHeight( 0 )
{}

KCompletionTable::~KCompletionTable()
{}

QValueList<int> KCompletionTable::columnWidths() const
{
    QValueList<int> widths;
    for ( uint i = 0; i < m_colWidths.count(); ++i )
        widths.append( m_colWidths[ i ] );
    return widths;
}

void KCompletionTable::setColumnWidths( QValueList<int> &colWidths )
{
    m_colWidths.clear();
    for ( uint i = 0; i < colWidths.count(); ++i )
        m_colWidths[ i ] = colWidths[ i ];

    m_boxWidth = columnPosition( m_colWidths.count() ) + 4;
    m_boxHeight = items().count() * itemHeight( currentItem() ) + 3;
}

void KCompletionTable::clearWidths()
{
    m_boxWidth = 0;
    m_boxHeight = 0;
    m_colWidths.clear();
}

void KCompletionTable::calculateWidths( QStringList &list )
{
    QFontMetrics fm = fontMetrics();
    for ( uint i = 0; i < list.count(); ++i )
    {
        int spacing = i == 0 ? 2 : 3;
        int width = fm.width( list[ i ] );
        if ( m_colWidths[ i ] < width + spacing * 2 )
        {
            m_colWidths[ i ] = width + spacing * 2;
        }
    }

    m_boxWidth = columnPosition( m_colWidths.count() ) + 4;
    m_boxHeight = items().count() * itemHeight( currentItem() ) + 3;
}

int KCompletionTable::columnPosition( uint column )
{
    int width = 0;
    for ( uint i = 0; i < column; ++i )
    {
        width += m_colWidths[ i ];
    }
    return width;
}

QSize KCompletionTable::sizeHint() const
{
    return QSize( m_boxWidth, m_boxHeight );
}

void KCompletionTable::popup()
{
    KCompletionBox::popup();
}

void KCompletionTable::show()
{
    KCompletionBox::show();
    resize( m_boxWidth, m_boxHeight );
}

void KCompletionTable::resize( int /*w*/, int /*h*/ )
{
    m_boxHeight = items().count() * itemHeight( currentItem() ) + 3;
    if ( m_boxHeight > 300 )
        m_boxHeight = 300;

    int temp = m_boxWidth;
    if ( verticalScrollBar() ->isVisible() )
        temp += verticalScrollBar() ->width();

    if ( QWidget *p = ::qt_cast<QWidget*>( parent() ) )
        if ( temp < p->size().width() )
            temp = p->size().width();

    KCompletionBox::resize( temp, m_boxHeight );
}

void KCompletionTable::setupPainter( QPainter *p, QListBoxItem *i )
{
    bool drawActiveSelection = hasFocus() || !style().styleHint( QStyle::SH_ItemView_ChangeHighlightOnFocus, this );
    const QColorGroup &g = ( drawActiveSelection ? colorGroup() : palette().inactive() );

    QRect rect = itemRect( i );
    int cw = rect.right();
    int ch = itemHeight( currentItem() );

    p->save();
    if ( i->isSelected() )
    {
        if ( numColumns() == 1 )
        {
            p->fillRect( 0, 0, cw, ch, g.brush( QColorGroup::Highlight ) );
            p->setPen( g.highlightedText() );
            p->setBackgroundColor( g.highlight() );
        }
        else
        {
            int iw = i->width( this );
            p->fillRect( 0, 0, iw, ch, g.brush( QColorGroup::Highlight ) );
            p->fillRect( iw, 0, cw - iw + 1, ch,
                         g.brush( QPalette::backgroundRoleFromMode( viewport() ->backgroundMode() ) ) );
            p->setPen( g.highlightedText() );
            p->setBackgroundColor( g.highlight() );
        }
    }
    else
    {
        p->fillRect( 0, 0, cw, ch,
                     g.brush( QPalette::backgroundRoleFromMode( viewport() ->backgroundMode() ) ) );
    }

    if ( i->isCurrent() && hasFocus() )
    {
        if ( numColumns() > 1 )
            cw = i->width( this );

        style().drawPrimitive( QStyle::PE_FocusRect, p, QRect( 0, 0, cw, ch ), g,
                               QStyle::Style_FocusAtBorder,
                               QStyleOption( i->isSelected() ? g.highlight() : g.base() ) );
    }

    p->restore();
}

void KCompletionTable::paintCell( QPainter * p, int row, int /*col*/ )
{
    if ( m_countCache != count() )
        resize( m_boxWidth, m_boxHeight );

    QListBoxItem *cur = item( row );
    setupPainter( p, cur );

    QStringList rows = QStringList::split( '|', cur->text(), true );
    QFontMetrics fm = p->fontMetrics();

    for ( uint i = 0; i < m_colWidths.count() || i == 0 ; ++i )
    {
        int spacing = i == 0 ? 2 : 3;
        if ( i < rows.count() )
        {
            p->drawText(
                         spacing + columnPosition( i ),                 // x
                         0,                                             // y
                         m_colWidths[ i ] - spacing,                    // w
                         itemHeight( row ),                             // h
                         Qt::AlignLeft | Qt::AlignTop | Qt::SingleLine,
                         rows[ i ] );
        }

        if ( i != m_colWidths.count() - 1 )
        {
            QPen pen( p->pen() );
            p->setPen( colorGroup().mid() );
            p->drawLine( columnPosition( i + 1 ), 0, columnPosition( i + 1 ), itemHeight( row ) );
            p->setPen( pen );
        }
    }

    m_countCache = count();
}

#include "kcompletiontable.moc"
