/* 
** Utility header file
**
** This library and program is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
**   
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
**
** You should have received a copy of the GNU Library General Public
** License along with this library; if not, write to the Free
** Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
** MA 02111-1307, USA
*/

#ifndef __UTIL_H__
#define __UTIL_H__

#include <syslog.h>
#include <unistd.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "config.h"

#define DAV_HOME "http://dav.sourceforge.net"
#define DAV_VFS_TYPE "davfs"
#ifndef MAX_CODADEVS
#define MAX_CODADEVS 255
#endif
#ifndef CODA_PSDEV_MAJOR
#define CODA_PSDEV_MAJOR 67
#endif
#define DAV_DEV_NAME    "/dev/davfs"
#define DAV_PROC_MOUNTS "/proc/mounts"
#define DAV_TMP_DIR "/tmp/"
#define DAV_MAX_MKTMP_TRY 3   /* Let's try three time to get random name */

/* magic flag number to indicate "new" flags */
#ifndef MS_MGC_VAL
#define MS_MGC_VAL 0xC0ED0000 
#endif

#define DEFAULT_PROXY_PORT 80
#define DEFAULT_MODE 00666

#ifdef DEBUG
#define DBG0(x)   syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_DEBUG), x);
#define DBG1(x, a1)   syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_DEBUG), x, a1);
#define DBG2(x, a1, a2)   syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_DEBUG), x, a1, a2);
#define DBG3(x, a1, a2, a3)   syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_DEBUG), x, a1, a2, a3);
#define TRACE() TRACE() DBG0("- TRACE :"  __FUNCTION__ )
#else
#define DBG0(x) 
#define DBG1(x, a1)
#define DBG2(x, a1, a2)
#define DBG3(x, a1, a2, a3)
#define TRACE() 
#endif

/* Data structure for commandline arguments and options*/
typedef struct {
    /* Arguments and options concerning mounting */
    char *device;
    /*2005-05-29, werner, security fix, fixed mode
    uid_t uid;
    gid_t gid;
    mode_t file_mode;
    mode_t dir_mode;*/
    unsigned long int ro;
    unsigned long int nodev;
    unsigned long int noexec;
    unsigned long int nosuid;
    int user;
    /* Arguments and options concerning the WebDAV-resource */
    char *p_host;
    int p_port;
    int useproxy;
    int askauth;
    int locks;
} dav_args;

/* Frees all strings held by args and finally frees args. */
void dav_delete_args(dav_args *args);

/* Parses commandline arguments and options and stores them in args
 * or the appropriate global variables.
 * For arguments and options please see the usage()-funktion.
 * As soon as 'version' or 'help' is found, an appropriate message is printed
 * and exit(EXIT_SUCCESS) is called.
 * argc    : the number of arguments
 * argv[]  : array of argument strings
 * return value : args, containig the parsed options and arguments
 * Note: All strings returned in args are newly allocated.
 *       The calling function is responsible to free them. */
dav_args *dav_get_args(int argc, char *argv[]);

/* Reads the config files and checks and completes arguments and options.
 * It checks whether the program is setuid.
 * It checks whether the user has permissions for mounting.
 * In case of a non-root user the values in args are checked against an entry
 * in fstab for the given mount point to prevent normal users from calling
 * mount.davfs directly and thus circumvent control by mount and fstab.
 * It checks whether the davfs-device is free or searches for a free device
 * if none is given.
 * It does not check whether the mount point is used, as mounts may be
 * overridden.
 * If useproxy is set (default) the name of the proxy is looked up in the
 * config files. If not found the environment variable http_proxy is used
 * instead. If no proxy is found, useproxy is reset.
 * The creditentials for proxy and WebDAV-server are looked up in the
 * secrets files.
 * It checks for the existance of /proc/mounts.
 * args  : structure containing arguents and options
 * Note: All strings returned in args are newly allocated.
 *       The calling function is responsible to free them. */
void dav_check_config(dav_args *args);

/* Calls the mount()-function to mount the file system.
 * It uses dav_url as device, args->mpoint as mount point and coda as
 * file system type. The options ro/rw, noexec and suid are set according
 * the values in args. The dive file descriptor fd and the CODA_MOUNT_VERSION
 * are supplied in the mountdata argument for use by coda kernel code.
 * It also writes an entry into mtab. The options field contains ro/rw, noexec
 * and nosuid if appropriate according to args. It allways contains nodev,
 * device=<name of davfs-device> and user=<name of user>.
 * fd   : file descriptor of the davfs-device
 * args : structure containig mount-information */
int dav_mount(int fd, const dav_args *args);

/* Saves the pid of the mount-davfs daemon in the pid-file. The name of the
 * pid-file is taken from the global variable dav_pidfile. If an error
 * occurs during opening of the pid-file, an error message is printed and
 * exit(EXIT_FAILURE) is called. */
void dav_save_mount_pid(void);

/* The glabal variable dav_mounts[0] stores the name a the mtab-file. This is
 * looked up for an entry matching device dav_url and mount point dav_mpoint.
 * If an matching entry is found, 1 is returned, else 0.
 * return value : 0 - no matching entry in the mtab-file (not mounted)
 *                1 - matching entry in the mtab-file (mounted) */
int dav_still_mounted();

/* Remove trailing '/' from path str */
void dav_no_trail(char *str);

/* Creates a temporary filname  DAV_TMP_DIR/PACKAGE_NAME_prefix_XXXXXX,
 * where XXXXXX is choosen at random. The file is opened and the file
 * descriptor is stored in *fd. The filname is returned.
 * prefix :  will be part of the filename
 * *fd    : pointer to int, where the file descriptor will be stored
 * return value : string containing the filename. This string is newly
 *                allocated and the calling function is responsible to
 *                free it. */
char *dav_get_tempnam(const char *prefix, int *fd);

#endif

/* start emacs stuff */
/* 
 * local variables: 
 * eval: (load-file "../tools/davfs-emacs.el") 
 * end: 
 */ 

