/* 
** Interface to the Coda kernel module.
 */

/*
 *  This file is part of davfs2.
 *
 *  davfs2 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  davfs2 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with davfs2; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include "config.h"

#include <errno.h>
#include <error.h>
#include <fcntl.h>
#include <grp.h>
#include <limits.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <syslog.h>

#include <sys/mount.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <ne_alloc.h>
#include <ne_string.h>

#include "dav_debug.h"
#include "defaults.h"
#include "mount_davfs.h"
#include "cache.h"
#include "dav_coda.h"


/* Data Types */
/*============*/

#if SIZEOF_VOID_P == 8
typedef u_int64_t dav_upointer;
#else
typedef u_int32_t dav_upointer;
#endif


/* Private constants */
/*===================*/

/* Size of the buffer used for communication with the kernel module.
   Buffer will contain max. 1 PATH, 1 NAME and ca. 40 Byte of other data. */
#define DAV_BUFLEN 2 * CODA_MAXPATHLEN

/* The interface of the coda kernel module defines a data structure for
   every type of upcall. This structures are different for different
   CODA_KERNEL_VERSIONs. To allow the same mount.davfs-binary to acces the
   members of this different structures, davfs2 uses a two dimensional
   array that holds the offsets into this structures. The first dimension
   references the member variable to access, the second dimension selects
   the CODA_KERNEL_VERSION. dav_init_coda() will evaluate the
   CODA_KERNEL_VERSION and set the variable version accordingly. version will
   allways be used as the second index into the array. */

/* Enumeration of symbolic constants referencing the different member variables
   of all the structs coda_xxxx_xx. They will be used as first index into the
   array of offsets. */
enum {
    OC = 0,   /* Opcode specifying the kind of upcall. */
    ID,       /* ID that is unique for every upcall. */
    UID,
    RES,      /* Return code. */
    VOL_O,    /* Expected by coda, no meaning for davfs2. */
    VN_O,     /* Expected by coda, no meaning for davfs2. */
    FID_I,    /* Pointer to the dav_node; input from coda. */
    FID_O,    /* Pointer to the dav_node; output to coda. */
    FLAGS,
    NAME,     /* Offset to a name string. */
    TYPE,     /* Node type (directory or regular file). */
    ATTR_I,   /* coda_vattr structure; input from coda. */
    ATTR_O,   /* coda_vattr structure; output to coda. */
    GET_ATTR, /* coda_vattr structure; used in GETATTR upcall. */
    EXCL,     /* Exclusive flag in OPEN upcalls. */
    MODE,
    C_NAME,   /* Offset to a name string; use in CREATE upcall. */
    DIRNAME,  /* Offset to a name string; use in MKDIR upcall. */
    FD,
    DEV,
    INO,
    DST_FID,
    DST_NAME,
    STAT
};

/* Offsets for different member varialbles of struct coda_xxxx_xx.
   Currently two different versions of the coda interface are supported. */
size_t offset[][2] = {
    [OC]    = {
        offsetof(struct coda2_in_hdr, opcode),
        offsetof(struct coda3_in_hdr, opcode)
    },
    [ID]    = {
        offsetof(struct coda2_in_hdr, unique),
        offsetof(struct coda3_in_hdr, unique)
    },
    [UID]   = {
        offsetof(struct coda2_in_hdr, cred)
            + offsetof(struct coda_cred, cr_uid),
        offsetof(struct coda3_in_hdr, uid)
    },
    [RES]   = {
        offsetof(struct coda_out_hdr, result),
        offsetof(struct coda_out_hdr, result)
    },
    [VOL_O] = {
        offsetof(struct coda2_lookup_out, VFid.vol),
        offsetof(struct coda2_lookup_out, VFid.vol)
    },
    [VN_O] = {
        offsetof(struct coda2_lookup_out, VFid.vnode),
        offsetof(struct coda2_lookup_out, VFid.vnode)
    },
    [FID_I] = {
        offsetof(struct coda2_lookup_in, VFid.id),
        offsetof(struct coda3_lookup_in, VFid.id)
    },
    [FID_O] = {
        offsetof(struct coda2_lookup_out, VFid.id),
        offsetof(struct coda2_lookup_out, VFid.id)
    },
    [FLAGS] = {
        offsetof(struct coda2_lookup_in, name),
        offsetof(struct coda3_lookup_in, name)
    },
    [NAME] = {
        offsetof(struct coda2_lookup_in, name),
        offsetof(struct coda3_lookup_in, name)
    },
    [TYPE] = {
        offsetof(struct coda2_lookup_out, vtype),
        offsetof(struct coda3_lookup_out, vtype)
    },
    [ATTR_I]  = {
        offsetof(struct coda2_create_in, attr),
        offsetof(struct coda3_create_in, attr)
    },
    [ATTR_O]  = {
        offsetof(struct coda2_create_out, attr),
        offsetof(struct coda3_create_out, attr)
    },
    [GET_ATTR]  = {
        offsetof(struct coda_getattr_out, attr),
        offsetof(struct coda_getattr_out, attr)
    },
    [EXCL]  = {
        offsetof(struct coda2_create_in, excl),
        offsetof(struct coda3_create_in, excl)
    },
    [MODE]  = {
        offsetof(struct coda2_create_in, mode),
        offsetof(struct coda3_create_in, mode)
    },
    [C_NAME]  = {
        offsetof(struct coda2_create_in, name),
        offsetof(struct coda3_create_in, name)
    },
    [DIRNAME]  = {
        offsetof(struct coda2_create_in, excl),
        offsetof(struct coda3_create_in, excl)
    },
    [FD] = {
        offsetof(struct coda_open_by_fd_out, fd),
        offsetof(struct coda_open_by_fd_out, fd)
    },
    [DEV] = {
        offsetof(struct coda_open_out, dev),
        offsetof(struct coda_open_out, dev)
    },
    [INO] = {
        offsetof(struct coda_open_out, inode),
        offsetof(struct coda_open_out, inode)
    },
    [DST_FID] = {
        offsetof(struct coda2_rename_in, destFid.id),
        offsetof(struct coda3_rename_in, destFid.id)
    },
    [DST_NAME] = {
        offsetof(struct coda2_rename_in, destname),
        offsetof(struct coda3_rename_in, destname)
    },
    [STAT] = {
        offsetof(struct coda_statfs_out, stat),
        offsetof(struct coda_statfs_out, stat)
    }
};

/* Macro used to acces the data in buf. It does a typecast and uses
   the offsets from array offset. Global variable version is used as
   the second index into this array. */
#define var(type, member) (*((type *) (buf + offset[member][version])))

/* This array holds the size of the data returned to the kernel module.
   Note: For some upcalls there is an additional string. The length of
   this string must be added. */
size_t size_out[][2] = {
    [RES]      = {sizeof(struct coda_out_hdr),
                  sizeof(struct coda_out_hdr)},
    [TYPE]     = {sizeof(struct coda2_lookup_out),
                  sizeof(struct coda3_lookup_out)},
    [FID_O]    = {sizeof(struct coda2_root_out),
                  sizeof(struct coda3_root_out)},
    [ATTR_O]   = {sizeof(struct coda_getattr_out),
                  sizeof(struct coda_getattr_out)},
    [GET_ATTR] = {sizeof(struct coda_getattr_out),
                  sizeof(struct coda_getattr_out)},
    [FD]       = {sizeof(struct coda_open_by_fd_out),
                  sizeof(struct coda_open_by_fd_out)},
    [INO]      = {sizeof(struct coda_open_out),
                  sizeof(struct coda_open_out)},
    [STAT]      = {sizeof(struct coda_statfs_out),
                  sizeof(struct coda_statfs_out)}
};

/* Macro to get the size of data return to the kernel module. */
#define bytes_out(last_member) size_out[last_member][version]


/* Public global variables */
/*=========================*/

volatile int keep_on_running = 1;


/* Private global variables */
/*==========================*/

/* When there are no upcalls for that long, call idle function. */
time_t idle_time;

/* When to call the idle function even if not idle (in case there are a lot
   of useless file system calls, that prevent coda from getting bored). */
time_t next_idle;

/* The device, size of output data and buffers to communicate with
   coda kernel code. */
/* File descriptor of the open coda device. Will be set by dav_init_coda(). */
int device;

/* Buffer used for communication with the kernel module (in and out).
   It contains one of the struct coda_xxxx_xx. */
char buf[DAV_BUFLEN];

/* Will be set by dav_init_coda().
   Must be 0 for CODA_KERNEL_VERSION 2,
   and     1 for CODA_KERNEL_VERSION 3. */
int version;

/* Directory where to store the directory files. */
static const char *cache_dir;


/* Private function prototypes */
/*=============================*/

/* Functions to handle upcalls fromthe kernel module. */
static size_t dav_coda_access(void);
static size_t dav_coda_close(void);
static size_t dav_coda_create(void);
static size_t dav_coda_getattr(void);
static size_t dav_coda_lookup(void);
static size_t dav_coda_mkdir(void);
static size_t dav_coda_open(void);
static size_t dav_coda_open_by_fd(void);
static size_t dav_coda_open_by_path(void);
static size_t dav_coda_root(void);
static size_t dav_coda_setattr(void);

/* Functions that will do a downcall to the kernel module. */
static void dav_coda_zap_file(dav_node *node);

/* Auxiliary functions. */
static int common_open(int *flags);
static int create_dir_cache_file(dav_node *node);
static int eval_dir_size(const dav_node *node);
static int open_device(dav_args *args);
static void set_attr(struct coda_vattr *attr, const dav_node *node);


/* Public functions */
/*==================*/

void dav_close_coda(void) {

    close(device);
}


void dav_init_coda(dav_args *args) {

    idle_time = args->idle_time;

    seteuid(0);
    if (open_device(args) != 0)
        error(EXIT_FAILURE, 0, "No free coda device to mount.");
    seteuid(getuid());

    u_int coda_version;
    ioctl(device, CIOC_KERNEL_VERSION, &coda_version);
    DBG1("  Coda version %u.", coda_version);
    if (coda_version == 2) {
        version = 0;
    } else if (coda_version == 3) {
        version = 1;
    } else {
        error(EXIT_FAILURE, 0, "CODA version %u not supported.", coda_version);
    }

    cache_dir = dav_register_kernel_interface(&dav_coda_zap_file);
}


void dav_run_messageloop(void) {

    next_idle = time(NULL) + 2 * idle_time;
    while (keep_on_running) {

        fd_set fds;
        FD_ZERO(&fds);
        FD_SET(device, &fds);
        struct timeval tv;
        tv.tv_sec = idle_time;
        tv.tv_usec = 0;
        int ret = select(device + 1, &fds, NULL, NULL, &tv);

        if (time(NULL) > next_idle) {
            DBG0("IDLE:");
            dav_idle();
            next_idle = time(NULL) + idle_time;
        }

        if (ret > 0) {
            if (read(device, buf, DAV_BUFLEN) <= 0)
                continue;
        } else if (ret == 0) {
            if (!dav_is_mounted())
                break;
            continue;
        } else {
            break;
        }

        size_t len;
        switch (var(u_int32_t, OC)) {
        case CODA_ROOT:
            DBG0("CODA_ROOT:");
            len = dav_coda_root();
            break;
        case CODA_OPEN_BY_FD:
            DBG0("CODA_OPEN_BY_FD:");
            len = dav_coda_open_by_fd();
            break;
        case CODA_OPEN:
            DBG0("CODA_OPEN:");
            len = dav_coda_open();
            break;
        case CODA_CLOSE:
            DBG0("CODA_CLOSE:");
            len =  dav_coda_close();
            break;
        case CODA_IOCTL:
            DBG0("CODA_IOCTL: not supported");
            DBG1("  n %p", var(dav_node *, FID_I));
            var(u_int32_t, RES) = ENOTSUP;
            len =  bytes_out(RES);
            break;
        case CODA_GETATTR:
            DBG0("CODA_GETATTR:");
            len = dav_coda_getattr();
            break;
        case CODA_SETATTR:
            DBG0("CODA_SETATTR:");
            len = dav_coda_setattr();
            break;
        case CODA_ACCESS:
            DBG0("CODA_ACCESS:");
            len = dav_coda_access();
            break;
        case CODA_LOOKUP:
            DBG0("CODA_LOOKUP:");
            len = dav_coda_lookup();
            break;
        case CODA_CREATE:
            DBG0("CODA_CREATE:");
            len = dav_coda_create();
            break;
        case CODA_REMOVE:
            DBG0("CODA_REMOVE:");
            DBG2("  p %p, %s", var(dav_node *, FID_I),
                 buf + var(int, NAME));
            var(u_int32_t, RES) = dav_remove(var(dav_node *, FID_I),
                                             buf + var(int, NAME),
                                             var(vuid_t, UID));
            DBG1("  return: %i", var(u_int32_t, RES));
            len = bytes_out(RES);
            break;
        case CODA_LINK:
            DBG0("CODA_LINK: not supported");
            var(u_int32_t, RES) = ENOTSUP;
            len =  bytes_out(RES);
            break;
        case CODA_RENAME:
            DBG0("CODA_RENAME:");
            DBG2("  sp %p, %s", var(dav_node *, FID_I),
                 buf + var(int, NAME));
            DBG2("  dp %p, %s", var(dav_node *, DST_FID),
                 buf + var(int, DST_NAME));
            var(u_int32_t, RES) = dav_rename(var(dav_node *, FID_I),
                                             buf + var(int, NAME),
                                             var(dav_node *, DST_FID),
                                             buf + var(int, DST_NAME),
                                             var(vuid_t, UID));
            DBG1("  return: %i", var(u_int32_t, RES));
            len = bytes_out(RES);
            break;
        case CODA_MKDIR:
            DBG0("CODA_MKDIR:");
            len = dav_coda_mkdir();
            break;
        case CODA_RMDIR:
            DBG0("CODA_RMDIR:");
            DBG2("  p %p, %s", var(dav_node *, FID_I),
                 buf + var(int, NAME));
            var(u_int32_t, RES) = dav_rmdir(var(dav_node *, FID_I),
                                            buf + var(int, NAME),
                                            var(vuid_t, UID));
            DBG1("  return: %i", var(u_int32_t, RES));
            len = bytes_out(RES);
            break;
        case CODA_SYMLINK:
            DBG0("CODA_SYMLINK: not supported");
            var(u_int32_t, RES) = ENOTSUP;
            len =  bytes_out(RES);
            break;
        case CODA_READLINK:
            DBG0("CODA_READLINK: not supported");
            var(u_int32_t, RES) = ENOTSUP;
            len =  bytes_out(RES);
            break;
        case CODA_FSYNC:
            DBG0("CODA_FSYNC:");
            DBG1("  n %p", var(dav_node *, FID_I));
            var(u_int32_t, RES) = dav_sync(var(dav_node *, FID_I));
            len =  bytes_out(RES);
            break;
        case CODA_VGET:
            DBG0("CODA_VGET: not supported");
            var(u_int32_t, RES) = ENOTSUP;
            len =  bytes_out(RES);
            break;
        case CODA_OPEN_BY_PATH:
            DBG0("CODA_OPEN_BY_PATH:");
            len = dav_coda_open_by_path();
            break;
        case CODA_STATFS:
            DBG0("CODA_STATFS:");
            var(struct coda_statfs, STAT).f_blocks = 0xC00000;
            var(struct coda_statfs, STAT).f_bfree = 0x6A6000;
            var(struct coda_statfs, STAT).f_bavail = 0x6A0000;
            var(struct coda_statfs, STAT).f_files = 0x400000;
            var(struct coda_statfs, STAT).f_ffree = 0x237000;
            var(u_int32_t, RES) = 0;
            len =  bytes_out(STAT);
            break;
        case CODA_STORE:
            DBG0("CODA_STORE:");
            DBG2("  n %p, f %x", var(dav_node *, FID_I), var(int, FLAGS));
            var(u_int32_t, RES) = dav_sync(var(dav_node *, FID_I));
            len =  bytes_out(RES);
            break;
        case CODA_RELEASE:
            DBG0("CODA_RELEASE:");
            len =  dav_coda_close();
            break;
        default:
            DBG1("UNKNOWN CODA CALL %i", var(u_int32_t, OC));
            var(u_int32_t, RES) = ENOTSUP;
            len =  bytes_out(RES);
            break;
        }
        
        write(device, buf, len);
    }
}


/* Private functions */
/*===================*/

/* Functions to handle upcalls fromthe kernel module.
   The cache module only uses data types from the C-library. For file access,
   mode and the like it only uses symbolic constants defined in the C-library.
   So the main porpose of this functions is to translate from kernel specific
   types and constants to types and constants from the C-library, and back.
   The dav_coda_open_xxx functions, when called for a directory, will
   additionally create the cache file, that holds the direntries, and fill
   in the member size of dav_node. This is necessary, as the structure of
   direntries also depends from the kernel module and will not be indentical
   to the one defined in the C-library.
   All of this functions return the amount of data in buf that is to be
   send to the kernel module. */

static size_t dav_coda_access(void) {

    DBG2("  n %p, f %x", var(dav_node *, FID_I), var(int, FLAGS));
    int how = (var(int, FLAGS) & C_A_R_OK) ? R_OK : 0;
    how |= (var(int, FLAGS) & C_A_W_OK) ? W_OK : 0;
    how |= (var(int, FLAGS) & C_A_X_OK) ? X_OK : 0;
    how |= (var(int, FLAGS) & C_A_F_OK) ? F_OK : 0;
    
    var(u_int32_t, RES) = dav_access(var(dav_node *, FID_I),
                                     var(vuid_t, UID), how);
    DBG1("  return: %i", var(u_int32_t, RES));
    return bytes_out(RES);
}


/* I don't know, when the kernel module sends upcall CLOSE and when RELEASE.
   There seems to be no difference in the meaning. */
static size_t dav_coda_close(void) {

    DBG2("  n %p, f %x", var(dav_node *, FID_I), var(int, FLAGS));
    int flags = 0;
    if ((var(int, FLAGS) & C_O_READ) && (var(int, FLAGS) & C_O_WRITE)) {
        flags = O_RDWR;
    } else if (var(int, FLAGS) & C_O_READ) {
        flags = O_RDONLY;
    } else if (var(int, FLAGS) & C_O_WRITE) {
        flags = O_WRONLY;
    }

    dav_node *node = var(dav_node *, FID_I);
    var(u_int32_t, RES) = dav_close(node, var(vuid_t, UID), flags);
    if (var(u_int32_t, RES) != 0) {
        DBG1("  return: %i", var(u_int32_t, RES));
        return bytes_out(RES);
    }

    if (node->o_read == 0 && node->fd_read > 0) {
        close(node->fd_read);
        node->fd_read = 0;
    }

    if (node->o_write == 0 && node->fd_write > 0) {
        close(node->fd_write);
        node->fd_write = 0;
    }

    DBG1("  return: %i", var(u_int32_t, RES));
    return bytes_out(RES);
}


static size_t dav_coda_create(void) {

    DBG2("  p %p, m %o", var(dav_node *, FID_I), var(int, MODE));
    DBG1("  %s", buf + var(int, C_NAME));
    dav_node *node = NULL;
    var(u_int32_t, RES) = dav_create(&node, var(dav_node *, FID_I),
                                     buf + var(int, C_NAME),
                                     var(uid_t, UID), var(int, EXCL),
                                     var(int, MODE) & DAV_A_MASK);

    if (var(u_int32_t, RES) != 0 || node == NULL) {
        DBG1("  return: %i", var(u_int32_t, RES));
        return bytes_out(RES);
    }

    var(u_int32_t, VOL_O) = DAV_VOL;
    var(u_int32_t, VN_O) = DAV_VNODE;
    var(dav_node *, FID_O) = node;
    set_attr(&var(struct coda_vattr, ATTR_O), node);
    var(u_int32_t, RES) = 0;
    DBG0("  return: OK");
    return bytes_out(ATTR_O);
}


static size_t dav_coda_getattr(void) {

    DBG1("  n %p", var(dav_node *, FID_I));
    dav_node *node = var(dav_node *, FID_I);
    var(u_int32_t, RES) = dav_getattr(node, var(vuid_t, UID));
    if (var(u_int32_t, RES) != 0) {
        DBG1("  return: %i", var(u_int32_t, RES));
        return bytes_out(RES);
    }

    if (S_ISDIR(node->mode))
        node->size = eval_dir_size(node);
    set_attr(&var(struct coda_vattr, GET_ATTR), node);
    DBG0("  return: OK");
    return bytes_out(GET_ATTR);
}


static size_t dav_coda_lookup(void) {

    DBG2("  p %p, %s", var(dav_node *, FID_I), buf + var(int, NAME));
    dav_node *node = NULL;
    var(u_int32_t, RES) = dav_lookup(&node, var(dav_node *, FID_I),
                                     buf + var(int, NAME), var(vuid_t, UID));
    if (var(u_int32_t, RES) != 0 || node == NULL) {
        DBG1("  return: %i", var(u_int32_t, RES));
        return bytes_out(RES);
    }

    var(u_int32_t, VOL_O) = DAV_VOL;
    var(u_int32_t, VN_O) = DAV_VNODE;
    var(dav_node *, FID_O) = node;
    var(u_int32_t, TYPE) = (node->mode & S_IFDIR) ? CDT_DIR : CDT_REG;
    var(u_int32_t, RES) = 0;
    DBG0("  return: OK");
    return bytes_out(TYPE);
}


static size_t dav_coda_mkdir(void) {

    DBG2("  p %p, %s", var(dav_node *, FID_I), buf + var(int, DIRNAME));
    dav_node *node = NULL;
    var(u_int32_t, RES) = dav_mkdir(&node, var(dav_node *, FID_I),
                                    buf + var(int, DIRNAME), var(uid_t, UID),
                                    var(struct coda_vattr, ATTR_I).va_mode
                                        & DAV_A_MASK);

    if (var(u_int32_t, RES) != 0 || node == NULL) {
        DBG1("  return: %i", var(u_int32_t, RES));
        return bytes_out(RES);
    }

    node->size = eval_dir_size(node);
    var(u_int32_t, VOL_O) = DAV_VOL;
    var(u_int32_t, VN_O) = DAV_VNODE;
    var(dav_node *, FID_O) = node;
    set_attr(&var(struct coda_vattr, ATTR_O), node);
    var(u_int32_t, RES) = 0;
    DBG0("  return: OK");
    return bytes_out(ATTR_O);
}


/* Note: The coda kernel module defines three types of open upcalls.
   Each of this may be used for regular files als well as for directories. */
static size_t dav_coda_open(void) {

    int flags = 0;
    if (common_open(&flags) != 0)
        return bytes_out(RES);

    struct stat st;
    var(u_int32_t, RES) = stat(var(dav_node *, FID_I)->cache_path, &st);
    if (var(u_int32_t, RES) != 0) {
        DBG1("  return: %i", var(u_int32_t, RES));
        return bytes_out(RES);
    }

    var(cdev_t, DEV) = st.st_dev;
    var(ino_t, INO) = st.st_ino;
    DBG0("  return: OK");
    return bytes_out(INO);
}


/* Will open the node->cache_path for read or write and store the file
   descriptor in node->fd_read or node->fd_write. When node->o_read or
   node->o_write becomes 0, the file is closed again and node->fd_xxxx
   is set to 0. */
static size_t dav_coda_open_by_fd(void) {

    int flags = 0;
    if (common_open(&flags) != 0)
        return bytes_out(RES);

    if ((O_ACCMODE & flags) == O_RDONLY) {
        if (var(dav_node *, FID_I)->fd_read <= 0 )
            var(dav_node *, FID_I)->fd_read =
                    open(var(dav_node *, FID_I)->cache_path, flags);
        if (var(dav_node *, FID_I)->fd_read <= 0) {
            var(u_int32_t, RES) = errno;
            DBG1("  return: %i", var(u_int32_t, RES));
            return bytes_out(RES);
        }
        var(int, FD) = var(dav_node *, FID_I)->fd_read;
    } else {
        if (var(dav_node *, FID_I)->fd_write <= 0 )
            var(dav_node *, FID_I)->fd_write =
                    open(var(dav_node *, FID_I)->cache_path, flags);
        if (var(dav_node *, FID_I)->fd_write <= 0) {
            var(u_int32_t, RES) = errno;
            DBG1("  return: %i", var(u_int32_t, RES));
            return bytes_out(RES);
        }
        var(int, FD) = var(dav_node *, FID_I)->fd_write;
    }

    DBG0("  return: OK");
    return bytes_out(FD);
}


static size_t dav_coda_open_by_path(void) {

    int flags = 0;
    if (common_open(&flags) != 0)
        return bytes_out(RES);

    char *name = var(dav_node *, FID_I)->cache_path;
    if ((strlen(name) + 1) > CODA_MAXPATHLEN) {
        var(u_int32_t, RES) = EIO;
        DBG1("  return: %i", var(u_int32_t, RES));
        return bytes_out(RES);
    }
    var(int, FD) = bytes_out(FD);
    strcpy(buf + bytes_out(FD), name);
    DBG0("  return: OK");
    return bytes_out(FD) + strlen(name) + 1;
}


static size_t dav_coda_root(void) {

    var(u_int32_t, RES) = dav_root(&var(dav_node *, FID_O), var(vuid_t, UID));
    if (var(u_int32_t, RES) != 0) {
        DBG1("  return: %i", var(u_int32_t, RES));
        return bytes_out(RES);
    }

    var(u_int32_t, VOL_O) = DAV_VOL;
    var(u_int32_t, VN_O) = DAV_VNODE;
    var(u_int32_t, RES) = 0;
    DBG0("  return: OK");
    return bytes_out(FID_O);
}

/* From the documentation of the coda kernel module:
   The only attributes which the FS driver may request to change are the
   mode, owner, groupid, atime, mtime and ctime. */
static size_t dav_coda_setattr(void) {

    struct coda_vattr *attr = &var(struct coda_vattr, ATTR_I);
    DBG2("  n %p, m %o", var(dav_node *, FID_I), attr->va_mode);
    DBG2("  uid: %i, gid: %i", attr->va_uid, attr->va_gid);
    DBG2("  at %li, mt %li", attr->va_atime.tv_sec, attr->va_mtime.tv_sec);
    var(u_int32_t, RES) = dav_setattr(var(dav_node *, FID_I), var(vuid_t, UID),
                                      attr->va_mode != USHRT_MAX,
                                      attr->va_mode & DAV_A_MASK,
                                      attr->va_uid != DAV_ID_MAX, attr->va_uid,
                                      attr->va_gid != DAV_ID_MAX, attr->va_gid,
                                      attr->va_atime.tv_sec != -1,
                                      attr->va_atime.tv_sec,
                                      attr->va_mtime.tv_sec != -1,
                                      attr->va_mtime.tv_sec);

    DBG1("  return: %i", var(u_int32_t, RES));
    return bytes_out(RES);
}


/* Downcall to inform the kernel node does no longer exist. */
static void dav_coda_zap_file(dav_node *node) {

    DBG0("  CODA_ZAP_FILE:");
    DBG2("  n %p, %s", node, node->name);
    char buf[bytes_out(FID_O)];
    var(u_int32_t, OC) = CODA_ZAPFILE;
    var(u_int32_t, ID) = 0;
    var(u_int32_t, RES) = 0;
    var(u_int32_t, VOL_O) = DAV_VOL;
    var(u_int32_t, VN_O) = DAV_VNODE;
    var(dav_node *, FID_O) = node;
    write(device, buf, bytes_out(FID_O));
}


/* Except form the usual translation of data, in case of an open on a directory
   it will guarantee that node->cache_path is a file containing the
   dir entries. */
static int common_open(int *flags) {

    DBG2("  n %p, f %x", var(dav_node *, FID_I), var(int, FLAGS));
    if ((var(int, FLAGS) & C_O_READ) && (var(int, FLAGS) & C_O_WRITE)) {
        *flags = O_RDWR;
    } else if (var(int, FLAGS) & C_O_READ) {
        *flags = O_RDONLY;
    } else if (var(int, FLAGS) & C_O_WRITE) {
        *flags = O_WRONLY;
    }
    *flags |= (var(int, FLAGS) & C_O_CREAT) ? O_CREAT : 0;
    *flags |= (var(int, FLAGS) & C_O_EXCL) ? O_EXCL : 0;
    *flags |= (var(int, FLAGS) & C_O_TRUNC) ? O_TRUNC : 0;

    var(u_int32_t, RES) = dav_open(var(dav_node *, FID_I),
                                   var(vuid_t, UID), *flags);
    if (var(u_int32_t, RES) != 0) {
        DBG1("  return: %i", var(u_int32_t, RES));
        return -1;
    }

    if ((var(dav_node *, FID_I)->mode & S_IFDIR)
            && var(dav_node *, FID_I)->cache_path == NULL) {
        var(u_int32_t, RES) = create_dir_cache_file(var(dav_node *, FID_I));
        if (var(u_int32_t, RES) != 0) {
            DBG0("  return: EIO");
            return -1;
        }
    }

    return 0;
}

/* Creates the file node->cache_path containig the dir entries.
   Note: Member d_fileno of struct venus_dirent has no meaning for davfs:
   It tries to make it unique. It takes the address of node and removes
   those least significant bits that are, due to alignment, allways zero.
   On 64-bit systems there is still a chance of duplicat d_filno values,
   as d_fileno is only 32 bit wide. */
static int create_dir_cache_file(dav_node *node) {

    node->cache_path = ne_concat(cache_dir, "/dir_", node->name, "_XXXXXX",
                                 NULL);
    int fd = mkstemp(node->cache_path);
    if (fd <= 0) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "Error %i creating: %s", errno, node->cache_path);
        NE_FREE(node->cache_path);
        return EIO;
    }

    size_t offset = 0;
    struct venus_dirent entry;
    entry.d_fileno = (dav_upointer) node / alignment;
    entry.d_type = CDT_DIR;
    entry.d_namlen = 1;
    entry.d_reclen = DIRSIZ(&entry);
    strncpy(entry.d_name, ".", 4);
    int ret = write(fd, &entry, entry.d_reclen);
    if (ret != entry.d_reclen) {
        close(fd);
        remove(node->cache_path);
        NE_FREE(node->cache_path);
        return EIO;
    }
    offset += ret;

    entry.d_fileno = (node->parent == NULL)
                     ? 1 : (dav_upointer) node->parent / alignment;
    entry.d_type = CDT_DIR;
    entry.d_namlen = 2;
    entry.d_reclen = DIRSIZ(&entry);
    strncpy(entry.d_name, "..", 4);
    ret = write(fd, &entry, entry.d_reclen);
    if (ret != entry.d_reclen) {
        close(fd);
        remove(node->cache_path);
        NE_FREE(node->cache_path);
        return EIO;
    }
    offset += ret;

    dav_node *n = node->childs;
    while (n != NULL) {
        entry.d_fileno = (dav_upointer) n / alignment;
        entry.d_type = (n->mode & S_IFDIR) ? CDT_DIR : CDT_REG;
        entry.d_namlen = (strlen(n->name) > CODA_MAXNAMLEN)
                         ? CODA_MAXNAMLEN : strlen(n->name);
        entry.d_reclen = DIRSIZ(&entry);
        int len = (entry.d_namlen + 1 + 3) & ~3;
        strncpy(entry.d_name, n->name, len);
        ret = write(fd, &entry, entry.d_reclen);
        if (ret != entry.d_reclen) {
            close(fd);
            remove(node->cache_path);
            NE_FREE(node->cache_path);
            return EIO;
        }
        offset += ret;
        n = n->next;
    }

    memset(&entry, 0, sizeof(struct venus_dirent));
    node->size = (offset + DAV_MIN_DENTRY + (DAV_BLKSIZE - 1))
                 & ~(DAV_BLKSIZE - 1);
    while (offset < node->size) {
        entry.d_reclen = ((node->size - offset) > sizeof(struct venus_dirent))
                         ? sizeof(struct venus_dirent) : (node->size - offset);
        ret = write(fd, &entry, entry.d_reclen);
        if (ret != entry.d_reclen) {
            close(fd);
            remove(node->cache_path);
            NE_FREE(node->cache_path);
            return EIO;
        }
        offset += ret;
    }
    
    if (fchmod(fd, node->mode & (S_IRWXU | S_IRWXG | S_IRWXO)) != 0) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "Error %i changing mode: %s", errno, node->cache_path);
        close(fd);
        remove(node->cache_path);
        NE_FREE(node->cache_path);
        return EIO;
    }

    TEMP_FAILURE_RETRY (close(fd));
    return 0;
}


/* Evaluates the directory size. Number of file, length of file names
   and size of struct venus_dirent are taken into account. The value
   is always a multiple of DAV_BLKSIZE. */
static int eval_dir_size(const dav_node *node) {

    struct venus_dirent entry;
    entry.d_namlen = 2;
    size_t size = DIRSIZ(&entry);
    entry.d_namlen = 1;
    size += DIRSIZ(&entry);

    dav_node *n = node->childs;
    while (n != NULL) {
        entry.d_namlen = strlen(n->name);
        size += DIRSIZ(&entry);
        n = n->next;
    }
    
    size = (size + DAV_MIN_DENTRY + (DAV_BLKSIZE - 1)) & ~(DAV_BLKSIZE - 1);
    return size;
}


/* Starting with minor number 0 a free coda device is searched and if
   necessary created. The device is opend and the file descriptor stored
   in private global variable device and in args->device. Group of the
   device is set to args->dav_group and mode is changed appropriately. */
static int open_device(dav_args *args) {

    args->minor = 0;
    while (args->minor < MAX_CODADEVS) {
        char *path;
        if (asprintf(&path, "%s/%s%i", DAV_DEV_DIR, DAV_DEV_NAME, args->minor)
                < 0)
            abort();
        if (access(path, F_OK) != 0)
            mknod(path, S_IFCHR, makedev(CODA_PSDEV_MAJOR, args->minor));
        args->device = open(path, O_RDWR);
        if (args->device > 0) {
            chown(path, 0, args->dav_group);
            chmod(path, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
            NE_FREE(path);
            device = args->device;
            return 0;
        }
        NE_FREE(path);
        ++args->minor;
    }
    return -1;
}


/* Translates attribute from node to attr.
   Note: Members va_fileid, v_gen, va_flags, va_rdev and va_filerev have no
   meaning for davfs. va_fileid is treated like d_fileno in struct venus_dirent,
   the other are set to zero. The meaning of va_type is not clear at all.
   Times are only set with 1 second precision, as this is the precision of the
   last-modified time in HTTP. */
static void set_attr(struct coda_vattr *attr, const dav_node *node) {

    attr->va_type = 0;
    attr->va_mode = node->mode;
    if (S_ISDIR(node->mode)) {
        attr->va_nlink = node->nref;
    } else {
        attr->va_nlink = 1;
    }
    attr->va_uid = node->uid;
    attr->va_gid = node->gid;
    attr->va_fileid = (dav_upointer) node / alignment;
    attr->va_size = node->size;
    attr->va_blocksize = DAV_BLKSIZE;
    attr->va_atime.tv_sec = node->atime;
    attr->va_atime.tv_nsec = 0;
    attr->va_mtime.tv_sec = node->mtime;
    attr->va_mtime.tv_nsec = 0;
    attr->va_ctime.tv_sec = node->ctime;
    attr->va_ctime.tv_nsec = 0;
    attr->va_gen = 0;
    attr->va_flags = 0;
    attr->va_rdev = 0;
    attr->va_bytes = node->size;
    attr->va_filerev = 0;

}
