/* 
** Send requests to the WebDAV server.
 */

/*
 *  This file is part of davfs2.
 *
 *  davfs2 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  davfs2 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with davfs2; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef __WEBDAV_H__
#define __WEBDAV_H__


/* Data Types */
/*============*/

/* This structure holds the properties retrieved from the server.
   Usually a linked list of these is returned by dav_get_collection().
   Unused pointers should be set to NULL, integer types to 0. */
typedef struct dav_props dav_props;
struct dav_props {
    char *name;         /* The name of the file or directory. Only the last
                           component (no path), no slashes. */
    off_t size;         /* File size in bytes (regular files only). */
    time_t ctime;       /* Creation date. */
    time_t mtime;       /* Date of last modification. */
    int is_dir;         /* Boolean; 1 if a directory. */
    int is_exec;        /* -1 if not specified; 1 is executeable;
                           0 not executeable. */
    dav_props *next;    /* Next in the list. */
};


/* Function prototypes */
/*=====================*/

/* Creates and initializes a neon_session, using configuration information
   from args.
   The server, path, lock store and credetnials are store in the session
   using ne_set_session_private().
   args : structure containing configuration data. */
void dav_init_webdav(const dav_args *args);


/* Frees the session and its context.
   All locks held on the server are released, if possible. */
void dav_close_webdav(void);


/* Deletes the file path on the server.
   path : file name, relative to the path of the root collection. With leading
          but without trailing slash.
   return value : 0 on success; an appropriate file error code otherwise. */
int dav_delete(const char *path);


/* Deletes the collection path on the server.
   path : name of the collection, relative to the path of the root collection.
          With leading but without trailing slash. The path of the root
          collection is the empty string.
   return value : 0 on success; an appropriate file error code otherwise. */
int dav_delete_collection(const char *path);


/* Retrieves properties for the directory named by path and its
   direct childs (depth 1) from the server.
   Path path is relative to the mountpoint. It is concatenated with the URL
   stored in the session to build the complete URL for this directory.
   The properties are returned as a linked list of dav_props. If successfull,
   this list contains at least one entry (the directory itself; its name is
   the empty string). The calling function is responsible for freeing the list
   and all the strings included.
   path : The path of the directory relative to the root collection.
          It must not have a trailing slash, but a leading slash, unless it
          is the root of the file system (the mount point). The path of the
          root of the filesystem is just the empty string (not a slash).
   *props : will point to the list of properties on return.
   return value : 0 on success; an appropriate file error code otherwise. */
int dav_get_collection(dav_props **props, const char *path);


/* Fetches file path from the server and stores it in file cache_path. The
   values of ETag, Last-Modified time and file size are stored in etag, mtime
   and size. If *etag is not NULL and/or mtime is not 0, a conditional GET
   will be performed and nothing will be changed it the file has not been
   modified on the server.
   cache_path : name of the cache file to store the file in.
   path       : file name, relative to the path of the root collection. With
                leading but without trailing slash.
   etag       : contains the ETag of the cached version and will be replaced
                by the new value of the file has been changed.
   mtime      : the Last-Modified value of the cached version. Will be
                updated if the file has been changed.
   size       : will be set to the size of the file.
   return value : 0 on success; an appropriate file error code otherwise.
                  Not-Modified counts as success. */
int dav_get_file(const char *cache_path, const char *path, char **etag,
                 time_t *mtime, off_t *size);


/* Tests for the existence of file path.
   path : file name, relative to the path of the root collection. With leading
          but without trailing slash.
   return value : 0 if the file exists; an appropriate file error code
                  otherwise. */
int dav_head(const char *path);


/* Locks the file path on the server with an excluse write lock. On return
   the boolean parameter locked indicates whether the file is locked.
   If the session is initialized with the nolocks option, it does not lock
   the resource but allways returns success.
   If the file is already locked, but not by this instance of davfs2, it will
   try of the lock is from the same user using davfs2, and if so, to use this
   lock.
   path : file name, relative to the path of the root collection. With leading
          but without trailing slash.
   return value : 0 on success; an appropriate file error code
                  otherwise. */
int dav_lock(const char *path, int *locked);


/* Checks of there is already a lock on file path. If so it checks if the lock
   owner is the same user using davfs2. In this case it will regard itself as
   owner of the lock and use it.
   If the session is initialized with the nolocks option, it does nothing but
   return success.
   path : file name, relative to the path of the root collection. With leading
          but without trailing slash.
   return value : 0 if no error occured; an appropriate file error code
                  otherwise. */
int dav_lock_discover(const char *path);


/* Creates a new collection on the server.
   path : name of the collection, relative to the path of the root collection.
          With leading but without trailing slash.
   return value : 0 on success; an appropriate file error code otherwise. */
int dav_make_collection(const char *path);


/* Moves file src to the new name/location dst.
   src : old file name, relative to the path of the root collection. With
         leading but without trailing slash.
   dst : new file name, relative to the path of the root collection. With
         leading but without trailing slash.
   return value : 0 on success; an appropriate file error code otherwise. */
int dav_move(const char *src, const char *dst);


/* Moves collection src to the new name/location dst.
   src : old name of the collection, relative to the path of the root
         collection. With leading but without trailing slash.
   dst : new name of the collection, relative to the path of the root
         collection. With leading but without trailing slash.
   return value : 0 on success; an appropriate file error code otherwise. */
int dav_move_collection(const char *src, const char *dst);


/* Stores the contents of file cache_path on the server location path and
   updates the value of etag and mtime.
   cache_path : name of the file the content of which will be stored on the
                server.
   path       : file name, relative to the path of the root collection. With leading
                leading but without trailing slash.
   etag       : pointer to a string to return the new ETag value there.
   mtime      : pointer to time_t to return the Last_Modified value.
   return value : 0 on success; an appropriate file error code otherwise. */
int dav_put(const char *cache_path, const char *path, char **etag,
            time_t *mtime);

/* Sets or resets the execute propertie of file path.
   path : file name, relative to the path of the root collection. With leading
          leading but without trailing slash.
   set  : boolean value; 0 reset execute propertie; 1 set execute propertie. */
int dav_set_execute(const char *path, int set);


/* Tells webdav that no more terminal is available, so errors can only
 * be logged. Before this function is invoced webdav tries to 
 * communicate with the user when problems occur. */
void dav_set_no_terminal(void);


/* Releases the lock on file path on the server. On return the boolean
   parameter locked indicates whether the file is locked or not.
   If the session is initialized with the nolocks option, it does nothing but
   return success.
   path : file name, relative to the path of the root collection. With leading
          but without trailing slash.
   return value : 0 if no error occured; an appropriate file error code
                  otherwise. */
int dav_unlock(const char *path, int *locked);

#endif
