/* Example program demonstrating the capabilities/interface of Dazuko.
   Written by John Ogness <jogness@antivir.de>

   Copyright (C) 2002-2003, H+BEDV Datentechnik GmbH
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

   3. Neither the name of Dazuko nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE.
*/

/* This program is a simple application demonstrating how to interface
   with Dazuko. The program instructs Dazuko to detect open, close, and
   execute accesses within the /home and /usr/home directories (or any
   subdirectory thereof). The program than prints out the accesses and
   instructs Dazuko to allow access.

   An application communicates with Dazuko through a device. After
   installing Dazuko (as kernel module or after building in the
   kernel itself), the device node must be created. The major device
   number is needed. This can be found by looking at /proc/devices.

   example:
   > grep dazuko /proc/devices
   < 254 dazuko
   > mknod -m 600 /dev/dazuko c 254 0

   After Dazuko is loaded and the device node is created, this example
   program SHOULD run (as is) without any problems.

   Note: For security reasons, Dazuko will only interact with applications
         running as root. Therefore, this example program must also be run
         as root.
 */

#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include "dazukoio.h"

int	RUNNING = 1;	/* flag for main loop */

void sigterm(int sig)
{
	/* exit the loop on TERM or INT signal */

	printf("detected signal: %d\n", sig);

	RUNNING = 0;
	signal(sig, sigterm);
}

void print_access(struct access_t *acc_ptr)
{
	/* print access data */

	switch (acc_ptr->event)
	{
		case ON_OPEN:
			printf("OPEN ");
			break;
		case ON_CLOSE:
			printf("CLOSE ");
			break;
		case ON_CLOSE_MODIFIED:
			printf("CLOSE (modified) ");
			break;
		case ON_EXEC:
			printf("EXEC ");
			break;
		default:
			printf("???? ");
			break;
	}
	printf("flags:%d ", acc_ptr->o_flags);
	printf("mode:%d ", acc_ptr->o_mode);
	printf("uid:%d ", acc_ptr->uid);
	printf("pid:%d ", acc_ptr->pid);
	printf("file:%s ", acc_ptr->filename);
	printf("\n");

	fflush(stdout);
}

int main(void)
{
	struct access_t	acc;

	/* make sure we are root */
	if (getuid() != 0)
	{
		printf("only root can run this program\n");
		return -1;
	}

	/* register with dazuko */
	if (dazukoRegister("DAZUKO_EXAMPLE") != 0)
	{
		printf("error: failed to register with Dazuko\n");
		return -1;
	}

	printf("registered with Dazuko successfully\n");

	/* detect TERM signals */
	signal(SIGTERM, sigterm);

	/* detect INT signals */
	signal(SIGINT, sigterm);

	/* set access mask */
	if (dazukoSetAccessMask(ON_OPEN | ON_CLOSE | ON_CLOSE_MODIFIED | ON_EXEC) != 0)
	{
		printf("error: failed to set access mask\n");
		dazukoUnregister();
		return -1;
	}

	printf("set access mask successfully\n");

	/* set scan path */
	if (dazukoAddIncludePath("/home/") != 0)
	{
		printf("error: failed to add /home/ include path\n");
		dazukoUnregister();
		return -1;
	}

	/* set scan path */
	if (dazukoAddIncludePath("/usr/home/") != 0)
	{
		printf("error: failed to add /usr/home/ include path\n");
		dazukoUnregister();
		return -1;
	}

	printf("set scan path successfully\n");

	while (RUNNING)
	{
		/* get an access */
		if (dazukoGetAccess(&acc) == 0)
		{
			print_access(&acc);

			/* always allow access */
			acc.deny = 0;

			/* return access (IMPORTANT, the kernel is waiting for us!) */
			if (dazukoReturnAccess(&acc) != 0)
			{
				printf("error: failed to return access\n");
				RUNNING = 0;
			}
		}
		else
		{
			printf("warning: failed to get an access\n");
			RUNNING = 0;
		}
	}

	/* unregister with Dazuko */
	if (dazukoUnregister() != 0)
	{
		printf("error: failed to unregister with Dazuko\n");
		return -1;
	}

	printf("unregistered with Dazuko successfully\n");

	return 0;
}
