/* Copyright (C) 2004 - 2007  db4objects Inc.  http://www.db4o.com

This file is part of the db4o open source object database.

db4o is free software; you can redistribute it and/or modify it under
the terms of version 2 of the GNU General Public License as published
by the Free Software Foundation and as clarified by db4objects' GPL 
interpretation policy, available at
http://www.db4o.com/about/company/legalpolicies/gplinterpretation/
Alternatively you can write to db4objects, Inc., 1900 S Norfolk Street,
Suite 350, San Mateo, CA 94403, USA.

db4o is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */
namespace Db4objects.Db4o.Internal.Fileheader
{
	/// <exclude></exclude>
	public class TimerFileLockEnabled : Db4objects.Db4o.Internal.Fileheader.TimerFileLock
	{
		private readonly Db4objects.Db4o.IO.IoAdapter _timerFile;

		private readonly object _timerLock;

		private byte[] _longBytes = new byte[Db4objects.Db4o.Internal.Const4.LONG_LENGTH];

		private byte[] _intBytes = new byte[Db4objects.Db4o.Internal.Const4.INT_LENGTH];

		private int _headerLockOffset = 2 + Db4objects.Db4o.Internal.Const4.INT_LENGTH;

		private readonly long _opentime;

		private int _baseAddress = -1;

		private int _openTimeOffset;

		private int _accessTimeOffset;

		private bool _closed = false;

		public TimerFileLockEnabled(Db4objects.Db4o.Internal.IoAdaptedObjectContainer file
			)
		{
			_timerLock = file.Lock();
			_timerFile = file.TimerFile();
			_opentime = UniqueOpenTime();
		}

		public override void CheckHeaderLock()
		{
			try
			{
				if (((int)_opentime) == ReadInt(0, _headerLockOffset))
				{
					WriteHeaderLock();
					return;
				}
			}
			catch (System.IO.IOException)
			{
			}
			throw new Db4objects.Db4o.Ext.DatabaseFileLockedException(_timerFile.ToString());
		}

		public override void CheckOpenTime()
		{
			try
			{
				long readOpenTime = ReadLong(_baseAddress, _openTimeOffset);
				if (_opentime == readOpenTime)
				{
					WriteOpenTime();
					return;
				}
			}
			catch (System.IO.IOException)
			{
			}
			throw new Db4objects.Db4o.Ext.DatabaseFileLockedException(_timerFile.ToString());
		}

		public override void CheckIfOtherSessionAlive(Db4objects.Db4o.Internal.LocalObjectContainer
			 container, int address, int offset, long lastAccessTime)
		{
			if (_timerFile == null)
			{
				return;
			}
			long waitTime = Db4objects.Db4o.Internal.Const4.LOCK_TIME_INTERVAL * 5;
			long currentTime = Sharpen.Runtime.CurrentTimeMillis();
			while (Sharpen.Runtime.CurrentTimeMillis() < currentTime + waitTime)
			{
				Db4objects.Db4o.Foundation.Cool.SleepIgnoringInterruption(waitTime);
			}
			try
			{
				long currentAccessTime = ReadLong(address, offset);
				if ((currentAccessTime > lastAccessTime))
				{
					throw new Db4objects.Db4o.Ext.DatabaseFileLockedException(container.ToString());
				}
			}
			catch (System.IO.IOException)
			{
			}
		}

		public override void Close()
		{
			WriteAccessTime(true);
			_closed = true;
		}

		public override bool LockFile()
		{
			return true;
		}

		public override long OpenTime()
		{
			return _opentime;
		}

		public override void Run()
		{
			Sharpen.Lang.Thread t = Sharpen.Lang.Thread.CurrentThread();
			t.SetName("db4o file lock");
			try
			{
				while (WriteAccessTime(false))
				{
					Db4objects.Db4o.Foundation.Cool.SleepIgnoringInterruption(Db4objects.Db4o.Internal.Const4
						.LOCK_TIME_INTERVAL);
					if (_closed)
					{
						break;
					}
				}
			}
			catch (System.IO.IOException)
			{
			}
		}

		public override void SetAddresses(int baseAddress, int openTimeOffset, int accessTimeOffset
			)
		{
			_baseAddress = baseAddress;
			_openTimeOffset = openTimeOffset;
			_accessTimeOffset = accessTimeOffset;
		}

		public override void Start()
		{
			WriteAccessTime(false);
			_timerFile.Sync();
			CheckOpenTime();
			new Sharpen.Lang.Thread(this).Start();
		}

		private long UniqueOpenTime()
		{
			return Sharpen.Runtime.CurrentTimeMillis();
		}

		private bool WriteAccessTime(bool closing)
		{
			if (NoAddressSet())
			{
				return true;
			}
			long time = closing ? 0 : Sharpen.Runtime.CurrentTimeMillis();
			bool ret = WriteLong(_baseAddress, _accessTimeOffset, time);
			Sync();
			return ret;
		}

		private bool NoAddressSet()
		{
			return _baseAddress < 0;
		}

		public override void WriteHeaderLock()
		{
			try
			{
				WriteInt(0, _headerLockOffset, (int)_opentime);
				Sync();
			}
			catch (System.IO.IOException)
			{
			}
		}

		public override void WriteOpenTime()
		{
			try
			{
				WriteLong(_baseAddress, _openTimeOffset, _opentime);
				Sync();
			}
			catch (System.IO.IOException)
			{
			}
		}

		private bool WriteLong(int address, int offset, long time)
		{
			lock (_timerLock)
			{
				if (_timerFile == null)
				{
					return false;
				}
				_timerFile.BlockSeek(address, offset);
				Db4objects.Db4o.Foundation.PrimitiveCodec.WriteLong(_longBytes, time);
				_timerFile.Write(_longBytes);
				return true;
			}
		}

		private long ReadLong(int address, int offset)
		{
			lock (_timerLock)
			{
				if (_timerFile == null)
				{
					return 0;
				}
				_timerFile.BlockSeek(address, offset);
				_timerFile.Read(_longBytes);
				return Db4objects.Db4o.Foundation.PrimitiveCodec.ReadLong(_longBytes, 0);
			}
		}

		private bool WriteInt(int address, int offset, int time)
		{
			lock (_timerLock)
			{
				if (_timerFile == null)
				{
					return false;
				}
				_timerFile.BlockSeek(address, offset);
				Db4objects.Db4o.Foundation.PrimitiveCodec.WriteInt(_intBytes, 0, time);
				_timerFile.Write(_intBytes);
				return true;
			}
		}

		private long ReadInt(int address, int offset)
		{
			lock (_timerLock)
			{
				if (_timerFile == null)
				{
					return 0;
				}
				_timerFile.BlockSeek(address, offset);
				_timerFile.Read(_longBytes);
				return Db4objects.Db4o.Foundation.PrimitiveCodec.ReadInt(_longBytes, 0);
			}
		}

		private void Sync()
		{
			_timerFile.Sync();
		}
	}
}
