/* This file is part of the db4o object database http://www.db4o.com

Copyright (C) 2004 - 2009  Versant Corporation http://www.versant.com

db4o is free software; you can redistribute it and/or modify it under
the terms of version 3 of the GNU General Public License as published
by the Free Software Foundation.

db4o is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program.  If not, see http://www.gnu.org/licenses/. */
using Db4objects.Db4o.Ext;
using Db4objects.Db4o.Foundation;
using Db4objects.Db4o.IO;
using Db4objects.Db4o.Internal;
using Db4objects.Db4o.Internal.Fileheader;
using Sharpen;
using Sharpen.Lang;

namespace Db4objects.Db4o.Internal.Fileheader
{
	/// <exclude></exclude>
	public class TimerFileLockEnabled : TimerFileLock
	{
		private readonly IoAdapter _timerFile;

		private readonly object _timerLock;

		private byte[] _longBytes = new byte[Const4.LongLength];

		private byte[] _intBytes = new byte[Const4.IntLength];

		private int _headerLockOffset = 2 + Const4.IntLength;

		private readonly long _opentime;

		private int _baseAddress = -1;

		private int _openTimeOffset;

		private int _accessTimeOffset;

		private bool _closed = false;

		public TimerFileLockEnabled(IoAdaptedObjectContainer file)
		{
			_timerLock = file.Lock();
			// FIXME: No reason to sync over the big master lock.
			//        A local lock should be OK.
			// _timerLock = new Object();
			_timerFile = file.TimerFile();
			_opentime = UniqueOpenTime();
		}

		public override void CheckHeaderLock()
		{
			if (((int)_opentime) != ReadInt(0, _headerLockOffset))
			{
				throw new DatabaseFileLockedException(_timerFile.ToString());
			}
			WriteHeaderLock();
		}

		public override void CheckOpenTime()
		{
			long readOpenTime = ReadLong(_baseAddress, _openTimeOffset);
			if (_opentime != readOpenTime)
			{
				throw new DatabaseFileLockedException(_timerFile.ToString());
			}
			WriteOpenTime();
		}

		/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
		public override void CheckIfOtherSessionAlive(LocalObjectContainer container, int
			 address, int offset, long lastAccessTime)
		{
			if (_timerFile == null)
			{
				// need to check? 
				return;
			}
			long waitTime = Const4.LockTimeInterval * 5;
			long currentTime = Runtime.CurrentTimeMillis();
			// If someone changes the system clock here, he is out of luck.
			while (Runtime.CurrentTimeMillis() < currentTime + waitTime)
			{
				Cool.SleepIgnoringInterruption(waitTime);
			}
			long currentAccessTime = ReadLong(address, offset);
			if ((currentAccessTime > lastAccessTime))
			{
				throw new DatabaseFileLockedException(container.ToString());
			}
		}

		/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
		public override void Close()
		{
			WriteAccessTime(true);
			lock (_timerLock)
			{
				_closed = true;
			}
		}

		public override bool LockFile()
		{
			return true;
		}

		public override long OpenTime()
		{
			return _opentime;
		}

		public override void Run()
		{
			while (true)
			{
				lock (_timerLock)
				{
					if (_closed)
					{
						return;
					}
					WriteAccessTime(false);
				}
				Cool.SleepIgnoringInterruption(Const4.LockTimeInterval);
			}
		}

		public override void SetAddresses(int baseAddress, int openTimeOffset, int accessTimeOffset
			)
		{
			_baseAddress = baseAddress;
			_openTimeOffset = openTimeOffset;
			_accessTimeOffset = accessTimeOffset;
		}

		/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
		public override void Start()
		{
			WriteAccessTime(false);
			_timerFile.Sync();
			CheckOpenTime();
			Thread thread = new Thread(this);
			thread.SetName("db4o file lock");
			thread.SetDaemon(true);
			thread.Start();
		}

		private long UniqueOpenTime()
		{
			return Runtime.CurrentTimeMillis();
		}

		// TODO: More security is possible here to make this time unique
		// to other processes. 
		/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
		private bool WriteAccessTime(bool closing)
		{
			if (NoAddressSet())
			{
				return true;
			}
			long time = closing ? 0 : Runtime.CurrentTimeMillis();
			bool ret = WriteLong(_baseAddress, _accessTimeOffset, time);
			Sync();
			return ret;
		}

		private bool NoAddressSet()
		{
			return _baseAddress < 0;
		}

		public override void WriteHeaderLock()
		{
			WriteInt(0, _headerLockOffset, (int)_opentime);
			Sync();
		}

		public override void WriteOpenTime()
		{
			WriteLong(_baseAddress, _openTimeOffset, _opentime);
			Sync();
		}

		/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
		private bool WriteLong(int address, int offset, long time)
		{
			lock (_timerLock)
			{
				if (_timerFile == null)
				{
					return false;
				}
				_timerFile.BlockSeek(address, offset);
				PrimitiveCodec.WriteLong(_longBytes, time);
				_timerFile.Write(_longBytes);
				return true;
			}
		}

		/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
		private long ReadLong(int address, int offset)
		{
			lock (_timerLock)
			{
				if (_timerFile == null)
				{
					return 0;
				}
				_timerFile.BlockSeek(address, offset);
				_timerFile.Read(_longBytes);
				return PrimitiveCodec.ReadLong(_longBytes, 0);
			}
		}

		private bool WriteInt(int address, int offset, int time)
		{
			lock (_timerLock)
			{
				if (_timerFile == null)
				{
					return false;
				}
				_timerFile.BlockSeek(address, offset);
				PrimitiveCodec.WriteInt(_intBytes, 0, time);
				_timerFile.Write(_intBytes);
				return true;
			}
		}

		private long ReadInt(int address, int offset)
		{
			lock (_timerLock)
			{
				if (_timerFile == null)
				{
					return 0;
				}
				_timerFile.BlockSeek(address, offset);
				_timerFile.Read(_longBytes);
				return PrimitiveCodec.ReadInt(_longBytes, 0);
			}
		}

		/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
		private void Sync()
		{
			_timerFile.Sync();
		}
	}
}
