/*
  DB Concatenate
  ==============

  DBCat takes in sound data, converts it to the 16bit unsigned 44.1kHz format
  and then sends the data to the DB Fourier Synthesis Daemon.  

  DBCat sends the data to whichever dbfsd pipe is free at the time. If no
  pipe is free, the program will exit with an error message.

  Version: B.2
  Author:  Bob Dean
  Copyright (c) 1999


   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public Licensse as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/



#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <stdarg.h>
#include <dlfcn.h> /* functions used to dynamically link to libs */

#include <dbaudiolib.h>
#include <dbdebug.h>

#define DBCAT_DEFAULT_NAME "DBCat"
#define DBCAT_SAMPLE_NAME  "DBSamp"

extern int errno;
extern int debug_level;

int  loop_flag;
int  sample_flag;
int  in_fd,out_fd;
int  numch,sample_size,sample_rate,chindex;
enum channel_type_e type;

int pause_flag;

#ifdef USE_DYNAMIC_LIBRARY
DBAudioLibFunctions * dbaudiofxns;
void * h;
#endif

/***************************************************************************/
/*  Initialization functions                                               */
/***************************************************************************/

/*
  print_help - outputs help to the console. 

  inputs - none
  outputs - text to the console.
 */

void print_version()
{
	printf("DBCat - sound concatenation utility for the DB Fourier Synthesis Daemon\n");
	printf("%s\n",DBMIX_COPYRIGHT);
	printf("DBMix Version: %s\n",DBMIX_VERSION);
}

void print_help()
{
	print_version();

	printf("\n");
	printf("Supported options [defaults in brackets]:\n");
	printf("   -d    turn on debug messages to console      [Off]\n");
	printf("   -v    print version information.             [None]\n");
	printf("   -m    input data is mono.                    [Stereo]\n");
	printf("   -r N  input data has a sampling rate of N Hz [44100 Hz]\n");
	printf("   -8    input data is 8 bit unsigned samples   [16 bit signed]\n");
	printf("   -c X  use channel X if available             [first available]\n");
	printf("   -S    sample only mode.                      [none]\n");

	printf("\n");
}



/*
  parse_cmdline - steps through argv and sets all the variables

  inputs - argc argv
  outputs - sets system variables
 */
void parse_cmdline(int argc, char* argv[])
{
	int opt;
   
	in_fd = 0;
	debug_level = 0;
	numch = 2;
	sample_size = 16;
	sample_rate = 44100;
	chindex = 0;
	pause_flag = 0;
	sample_flag = 0;

	while((opt = getopt(argc,argv,"hvdr:c:m8S")) != -1)
	{
		switch(opt)
		{
			case 'd':
				debug_level = 1;
				break;
			case 'h':
				print_help();
				exit(0);
				break;
			case 'm':
				numch = 1;
				break;
			case '8':
				sample_size = 8;
				break;
			case 'r':
				sample_rate = atoi(optarg);
				Debug("sample_rate is %d\n",sample_rate);
				break;
			case 'c':
				chindex = atoi(optarg);
				Debug("chindex is %d\n",chindex);
				break;
			case 'v':
				print_version();
			    exit(0);
			case 'S':
				sample_flag = 1;
				Debug("Set sample only mode\n");
				break;
			case ':':
				Debug("option needs a value\n");
				exit(0);
			case '?':
				Debug("unknown option: %c\n",optopt);
				exit(0);
			default: break;
		}
	}
}


/*
  sigexit - changes system variables to force clean shutdown
    inputs - none
    outputs - none
 */
void sigexit(int signum)
{
	loop_flag = FALSE;

#ifdef USE_DYNAMIC_LIBRARY
	if(dbaudiofxns != NULL) dbaudiofxns->DBAudio_Close();
	dlclose(h);
#else
	DBAudio_Close();
#endif

	Debug("DBCat has completed. Meow.\n");
	exit(0);
}



/* function to sleep a thread for usec microseconds*/
void dbcat_usleep(gint usec)
{
	struct timeval tv;
	
	tv.tv_sec = usec / 1000000;
	usec -= tv.tv_sec * 1000000;
	tv.tv_usec = usec;
	select(0, NULL, NULL, NULL, &tv);
}


void dbcat_message_handler(dbfsd_msg msg)
{	
	switch (msg.msg_type)
	{
		case DBMSG_PAUSE:
			pause_flag = 1;
			DBAudio_Pause(pause_flag);
			break;
		case DBMSG_UNPAUSE:
			pause_flag = 0;
			DBAudio_Pause(pause_flag);
			break;
		case DBMSG_PLAY:
			pause_flag = 0;
			DBAudio_Pause(pause_flag);
			break;
		case DBMSG_MUTE:

			break;
		case DBMSG_UNMUTE:

			break;
		case DBMSG_STOP:
			break;
		case DBMSG_EJECT:
			break;
		case DBMSG_REWIND:
			break;
		case DBMSG_FFORWARD:
			break;
		case DBMSG_NEXT:
			break;
		case DBMSG_PREV:
			break;
		default:
			Error("dbcat_message_handler: unknown or unsupported IPC message %d\n ",msg);
	}

	return;
}



/***************************************************************************/
/*                MAIN                                                     */
/***************************************************************************/

int main(int argc, char **argv)
{
	char   buf[PIPE_BUF*2];
	char * dbcat_name;
	int    len,error,count;

#ifdef USE_DYNAMIC_LIBRARY	
	void * (*get_fxns) (); /* define function pointer to hold pntr to DBAudio_Get_Functions() */
	
	dbaudiofxns = NULL;
#endif

	debug_level = 0;
	type = PIPE_CHANNEL;
	pause_flag = 0;

	memset(buf,0,PIPE_BUF*2);

	/*install signal handlers*/
	signal(SIGINT,sigexit);
	signal(SIGTERM,sigexit);

	parse_cmdline(argc,argv);

	if (sample_flag)
	{
		dbcat_name = DBCAT_SAMPLE_NAME;
	}
	else
	{
		dbcat_name = DBCAT_DEFAULT_NAME;
	}

#ifdef USE_DYNAMIC_LIBRARY	
	Debug("USING DLOPEN()");
	/* link against DBAudioLib and retrieve fucntion structure */
	if((h = dlopen("libdbaudiolib.so",RTLD_NOW)) == NULL) 
	{
		printf("Failed to open DBAudioLib:: %s\n",dlerror());
		return 1;
	}

	if((get_fxns = dlsym(h,"DBAudio_Get_Functions")) == NULL)
	{
		printf("Failed to retrieve pntr to DBAudio_Get_Functions: %s\n",dlerror());
		return 1;
	}

	dbaudiofxns = (DBAudioLibFunctions*)get_fxns();

	error = dbaudiofxns->DBAudio_Init(dbcat_name,sample_size,sample_rate,numch,type,chindex);
/* 	error = DBAudio_Init("Dbcat",sample_size,sample_rate,numch,type,chindex); */
#else
	Debug("USING SHARED LIBRARY");
	error = DBAudio_Init(dbcat_name,sample_size,sample_rate,numch,type,chindex);
#endif

	if(error != SUCCESS)
	{
		printf("Error. Is dbfsd running?\n");
		return 1;
	}
	
#ifdef USE_DYNAMIC_LIBRARY
	dbaudiofxns->DBAudio_Set_Message_Handler(dbcat_message_handler,DBMSG_ALL);
#else
	DBAudio_Set_Message_Handler(dbcat_message_handler,DBMSG_ALL);
#endif

	/*init variables for main loop*/
	loop_flag = TRUE;
	count = 0;

	/*main loop*/
	while(loop_flag)
    {
		char c1;

		if (pause_flag)
		{
#ifdef USE_DYNAMIC_LIBRARY
			dbaudiofxns->DBAudio_Handle_Message_Queue();
#else
			DBAudio_Handle_Message_Queue();
#endif
			usleep(50000);
		}
		else
		{
			/* if not in sample mode, read in data */
			if (!sample_flag)
			{
				/* read in data, adjusting for count bytes already in the buffer */
				if((len = read(in_fd,buf+count,PIPE_BUF-count)) < 0)
				{
					loop_flag = FALSE;
					
#ifdef USE_DYNAMIC_LIBRARY
					dbaudiofxns->DBAudio_Close();
					dlclose(h);
#else
					DBAudio_Close();
#endif
				}

				len+=count;
			}
			else
			{
				/* sample mode, write full empty buffer */
				len = PIPE_BUF*2;
			}
			
#ifdef USE_DYNAMIC_LIBRARY
			if((count = dbaudiofxns->DBAudio_Write(buf,len)) == FAILURE)
#else
			if((count = DBAudio_Write(buf,len)) == FAILURE)	
#endif
			{
				perror("DBCAT: DBAudio_Write failed::");
			}
			
			/* if we could not write the full buffer, copy the unwritten
			   data to the front of the buffer */
			if((count < len) && (count > 0))
			{
				c1 = *(buf+count);
				memmove(buf,buf+count,len - count);
				count = len - count;
			}
			else
			{
				/* if we could not write *any* data, we are probably PAUSED
				   adjust count to compensate */
				if (count == 0)
				{
					count = PIPE_BUF;
				}
				else
				{
					count = 0;
				}
			}
		}
    }

	return 0;
}
