/*! \file commandhandlers.cpp
    \brief Implements the exposed RPC methods.
    
    This file contains implementations of the handleCommand() methods of the RPC handler classes.
 */

#include "commandhandlers.h"
#include "global.h"
#include "log.h"
#include "user.h"
#include "shareitemmodel.h"
#include "finisheditem.h"

#include <QThread>
#include <iostream>
#include <rpcdriver/types.h>
using namespace std;
using namespace boost;

namespace ui_cmd_handlers
{

using std::list;

/**
 * Parses an anylist of user structs and returns a QList which contains user pointers.
 *
 */
QList<User*> parseUserList(const anylist& users)
{
    anylist::const_iterator uit = users.begin();
    QList<User*> parsedUsers;

    for(;uit!=users.end();uit++)
    {
        User* user = new User;		// Fast, not slow! :)
        anylist userprops = boost::any_cast<anylist>(*uit);
        anylist::const_iterator pit = userprops.begin();
        user->id = boost::any_cast<int>(*pit);
        pit++;
        user->nick = boost::any_cast<string>(*pit).c_str();
        pit++;
        user->flags = any_cast<int>(*pit);
        pit++;
        user->email = any_cast<string>(*pit).c_str();
        pit++;
        user->description = any_cast<string>(*pit).c_str();
        pit++;
        user->connection = any_cast<string>(*pit).c_str();
        pit++;
        user->tag = any_cast<string>(*pit).c_str();
        pit++;
        user->shared = any_cast<int64>(*pit);
        pit++;
        user->numSlots = any_cast<int>(*pit);
        parsedUsers.append(user);
    }

    return parsedUsers;
}

void HubConnectedHandler::handleCommand(const anylist& params)
{
    try
    {
        logger->info("hubConnected!");
        int sessionId = boost::any_cast<int>(params.front());
        emit hubConnected(sessionId);
    }
    catch( const boost::bad_any_cast& e )
    {}
}

void HubUpdatedHandler::handleCommand(const anylist& params)
{
    try
    {
        logger->info("hubUpdated!");
        anylist::const_iterator it = params.begin();
        int sessionId = boost::any_cast<int>(*it);
        it++;
        QString hubName = (boost::any_cast<string>(*it)).c_str();

        emit hubUpdated(sessionId,hubName);
    }
    catch( const boost::bad_any_cast& e )
    {}
}

void HubStatsHandler::handleCommand(const anylist& params)
{
	try
	{
		anylist::const_iterator it = params.begin();
		int sessionId = boost::any_cast<int>(*it);
		it++;
		int64 shared = boost::any_cast<int64>(*it);

		emit hubStats(sessionId,shared);
	}
	catch( const boost::bad_any_cast& e )
	{
		throw RpcHandlerException("HubStatsHandler","Bad AnyCast");
	}
}


void ConnectionFailedHandler::handleCommand(const anylist& params)
{
    try
    {
        logger->info("connectionFailed!");
        anylist::const_iterator it = params.begin();
        int sessionId = boost::any_cast<int>(*it);
        it++;
        QString reason = (boost::any_cast<string>(*it)).c_str();

        emit connectionFailed(sessionId,reason);
    }
    catch( const boost::bad_any_cast& e )
    {}
}

void PrivateChatHandler::handleCommand( const anylist& params)
{
    try
    {
        logger->debug("privateChat!");
        anylist::const_iterator it = params.begin();
        int sessionId = boost::any_cast<int>(*it);
        it++;
        QString from = (boost::any_cast<string>(*it)).c_str();
        it++;
        QString msg((boost::any_cast<string>(*it)).c_str());
        emit privateChat(sessionId,from,msg);
    }
    catch( const boost::bad_any_cast& e )
    {}

}

void ChatMessageHandler::handleCommand( const anylist& params)
{
    try
    {
        logger->debug("publicChat!");
        anylist::const_iterator it = params.begin();
        int sessionId = boost::any_cast<int>(*it);
        it++;
        QString msg = (boost::any_cast<string>(*it)).c_str();
        emit chatMessage(sessionId,msg);
    }
    catch( const boost::bad_any_cast& e )
    {
        logger->error("bad anycast in chatmessage");
    }

}

void UsersUpdatedHandler::handleCommand( const anylist& params)
{
    try
    {
        QList<User*> parsedUsers;

        //logger->debug("usersUpdated!");
        anylist::const_iterator it = params.begin();
        int sessionId = boost::any_cast<int>(*it);
        it++;

        // ................................................................
        // TODO:  This will result in a deep copy of the entire list??!!
        // We need to solve it!
        anylist users = boost::any_cast<anylist>(*it);

        /*anylist::const_iterator uit = users.begin();

        for(;uit!=users.end();uit++)
        {
        	User* user = new User;		// Fast, not slow! :)
        	anylist userprops = boost::any_cast<anylist>(*uit);
        	anylist::const_iterator pit = userprops.begin();
        	user->id = boost::any_cast<int>(*pit); pit++;
        	user->nick = boost::any_cast<string>(*pit).c_str(); pit++;
        	user->flags = any_cast<int>(*pit);pit++;
        	user->email = any_cast<string>(*pit).c_str();pit++;
        	user->description = any_cast<string>(*pit).c_str();pit++;
        	user->connection = any_cast<string>(*pit).c_str();pit++;
        	user->tag = any_cast<string>(*pit).c_str(); pit++;
        	user->shared = any_cast<int64>(*pit); pit++;
        	user->numSlots = any_cast<int>(*pit);
        	parsedUsers.append(user);
        }*/
        parsedUsers = parseUserList(users);

        emit usersUpdated( sessionId, parsedUsers );

    }
    catch( const boost::bad_any_cast& e )
    {
        logger->error("badanycast in usersupdated.");
    }

}



void settingsInfoHandler::handleCommand(const anylist& params)
{
    try
    {
        QList<QString> keys;
        QList<QVariant> values;
        anylist::const_iterator pi = params.begin();
        const anylist& k = boost::any_cast<anylist>(*pi);
        pi++;
        const anylist& v = boost::any_cast<anylist>(*pi);

        anylist::const_iterator ki = k.begin();
        while(ki!=k.end())
        {
            keys.append(QString(boost::any_cast<string>(*ki).c_str()));
            ki++;
        }
        anylist::const_iterator vi = v.begin();
        while(vi!=v.end())
        {
            if((*vi).type()==typeid(string))
                values.append(QString(boost::any_cast<string>(*vi).c_str()));
            else if((*vi).type()==typeid(int))
                values.append(boost::any_cast<int>(*vi));
            else if((*vi).type()==typeid(bool))
                values.append(boost::any_cast<bool>(*vi));
            else if((*vi).type()==typeid(int64))
				values.append((qlonglong)boost::any_cast<int64>(*vi));
            vi++;
        }
        emit settingsInfo( keys, values ) ;
    }
    catch (const boost::bad_any_cast& e)
    {

        cout << "bad any cast" << endl;
    }
    catch (...)
    {
        cout << "exception" << endl;
    }
}

void PublicHubListHandler::handleCommand( const anylist& params )
{
	try {
		// The parameters should consist of a list of lists.
		// Iterate through the list and cast each item to a list which 
		// is treated as a hub entry and converted.
		
		QList<rpc_types::HubEntry> hubEntries;
		const anylist& hubs = boost::any_cast<const anylist&>(params.front());
		anylist::const_iterator hubsIt = hubs.begin();
		while(hubsIt != hubs.end()) {
			
			const anylist& hub = boost::any_cast<const anylist&>(*hubsIt);
	
			if( hub.size() == rpc_types::HubEntry::HUB_ENTRY_SIZE ) {
				
				anylist::const_iterator hubIt = hub.begin();			
				rpc_types::HubEntry hubEntry;
				hubEntry.name = boost::any_cast<string>(*hubIt);
				++hubIt;
				hubEntry.server = boost::any_cast<string>(*hubIt);
				++hubIt;
				hubEntry.description = boost::any_cast<string>(*hubIt);
				++hubIt;
				hubEntry.country = boost::any_cast<string>(*hubIt);
				++hubIt;
				hubEntry.rating = boost::any_cast<string>(*hubIt);
				++hubIt;
				hubEntry.reliability = boost::any_cast<int>(*hubIt);
				++hubIt;
				hubEntry.shared = boost::any_cast<int64>(*hubIt);
				++hubIt;
				hubEntry.minShare = boost::any_cast<int64>(*hubIt);
				++hubIt;
				hubEntry.users = boost::any_cast<int>(*hubIt);
				++hubIt;
				hubEntry.minSlots = boost::any_cast<int>(*hubIt);
				++hubIt;
				hubEntry.maxHubs = boost::any_cast<int>(*hubIt);
				++hubIt;
				hubEntry.maxUsers = boost::any_cast<int>(*hubIt);
				
				hubEntries.append(hubEntry);
			}
			
			++hubsIt;
		}
		
		emit hubList(hubEntries);
	}
	catch( const boost::bad_any_cast& e ){
		logger->error(e.what());
		throw RpcHandlerException("HubList","Bad AnyCast");
	}
}

void SessionCreatedHandler::handleCommand(const anylist& params)
{
    try
    {
        logger->info("SessionCreated");
        int sessionId = boost::any_cast<int>(params.front());
        emit sessionCreated(sessionId);
    }
    catch( const boost::bad_any_cast& e )
    {
        logger->error("sessioncreated bad anycast");
    }
}

void RunningSessionsHandler::handleCommand( const anylist& params)
{
    try
    {
        logger->debug("runningSessions!");
        anylist::const_iterator it = params.begin();

        anylist sessions = any_cast<anylist>(*it);
        it++;
        anylist queue = any_cast<anylist>(*it);

        // Parse the sessions list
        anylist::const_iterator sessionIt = sessions.begin();
        for(;sessionIt!=sessions.end();sessionIt++)
        {
            // Each element in the sessions list is a SessionInfo struct, which is sent as a list of course.
            anylist si = any_cast<anylist>(*sessionIt);
            anylist::const_iterator infoIt = si.begin();
            int sessionId = boost::any_cast<int>(*infoIt);
            infoIt++;
            QString hubName = (boost::any_cast<string>(*infoIt)).c_str();
            infoIt++;
            QString url = (boost::any_cast<string>(*infoIt)).c_str();
            infoIt++;
            // Now comes the list of users
            anylist userlist = any_cast<anylist>(*infoIt);
            QList<User*> parsedUsers = parseUserList(userlist);
            emit sessionInfo( sessionId,hubName,url,parsedUsers );
        }
		
		QList<QueueItem> queueItemList;
		
		for(anylist::const_iterator qlit=queue.begin();qlit!=queue.end();qlit++)
		{
			anylist qitem = any_cast<anylist>(*qlit);
			anylist::const_iterator qit = qitem.begin();
			QueueItem qi;
			qi.id = any_cast<int>(*qit);
			qit++;
			qi.target = any_cast<string>(*qit).c_str();
			qit++;
			qi.size = any_cast<int64>(*qit);
			qit++;
			qi.downloadedBytes = any_cast<int64>(*qit);
			qit++;
			qi.status = any_cast<int>(*qit);
			qit++;
			qi.priority = any_cast<int>(*qit);
			qit++;
			qi.currentSource = any_cast<int>(*qit);
			qit++;
			if(qit!=qitem.end()) {
				anylist sources = any_cast<anylist>(*qit);
				anylist::const_iterator slit = sources.begin();
				while(slit!=sources.end())
				{
					qi.sources += any_cast<int>(*slit);
					slit++;
				}
				
			}
			queueItemList+=qi;
		}
		emit queueList(queueItemList);

    }
    catch( const boost::bad_any_cast& e )
    {
        logger->error("bad anycast in runningSessions");
		throw RpcHandlerException("runningSessions","Bad anycast");
    }

}

void SharedDirsHandler::handleCommand( const anylist & params )
{
    try
    {
        logger->debug("sharedDirs");
        anylist dirs = any_cast<anylist>(params.front());
        anylist::const_iterator it = dirs.begin();
        QList<ShareItem> shares;

        for(;it!=dirs.end();it++)
        {
            anylist item = any_cast<anylist>(*it);
            anylist::iterator sit = item.begin();
            QString virtualName((any_cast<string>(*sit)).c_str());
            sit++;
            QString realName((any_cast<string>(*sit)).c_str());
            sit++;
            int64 bytesShared(any_cast<int64>(*sit));
            ShareItem shareItem(virtualName,realName,bytesShared);
            shares+=shareItem;
        }
        emit sharedDirs( shares );
    }
    catch( const boost::bad_any_cast& e )
    {
        logger->error("shareddirs bad anycast");
    }
}

void SearchResultsHandler::handleCommand( const anylist & params )
{
    try
    {
        QList<SearchEntry> results;

        logger->debug("searchResults");
        anylist::const_iterator it = params.begin();
        int session = any_cast<int>(*it);
        it++;
        anylist srs = any_cast<anylist>(*it);

        for(it=srs.begin();it!=srs.end();it++)
        {
            anylist sr = any_cast<anylist>(*it);
            //logger->debug(QString("a struct with %1 elements").arg(sr.size()));
            anylist::const_iterator srit = sr.begin();

            string file = any_cast<string>(*srit);
            srit++;
            int fileisutf8 = any_cast<int>(*srit);
            srit++;
            QString fileName(any_cast<string>(*srit).c_str());
            srit++;
            QString userName(any_cast<string>(*srit).c_str());
            srit++;
            QString hubName(any_cast<string>(*srit).c_str());
            srit++;
            int userId(any_cast<int>(*srit));
            srit++;
            int64 fileSize = any_cast<int64>(*srit);
            srit++;
            int sluts = any_cast<int>(*srit);
            srit++;
            int freeSlots = any_cast<int>(*srit);
            srit++;
            int type = any_cast<int>(*srit);
            srit++;
            QString tth(any_cast<string>(*srit).c_str());
            SearchEntry se(file,userName,hubName,userId,fileSize,sluts,freeSlots,type,tth,fileisutf8!=0);
            results += se;
        }

        emit searchResults( session, results );

    }
    catch( const boost::bad_any_cast& e )
    {
        logger->error("searchresults bad anycast");
    }
}


void QueueEventHandler::handleCommand( const anylist & params )
{
    try
    {
        logger->debug("queueEvent");
        anylist::const_iterator it = params.begin();
        int eventType = any_cast<int>(*it);
        it++;

        switch(eventType)
        {
        case 1:					// add
            {
                // Next param is a queueItem
                anylist qitem = any_cast<anylist>(*it);
                anylist::const_iterator qit = qitem.begin();
                QueueItem qi;
                qi.id = any_cast<int>(*qit);
                qit++;
                qi.target = any_cast<string>(*qit).c_str();
                qit++;
                qi.size = any_cast<int64>(*qit);
                qit++;
                qi.downloadedBytes = any_cast<int64>(*qit);
                qit++;
                qi.status = any_cast<int>(*qit);
                qit++;
                qi.priority = any_cast<int>(*qit);
                qit++;
                qi.currentSource = any_cast<int>(*qit);
                //qit++;
                //if(qit!=params.end()) {
                //	anylist sourcearr = any_cast<anylist>(*qit);
                //	anylist::iterator sourceit = sourcearr.begin();
                //	for(;sourceit!=sourcearr.end();sourceit++) qi.sources+=any_cast<int>(*sourceit);
                //}
                logger->debug("emitting queueItemAdded");
                emit queueItemAdded( qi );

                break;
            }
        case 2:					// remove
            {
                // Second param: id
                int id = any_cast<int>(*it);

                logger->debug("emitting queueItemRemoved");

                emit queueItemRemoved( id );
                break;
            }
        case 3:					// finished
            {
                int id = any_cast<int>(*it);

                logger->debug("emitting queueItemFinshed");

                emit queueItemFinshed( id );
                break;
            }
        case 4:					// sourcesupdated
            {
                int id = any_cast<int>(*it);
                it++;
                anylist sources = any_cast<anylist>(*it);
                QList<int> sourceList;
                for(anylist::iterator sit=sources.begin();sit!=sources.end();sit++)
                    sourceList+=any_cast<int>(*sit);

                logger->debug("emitting sourcesUpdated");

                emit sourcesUpdated( id, sourceList );
                break;
            }
        case 5: {						// statusupdated
            int id = any_cast<int>(*it);
            it++;
            int status = any_cast<int>(*it);
            it++;
            int currSrc = any_cast<int>(*it);
            it++;
            int priority = any_cast<int>(*it);
            it++;

            logger->debug("emitting statusUpdated");

            emit statusUpdated( id, status, currSrc, priority );
            break;
	}
        default:
            logger->error("Wrong eventType in queueEvent!");
        }

    }
    catch( const boost::bad_any_cast& e )
    {
        logger->error("queueevent bad anycast");
    }
}

void FinishedEventHandler::handleCommand( const anylist & params )
{
	try {
        logger->debug("finishedEvent");
        anylist::const_iterator it = params.begin();
        int eventType = any_cast<int>(*it);
        it++;
        int transferType = any_cast<int>(*it);
        it++;

        switch(eventType)
        {
        case 1:					// add
            {
                // A FinishedItem
                anylist fi = any_cast<anylist>(*it);
                anylist::iterator fit = fi.begin();
                FinishedItem fitem;
                fitem.id = any_cast<int>(*fit);
                fit++;
                fitem.target = any_cast<string>(*fit).c_str();
                fit++;
                fitem.user = any_cast<string>(*fit).c_str();
                fit++;
                fitem.hub = any_cast<string>(*fit).c_str();
                fitem.type = transferType;
                emit itemAdded(fitem);

            }
            break;
        case 2:					// remove
		{
			int id = any_cast<int>(*it);
			emit itemRemoved( id );
		}
            break;
        case 3:					// removeAll
			emit allRemoved(transferType);
			break;
        }
    }
	catch( const boost::bad_any_cast& e ) {
        logger->error("finishedevent bad anycast");
	}
}

void TransferEventHandler::handleCommand( const anylist & params )
{
	//logger->debug("transferEvent");
	anylist::const_iterator it = params.begin();
	int eventType = any_cast<int>(*it);
	//logger->debug(QString("eventType: %1").arg(eventType));
	it++;
	FileTransfer::Type transferType = static_cast<FileTransfer::Type>(any_cast<int>(*it));
	//logger->debug(QString("transfertype: %1").arg(transferType));
	it++;
	
	
	if(eventType==0 ) emit transferStarting( createTransfer(transferType,any_cast<anylist>(*it)) );
	else if(eventType==1) {
		QList<FileTransfer> tl;
		anylist ftlst = any_cast<anylist>(*it);
		for(anylist::iterator fit = ftlst.begin();fit!=ftlst.end();fit++)
			tl+=createTransfer( transferType, any_cast<anylist>(*fit));
		emit transferTick( tl );
	}
	else if(eventType==2) emit transferComplete( createTransfer(transferType,any_cast<anylist>(*it)) );
	else if(eventType==3) {
		anylist ft = any_cast<anylist>(*it);
		it++;
		QString errorMsg = any_cast<string>(*it).c_str();
		emit transferFailed( createTransfer(transferType,ft), errorMsg );
	}
	
}

FileTransfer TransferEventHandler::createTransfer( FileTransfer::Type type, const anylist & ft )
{
	anylist::const_iterator it = ft.begin();
	FileTransfer f;
	f.type = type;
	f.pos = any_cast<int64>(*it);
	it++;
	f.startpos = any_cast<int64>(*it);
	it++;
	f.actual = any_cast<int64>(*it);it++;
	f.size = any_cast<int64>(*it);it++;
	f.averageSpeed = any_cast<int64>(*it);it++;
	f.secondsLeft = any_cast<int64>(*it);it++;
	f.bytesLeft = any_cast<int64>(*it);it++;
	f.filename = any_cast<string>(*it).c_str();it++;
	if(type==FileTransfer::UPLOAD) {f.localfilename =  any_cast<string>(*it).c_str();it++;}
	f.tth = any_cast<string>(*it).c_str();it++;
	f.userid =  any_cast<int>(*it);it++;
	return f;
}

void UserFileListHandler::handleCommand( const anylist& params )
{
	try {
		logger->debug("File lising");
		anylist::const_iterator it = params.begin();
		int userId = any_cast<int>(*it);
		++it;
		string fileTree = any_cast<string>(*it);
		QString temp = fileTree.c_str();
		// Create a new model and emit it
		UserFileModelPtr model( new UserFileModel( userId, temp ) );
		
		emit fileListing( model );
	}
	catch( const boost::bad_any_cast& e ) {
		logger->error(e.what());
	}
}


void FavouriteHubListHandler::handleCommand( const anylist& params )
{
	try {
		logger->debug("Favourite hub list");
		
		anylist::const_iterator paramIt = params.begin();
		const anylist& favHubList = any_cast<const anylist&>(*paramIt);
		anylist::const_iterator favHubListIt = favHubList.begin();
		
		QList<rpc_types::FavouriteHub> hubs;
		
		while( favHubListIt != favHubList.end() ) {
			
			const anylist& favHub = any_cast<const anylist&>(*favHubListIt);
			anylist::const_iterator favHubIt = favHub.begin();
			rpc_types::FavouriteHub hubEntry;
			hubEntry.nick = any_cast<string>(*favHubIt);
			++favHubIt;
			hubEntry.userDescription = any_cast<string>(*favHubIt);
			++favHubIt;
			hubEntry.name = any_cast<string>(*favHubIt);
			++favHubIt;
			hubEntry.server = any_cast<string>(*favHubIt);
			++favHubIt;
			hubEntry.description = any_cast<string>(*favHubIt);
			++favHubIt;
			hubEntry.password = any_cast<string>(*favHubIt);
			++favHubIt;
			hubEntry.autoConnect = any_cast<bool>(*favHubIt);
		
			hubs.push_back(hubEntry);
			
			++favHubListIt;
		}
		
		emit hubList( hubs );
	}
	catch( const boost::bad_any_cast& e ) {
		logger->error(e.what());
	}
}

void UserRemovedHandler::handleCommand( const anylist& params )
{
	try {
		anylist::const_iterator it = params.begin();
		int sessionId = any_cast<int>(*it);
		++it;
		int userId = any_cast<int>(*it);
		emit userRemoved( sessionId, userId );
	}
	catch( const boost::bad_any_cast& e )
	{
		logger->error(e.what());
		throw RpcHandlerException("userRemoved","Bad anycast");
	}
}

void FavouriteHubAddedHandler::handleCommand( const anylist& params )
{
	try {
		logger->debug("Favourite hub added");
		
		anylist::const_iterator paramIt = params.begin();
		
		const anylist& favHub = any_cast<const anylist&>(*paramIt);
		anylist::const_iterator favHubIt = favHub.begin();
		rpc_types::FavouriteHub hubEntry;
		hubEntry.nick = any_cast<string>(*favHubIt);
		++favHubIt;
		hubEntry.userDescription = any_cast<string>(*favHubIt);
		++favHubIt;
		hubEntry.name = any_cast<string>(*favHubIt);
		++favHubIt;
		hubEntry.server = any_cast<string>(*favHubIt);
		++favHubIt;
		hubEntry.description = any_cast<string>(*favHubIt);
		++favHubIt;
		hubEntry.password = any_cast<string>(*favHubIt);
		++favHubIt;
		hubEntry.autoConnect = any_cast<bool>(*favHubIt);
		
		emit favHubAdded( hubEntry );
		
	}
	catch( const boost::bad_any_cast& e ) {
		logger->error(e.what());
	}
}


void FavouriteHubRemovedHandler::handleCommand( const anylist& params )
{
	try {
		logger->debug("Favourite hub removed");
		
		anylist::const_iterator paramIt = params.begin();
		
		const string server = any_cast<string>(*paramIt);
		
		emit favHubRemoved( server );
		
	}
	catch( const boost::bad_any_cast& e ) {
		logger->error(e.what());
	}
}

void HashInfoHandler::handleCommand( const anylist& params )
{
	// TODO
}

}
