/* Distributed Checksum Clearinghouse
 *
 * sendmail milter interface
 *
 * Copyright (c) 2004 by Rhyolite Software
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.2.66-1.146 $Revision$
 */

#include "libmilter/mfapi.h"
#include "cmn_defs.h"


static u_char background = 1;

static const char *userdirs;
static DCC_PATH userdirs_path;

/* These counts are not protected against corruption by simultaneous
 * increments by threads.  They are not worth the cost of proper locking */
static struct {
    int	    msgs;			/* total messages */
    int	    tgts;			/* total addressees */
    int	    tgts_discarded;		/* discarded for this many addressess */
    int	    tgts_rejected;
    int	    tgts_embargoed;
    int	    msgs_embargoed;
    int	    msgs_spam;
    time_t  msg_prev;
} totals;

static char sm_isspam_macro_def[] = "{dcc_isspam}";
static char *sm_isspam_macro = sm_isspam_macro_def;
static char sm_notspam_macro_def[] = "{dcc_notspam}";
static char *sm_notspam_macro = sm_notspam_macro_def;

typedef struct {
    char    rcode[sizeof("5yz")];
    char    xcode[sizeof("1.23.45")];
#    define  REPLY_BUF 200
    char    pat[REPLY_BUF];
    u_char  is_pat;
} REPLY;
static REPLY rej_reply;
static REPLY grey_reply;
static REPLY temp_reply;
#define DCC_XCODE   "5.7.1"
#define DCC_RCODE   "550"
#define GREY_XCODE  "4.2.1"
#define GREY_RCODE  "452"

static CMN_ACTION action = CMN_REJECT;

static enum {
    DCCM_SETHDR, DCCM_ADDHDR, DCCM_NOHDR
} chghdr = DCCM_SETHDR;

/* DCC-milter state or context */
typedef struct work {
    SMFICTX	*milter_ctx;
#    define	 WORK_MILTER_CTX_IDLE ((SMFICTX *)DCC_SRVR_PORT)
#    define	 WORK_ID_LEN 24
    char	id[WORK_ID_LEN];
    char	reply_buf[REPLY_BUF+WORK_ID_LEN+INET6_ADDRSTRLEN];
    CMN_WORK	cw;
    /* from here down is zeroed when the structure is allocated */
#define WORK_ZERO fwd
    struct work *fwd;
    u_char	have_helo;
    /* from here down is zeroed when the structure is used for a 2nd msg */
#define WORK_REZERO reply_ptr
    char	*reply_ptr;
    REPLY	reply;
    int		num_x_dcc;		/* # of X-DCC headers in message */
    u_char	flags;
#    define	 FG_SOH	    0x01	/* seen first SMTP header */
} WORK;

#define WORK_EXCESS ((WORK *)1)


/* use a free list to avoid malloc() overhead */
static WORK *work_free;
static int work_too_many;
static time_t work_msg_time;

/* Every job involves
 *	a socket connected to sendmail,
 *	a log file,
 *	and a socket to talk to the DCC server.
 * While processing per-user whitelists there are
 *	another log file
 *	and an extra file descriptor for the main log file.
 * The file descriptors for the whitelists are accounted for in EXTRA_FILES */
#define FILES_PER_JOB	5

#define MAX_SELECT_WORK ((FD_SETSIZE-EXTRA_FILES)/FILES_PER_JOB)
#define DEF_MAX_WORK	200
#define MIN_MAX_WORK	2
static int max_max_work = MAX_SELECT_WORK;
static int max_work = DEF_MAX_WORK;

static sfsistat dccm_conn(SMFICTX *, char *, _SOCK_ADDR *);
static sfsistat dccm_helo(SMFICTX *, char *);
static sfsistat dccm_envfrom(SMFICTX *, char **);
static sfsistat dccm_envrcpt(SMFICTX *, char **);
static sfsistat dccm_header(SMFICTX *, char *, char *);
static sfsistat dccm_eoh(SMFICTX *);
static sfsistat dccm_body(SMFICTX *, u_char *, size_t);
static sfsistat dccm_eom(SMFICTX *);
static sfsistat dccm_abort(SMFICTX *);
static sfsistat dccm_close(SMFICTX *);

static char dccm_name[] = {"DCC"};
static struct smfiDesc smfilter = {
    dccm_name,				/* filter name */
    SMFI_VERSION,			/* version code -- do not change */
    SMFIF_CHGHDRS | SMFIF_ADDHDRS | SMFIF_DELRCPT,  /* flags */
    dccm_conn,				/* connection info filter */
    dccm_helo,				/* SMTP HELO command filter */
    dccm_envfrom,			/* envelope sender filter */
    dccm_envrcpt,			/* envelope recipient filter */
    dccm_header,			/* header filter */
    dccm_eoh,				/* end of header */
    dccm_body,				/* body block filter */
    dccm_eom,				/* end of message */
    dccm_abort,				/* message aborted */
    dccm_close				/* connection finished */
};


static char *add_braces(const char *);
static void *totals_msg_thread(void *);
static void totals_msg(void);
static void del_sock(const char *);
static void parse_reply(REPLY *, u_char,
			const char *, const char *, const char *);
static void make_reply(WORK *, REPLY *);
static sfsistat set_reply(WORK *);


static void
usage(u_char die)
{
	const char str[] = {
	    "usage: [-VdbxANQW] [-G on | off | noIP | IPmask/xx] [-h homedir]\n"
	    "    [-p protocol:filename | protocol:port@host] [-m map]\n"
	    "    [-w whiteclnt] [-U userdirs] [-a IGNORE | REJECT | DISCARD]\n"
	    "    [-t type,[log-thold,][spam-thold]]"
	    " [-g [not-]type] [-S header]\n"
	    "    [-l logdir] [-R rundir] [-r rejection-msg] [-j maxjobs]\n"
	    "    [-L ltype,facility.level]"
	};
	static u_char complained;

	/* its important to try to run, so don't give up unless necessary */
	if (die) {
		dcc_logbad(EX_USAGE, complained ? "giving up" : str);
	} else if (!complained) {
		dcc_error_msg("%s\ncontinuing", str);
		complained = 1;
	}
}


int NRATTRIB
main(int argc, char **argv)
{
	DCC_EMSG emsg;
#ifdef HAVE_RLIMIT_NOFILE
	struct rlimit nofile;
	int old_rlim_cur;
#endif
	FILE *f;
	long l;
	u_char log_tgts_set = 0;
	const char *logdir = 0;
	pthread_t msg_tid;
	WORK *wp;
	time_t smfi_main_start;
	char *p;
	const char *rundir = DCC_RUNDIR;
	const char *homedir = 0;
	DCC_PATH conn_def;		/* set later but before used */
	char *conn = conn_def;		/* MILTER socket specification */
	int result, i;

	dcc_syslog_init(1, argv[0], 0);
	dcc_init_tholds();

#ifdef HAVE_RLIMIT_NOFILE
	if (0 > getrlimit(RLIMIT_NOFILE, &nofile)) {
		dcc_error_msg("getrlimit(RLIMIT_NOFILE): %s", ERROR_STR());
		old_rlim_cur = 1000*1000;
	} else {
		old_rlim_cur = nofile.rlim_cur;
		if (nofile.rlim_max < 1000*1000) {
			i = nofile.rlim_max;
#ifndef USE_POLL
			if (i > FD_SETSIZE)
				i = FD_SETSIZE;
#endif
			max_max_work = (i - EXTRA_FILES)/FILES_PER_JOB;
			if (max_max_work <= 0) {
				dcc_error_msg("only %d open files allowed",
					      (int)nofile.rlim_max);
				max_max_work = 1;
			}
		}
	}
#else /* HAVE_RLIMIT_NOFILE */
	if (max_work <= 0) {
		dcc_error_msg(EX_OSERR, "too few open files allowed");
		max_max_work = 1;
	}
#endif /* HAVE_RLIMIT_NOFILE */
	max_work = min(DEF_MAX_WORK, max_max_work);

	while (EOF != (i = getopt(argc, argv,
				  "VdbxANQW"
				  "G:h:p:m:w:U:a:t:g:S:l:R:r:s:o:j:L:"))) {
		switch (i) {
		case 'V':
			fprintf(stderr, DCC_VERSION"\n");
			exit(EX_OK);
			break;

		case 'd':
			++dcc_clnt_debug;
			break;

		case 'b':
			background = 0;
			break;

		case 'x':
			try_extra_hard = DCC_CLNT_FG_NO_FAIL;
			break;

		case 'A':
			chghdr = DCCM_ADDHDR;
			smfilter.xxfi_flags &= ~SMFIF_CHGHDRS;
			smfilter.xxfi_flags |= SMFIF_ADDHDRS;
			break;

		case 'N':
			chghdr = DCCM_NOHDR;
			smfilter.xxfi_flags &= ~(SMFIF_ADDHDRS | SMFIF_CHGHDRS);
			break;

		case 'Q':
			dcc_query_only = 1;
			break;

		case 'W':
			to_white_only = 1;
			break;

		case 'G':
			if (!dcc_parse_client_grey(optarg))
				usage(0);
			break;

		case 'h':
			homedir = optarg;
			break;

		case 'p':
			conn = optarg;
			break;

		case 'm':
			mapfile_nm = optarg;
			break;

		case 'w':
			main_white_nm = optarg;
			break;

		case 'U':
			/* add '/' to end of the path without converting
			 * it to "/" */
			if (*optarg == '\0') {
				userdirs_path[0] = '\0';
			} else {
				strncpy(userdirs_path, optarg,
					sizeof(userdirs_path));
				i = strlen(userdirs_path)-1;
				while (i > 1 && userdirs_path[i] == '/')
					--i;
				userdirs_path[++i] = '/';
				userdirs_path[++i] = '\0';
			}
			userdirs = userdirs_path;
			break;

		case 'a':
			if (!strcasecmp(optarg, "IGNORE")) {
				action = CMN_IGNORE;
			} else if (!strcasecmp(optarg, "REJECT")) {
				action = CMN_REJECT;
			} else if (!strcasecmp(optarg, "DISCARD")) {
				action = CMN_DISCARD;
			} else {
				dcc_error_msg("unrecognized -a action: %s",
					      optarg);
			}
			break;

		case 't':
			if (dcc_parse_tholds('t', optarg))
				log_tgts_set = 1;
			break;

		case 'g':		/* honor not-spam "counts" */
			dcc_parse_honor(optarg);
			break;

		case 'S':
			dcc_add_sub_hdr(0, optarg);
			break;

		case 'l':		/* log rejected mail here */
			logdir = optarg;
			break;

		case 'R':
			rundir = optarg;
			break;

		case 'r':
			if (rej_reply.rcode[0] == '\0') {
				parse_reply(&rej_reply, 1,
					    DCC_XCODE, DCC_RCODE, optarg);
			} else if (grey_reply.rcode[0] == '\0') {
				parse_reply(&grey_reply, 1,
					    GREY_XCODE, GREY_RCODE, optarg);
				if (grey_reply.rcode[0] != '4'
				    || grey_reply.xcode[0] != '4') {
					dcc_error_msg("invalid greylist"
						      " message: %s", optarg);
					grey_reply.rcode[0] = '\0';
				}
			} else {
				dcc_error_msg("more than 2 -r settings");
			}
			break;

		case 's':		/* deprecated: set dcc_isspam */
			sm_isspam_macro = add_braces(optarg);
			break;

		case 'o':		/* deprecated: set dcc_notspam */
			sm_notspam_macro = add_braces(optarg);
			break;

		case 'j':		/* maximum simultaneous jobs */
			l = strtoul(optarg, &p, 0);
			if (*p != '\0' || l < MIN_MAX_WORK) {
				dcc_error_msg("invalid queue length %s",
					      optarg);
			} else if (l > max_max_work) {
				dcc_error_msg("queue length %s"
					      " larger than limit %d",
					      optarg, max_max_work);
				max_work = max_max_work;
			} else {
				max_work = l;
			}
			break;

		case 'L':
			dcc_parse_log_opt(optarg);
			break;

		default:
			usage(0);
		}
	}
	if (argc != optind)
		usage(0);

	if (!dcc_cdhome(emsg, homedir))
		dcc_error_msg("%s", emsg);

	snprintf(conn_def, sizeof(conn_def), "%s/%s", rundir, dcc_progname);

	if (logdir)
		dcc_log_init(0, logdir);
	if (!dcc_have_logdir) {
		if (log_tgts_set)
			dcc_error_msg("log thresholds set with -t"
				      " but no -l directory");
		/* if not logging,
		 * tell sendmail to not bother with some stuff */
		smfilter.xxfi_helo = 0;
	}


#ifdef HAVE_RLIMIT_NOFILE
	i = max_work*FILES_PER_JOB+EXTRA_FILES;
	if (old_rlim_cur < i) {
		nofile.rlim_cur = i;
		if (0 > setrlimit(RLIMIT_NOFILE, &nofile)) {
			dcc_error_msg("setrlimit(RLIMIT_NOFILE,%d): %s",
				      i, ERROR_STR());
			max_work = old_rlim_cur/FILES_PER_JOB - EXTRA_FILES;
			if (max_work <= 0) {
				dcc_error_msg("only %d open files allowed",
					      old_rlim_cur);
				max_work = MIN_MAX_WORK;
			}
		}
	}
#endif /* HAVE_RLIMIT_NOFILE */

	parse_reply(&temp_reply, 1,
		    "4.7.1", "451", "DCC failure");
	if (rej_reply.rcode[0] == '\0')
		parse_reply(&rej_reply, 1, DCC_XCODE, DCC_RCODE,
			    "mail %s from %s rejected by DCC");
	if (grey_reply.rcode[0] == '\0')
		parse_reply(&grey_reply, 1, GREY_XCODE, GREY_RCODE,
			    "mail %s from %s temporary greylist embargoed");

	/* Create the contexts. */
	i = max_work;
	wp = dcc_malloc(sizeof(*wp)*i);
	work_free = wp;
	memset(wp, 0, sizeof(*wp)*i);
	while (--i > 0) {
		wp->milter_ctx = WORK_MILTER_CTX_IDLE;
		cmn_create(&wp->cw);
		wp->fwd = wp+1;
		++wp;
	}
	wp->milter_ctx = WORK_MILTER_CTX_IDLE;
	cmn_create(&wp->cw);

	rcpts_create(max_work*2);

	if (MI_SUCCESS != smfi_setconn(conn))
		dcc_logbad(EX_USAGE, "illegal sendmail connection"
			   " \"%s\"\n", optarg);

	del_sock(conn);

	if (smfi_register(smfilter) == MI_FAILURE)
		dcc_logbad(EX_UNAVAILABLE, "smfi_register failed\n");

	if (background) {
		DCC_PATH pidpath;

		if (daemon(1, 0) < 0)
			dcc_logbad(EX_OSERR, "daemon(): %s", ERROR_STR());
		snprintf(pidpath, sizeof(pidpath), "%s/%s.pid",
			 rundir, dcc_progname);
		unlink(pidpath);
		f = fopen(pidpath, "w");
		if (!f) {
			dcc_error_msg("fopen(%s): %s",
				      pidpath, ERROR_STR());
		} else {
#ifdef linux
			/* Linux threads are broken.  Signals given the
			 * original process are delivered to only the
			 * thread that happens to have that PID.  The
			 * sendmail libmilter thread that needs to hear
			 * SIGINT and other signals does not, and that breaks
			 * scripts that need to stop dccm.
			 * However, signaling the process group works. */
			fprintf(f, "-%d\n", (u_int)getpgrp());
#else
			fprintf(f, "%d\n", (u_int)getpid());
#endif
			fclose(f);
		}
	}
	/* Be careful to start all threads only after the fork() in daemon(),
	 * because some POSIX threads packages (e.g. FreeBSD) get confused
	 * about threads in the parent.  */


	cmn_init(emsg);

	i = pthread_create(&msg_tid, 0, totals_msg_thread, 0);
	if (i)
		dcc_logbad(EX_SOFTWARE, "pthread_create(totals msg): %s",
			   ERROR_STR1(i));
	i = pthread_detach(msg_tid);
	if (i)
		dcc_logbad(EX_SOFTWARE, "pthread_detach(totals msg): %s",
			   ERROR_STR1(i));

	dcc_trace_msg(DCC_VERSION" listening to %s with %s", conn, dcc_homedir);
	if (dcc_clnt_debug)
		dcc_trace_msg("max_work=%d max_max_work=%d",
			      max_work, max_max_work);

	/* It would be nice to remove the UNIX domain socket and PID file
	 * when smfi_main() returns, but we dare not because the library
	 * delays for several seconds after being signalled to stop.
	 * Our files might have been unlinked and the files now in
	 * the filesystem might belong to some other process. */
	smfi_main_start = time(0);
	result = smfi_main();

	totals_msg();

	/* The sendmail libmilter machinery sometimes gets confused and
	 * gives up.  Try to start over if we had been running for at least
	 * 10 minutes */
	if (result != MI_SUCCESS
	    && time(0) > smfi_main_start+10*60) {
		dcc_error_msg("trying to restart after smfi_main() = %d",
			      result);
		execvp(DCC_LIBEXECDIR"/dccm", argv);
	}

	if (result != MI_SUCCESS)
		dcc_error_msg("smfi_main() = %d", result);
	exit((result == MI_SUCCESS) ? 0 : EX_UNAVAILABLE);
}



static char *
add_braces(const char *s)
{
	int i;
	char *new;

	i = strlen(s);
	if (i >= 2 && s[0] == '{' && s[i-1] == '}')
		return strdup(s);
	new = dcc_malloc(i+3);
	new[0] = '{';
	memcpy(new+1, s, i);
	new[i+1] = '}';
	new[i+2] = '\0';
	return new;
}



/* brag about our accomplishments */
static void
totals_msg(void)
{
	struct timeval now;
	struct tm tm;
	char tbuf[20];

	lock_work();
	gettimeofday(&now, 0);
	strftime(tbuf, sizeof(tbuf), "%x %X",
		 dcc_localtime(totals.msg_prev, &tm));
	if (grey_on) {
		dcc_trace_msg(DCC_VERSION
			      " greylist embargoed %d messages to %d targets;"
			      " %srejected %d messages for %d targets and"
			      " discarded for %d targets among"
			      " %d total messages for %d targets since %s",
			      totals.msgs_embargoed, totals.tgts_embargoed,
			      (action == CMN_IGNORE) ? "would have " : "",
			      totals.msgs_spam, totals.tgts_rejected,
			      totals.tgts_discarded,
			      totals.msgs, totals.tgts, tbuf);
	} else {
		dcc_trace_msg(DCC_VERSION
			      " %srejected %d messages for %d targets and"
			      " discarded for %d targets among"
			      " %d total messages for %d targets since %s",
			      (action == CMN_IGNORE) ? "would have " : "",
			      totals.msgs_spam, totals.tgts_rejected,
			      totals.tgts_discarded,
			      totals.msgs, totals.tgts, tbuf);
	}
	memset(&totals, 0, sizeof(totals));
	totals.msg_prev = now.tv_sec;
	unlock_work();
}



static void *
totals_msg_thread(void *ign UATTRIB)
{
	time_t now, next;
	struct tm tm;
	int secs;

	totals.msg_prev = time(0);
	for (;;) {
		now = time(0);
		dcc_localtime(now, &tm);
		tm.tm_sec = 0;
		tm.tm_min = 0;
		tm.tm_hour = 0;
		++tm.tm_mday;
		next = mktime(&tm);
		if (next == -1) {
			dcc_error_msg("mktime() failed");
			secs = 60*60;
		} else {
			secs = next - now;
		}
		do {
			secs = sleep(secs);
		} while (secs > 0);
		totals_msg();
	}
}



/* remove the Unix domain socket of a previous instance of this daemon */
static void
del_sock(const char *conn0)
{
	int s;
	struct stat sb;
	const char *conn;
	struct sockaddr_un conn_sun;
	int len, i;

	/* Ignore the sendmail milter "local|whatever:" prefix.
	 * If it is a UNIX domain socket, fine.  If not, no harm is done */
	conn = strchr(conn0, ':');
	if (conn)
		++conn;
	else
		conn = conn0;

	len = strlen(conn);
	if (len >= ISZ(conn_sun.sun_path))
		return;			/* perhaps not a UNIX domain socket */

	memset(&conn_sun, 0, sizeof(conn_sun));
	conn_sun.sun_family = AF_LOCAL;
	strcpy(conn_sun.sun_path, conn);
#ifdef DCC_HAVE_SA_LEN
	conn_sun.sun_len = SUN_LEN(&conn_sun);
#endif

	if (0 > stat(conn_sun.sun_path, &sb))
		return;
	if (!(S_ISSOCK(sb.st_mode) || S_ISFIFO(sb.st_mode)))
		dcc_logbad(EX_UNAVAILABLE, "non-socket present at %s",
			   conn_sun.sun_path);

	/* The sendmail libmilter seems to delay as long as 5 seconds
	 * before stopping.  It delays indefinitely if an SMTP client
	 * is stuck. */
	i = 0;
	for (;;) {
		s = socket(AF_UNIX, SOCK_STREAM, 0);
		if (s < 0) {
			dcc_logbad(EX_OSERR, "socket(AF_UNIX): %s",
				   ERROR_STR());
			return;
		}
		if (++i > 5*10)
			dcc_logbad(EX_UNAVAILABLE,
				   "DCCM or something already or still running"
				   " with socket at %s",
				   conn_sun.sun_path);
		if (0 > connect(s, (struct sockaddr *)&conn_sun,
				sizeof(conn_sun))) {
			/* unlink it only if it looks like a dead socket */
			if (errno == ECONNREFUSED || errno == ECONNRESET
			    || errno == EACCES) {
				if (0 > unlink(conn_sun.sun_path))
					dcc_error_msg("unlink(old %s): %s",
						      conn_sun.sun_path,
						      ERROR_STR());
			} else {
				dcc_error_msg("connect(old %s): %s",
					      conn_sun.sun_path, ERROR_STR());
			}
			close(s);
			break;
		}
		close(s);
		usleep(100*1000);
	}
}



static WORK *
work_alloc(void)
{
	WORK *wp;

	lock_work();
	wp = work_free;
	if (!wp) {
		++work_too_many;
		unlock_work();
		return 0;
	}
	if (wp->milter_ctx != WORK_MILTER_CTX_IDLE) {
		dcc_error_msg("corrupt WORK area");
		abort();
	}
	work_free = wp->fwd;
	unlock_work();

	/* clear most of it */
	memset(&wp->WORK_ZERO, 0,
	       sizeof(*wp) - ((char*)&wp->WORK_ZERO - (char*)wp));
	cmn_clear(&wp->cw, wp);

	return wp;
}



static WORK *
get_wp(SMFICTX *milter_ctx, const char *hook)
{
	WORK *wp;

	wp = (WORK *)smfi_getpriv(milter_ctx);
	if (wp == WORK_EXCESS) {
		if (dcc_clnt_debug && hook)
			dcc_trace_msg("%s for excessive message", hook);
		return 0;
	}
	if (!wp) {
		dcc_error_msg("null SMFICTX pointer for %s", hook);
		return 0;
	}
	if (wp->milter_ctx != milter_ctx) {
		dcc_error_msg("bogus SMFICTX pointer or corrupt WORK area");
		smfi_setpriv(milter_ctx, 0);
		return 0;
	}

	if (!wp->cw.dcc_ctxt)
		return 0;		/* failed to find DCC server */

	return wp;
}



/* we are finished with one SMTP message.
 * get ready for the next from the same connection to an SMTP client */
static void
work_done(WORK *wp, const char *result)
{
	LOG_CAPTION(wp, "result: ");
	log_write(&wp->cw, result, 0);
	LOG_EOL(wp);
	log_stop(&wp->cw);

	rcpts_free(&wp->cw, 1);

	memset(&wp->WORK_REZERO, 0,
	       sizeof(*wp) - ((char*)&wp->WORK_REZERO - (char*)wp));
	cmn_clear(&wp->cw, wp);
}



/* start a new connection to an SMTP client */
static sfsistat
dccm_conn(SMFICTX *milter_ctx,
	  char *name,			/* SMTP client hostname */
	  _SOCK_ADDR *sender)
{
	WORK *wp;

	wp = (WORK *)smfi_getpriv(milter_ctx);
	if (wp) {
		dcc_error_msg("bogus initial SMFICTX pointer");
		return SMFIS_TEMPFAIL;
	}
	wp = work_alloc();
	if (!wp) {
		smfi_setpriv(milter_ctx, WORK_EXCESS);
		return SMFIS_TEMPFAIL;
	}
	smfi_setpriv(milter_ctx, wp);
	wp->milter_ctx = milter_ctx;

	if (!name) {
		if (dcc_clnt_debug)
			cmn_trace_msg(&wp->cw, "null sender name");
		strcpy(wp->cw.clnt_name, "(null name)");
	} else {
		BUFCPY(wp->cw.clnt_name, name);
	}

	if (!sender) {
		if (!strcasecmp(wp->cw.clnt_name, "localhost")) {
			wp->cw.clnt_addr.s6_addr32[3] = htonl(0x7f000001);
			wp->cw.clnt_addr.s6_addr32[0] = 0;
			wp->cw.clnt_addr.s6_addr32[1] = 0;
			wp->cw.clnt_addr.s6_addr32[2] = htonl(0xffff);
			strcpy(wp->cw.clnt_str, "127.0.0.1");
		} else {
			if (dcc_clnt_debug)
				cmn_trace_msg(&wp->cw,
					      "null sender address for \"%s\"",
					      wp->cw.clnt_name);
			strcpy(wp->cw.clnt_str, "(null address)");
		}
	} else if (sender->sa_family != AF_INET
		   && sender->sa_family != AF_INET6) {
		dcc_error_msg("unexpected sender address family %d",
			      sender->sa_family);
		snprintf(wp->cw.clnt_str, sizeof(wp->cw.clnt_str),
			 "(milter AF %d)", sender->sa_family);
	} else {
		if (sender->sa_family == AF_INET) {
			dcc_toipv6(&wp->cw.clnt_addr,
				   ((struct sockaddr_in *)sender)->sin_addr);
		} else if (sender->sa_family == AF_INET6) {
			memcpy(&wp->cw.clnt_addr,
			       &((struct sockaddr_in6 *)sender)->sin6_addr,
			       sizeof(wp->cw.clnt_addr));
		}
		if (!DCC_INET_NTOP(AF_INET6, &wp->cw.clnt_addr, wp->cw.clnt_str,
				   sizeof(wp->cw.clnt_str))) {
			dcc_error_msg("inet_ntop(addr of %s) failed",
				      wp->cw.clnt_name);
			strcpy(wp->cw.clnt_str, "(inet_ntop() ?)");
		}
	}

	log_start(&wp->cw);
	log_print(&wp->cw, DCC_XHDR_TYPE_IP": %s %s\n",
		  wp->cw.clnt_name, wp->cw.clnt_str);

	if (!ck_dcc_ctxt(&wp->cw)) {
		make_reply(wp, &temp_reply);
		return set_reply(wp);
	}

	/* This much is common for all of the messages that might
	 * arrive through this connection to the SMTP client */

	return SMFIS_CONTINUE;
}



/* log HELO */
static sfsistat
dccm_helo(SMFICTX *milter_ctx, char *helo)
{
	WORK *wp;
	int i;

	wp = get_wp(milter_ctx, "helo");
	if (!wp)
		return SMFIS_TEMPFAIL;

	i = strlen(helo);
	if (i < ISZ(wp->cw.helo)) {
		memcpy(wp->cw.helo, helo, i+1);
	} else {
		memcpy(wp->cw.helo, helo, ISZ(wp->cw.helo)-ISZ(DCC_HELO_CONT));
		strcpy(&wp->cw.helo[ISZ(wp->cw.helo)-ISZ(DCC_HELO_CONT)],
		       DCC_HELO_CONT);
	}

	wp->have_helo = 1;
	log_print(&wp->cw, "HELO: %s\n", wp->cw.helo);

	return SMFIS_CONTINUE;
}



/* deal with Mail From envelope value */
static sfsistat
dccm_envfrom(SMFICTX *milter_ctx, char **from)
{
	static char mail_host_macro[] = "{mail_host}";
	static char dcc_mail_host_macro[] = "{dcc_mail_host}";
	const char *mail_host;
	WORK *wp;

	wp = get_wp(milter_ctx, "envfrom");
	if (!wp)
		return SMFIS_TEMPFAIL;

	wp->cw.action = action;

	dcc_get_ipv6_ck(&wp->cw.cks, &wp->cw.clnt_addr);
	dcc_ck_get_sub(&wp->cw.cks, "helo", wp->have_helo ? wp->cw.helo : "");

	if (log_start(&wp->cw)) {
		log_print(&wp->cw, DCC_XHDR_TYPE_IP": %s %s\n",
			  wp->cw.clnt_name, wp->cw.clnt_str);
		if (wp->have_helo)
			log_print(&wp->cw, "HELO: %s\n", wp->cw.helo);
	}

	/* Even if sendmail.cf sets the ${dcc_mail_host} macro,
	 * FEATURE(delay_checks) can delay its setting until after
	 * the MAIL command has been processed and this milter function
	 * has been called. */
	mail_host = smfi_getsymval(milter_ctx, dcc_mail_host_macro);
	if (!mail_host || !*mail_host)
		mail_host = smfi_getsymval(milter_ctx, mail_host_macro);
	dcc_ck_get_sub(&wp->cw.cks, "mail_host",
		       (mail_host != 0) ? mail_host : "");

	LOG_CAPTION(wp, DCC_XHDR_TYPE_ENV_FROM": ");
	log_write(&wp->cw, from[0], 0);
	LOG_CAPTION(wp, "  mail_host=");
	if (mail_host)
		log_write(&wp->cw, mail_host, 0);
	LOG_EOL(wp);
	dcc_get_cks(&wp->cw.cks, DCC_CK_ENV_FROM, from[0], 1);

	/* get ready for the body checksums before the headers so that
	 * we can notice the MIME separator */
	dcc_ck_body_init(&wp->cw.cks);

	return SMFIS_CONTINUE;
}



/* log recipients and check the local whitelist for each one */
static sfsistat
dccm_envrcpt(SMFICTX *milter_ctx, char **rcpt)
{
	static char dollar_i[] = "i";
	static char rcpt_mailer[] = "{rcpt_mailer}";
	static char rcpt_addr[] = "{rcpt_addr}";
	static char dcc_userdir[] = "{dcc_userdir}";
	const char *id, *mailer, *addr, *dir;
	WORK *wp;
	RCPT_ST *rcpt_st;
	char c, *p;

	wp = get_wp(milter_ctx, "envrcpt");
	if (!wp)
		return SMFIS_TEMPFAIL;

	rcpt_st = rcpt_st_alloc(&wp->cw, 1);
	if (!rcpt_st)
		return SMFIS_TEMPFAIL;

	LOG_CAPTION(wp, DCC_XHDR_TYPE_ENV_TO": ");
	BUFCPY(rcpt_st->env_to, rcpt[0]);
	log_write(&wp->cw, rcpt_st->env_to, 0);

	id = smfi_getsymval(milter_ctx, dollar_i);
	if (id)
		BUFCPY(wp->id, id);
	else
		wp->id[0] = '\0';

	addr = smfi_getsymval(milter_ctx, rcpt_addr);
	if (addr
	    && (*addr == '\0' || !strcmp(addr, "..") || strchr(addr, '/'))) {
		if (dcc_clnt_debug)
			cmn_trace_msg(&wp->cw, "insecure {rcpt_addr} \"%s\"",
				      addr);
		addr = 0;
	}
	LOG_CAPTION(wp, "  addr=");
	if (addr) {
		/* Convert ASCII upper to lower case.
		 * Be simplistic about international character sets and
		 * avoid locale and portability complications. */
		p = rcpt_st->user;
		do {
			c = *addr++;
			if (c >= 'A' && c <= 'Z')
				c -= 'A' - 'a';
			*p++ = c;
		} while (c != '\0' && p < LAST(rcpt_st->user));
		log_write(&wp->cw, rcpt_st->user, 0);
	} else {
		rcpt_st->user[0] = '\0';
	}

	/* pick a per-user whitelist and log directory */
	rcpt_st->dir[0] = '\0';
	if (userdirs && dcc_have_logdir) {
		dir = smfi_getsymval(milter_ctx, dcc_userdir);
		if (dir) {
			snprintf(rcpt_st->dir, sizeof(rcpt_st->dir),
				 "%s%s", userdirs, dir);
			/* for compatiblity with not using ${dcc_userdir}
			 * and consistency with case confusion by mail senders,
			 * convert ASCII upper to lower case in the (probable)
			 * username */
			for (p = strrchr(rcpt_st->dir, '/');
			     (c = *p) != '/';
			     --p) {
				if (c >= 'A' && c <= 'Z')
					*p -= 'A' - 'a';
			}
			if (strstr(rcpt_st->dir, "/../")) {
				if (dcc_clnt_debug)
					cmn_trace_msg(&wp->cw,
						      "insecure {dcc_userdir}"
						      " \"%s\"",
						      dir);
				rcpt_st->dir[0] = '\0';
			}
		}
		if (rcpt_st->dir[0] == '\0'
		    && (mailer = smfi_getsymval(milter_ctx, rcpt_mailer)) != 0
		    && addr) {
			snprintf(rcpt_st->dir, sizeof(rcpt_st->dir),
				 "%s%s/%s",
				 userdirs, mailer, rcpt_st->user);
		}
	}
	LOG_CAPTION(wp, "  dir=");
	log_write(&wp->cw, rcpt_st->dir, 0);
	LOG_EOL(wp);

	++wp->cw.tgts;

	wp->cw.log_pos_to_end = log_lseek(&wp->cw, SEEK_END);

	return SMFIS_CONTINUE;
}



static sfsistat
dccm_header(SMFICTX *milter_ctx, char *headerf, char *headerv)
{
	WORK *wp;

	wp = get_wp(milter_ctx, "header");
	if (!wp)
		return SMFIS_TEMPFAIL;

	if (wp->cw.log_fd >= 0) {
		if (!(wp->flags & FG_SOH)) {
			wp->flags |= FG_SOH;
			LOG_EOL(wp);
		}
		log_write(&wp->cw, headerf, 0);
		LOG_CAPTION(wp, ": ");
		log_write(&wp->cw, headerv, 0);
		LOG_EOL(wp);
	}

	/* compute DCC checksums for favored headers */
#define GET_HDR_CK(t) if (!strcasecmp(headerf, DCC_XHDR_TYPE_##t)) {	\
		dcc_get_cks(&wp->cw.cks, DCC_CK_##t, headerv, 1);	\
		return SMFIS_CONTINUE;}
	GET_HDR_CK(FROM);
	GET_HDR_CK(MESSAGE_ID);
	GET_HDR_CK(RECEIVED);
#undef GET_HDR_CK

	if (!CSTRCMP(headerf, DCC_XHDR_START))
		++wp->num_x_dcc;

	dcc_ck_get_sub(&wp->cw.cks, headerf, headerv);

	/* Notice MIME multipart boundary definitions */
	dcc_ck_mime_hdr(&wp->cw.cks, headerf, headerv);

	return SMFIS_CONTINUE;
}



static sfsistat
dccm_eoh(SMFICTX *milter_ctx)
{
	WORK *wp;

	wp = get_wp(milter_ctx, "eoh");
	if (!wp)
		return SMFIS_TEMPFAIL;

	/* Create a checksum for a null Message-ID header if there
	 * was no Message-ID header.  */
	if (wp->cw.cks.sums[DCC_CK_MESSAGE_ID].type != DCC_CK_MESSAGE_ID)
		dcc_get_cks(&wp->cw.cks, DCC_CK_MESSAGE_ID, "", 0);

	/* log the blank line between the header and the body */
	LOG_EOL(wp);

	return SMFIS_CONTINUE;
}



static sfsistat
dccm_body(SMFICTX *milter_ctx, u_char *bodyp, size_t bodylen)
{
#if MAX_LOG_SIZE > 0
	int len;
	const u_char *p, *lim;
#endif
	WORK *wp;

	wp = get_wp(milter_ctx, "body");
	if (!wp)
		return SMFIS_TEMPFAIL;

	/* Log the body block
	 * avoid filling the disk by truncating the body */
	if (wp->cw.log_fd >= 0) {
#if MAX_LOG_SIZE > 0
		len = MAX_LOG_SIZE - wp->cw.log_size;
		if (len >= 0) {
			if (len >= (int)bodylen) {
				log_write(&wp->cw, bodyp, bodylen);
			} else {
				lim = bodyp;
				p = lim+len;
				if (len > 80)
					lim += len-80;
				while (--p > lim) {
					if (*p == '\n') {
					    len = p-bodyp+1;
					    break;
					}
				}
				log_write(&wp->cw, bodyp, len);
				if (bodyp[len-1] != '\n')
					LOG_EOL(wp);
				LOG_CAPTION(wp, DCC_LOG_TRN_MSG_CR);
				wp->cw.log_size = MAX_LOG_SIZE+1;
			}
		}
#else
		log_write(&wp->cw, bodyp, bodylen);
#endif
	}

	dcc_ck_body(&wp->cw.cks, bodyp, bodylen);

	return SMFIS_CONTINUE;
}



/* Parse a string into RFC 821 and RFC 2034 codes and a pattern to make a
 * message, or generate the codes for a string that has a message
 * without codes.
 * The should be something like "5.7.1 550 spammer go home"
 * or "spammer go home".
 * The pattern can have as many as two "%s" references to the sendmail
 * queue-ID and SMTP client IP address.
 */
static void
parse_reply(REPLY *reply,		/* to here */
	    u_char pat_ok,
	    const char *xcode,		/* default value such as "5.7.1" */
	    const char *rcode,		/* default value such as "550" */
	    const char *str)		/* from here */
{
	const char *p;
	int dot1, dot2, e;
	char c, *pat;
	int p_cnt;

	p = dcc_parse_word(0, reply->xcode, sizeof(reply->xcode), str,
			   0, 0, 0);
	p = dcc_parse_word(0, reply->rcode, sizeof(reply->rcode), p,
			   0, 0, 0);
	if (!p
	    || (reply->rcode[0] != '4' && reply->rcode[0] != '5')
	    || reply->rcode[0] != reply->xcode[0]
	    || strspn(reply->rcode, "0123456789") != 3
	    || (dot1 = strspn(reply->xcode, "0123456789")) == 0
	    || reply->xcode[dot1] != '.'
	    || (dot2 = strspn(&reply->xcode[dot1+1], "0123456789")) == 0
	    || reply->xcode[dot1+1+dot2] != '.'
	    || (e = strspn(&reply->xcode[dot1+1+dot2+1], "0123456789")) == 0
	    || reply->xcode[dot1+1+dot2+1+e] != '\0') {
		BUFCPY(reply->xcode, xcode);
		BUFCPY(reply->rcode, rcode);
		p = str;
	}

	p += strspn(p, DCC_WHITESPACE);
	if (!strchr(p, '%')) {
		reply->is_pat = 0;
		BUFCPY(reply->pat, p);
	} else {
		reply->is_pat = 1;
		p_cnt = pat_ok ? 0 : 2;
		pat = reply->pat;
		do {
			c = *p++;
			if (c == '\0')
				break;
			*pat++ = c;
			if (c == '%') {
				if (*p != 's' || ++p_cnt > 2)
					*pat++ = '%';
			}
		} while (pat < LAST(reply->pat)-2);
		*pat = '\0';
	}
}



static void
make_reply(WORK *wp, REPLY *reply)
{
	const char *clnt;

	memcpy(wp->reply.rcode, reply->rcode, sizeof(wp->reply.rcode));
	memcpy(wp->reply.xcode, reply->xcode, sizeof(wp->reply.xcode));
	if (!reply->is_pat) {
		wp->reply_ptr = reply->pat;
	} else {
		clnt = wp->cw.clnt_str;
		if (!strncmp("::ffff:", clnt, 7))
			clnt += 7;	/* strip IPv6 prefix */
		snprintf(wp->reply_buf, sizeof(wp->reply_buf),
			 reply->pat, wp->id, clnt);
		wp->reply_ptr = wp->reply_buf;
	}
}



static sfsistat
set_reply(WORK *wp)
{
	int i;

	/* temporize if we have not figured out what to tell sendmail */
	if (!wp->reply_ptr)
		make_reply(wp, &temp_reply);

	i = smfi_setreply(wp->milter_ctx,
			  wp->reply.rcode, wp->reply.xcode, wp->reply_ptr);
	if (i != MI_SUCCESS)
		cmn_error_msg(&wp->cw, "smfi_setreply(\"%s\",\"%s\",\"%s\")=%d",
			      wp->reply.rcode, wp->reply.xcode, wp->reply_ptr,
			      i);

	LOG_CAPTION(wp, "rejection message: ");
	log_write(&wp->cw, wp->reply.rcode, 0);
	LOG_CAPTION(wp, " ");
	log_write(&wp->cw, wp->reply.xcode, 0);
	LOG_CAPTION(wp, " ");
	log_write(&wp->cw, wp->reply_ptr, 0);
	LOG_EOL(wp);

	if (wp->reply.rcode[0] == '4') {
		work_done(wp, "temporary reject");
		return SMFIS_TEMPFAIL;
	} else {
		work_done(wp, "reject");
		return SMFIS_REJECT;
	}
}



/* tell sendmail to delete our header */
static void
delete_xhdr(WORK *wp, int hdr_num)
{
	static char null[] = "";	/* libmilter doesn't know about const */

	if (chghdr != DCCM_SETHDR)
		return;

	for (; hdr_num <= wp->num_x_dcc; ++hdr_num) {
		if (MI_SUCCESS != smfi_chgheader(wp->milter_ctx, wp->cw.xhdr,
						 hdr_num, null))
			break;
	}
}



/* see what sendmail had to say about the message */
static void
ask_sm(SMFICTX *milter_ctx, WORK *wp)
{
	const char *sm_notspam, *sm_isspam;
	char c, *p;

	wp->reply.rcode[0] = 0;

	if (0 != (sm_notspam = smfi_getsymval(milter_ctx, sm_notspam_macro))
	    && *sm_notspam != '\0') {
		/* If we have a sendmail macro name that indicates a
		 * white listing from sendmail rules and databases,
		 * check to see if the macro is set.
		 * If not, check for a spam sendmail macro name. */
		wp->cw.honor |= DCC_HONOR_MTA_NOTSPAM;
		LOG_CAPTION(wp, "sendmail.cf-->OK: \"");
		log_write(&wp->cw, sm_notspam, 0);
		LOG_CAPTION(wp, "\"\n");

	} else if (0 != (sm_isspam = smfi_getsymval(milter_ctx,
						    sm_isspam_macro))) {
		/* If sendmail says it is spam, our white lists are irrelevant
		 *
		 * Strip double quotes from the sendmail message.
		 * Sendmail needs quotes around the message so it
		 * won't convert blanks to dots. */
		p = wp->reply_buf;
		while (p < &wp->reply_buf[sizeof(wp->reply_buf)-1]
		       && (c = *sm_isspam++) !='\0') {
			if (c != '"')
				*p++ = c;
		}
		*p = '\0';
		p = wp->reply_buf+strspn(wp->reply_buf, DCC_WHITESPACE);
		wp->cw.honor |= DCC_HONOR_MTA_ISSPAM;
		log_print(&wp->cw, "sendmail.cf-->%s: \"%s\"\n",
			  sm_isspam_macro, p);
		if (!CSTRCMP(p, "DISCARD")) {
			p += STRZ("DISCARD");
			p += strspn(p, DCC_WHITESPACE":");
			wp->cw.action = CMN_DISCARD;
		} else {
			wp->cw.action = CMN_REJECT;
		}
		parse_reply(&wp->reply, 0, DCC_XCODE, DCC_RCODE, p);
		make_reply(wp, &wp->reply);
	}
}



void
user_reject(CMN_WORK *cwp, RCPT_ST *rcpt_st,
	    u_char rej_type)		/* 0=recipient only, 1=all, 2=grey */
{
	WORK *wp = cwp->wp;
	int i;

	/* if one of the other targets wants this message,
	 * try to remove this address from sendmail's list */
	if (rej_type == 0) {
		i = smfi_delrcpt(wp->milter_ctx, rcpt_st->env_to);
		if (MI_SUCCESS != i)
			cmn_error_msg(cwp, "delrcpt(%s)=%d",
				      rcpt_st->env_to, i);
		++totals.tgts_discarded;
		--cwp->reject_tgts;
		return;
	}

	if (!wp->reply_ptr)
		make_reply(wp, rej_type == 2 ? &grey_reply : &rej_reply);
	USER_LOG_CAPTION(rcpt_st, "rejection message: ");
	user_log_write(rcpt_st, wp->reply_ptr, 0);
	USER_LOG_EOL(rcpt_st);
}



/* deal with the end of the SMTP message as announced by sendmail */
static sfsistat
dccm_eom(SMFICTX *milter_ctx)
{
	WORK *wp;
	RCPT_ST *rcpt_st;
	char *hdr;
	int i;

	wp = get_wp(milter_ctx, "eom");
	if (!wp)
		return SMFIS_TEMPFAIL;

	dcc_ck_body_fin(&wp->cw.cks);

	LOG_CAPTION(wp, DCC_LOG_MSG_SEP);

	ask_sm(milter_ctx, wp);

	/* check the grey and white lists */
	cmn_ask_white(&wp->cw, 0);

	LOG_CAPTION(wp, "\n");
	wp->cw.header.buf[0] = '\0';
	wp->cw.header.used = 0;
	if (wp->cw.tgts != wp->cw.white_tgts) {
		/* Report to the DCC and add our header if allowed.
		 * Request a temporary failure if the DCC failed and we
		 * are trying hard */
		i = cmn_ask_dcc(&wp->cw, 0);
		if (i <= 0) {
			if (!i && try_extra_hard) {
				make_reply(wp, &temp_reply);
				return set_reply(wp);
			}
			/* after serious errors,
			 * act as if DCC server said not-spam */
			delete_xhdr(wp, 1);
		}

		/* install the X-DCC header */
		hdr = &wp->cw.header.buf[wp->cw.xhdr_len]+2;
		switch (chghdr) {
		case DCCM_NOHDR:
			break;
		case DCCM_SETHDR:
			/* delete extra headers to foil tricky spammers */
			delete_xhdr(wp, 2);
			i = smfi_chgheader(wp->milter_ctx, wp->cw.xhdr, 0, hdr);
			if (MI_SUCCESS != i)
				cmn_error_msg(&wp->cw,
					      "smfi_chgheader(\"%s\",0,\"%s\")="
					      "%d",
					      wp->cw.xhdr, hdr, i);
			break;
		case DCCM_ADDHDR:
			i = smfi_addheader(wp->milter_ctx, wp->cw.xhdr, hdr);
			if (MI_SUCCESS != i)
				cmn_error_msg(&wp->cw,
					      "smfi_addheader(\"%s\",\"%s\")="
					      "%d",
					      wp->cw.xhdr, hdr, i);
			break;
		}

	} else {
		/* remove our X-DCC header if not ok to report to the DCC */
		wp->cw.cks.flags &= ~DCC_CKS_HAVE_SUM;
		delete_xhdr(wp, 1);
		/* reject and/or log it if the target count is high enough */
		dcc_honor_log_cnts(&wp->cw.honor, &wp->cw.cks,
				   ((wp->cw.honor & DCC_HONOR_LOCAL_ISSPAM)
				    ? DCC_TGTS_TOO_MANY : wp->cw.tgts));
	}

	++totals.msgs;
	totals.tgts += wp->cw.tgts;
	users_process(&wp->cw);

	if (wp->cw.reject_tgts == 0) {
		/* it is not spam for any target */
		if (wp->cw.honor & DCC_HONOR_GREY_EMBARGO) {
			totals.tgts_embargoed += wp->cw.tgts;
			++totals.msgs_embargoed;
			return set_reply(wp);
		}
		/* deliver it if all (remaining) targets want it */
		work_done(wp, "accept");
		return SMFIS_ACCEPT;
	}

	/* it is spam for at least some targets */

	if (wp->cw.action == CMN_IGNORE) {
		/* if we are ignoring spam but can greylist it, do so */
		if (wp->cw.honor & DCC_HONOR_GREY_EMBARGO) {
			totals.tgts_embargoed += wp->cw.tgts;
			++totals.msgs_embargoed;
			return set_reply(wp);
		}
		if (wp->cw.reject_tgts != 0) {
			totals.tgts_rejected += wp->cw.reject_tgts;
			++totals.msgs_spam;
		}
		work_done(wp, "ignore and accept");
		return SMFIS_ACCEPT;
	}

	/* Reject it.
	 * Tell greylist to restore the embargo for those targets
	 * that believe the message was spam and did not
	 * white- or blacklist it */
	if (grey_on) {
		for (rcpt_st = wp->cw.rcpt_st;
		     rcpt_st;
		     rcpt_st = rcpt_st->fwd) {
			if (rcpt_st->grey_result == DCC_GREY_ASK_OFF
			    || rcpt_st->grey_result == DCC_GREY_ASK_FAIL)
				continue;
			if (!dcc_grey_spam(wp->cw.emsg, wp->cw.dcc_ctxt,
					   &wp->cw.cks, rcpt_st->triple_sum))
				cmn_error_msg(&wp->cw, "%s", wp->cw.emsg);
		}
	}
	if (wp->cw.action == CMN_DISCARD) {
		/* discard it if that is our choice
		 * or if sendmail said to */
		if (wp->cw.reject_tgts != 0) {
			totals.tgts_discarded += wp->cw.reject_tgts;
			++totals.msgs_spam;
		}
		work_done(wp, "discard");
		return SMFIS_DISCARD;

	} else {
		if (wp->cw.reject_tgts != 0) {
			totals.tgts_rejected += wp->cw.reject_tgts;
			++totals.msgs_spam;
		}
	}

	/* grey listed or wp->cw.action is CMN_REJECT so reject it */
	return set_reply(wp);
}



static sfsistat
dccm_close(SMFICTX *milter_ctx)
{
	int msg_cnt;
	struct timeval tv;
	WORK *wp;

	wp = smfi_getpriv(milter_ctx);
	if (!wp)
		return SMFIS_TEMPFAIL;
	if (wp == WORK_EXCESS) {
		smfi_setpriv(milter_ctx, 0);
		return SMFIS_TEMPFAIL;
	}
	if (wp->milter_ctx != milter_ctx) {
		dcc_error_msg("corrupt WORK area at close");
		abort();
	}
	wp->milter_ctx = WORK_MILTER_CTX_IDLE;

	LOG_CAPTION(wp, "close\n");
	log_stop(&wp->cw);

	lock_work();
	rcpts_free(&wp->cw, 0);

	wp->fwd = work_free;
	work_free = wp;
	msg_cnt = work_too_many;
	if (msg_cnt != 0) {
		gettimeofday(&tv, 0);
		if (work_msg_time == tv.tv_sec) {
			msg_cnt = 0;
		} else {
			work_msg_time = tv.tv_sec;
			work_too_many = 0;
		}
	}
	unlock_work();
	if (msg_cnt != 0)
		dcc_error_msg("%d too many simultaneous mail messages",
			      msg_cnt);

	smfi_setpriv(milter_ctx, 0);

	return SMFIS_CONTINUE;
}



static sfsistat
dccm_abort(SMFICTX *milter_ctx)
{
	WORK *wp = get_wp(milter_ctx, "abort");
	if (!wp)
		return SMFIS_TEMPFAIL;

	if (dcc_clnt_debug)
		wp->cw.honor |= DCC_HONOR_LOGIT;
	log_print(&wp->cw, "SMTP transaction aborted by %s\n",
		  wp->cw.clnt_name);
	log_stop(&wp->cw);
	rcpts_free(&wp->cw, 1);

	return SMFIS_CONTINUE;
}
