/* Distributed Checksum Clearinghouse
 *
 * checksum routines
 *
 * Copyright (c) 2005 by Rhyolite Software
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.2.74-1.87 $Revision$
 */

#ifndef DCC_CK_H
#define DCC_CK_H

#include "dcc_clnt.h"
#include "dcc_md5.h"


/* MIME boundary
 * RFC 1341 says boundaries can be 70 bytes, but handle non-conformant spam */
#define DCC_CK_BND_MAX	    94
#define DCC_CK_BND_DIM	(DCC_CK_BND_MAX+2)  /* including leading -- */
#define DCC_CK_BND_MISS	(DCC_CK_BND_DIM+1)
typedef struct {
    u_char	bnd_len;		/* length including leading "--" */
    u_char	cmp_len;		/* compared so far */
    char	bnd[DCC_CK_BND_DIM];	/* "--"boundary */
} DCC_CK_BND;


typedef u_char DCC_CK_FC[256];
extern const DCC_CK_FC dcc_cset_1, dcc_cset_2;

/* state machine for ignoring parts of URLs */
typedef struct {
    enum {
	DCC_URL_ST_IDLE,		/* waiting for H or = */
	DCC_URL_ST_QUOTE,		/*    "     " '"' or 'H' after = */
	DCC_URL_ST_QH,			/*    "     "  H after '"' */
	DCC_URL_ST_T1,			/*    "     "  T */
	DCC_URL_ST_T2,			/*    "     "  T */
	DCC_URL_ST_P,			/*    "     "  P */
	DCC_URL_ST_S,			/*    "     "  [S] */
	DCC_URL_ST_COLON,		/*    "     "  : */
	DCC_URL_ST_SLASH1,		/*    "     "  / */
	DCC_URL_ST_SLASH2,		/*    "     "  / */
	DCC_URL_ST_SLASH3_START,
	DCC_URL_ST_SLASH3,		/*    "     "  third / */
	DCC_URL_ST_SKIP,		/* skipping rest of URL */
    } st;
    char	*start;			/* start of hostname in URL */
    char	*dot;			/* last '.' in hostname */
    int		total;			/* length of URL */
    u_char	flags;
#    define	 DCC_URL_QUOTED	    0x01
#    define	 DCC_URL_DEL_DOMAIN 0x02    /* waiting for domain to delete */
#    define	 DCC_URL_PERCENT1   0x04    /* waiting for 1st digit after % */
#    define	 DCC_URL_PERCENT2   0x08    /* waiting for 2nd digit after % */
#    define	 DCC_URL_SIMPLE (DCC_URL_DEL_DOMAIN	\
				 | DCC_URL_PERCENT1	\
				 | DCC_URL_PERCENT2)
    u_char	percent;
} DCC_URL_SKIP;
#define DCC_URL_MAX	    65		/* 63 is RFC 1034 limit on labels */
#define DCC_URL_FAILSAFE    2000	/* big bogus "username:password@" */


/* specify a DNS blacklist */
typedef char DNSBL_DOM[MAXHOSTNAMELEN];
typedef struct dnsbl {
    struct dnsbl *fwd;
    DCC_SOCKU	su;			/* usually 127.0.0.2 */
    enum DNSBL_TYPE {
	DNSBL_TYPE_IPV4,
	DNSBL_TYPE_IPV6,
	DNSBL_TYPE_NAME,
    } bl_type;
    int		dom_len;
    DNSBL_DOM	dom;
    u_char	flags;
#    define	 DNSBL_FG_ENVELOPE  0x01
#    define	 DNSBL_FG_BODY	    0x02
#    define	 DNSBL_FG_MX	    0x04
#    define	 DNSBL_FGS  (DNSBL_FG_ENVELOPE | DNSBL_FG_BODY | DNSBL_FG_MX)
} DNSBL;
extern DNSBL *dnsbls;

/* working storage for DNS blacklisting */
typedef enum {
    DNSBL_HIT_NONE,
    DNSBL_HIT_MTA,			/* MTA has already decided */
    DNSBL_HIT_CLIENT,			/* SMTP client */
    DNSBL_HIT_MAIL_HOST,		/* SMTP envelope mail_from domain */
    DNSBL_HIT_URL,			/* URL in body */
}  DNSBL_HIT;
typedef struct {
    DCC_CLNT_CTXT *ctxt;
    int		msg_secs;		/* total seconds remaining for msg */
    time_t	start, stop;
    const char *id;
    DNSBL_HIT	hit;
    u_char	timeouts;		/* # of timed out lookups */
    u_char	mx;			/* hit was on an MX server */
    u_short	dom_len;
    DNSBL_DOM	dom;			/* body URL or envelope sender domain */
    DNSBL_DOM	probe;			/* what was actually looked up */
    DNSBL_DOM	timeout_dom;		/* 1st domain timed out */
} DNSBL_WORK;


/* accumulated checksums for an SMTP message */
typedef struct {
    const char *hdr;			/* name if substitute checksum */
    DCC_TGTS	tgts;			/* server's answer or previous total */
    DCC_CK_TYPE	type;
    u_char	rpt;			/* 1=report to server */
    DCC_SUM	sum;
} DCC_GOT_SUM;

typedef union {
    u_int32_t	w32[4];
    u_char	b[16];
} DCC_FUZ2_WORD;
#define DCC_FUZ2_WORD_CLEAR(_w) ((_w)->w32[0]=0, (_w)->w32[1]=0,	\
				 (_w)->w32[2]=0, (_w)->w32[3]=0)

typedef struct {
    DCC_URL_SKIP url;
    u_int	total;			/* bytes */
    char	*cp;			/* end of data in buffer */
    char	*eol;			/* most recent eol */
#    define	 DCC_FUZ1_MAX_LINE  78
#    define	 DCC_HTTPS_STR    "https"
#    define	 DCC_HTTPS_LEN	(sizeof(DCC_HTTPS_STR)-1)
    char	buf[DCC_FUZ1_MAX_LINE*4
		    +DCC_HTTPS_LEN];
    MD5_CTX	md5;
} DCC_CTX_FUZ1;


typedef struct {			/* per-language counts */
    u_int	wsummed;		/* bytes of words summed */
    u_int	wtotal;			/* words summed */
    MD5_CTX	md5;
} FUZ2_LANG;

typedef struct {
    u_int	btotal;			/* total non-whitespace bytes seen */
    u_int	xsummed;		/* non-word bytes checksummed */
    u_int	wlen;			/* length of current word */
    u_int	cref_cnt;		/* fancy character length */
    u_int	tag_len;		/* HTML tag length */
    DCC_FUZ2_WORD w;
    DCC_FUZ2_WORD cref_w;
    DCC_FUZ2_WORD tag;
    int		urls;			/* # of URLs seen */
    char	*url_cp;		/* accumulated URLs */
#    define	 DCC_FUZ2_URL_MAX   (1+DCC_URL_MAX)
    char	url_buf[DCC_FUZ2_URL_MAX*2];
    DCC_URL_SKIP url;			/* state machine for skipping URLs */
    enum {
	DCC_CREF_ST_IDLE = 0,
	DCC_CREF_ST_START,		/* get HTML character reference */
	DCC_CREF_ST_NUM,		/*	hex or decimal */
	DCC_CREF_ST_DEC,
	DCC_CREF_ST_HEX,
	DCC_CREF_ST_NAME,
    } cref_st;
    enum {
	DCC_FUZ2_ST_WORD = 0,		/* gathering word */
	DCC_FUZ2_ST_START_TAG,		/* gathering start of HTML tag */
	DCC_FUZ2_ST_SKIP_TAG,		/* skipping HTML tag */
	DCC_FUZ2_ST_SKIP_COMMENT	/* skipping HTML comment */
    } st;
#    define FUZ2_LAN_NUM 3
    FUZ2_LANG	lang[FUZ2_LAN_NUM];
} DCC_CTX_FUZ2;

/* The maximum length of a checksumed HELO value and the continuation string
 * need to be the same for all users of the local white-/blacklist */
#define DCC_HELO_MAX	(MAXHOSTNAMELEN+8)
#define	DCC_HELO_CONT	"..."

/* do not checksum more than this much of a header line */
#define DCC_HDR_CK_MAX	1024

/* substitute or locally configured checksums */
#define DCC_MAX_SUB_CKS 6

#define DCC_WSUMS (DCC_DIM_CKS+DCC_MAX_SUB_CKS)	/* max whitelist checksums */
typedef struct {
    DCC_GOT_SUM	sums[DCC_WSUMS];
    struct in6_addr ip_addr;		/* SMTP client IP address */
    struct {				/*  quoted-printable state machine */
	int	    x;
	int	    y;
	u_char	    n;
	enum {
	    DCC_CK_QP_IDLE,		/* waiting for '=' */
	    DCC_CK_QP_EQ,		/* seen "=" */
	    DCC_CK_QP_1,		/* seen "=X" of "=XY" */
	    DCC_CK_QP_FAIL1,		/* have output '=' after bad =X */
	    DCC_CK_QP_FAIL2,		/* have output '=' after bad =XY */
	    DCC_CK_QP_FAIL3,		/* have output 'X' after bad =XY */
	} state;
    } qp;
    struct {				/* base64 state machine */
	u_int32_t   quantum;
	int	    quantum_cnt;
    } b64;
    enum {
	CK_MP_ST_PREAMBLE,		/* preamble before first boundary */
	CK_MP_ST_BND,			/* MIME boundary */
	CK_MP_ST_HDRS,			/* headers after a boundary */
	CK_MP_ST_TEXT,			/* body or entity */
	CK_MP_ST_EPILOGUE		/* text after last boundary */
    } mp_st;
    enum CK_MHDR_ST {			/* parse entity fields */
	CK_MHDR_ST_IDLE,
	CK_MHDR_ST_CE_CT,		/* matching "Content-T" */
	CK_MHDR_ST_CE,			/*	"ransfer-Encoding:" */
	CK_MHDR_ST_CE_WS,		/* white space after "encoding:" */
	CK_MHDR_ST_CT,			/*	"ype:" */
	CK_MHDR_ST_CT_WS,		/* white space after "type:" */
	CK_MHDR_ST_QP,			/*	"quoted-printable" */
	CK_MHDR_ST_B64,			/*	"base64" */
	CK_MHDR_ST_TEXT,		/*	"text" */
	CK_MHDR_ST_HTML,		/*	"/html" */
	CK_MHDR_ST_CSET_SKIP_PARAM,	/*	skip to ";" after "text" */
	CK_MHDR_ST_CSET_SPAN_WS,	/*	skip blanks before "charset" */
	CK_MHDR_ST_CSET,		/* match "charset=" */
	CK_MHDR_ST_CSET_ISO_8859,	/*	"ISO-8859-" */
	CK_MHDR_ST_CSET_ISO_X,		/* find the 8859 type number */
	CK_MHDR_ST_MULTIPART,		/* match "multipart" */
	CK_MHDR_ST_BND_SKIP_PARAM,	/* skip "/alternative;" or similar */
	CK_MHDR_ST_BND_SPAN_WS,		/* skip whitespace before "boundary" */
	CK_MHDR_ST_BND,			/* match "boundary=" */
	CK_MHDR_ST_BND_VALUE		/* collecting boundary */
    } mhdr_st;
    u_char	mhdr_pos;
    u_char	mime_nest;		/* # of nested MIME entities */
    u_char	mime_bnd_matches;	/* # of active boundary matches */
    DCC_CK_BND	mime_bnd[3];
    enum {
	DCC_CK_CT_TEXT = 0,
	DCC_CK_CT_HTML,
	DCC_CK_CT_BINARY
    } mime_ct;
    const u_char *mime_cset;
    enum {
	DCC_CK_CE_ASCII = 0,
	DCC_CK_CE_QP,
	DCC_CK_CE_B64
    } mime_ce;
    struct {
	u_int	    total;		/* non-whitespace text */
	u_char	    flen;		/* bytes of ">From" seen */
	MD5_CTX	    md5;
    } ctx_body;
    DCC_CTX_FUZ1 fuz1;
    DCC_CTX_FUZ2 fuz2;
    DNSBL_WORK	*dnsbl;
    u_char	flags;
#    define	 DCC_CKS_HAVE_SUM	0x01    /* have at least one checksum */
#    define	 DCC_CKS_MIME_BOL	0x02    /* header decoder is at BOL */
#    define	 DCC_CKS_MIME_QUOTED	0x04
} DCC_GOT_CKS;

typedef DCC_TGTS DCC_CKS_WTGTS[DCC_WSUMS];


#define DCC_WHITE_SUFFIX ".dccw"

typedef u_int DCC_WHITE_INX;		/* 1-based index into hash table */

typedef char DCC_WHITE_MAGIC[128];

typedef struct {
    DCC_TGTS	tgts;
    int		bits;
    struct in6_addr addr;
    struct in6_addr mask;
} DCC_WHITE_CIDR_ENTRY;

typedef struct {
    u_char	len;
    DCC_WHITE_CIDR_ENTRY e[64];
} DCC_WHITE_CIDR;

typedef struct {
    DCC_WHITE_INX fwd;
    DCC_TGTS	tgts;
    u_char	gen;
    u_char	pad[2];
    DCC_CK_TYPE type;
    DCC_SUM	sum;
} DCC_WHITE_ENTRY;

/* sscanf() a checksum */
#define DCC_CKSUM_HEX_PAT	"%8x %8x %8x %8x"

#ifdef DCC_WIN32
#define DCC_WHITE_TBL_BINS 521		/* should be prime */
#else
#define DCC_WHITE_TBL_BINS 8209		/* should be prime */
#endif
typedef struct {
    DCC_WHITE_MAGIC magic;
    struct {
	DCC_WHITE_INX entries;		/* # of entries in the file */
	u_int	flags;
#	 define  DCC_WHITE_FG_PER_USER	    0x0001  /* 1=no hostnames */
#	 define	 DCC_WHITE_FG_GREY_LOG_ON   0x0002
#	 define	 DCC_WHITE_FG_GREY_LOG_OFF  0x0004
#	 define	 DCC_WHITE_FG_GREY_ON	    0x0008
#	 define	 DCC_WHITE_FG_GREY_OFF	    0x0010
#	 define	 DCC_WHITE_FG_LOG_ALL	    0x0020
#	 define	 DCC_WHITE_FG_LOG_NORMAL    0x0040
#	 define	 DCC_WHITE_FG_DCC_ON	    0x0080
#	 define	 DCC_WHITE_FG_DCC_OFF	    0x0100
#	 define	 DCC_WHITE_FG_DNSBL_ON	    0x0200
#	 define	 DCC_WHITE_FG_DNSBL_OFF	    0x0400
	time_t	resolve;		/* when IP addresses last resolved */
#	 define	 DCC_WHITE_RESOLVE_BAD	    0   /* hash table broken */
#	 define	 DCC_WHITE_RESOLVE_NO_IP    1	/* never re-resolve */
	DCC_PATH white_incs[8];		/* included files */
	u_char	gen;			/* generation number */
	DCC_WHITE_CIDR cidr_cur;
	DCC_WHITE_CIDR cidr_new;
    } hdr;
    DCC_WHITE_INX bins[DCC_WHITE_TBL_BINS]; /* 1-based indeces or 0 for empty */
    DCC_WHITE_ENTRY tbl[1];
} DCC_WHITE_TBL;


/* everything there is to know about a currently active whitelist file */
typedef struct {
    DCC_WHITE_TBL *info;
    void    *mutex;			/* lock this structure */
    DCC_WHITE_INX free;
    u_int   info_entries;		/* # of entries mapped window */
    u_int   info_size;			/* bytes in mapped window */
    time_t  ht_broken;			/* when to check sick file */
    time_t  stat_secs;			/* when to check ASCII files */
    int	    ht_fd;			/* hash table file */
#ifdef DCC_WIN32
    HANDLE  ht_map;			/* WIN32 hash table map handle */
#endif
    FILE    *ascii_file;
    DCC_PATH ascii_nm;
    DCC_PATH ht_nm;
    u_int   info_flags;			/* lock-safe copy of info->hdr.flags */
    u_int   new_info_flags;		/*	while parsing */
    u_char  ht_locked;			/* 0=file unlocked, 1=shared, 2=write */
    u_char  flags;
#    define  DCC_WF_PER_USER	0x01	/* 1=hostnames not allowed */
#    define  DCC_WF_WLIST_CMD	0x02	/* 1=wlist command */
#    define  DCC_WF_RO		0x04	/* 1=read-only */
#    define  DCC_WF_INFO_DIRTY	0x08	/* 1=need msync() */
#    define  DCC_WF_HT_CHECKED	0x10	/* 1=file has good magic */
} DCC_WF;

extern DCC_WF cmn_wf;

typedef enum {				/* greylist result */
    DCC_GREY_ASK_FAIL,			/* greylist server or other failure */
    DCC_GREY_ASK_OFF,			/* client whitelist or blacklist */
    DCC_GREY_ASK_EMBARGO,
    DCC_GREY_ASK_EMBARGO_END,		/* first time server says ok */
    DCC_GREY_ASK_PASS,			/* greylist server says ok */
    DCC_GREY_ASK_WHITE,			/* greylist server says whitelisted */
} DCC_ASK_GREY_RESULT;

extern u_char grey_on;
extern u_char grey_query_only;


extern void dcc_wf_init(DCC_WF *, void *, u_char);
extern void dcc_wf_lock(DCC_WF *);
extern void dcc_wf_unlock(DCC_WF *);

extern void dcc_str2ck(DCC_SUM, const char *, u_int, const char *);
extern u_char dcc_get_cks(DCC_GOT_CKS *, DCC_CK_TYPES, const char *, u_char);
extern u_char dcc_ck_get_sub(DCC_GOT_CKS *, const char *, const char *);
extern u_char dcc_add_sub_hdr(DCC_EMSG, const char *);
extern void dcc_ck_ipv6(DCC_SUM, const struct in6_addr *);
extern void dcc_get_ipv6_ck(DCC_GOT_CKS *, const struct in6_addr *);
extern u_char dcc_get_str_ip_ck(DCC_GOT_CKS *, const char *, struct in6_addr *);
extern void dcc_print_cks(void *, const DCC_GOT_CKS *, DCC_CKS_WTGTS,
			  void (out)(void *, const void *, u_int));
extern void dcc_print_grey(void *, DCC_ASK_GREY_RESULT, u_int, u_char *,
			   const char *, const DCC_SUM, const DCC_SUM,
			   void (out)(void *, const void *, u_int));
extern u_char dcc_grey_spam(DCC_EMSG, DCC_CLNT_CTXT *,
			    const DCC_GOT_CKS *, const DCC_SUM);

extern void dcc_ck_body_init(DCC_GOT_CKS *);
extern void dcc_ck_mime_hdr(DCC_GOT_CKS *, const char *, const char *);
extern u_char parse_mime_hdr(DCC_GOT_CKS *, const char *, u_int, u_char);
extern void dcc_ck_body(DCC_GOT_CKS *, const void *, u_int);
extern void dcc_ck_body_fin(DCC_GOT_CKS *);

extern u_char dcc_get_white(DCC_EMSG, DCC_WHITE_INX);

typedef int (*DCC_PARSED_CK_FNC)(DCC_EMSG, DCC_WF *,
				const char *, int,  /* fnm and lineno */
				DCC_CK_TYPES,	/* type of checksum */
				DCC_SUM,    /* computed checksum */
				DCC_TGTS);  /* "OK2" etc */
typedef int (*DCC_PARSED_CIDR_CK_FNC)(DCC_EMSG, DCC_WHITE_TBL *,
				      const char *, int,    /* fnm and lineno */
				      int,
				      const struct in6_addr *,
				      const struct in6_addr *,
				      DCC_TGTS);
typedef u_char (*DCC_PARSED_CK_LOCK)(DCC_EMSG, DCC_WF*);
extern int dcc_parse_ck(DCC_EMSG, DCC_WF *wf, const char *, int,
			const char *, const char *, DCC_TGTS,
			DCC_PARSED_CK_FNC, DCC_PARSED_CIDR_CK_FNC,
			DCC_PARSED_CK_LOCK);
extern int dcc_parse_hex_ck(DCC_EMSG, DCC_WF *wf,
			    const char *, int, const char *,
			    const char *, DCC_TGTS, DCC_PARSED_CK_FNC);
extern u_char dcc_parse_whitefile(DCC_EMSG, DCC_WF *, const DCC_PATH, FILE *,
				  DCC_WHITE_TBL **,
				  DCC_PARSED_CK_FNC, DCC_PARSED_CIDR_CK_FNC,
				  DCC_PARSED_CK_LOCK);

u_char dcc_white_unlock(DCC_EMSG, DCC_WF *);
typedef enum {
    DCC_WHITE_ERROR,
    DCC_WHITE_HALF_LISTED,
    DCC_WHITE_LISTED,
    DCC_WHITE_UNLISTED,
    DCC_WHITE_BLACK
} DCC_WHITE_RESULT;
extern u_char dcc_white_rdy(DCC_EMSG, DCC_WF *, const char *,
			    u_char, DCC_WHITE_RESULT *);
#define DCC_WHITE_RDY_LOCK_NEED_MUTEX	0   /* lock both, keep only mutex */
#define DCC_WHITE_RDY_LOCK_NEED_BOTH	1   /* need both locks, and keep both */
#define DCC_WHITE_RDY_LOCK_KEEP_BOTH	2   /* already locked, and keep both */
extern DCC_WHITE_RESULT dcc_white_sum(DCC_EMSG, DCC_WF *,
				      DCC_CK_TYPES, const DCC_SUM, u_char);
extern DCC_WHITE_RESULT dcc_white_cks(DCC_EMSG, DCC_WF *, const DCC_GOT_CKS *,
				      DCC_CKS_WTGTS,  const char *,
				      u_char, u_char);

extern int dcc_ask(DCC_EMSG, DCC_CLNT_CTXT *, u_char,
		   DCC_HEADER_BUF *, DCC_GOT_CKS *, u_int *, DCC_TGTS);
extern u_char dcc_white_ask(DCC_EMSG, DCC_WF *, DCC_CLNT_CTXT *,
			    DCC_HEADER_BUF *, u_int *, const char *,
			    DCC_GOT_CKS *, DCC_CKS_WTGTS, DCC_TGTS);
extern void dcc_init_tholds(void);
extern void dcc_parse_honor(const char *);
extern u_char dcc_parse_tholds(char, const char *);
extern void dcc_honor_log_cnts(u_int *, const DCC_GOT_CKS *, DCC_TGTS);
extern u_char dcc_parse_client_grey(char *);
#   define  DCC_HONOR_SRVR_OK2	    0x0001  /* have honored DCC_TGTS_OK2 */
#   define  DCC_HONOR_SRVR_NOTSPAM  0x0002  /* known not spam by DCC */
#   define  DCC_HONOR_SRVR_ISSPAM   0x0004  /* spam by DCC server */
#   define  DCC_HONOR_MTA_NOTSPAM   0x0008  /* MTA says it is not spam */
#   define  DCC_HONOR_MTA_ISSPAM    0x0010  /* MTA says it is spam */
#   define  DCC_HONOR_LOCAL_ISSPAM  0x0020  /* blacklisted */
#   define  DCC_HONOR_LOCAL_NOTSPAM 0x0040  /* whitelisted */
#   define  DCC_HONOR_GREY_EMBARGO  0x0080  /* embargo this message */
#   define  DCC_HONOR_GREY_LOGIT    0x0100  /* greylist logging indicated */
#   define  DCC_HONOR_LOGIT	    0x0200  /* log message for all recipients */
#   define  DCC_HONOR_LOG_ONE	    0x0400  /* log for at least one */
extern DCC_ASK_GREY_RESULT dcc_ask_grey(DCC_EMSG, DCC_CLNT_CTXT *, DCC_OPS,
					DCC_SUM, DCC_SUM,
					const DCC_GOT_CKS *, const DCC_SUM,
					DCC_TGTS *, DCC_TGTS *, DCC_TGTS *);

extern u_char dcc_parse_dnsbl(DCC_EMSG, const char *, const char *);
extern void dcc_dnsbl_init(DCC_GOT_CKS *, DCC_CLNT_CTXT *, const char *, int);
extern void dcc_dnsbl_url(DNSBL_WORK *);
extern void dcc_mail_host_dnsbl(DNSBL_WORK *, const char *);
extern void dcc_sender_dnsbl(DNSBL_WORK *, const struct in6_addr *);
extern u_char dcc_dnsbl_lock_init(void);
extern void dcc_dnsbl_lock(void);
extern void dcc_dnsbl_signal(void);
extern void dcc_dnsbl_wait(time_t);
extern void dcc_dnsbl_unlock(void);
extern u_char dcc_dnsbl_log_print(const DNSBL_WORK *,
				  void(*)(void *, const char*, ...), void *);



#endif /* DCC_CK_H */
