/* Distributed Checksum Clearinghouse
 *
 * server
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.206 $Revision$
 */

#include "dccd_defs.h"
#include <signal.h>			/* for Linux and SunOS */
#include <sys/uio.h>
#include <sys/wait.h>
#include "dcc_ifaddrs.h"

DCC_EMSG dcc_emsg;

static const char *homedir;
static char *aargs[10];
static char **aarg = &aargs[0];
static DCC_PATH dbclean_def = DCC_LIBEXECDIR"/dbclean";
static char *dbclean = dbclean_def;
static pid_t dbclean_pid = -1;
static char *addr_str;			/* an IP address for dbclean */
static char dbclean_str[] = "dbclean";
static char dbclean_mode[8];
static const char *dbclean_argv[20] = {dbclean_str, dbclean_mode};
static int dbclean_argc = 2;
static char dbclean_arg_str[200];
static int dbclean_arg_len;
time_t dbclean_limit_secs = DBCLEAN_LIMIT_SECS;

u_long dccd_tracemask = DCC_TRACE_ON_DEF_BITS;

u_char background = 1;
int stopint;				/* !=0 if stopping or received signal */

static time_t flods_ck_secs = FLODS_CK_SECS;

const char *brand = "";

static char *my_srvr_id_str;
DCC_SRVR_ID my_srvr_id;

u_char use_ipv6 = 0;
u_int16_t def_port;			/* default port #, network byte order */
typedef struct port_list {
    struct port_list *fwd;
    u_int16_t	port;			/* network byte order */
} PORT_LIST;
static PORT_LIST *ports;
SRVR_SOC *srvr_socs;
static SRVR_SOC **srvr_socs_end = &srvr_socs;
int srvr_rcvbuf = 1024*1024;

static u_char db_mode;

DB_RCD host_id_rcd;			/* our host's globally unique ID */
time_t host_id_next, host_id_last;

u_char anon_off;			/* turn off anonymous access */
time_t anon_delay_us = DCC_ANON_DELAY_US_DEF;   /* delay anonymous clients */
u_int anon_delay_inflate = DCC_ANON_INFLATE_OFF;

static QUEUE *queue_free;
static QUEUE *queue_anon_head;

/* assume or hope we can handle 200 requests/second */
int queue_max = 200*DCC_MAX_RTT_SECS;	/* ultimate bound on queue size */
static int queue_max_cur;		/* current upper bound */
static int queue_anon_cur;		/* current queue size */

struct timeval wake_time;		/* when we awoke from select() */
DCC_TS future_ts;			/* the distant future */

const char *need_del_dbclean;
time_t del_dbclean_next;
time_t dbclean_limit;
static time_t dbclean_cron_sub;

static DB_NOKEEP_CKS set_new_nokeep_cks, reset_new_nokeep_cks;

DB_FLOD_THOLDS flod_tholds;		/* flood at or after these thresholds */

static void parse_thold(const char *);
static const char *parse_rl_rate(RL_RATE *, float, const char *, const char *);
static void add_dbclean_arg(const char *);
static void ck_dbclean(int);
static void run_dbclean(const char *, const char *);
static void sigterm(int);
static void sighup(int);
static void stop_children(void);
static u_char get_if_changes(u_char);
static SRVR_SOC *add_srvr_soc(u_char, int, const void *, u_int16_t);
static int open_srvr_socs(int);
static void recv_job(void) NRATTRIB;
static u_char new_job(SRVR_SOC *);
static void NRATTRIB db_quit(int, const char *, ...) PATTRIB(2,3);

static void
usage(const char* barg)
{
	static const char str[] = {
	    "usage: [-64dVbfFQ] -i server-ID [-n brand] [-h homedir]\n"
	    "   [-I host-ID[,user]] [-a [server-addr][,server-port]]"
	    " [-q qsize]\n"
	    "   [-G [on,][weak-body,][weak-IP,]embargo],[wait],[white]]\n"
	    "   [-t type,threshhold] [-K [no-]type] [-T tracemode]\n"
	    "   [-u anon-delay[,inflate] [-C dbclean]"
	    " [-L ltype,facility.level]\n"
	    "   [-R [RL_SUB],[RL_FREE],[RL_ALL_FREE],[RL_BUGS]]"
	};
	static u_char complained;

	if (!complained) {
		if (barg)
			dcc_error_msg("unrecognized \"%s\"\n%s\n... continuing",
				      barg, str);
		else
			dcc_error_msg("%s\n... continuing", str);
		complained = 1;
	}
}


int NRATTRIB
main(int argc, char **argv)
{
	char *p;
	const char *rest;
	u_char print_version = 0;
	DCC_CK_TYPES type;
	char hostname[MAXHOSTNAMELEN];
	DCC_SOCKU *sup;
	u_int16_t port;
	int new_embargo, new_window, new_white;
	int error, i;
	u_long l;

	dcc_syslog_init(1, argv[0], 0);

	if (DCC_DIM_CKS != DCC_COMP_DIM_CKS)
		dcc_logbad(EX_SOFTWARE,
			   "DCC_DIM_CKS != DCC_COMP_DIM_CKS;"
			   " check uses of both");

	/* get first bytes of hostname to name our server-ID */
	memset(hostname, 0, sizeof(hostname));
	if (0 > gethostname(hostname, sizeof(hostname)-1))
		dcc_logbad(EX_NOHOST, "gethostname(): %s", ERROR_STR());
	if (hostname[0] == '\0')
		dcc_logbad(EX_NOHOST, "null hostname from gethostname()");
	memcpy(host_id_rcd.cks[0].sum, hostname,
	       sizeof(host_id_rcd.cks[0].sum));

	parse_rl_rate(&rl_sub_rate, 0.5, "RL_SUB", "400");
	parse_rl_rate(&rl_anon_rate, RL_AVG_SECS, "RL_ANON", "50");
	parse_rl_rate(&rl_all_anon_rate, 0.5, "RL_ALL_ANON", "300");
	parse_rl_rate(&rl_bugs_rate, RL_AVG_SECS, "RL_BUGS", "0.1");

	/* this must match DCCD_GETOPTS in cron-dccd.in */
	while ((i = getopt(argc, argv,
			   "64dVbfFQi:n:h:a:I:q:G:t:K:T:u:C:L:R:")) != EOF) {
		switch (i) {
		case '6':
#ifndef NO_IPV6
			use_ipv6 = 2;
#endif
			break;
		case '4':
			use_ipv6 = 0;
			break;

		case 'd':
			++db_debug;
			break;

		case 'V':
			fprintf(stderr, DCC_VERSION"\n");
			print_version = 1;
			break;

		case 'b':
			background = 0;
			break;

		case 'f':
			db_mode &= ~DB_OPEN_NO_MMAP;
			break;

		case 'F':
			db_mode |= DB_OPEN_NO_MMAP;
			break;

		case 'Q':
			query_only = 1;
			break;

		case 'i':
			my_srvr_id_str = optarg;
			if (!dcc_get_srvr_id(dcc_emsg, &my_srvr_id,
					     my_srvr_id_str, 0, 0, 0))
				dcc_logbad(dcc_ex_code, "%s", dcc_emsg);
			add_dbclean_arg("-i");
			add_dbclean_arg(my_srvr_id_str);
			break;

		case 'n':
			if (*optarg == '\0'
			    || strlen(optarg) > sizeof(DCC_BRAND)
			    || strpbrk(optarg, ":"DCC_WHITESPACE)) {
				dcc_error_msg("invalid brand name \"-n %s\"",
					      optarg);
			} else {
				brand = optarg;
			}
			break;

		case 'h':
			homedir = optarg;
			/* tell dbclean "-h ." because we will already
			 * be there and that allows our -h to be relative */
			add_dbclean_arg("-h.");
			break;

		case 'I':
			p = strchr(optarg, ',');
			if (p) {
				*p++ = '\0';
				dcc_daemon_su(p);
				if (*optarg == '\0')
					break;
			}
			strncpy((char *)host_id_rcd.cks[0].sum, optarg,
				sizeof(host_id_rcd.cks[0].sum));
			break;

		case 'a':
			/* postpone checking host names until we know -6 */
			if (aarg > LAST(aargs)) {
				dcc_error_msg("too many -a args");
				break;
			}
			optarg += strspn(optarg, DCC_WHITESPACE);
			*aarg++ = optarg;
			break;

		case 'q':
			l = strtoul(optarg, &p, 0);
			if (*p != '\0' || l < 2 || l > 1000) {
				dcc_error_msg("invalid queue length \"%s\"",
					      optarg);
			} else {
				queue_max = l;
			}
			break;

		case 'G':
			grey_on = 1;
			dcc_syslog_init(1, argv[0], " grey");
			add_dbclean_arg("-Gon");
			/* handle leading "on" "weak-body", and "weak-IP" */
			while (*optarg) {
				if (dcc_ck_word_comma(&optarg, "weak-body")
				    || dcc_ck_word_comma(&optarg, "weak_body")
				    || dcc_ck_word_comma(&optarg, "weak")) {
					grey_weak_body = 1;
					continue;
				}
				if (dcc_ck_word_comma(&optarg, "weak-IP")
				    || dcc_ck_word_comma(&optarg, "weak_IP")) {
					grey_weak_ip = 1;
					continue;
				}
				if (!dcc_ck_word_comma(&optarg, "on"))
					break;
			}
			new_embargo = dcc_get_secs(optarg, &p,
						   0, MAX_GREY_EMBARGO,
						   grey_embargo);
			if (new_embargo < 0) {
				dcc_error_msg("invalid greylist embargo \"%s\"",
					      optarg);
				break;
			}
			new_window = dcc_get_secs(p, &p,
						  new_embargo, 10*24*60*60,
						  max(new_embargo,grey_window));
			if (new_window < 0) {
				dcc_error_msg("invalid greylist wait time"
					      " \"%s\"",
					      optarg);
				break;
			}
			new_white = dcc_get_secs(p, 0,
						 new_window, DB_EXPIRE_SECS_MAX,
						 max(new_window, grey_white));
			if (new_white < 0) {
				dcc_error_msg("invalid greylist whitelist time"
					      " \"%s\"",
					      optarg);
				break;
			}
			grey_embargo = new_embargo;
			grey_window = new_window;
			grey_white = new_white;
			break;

		case 't':
			parse_thold(optarg);
			break;

		case 'K':
			if (!strcasecmp(optarg, "all")) {
				reset_new_nokeep_cks = -1;
				break;
			}
			if (!CSTRCMP(optarg, "no-")) {
				optarg += STRZ("no-");
				i = 0;
			} else {
				i = 1;
			}
			type = dcc_str2type(optarg);
			if (type == DCC_CK_INVALID) {
				dcc_error_msg("bad checksum type in"
					      " \"-K %s\"", optarg);
				break;
			}
			if (i)
				DB_SET_NOKEEP(reset_new_nokeep_cks, type);
			else
				DB_SET_NOKEEP(set_new_nokeep_cks, type);
			break;

		case 'T':
			if (!strcasecmp(optarg, "ADMN")) {
				dccd_tracemask |= DCC_TRACE_ADMN_BIT;
			} else if (!strcasecmp(optarg, "ANON")) {
				dccd_tracemask |= DCC_TRACE_ANON_BIT;
			} else if (!strcasecmp(optarg, "CLNT")) {
				dccd_tracemask |= DCC_TRACE_CLNT_BIT;
			} else if (!strcasecmp(optarg, "RLIM")) {
				dccd_tracemask |= DCC_TRACE_RLIM_BIT;
			} else if (!strcasecmp(optarg, "QUERY")) {
				dccd_tracemask |= DCC_TRACE_QUERY_BIT;
			} else if (!strcasecmp(optarg, "RIDC")) {
				dccd_tracemask |= DCC_TRACE_RIDC_BIT;
			} else if (!strcasecmp(optarg, "FLOOD")) {
				dccd_tracemask |= DCC_TRACE_FLOD_BIT;
			} else if (!strcasecmp(optarg, "FLOOD2")) {
				dccd_tracemask |= DCC_TRACE_FLOD2_BIT;
			} else if (!strcasecmp(optarg, "IDS")) {
				dccd_tracemask |= DCC_TRACE_IDS_BIT;
			} else if (!strcasecmp(optarg, "BL")) {
				dccd_tracemask |= DCC_TRACE_BL_BIT;
			} else {
				dcc_error_msg("invalid trace mode \"%s\"",
					      optarg);
			}
			break;

		case 'u':
			if (!strcasecmp(optarg, "forever")) {
				anon_off = 1;
				break;
			}
			if (!parse_dccd_delay(dcc_emsg, &anon_delay_us,
					      &anon_delay_inflate, optarg,
					      0, 0)) {
				dcc_error_msg("%s", dcc_emsg);
				break;
			}
			anon_off = 0;
			break;

		case 'C':
			if (*optarg == '\0') {
				dcc_error_msg("no path to dbclean \"-C %s\"",
					      optarg);
				break;
			}
			dbclean = optarg;
			for (p = strpbrk(optarg, DCC_WHITESPACE);
			     p != 0;
			     p = strpbrk(p, DCC_WHITESPACE)) {
				*p++ = '\0';
				p += strspn(p, DCC_WHITESPACE);
				if (*p == '\0')
					break;
				add_dbclean_arg(p);
			}
			break;

		case 'L':
			if (dcc_parse_log_opt(optarg)) {
				add_dbclean_arg("-L");
				add_dbclean_arg(optarg);
			}
			break;

		case 'R':
			rest = parse_rl_rate(&rl_sub_rate, -1,
					     "RL_SUB", optarg);
			rest = parse_rl_rate(&rl_anon_rate, -1,
					     "RL_ANON", rest);
			rest = parse_rl_rate(&rl_all_anon_rate, -1,
					     "RL_ALL_ANON", rest);
			rest = parse_rl_rate(&rl_bugs_rate, -1,
					     "RL_BUGS", rest);
			break;

		default:
			usage(argv[optind-1]);
		}
	}
	argc -= optind;
	argv += optind;
	if (argc != 0)
		usage(argv[0]);

	if (my_srvr_id == DCC_ID_INVALID) {
		if (print_version)
			exit(EX_OK);
		dcc_logbad(EX_USAGE, "server-ID not set with -i");
	}

	if (db_mode & DB_OPEN_NO_MMAP)
		add_dbclean_arg("-F");

	if (grey_on) {
		anon_off = 1;
		dccd_tracemask |= DCC_TRACE_IDS_BIT;
		flods_ck_secs = GREY_FLODS_CK_SECS;
	}

	/* parse addresses after we know whether -6 was among the args */
	for (aarg = &aargs[0]; aarg <= LAST(aargs) && *aarg; ++aarg) {
		addr_str = *aarg;
		rest = dcc_parse_nm_port(dcc_emsg, *aarg, 0,
					 hostname, sizeof(hostname),
					 &port, 0, 0, 0, 0);
		if (!rest) {
			dcc_error_msg("%s", dcc_emsg);
			continue;
		}
		rest += strspn(rest, DCC_WHITESPACE);
		if (*rest != '\0')
			dcc_error_msg("unrecognized port number in"
				      "\"-a %s\"", *aarg);
		if (hostname[0] == '\0') {
			PORT_LIST *pport = dcc_malloc(sizeof(*pport));
			pport->port = port;
			pport->fwd = ports;
			ports = pport;
			continue;
		}
		if (!strcmp(hostname, "@")) {
			++addr_str;	/* "" but not a const */
			add_srvr_soc(SRVR_SOC_ADDR, AF_UNSPEC, 0, port);
			continue;
		}
		dcc_host_lock();
		if (!dcc_get_host(hostname, use_ipv6 ? 2 : 0, &error)) {
			dcc_host_unlock();
			dcc_error_msg("%s: %s",
				      *aarg, DCC_HSTRERROR(error));
			continue;
		}
		for (sup = dcc_hostaddrs; sup < dcc_hostaddrs_end; ++sup) {
			if (sup->sa.sa_family == AF_INET)
				add_srvr_soc(SRVR_SOC_ADDR, AF_INET,
					     &sup->ipv4.sin_addr, port);
			else
				add_srvr_soc(SRVR_SOC_ADDR, AF_INET6,
					     &sup->ipv6.sin6_addr, port);
		}
		dcc_host_unlock();
	}
	if (addr_str) {
		/* tell dbclean about one "-a addr" */
		add_dbclean_arg("-a");
		add_dbclean_arg(addr_str);
	}

	dcc_clnt_unthread_init();

	if (!dcc_cdhome(dcc_emsg, homedir))
		dcc_logbad(dcc_ex_code, "%s", dcc_emsg);

	i = load_ids(dcc_emsg, 0, my_srvr_id);
	if (i < 0)
		dcc_logbad(dcc_ex_code, "%s", dcc_emsg);
	else if (!i)
		dcc_error_msg("%s", dcc_emsg);

	if (!def_port)
		def_port = DCC_GREY2PORT(grey_on);
	if (!srvr_socs && !ports) {
		ports = dcc_malloc(sizeof(*ports));
		ports->fwd = 0;
		ports->port = def_port;
	}
	get_if_changes(db_debug != 0);

	/* make initial attempt to open the server UDP sockets
	 * This also sets use_ipv6 to 0 or 1 if it is still 2 */
	if (open_srvr_socs(45) <= 0)
		dcc_logbad(EX_OSERR, "failed to open any server sockets");

	add_dbclean_arg(use_ipv6 == 0 ? "-4" : "-6");

	if (background) {
		if (0 > daemon(1, 0))
			dcc_logbad(EX_OSERR, "daemon(): %s", ERROR_STR());
	}

	if (!background)
		signal(SIGHUP, sigterm);    /* SIGHUP fatal during debugging */
	else
		signal(SIGHUP, sighup);	/* speed check of configuration */
	signal(SIGTERM, sigterm);
	signal(SIGINT, sigterm);
	signal(SIGPIPE, SIG_IGN);

	atexit(stop_children);

	gettimeofday(&db_time, 0);
	wake_time = db_time;

	flod_mmap_path_set();

	/* open the database, and try once to fix it */
	if (!dccd_db_open(DB_OPEN_LOCK_NOWAIT)) {
		dcc_error_msg("%s", dcc_emsg);
		run_dbclean("RSP", "database initially broken");
		ck_dbclean(0);		/* stall until it ends */
		if (!dccd_db_open(DB_OPEN_LOCK_NOWAIT)) {
			dcc_error_msg("%s", dcc_emsg);
			dcc_logbad(EX_NOINPUT,
				   "could not start database %s", db_nm);
		}
	}

	boot_ok_time = db_time.tv_sec + LAST_ERROR_ACT_SECS;
	host_id_next = db_time.tv_sec + LAST_ERROR_ACT_SECS;
	flods_init();
	stats_clear();
	check_blacklist_file();
	if (flod_mmaps != 0
	    && flod_mmaps->dccd_stats.reset.tv_sec != 0) {
		memcpy(&dccd_stats, &flod_mmaps->dccd_stats,
		       sizeof(dccd_stats));
	}
	dcc_trace_msg(DCC_VERSION" listening to port %d with %s and %s",
		      ntohs(srvr_socs->arg_port), dcc_homedir, db_window_size);

	recv_job();
}



static SRVR_SOC *			/* 0 or new entry */
add_srvr_soc(u_char flags,
	     int family,		/* AF_UNSPEC or 0 if addrp==0 */
	     const void *addrp,		/* 0, *in_addr, or *in6_addr */
	     u_int16_t port)
{
	DCC_SOCKU su;
	SRVR_SOC *sp;

	dcc_mk_su(&su, family, addrp, port);
	for (sp = srvr_socs; sp; sp = sp->fwd) {
		if (sp->arg_family == family
		    && sp->arg_port == port
		    && !memcmp(&sp->arg_addr, addrp,
			       ((family == AF_INET)
				? sizeof(sp->arg_addr.in4)
				: sizeof(sp->arg_addr.in6))))
			return sp;
	}

	sp = dcc_malloc(sizeof(*sp));
	memset(sp, 0, sizeof(*sp));
	sp->flags = flags;
	sp->udp = -1;
	sp->listen = -1;
	sp->su = su;
	sp->arg_family = family;
	if (addrp)
		memcpy(&sp->arg_addr, addrp,
		       ((family == AF_INET)
			? sizeof(sp->arg_addr.in4)
			: sizeof(sp->arg_addr.in6)));
	sp->arg_port = port;

	*srvr_socs_end = sp;
	srvr_socs_end = &sp->fwd;

	return sp;
}



static int				/* # of sockets opened */
open_srvr_socs(int retry_secs)
{
	static u_char srvr_rcvbuf_set = 0;
	int *retry_secsp;
	u_char family;
	SRVR_SOC *sp;
	int num_socs = 0;

	if (stopint)
		return -1;

	retry_secsp = retry_secs ? &retry_secs : 0;

	for (sp = srvr_socs; sp; sp = sp->fwd) {
		if (sp->udp >= 0) {
			++num_socs;
			continue;
		}

		/* resolve default port number if we finally know it */
		if (!sp->arg_port)
			sp->arg_port = def_port;

		family = sp->arg_family;

		/* create the UDP socket
		 * If we are using INADDR_ANY
		 * and do not yet know if IPv6 works, just try it */
		if (family == AF_UNSPEC
		    && use_ipv6 == 2) {
			dcc_mk_su(&sp->su, AF_INET6,
				  &sp->arg_addr, sp->arg_port);
			if (!dcc_udp_bind(dcc_emsg, &sp->udp,
					  &sp->su, retry_secsp)) {
				dcc_error_msg("%s", dcc_emsg);
				/* still don't know about IPv6 */
				continue;
			}
			if (sp->udp >= 0) {
				/* we finished an INADDR_ANY socket
				 * and learned that IPv6 works */
				use_ipv6 = 1;
				continue;
			} else {
				use_ipv6 = 0;
			}
		}

		if (family == AF_UNSPEC) {
			/* using INADDR_ANY but know whether IPv6 works */
			family = use_ipv6 ? AF_INET6 : AF_INET;
		} else if (use_ipv6 == 2
			   && family == AF_INET6) {
			/* don't know if IPv6 works but have an IPv6 address */
			use_ipv6 = 1;
		}

		dcc_mk_su(&sp->su, family, &sp->arg_addr, sp->arg_port);
		if (!dcc_udp_bind(dcc_emsg, &sp->udp,
				  &sp->su, retry_secsp)
		    || sp->udp < 0) {
			dcc_error_msg("%s", dcc_emsg);
			continue;
		}

		/* set socket receive buffer size as large as possible */
		for (;;) {
			if (!setsockopt(sp->udp, SOL_SOCKET, SO_RCVBUF,
					&srvr_rcvbuf, sizeof(srvr_rcvbuf)))
				break;
			if (srvr_rcvbuf_set || srvr_rcvbuf <= 4096) {
				dcc_error_msg("setsockopt(%s,SO_RCVBUF=%d): %s",
					      dcc_su2str_err(&sp->su),
					      srvr_rcvbuf,
					      ERROR_STR());
				break;
			}
			srvr_rcvbuf -= 4096;
		}
		srvr_rcvbuf_set = 1;

		++num_socs;
	}

	/* Finally decide the IPv6 issue if we found no sign of IPv6 */
	if (use_ipv6 == 2)
		use_ipv6 = 0;

	return num_socs;
}



/* get ready to bind to all local IP addreses */
static u_char				/* 1=added an interface */
add_ifs(u_char not_quiet)
{
	u_char added;
	SRVR_SOC *sp;
	PORT_LIST *pport;
#ifdef HAVE_GETIFADDRS
	struct ifaddrs *ifap0, *ifap;
	int num_ifs;
#endif

	if (!ports)
		return 0;

	added = 0;

#ifdef HAVE_GETIFADDRS
	if (0 > getifaddrs(&ifap0)) {
		dcc_error_msg("getifaddrs(): %s", ERROR_STR());
		ifap0 = 0;
	}

	num_ifs = 0;
	for (pport = ports; pport; pport = pport->fwd) {
		const SRVR_SOC *listener = 0;

		for (ifap = ifap0; ifap; ifap = ifap->ifa_next) {
			if (!(ifap->ifa_flags & IFF_UP))
				continue;
			if (!ifap->ifa_addr)
				continue;
			switch (ifap->ifa_addr->sa_family) {
			case AF_INET:
				++num_ifs;
				sp = add_srvr_soc(SRVR_SOC_IF | SRVR_SOC_NEW,
						  ifap->ifa_addr->sa_family,
						  &((struct sockaddr_in *)ifap
						    ->ifa_addr)->sin_addr,
						  pport->port);
				break;
			case AF_INET6:
				if (use_ipv6 == 0)
					continue;
				if (use_ipv6 == 2)
					use_ipv6 = 1;
				++num_ifs;
				sp = add_srvr_soc(SRVR_SOC_IF | SRVR_SOC_NEW,
						  ifap->ifa_addr->sa_family,
						  &((struct sockaddr_in6*)ifap
						    ->ifa_addr)->sin6_addr,
						  pport->port);
				break;
			default:
				continue;
			}
			if (sp->flags & SRVR_SOC_NEW) {
				added = 1;
				if (not_quiet)
					dcc_trace_msg("start listening on %s",
						      dcc_su2str_err(&sp->su));
			}
			sp->flags &= ~(SRVR_SOC_MARK | SRVR_SOC_NEW);

			/* interfaces can have duplicate addresses */
			if (listener == sp)
				continue;
			if (!listener) {
				listener = sp;
				if (!(sp->flags & SRVR_SOC_LISTEN)) {
					sp->flags |= SRVR_SOC_LISTEN;
					added = 1;
				}
			} else {
				if (sp->flags & SRVR_SOC_LISTEN) {
					sp->flags &= ~SRVR_SOC_LISTEN;
					added = 1;
				}
			}
		}
	}
#ifdef HAVE_FREEIFADDRS
	/* since this is done only a few times when HAVE_FREEIFADDRS is not
	 * defined, don't worry if we cannot release the list of interfaces */
	freeifaddrs(ifap0);
#endif

	if (num_ifs > 0)
		return added;
#endif /* HAVE_GETIFADDRS */

	/* if we got no joy from getifaddrs(), use INADDR_ANY */
	for (pport = ports; pport; pport = pport->fwd) {
		sp = add_srvr_soc(SRVR_SOC_IF | SRVR_SOC_LISTEN | SRVR_SOC_NEW,
				  AF_UNSPEC, 0, pport->port);
		if (sp->flags & SRVR_SOC_NEW) {
			added = 1;
			if (not_quiet)
				dcc_trace_msg("fallback listen on %s",
					      dcc_su2str_err(&sp->su));
		}
		sp->flags &= ~(SRVR_SOC_MARK | SRVR_SOC_NEW);
	}

	return added;
}



/* deal with changes to network interfaces */
static u_char				/* 1=something changed */
get_if_changes(u_char not_quiet)
{
	SRVR_SOC *sp, **spp;
	u_char changed;

	for (sp = srvr_socs; sp; sp = sp->fwd) {
		if (sp->flags & SRVR_SOC_IF)
			sp->flags |= SRVR_SOC_MARK;
	}

	changed = add_ifs(not_quiet);

	spp = &srvr_socs;
	while ((sp = *spp) != 0) {
		/* an interface recognized by add_srvr_soc() will have
		 * its SRVR_SOC_MARK cleared */
		if (!(sp->flags & SRVR_SOC_MARK)) {
			spp = &sp->fwd;
			continue;
		}

		/* forget interfaces that have disappeared */
		dcc_trace_msg("stop listening on %s", dcc_su2str_err(&sp->su));
		changed = 1;
		if (srvr_socs_end == &sp->fwd)
			srvr_socs_end = spp;
		*spp = sp->fwd;
		if (sp->udp >= 0)
			close(sp->udp);
		if (sp->listen >= 0)
			close(sp->listen);
		dcc_free(sp);
	}

	return changed;
}



static void
parse_thold(const char *arg)
{
	DCC_TGTS tgts;
	char *duparg, *cnt;
	DCC_CK_TYPES type;

	arg += strspn(arg, DCC_WHITESPACE);
	duparg = dcc_strdup(arg);

	cnt = strchr(duparg, ',');
	if (!cnt) {
		dcc_error_msg("missing comma in \"-t %s\"",
			      arg);
		dcc_free(duparg);
		return;
	}
	*cnt++ = '\0';
	tgts = dcc_str2cnt(cnt);
	if (tgts == 0 || tgts > DCC_TGTS_TOO_MANY) {
		dcc_error_msg("unrecognized count in \"-t %s\"",
			      arg);
		dcc_free(duparg);
		return;
	}

	if (*duparg == '\0') {
		/* type not specified, so assume body checksums */
		for (type = 0; type < DIM(flod_tholds); type++) {
			if (DCC_CK_IS_BODY(type))
				flod_tholds[type] = tgts;
		}

	} else {
		/* type specified */
		type = dcc_str2type(duparg);
		if (type > DCC_CK_INVALID
		    && type <= DCC_CK_FUZ2) {
			flod_tholds[type] = tgts;
		} else {
			dcc_error_msg("bad checksum type in \"-t %s\"", arg);
		}
	}

	dcc_free(duparg);
}



static const char *
parse_rl_rate(RL_RATE *rate, float penalty_secs,
	      const char *label, const char *arg)
{
	char *p;
	int sec, hi;

	if (penalty_secs >= 0)
		rate->penalty_secs = penalty_secs;

	if (*arg == '\0')
		return arg;

	if (*arg == ',')
		return ++arg;

	sec = strtod(arg, &p) * RL_SCALE;
	hi = sec*RL_AVG_SECS;
	if ((*p != '\0' && *p != ',')
	    || hi < RL_SCALE || sec > RL_MAX_CREDITS) {
		dcc_error_msg("invalid %s value in \"%s\"",
			      label, arg);
		return "";
	}

	rate->sec = sec;
	rate->hi = hi;
	rate->lo = -sec * rate->penalty_secs;

	return (*p == ',') ? p+1 : p;
}



static void
add_dbclean_arg(const char *arg)
{
	int i;

	if (dbclean_argc >= DIM(dbclean_argv)-2)
		dcc_logbad(EX_SOFTWARE, "too many args for dbclean");
	dbclean_argv[dbclean_argc++] = arg;
	i = snprintf(dbclean_arg_str+dbclean_arg_len,
		     sizeof(dbclean_arg_str)-dbclean_arg_len,
		     " %s", arg);
	dbclean_arg_len += i;
	if (dbclean_arg_len >= ISZ(dbclean_arg_str)-2)
		dcc_logbad(EX_SOFTWARE, "too many args for dbclean");
}



/* check effort to repair database */
static void
ck_dbclean(int options)
{
	int status;
	pid_t pid;

	if (dbclean_pid < 0)
		return;

	pid = waitpid(dbclean_pid, &status, options);
	if (pid != dbclean_pid)
		return;

	dbclean_pid = -1;

	/* do not try failing dbclean too often */
#ifdef WEXITSTATUS
	status = WEXITSTATUS(status);
#else
	status &= 0xff;
#endif
	if (status == 0) {
		dbclean_limit_secs = DBCLEAN_LIMIT_SECS;
	} else {
		dbclean_limit_secs *= 2;
		if (dbclean_limit_secs > DEL_DBCLEAN_SECS)
			dbclean_limit_secs = DEL_DBCLEAN_SECS;
	}

	/* don't restart dbclean until after it has stopped running
	 * and cooled for a while */
	dbclean_limit = db_time.tv_sec + dbclean_limit_secs;
}



/* try to repair the database */
static void
run_dbclean(const char *mode,		/* combination of S, P, and R */
	    const char *reason)
{
	ck_dbclean(0);			/* wait until previous ends */

	strcpy(dbclean_mode, "-Dq");
	if (mode)
		strncat(dbclean_mode, mode, sizeof(dbclean_mode)-sizeof("-Dq"));

	dbclean_pid = fork();
	if (dbclean_pid < 0) {
		dcc_error_msg("dbclean fork(): %s", ERROR_STR());
	} else if (dbclean_pid == 0) {
		dcc_trace_msg("%s; starting `%s %s`",
			      reason, dbclean, dbclean_arg_str);
		execv(dbclean, (char **)dbclean_argv);
		dcc_error_msg("execv(%s %s): %s",
			      dbclean, dbclean_arg_str, ERROR_STR());
		exit(-1);
	}

	need_del_dbclean = 0;
	dbclean_limit = db_time.tv_sec + dbclean_limit_secs;
}



/* check for changes or other interesting events when the flood timer expires */
static time_t				/* seconds to wait */
ck_changes(void)
{
	static time_t misc_timer;
	time_t secs;
	DB_HADDR hash_free;
	const char *mode;
	const char *reason;
	char reason_buf[100];

	/* check nothing if it is not yet time */
	secs = next_flods_ck - db_time.tv_sec;
	if (secs > 0 && secs <= flods_ck_secs)
		return secs;
	next_flods_ck = db_time.tv_sec + flods_ck_secs;

	/* do not make some checks too often */
	if (DB_IS_TIME(misc_timer, MISC_CK_SECS)) {
		db_sync_some(MISC_CK_SECS);

		switch (check_load_ids(dcc_emsg, my_srvr_id)) {
		case 0:
			dcc_error_msg("check/reload: %s", dcc_emsg);
			break;
		case 1:
			dcc_trace_msg("reloaded %s", ids_path);
			flods_restart("restart flooding with new IDs");
			break;
		}

#ifdef HAVE_GETIFADDRS
		/* frequently check for network interface changes, but only
		 * if we know how to release the result of getifaddrs() */
		if (get_if_changes(1)) {
			int socs = open_srvr_socs(0);
			if (!socs)
				bad_stop("failed to open any server sockets");
			else if (socs > 0)
				flods_restart("network interfaces changed");
		}
#endif

		cycle_q_delay();

		check_blacklist_file();

		/* sound a claim to our server-ID if the database is locked */
		if (DB_IS_TIME(host_id_next, DCC_SRVR_ID_SEC)
		    && DB_IS_LOCKED()) {
			host_id_last = db_time.tv_sec;
			host_id_next = host_id_last + DCC_SRVR_ID_SEC;
			dcc_timeval2ts(host_id_rcd.ts, &db_time, 0);

			host_id_rcd.srvr_id_auth = my_srvr_id;
			DB_TGTS_RCD_SET(&host_id_rcd, 1);
			host_id_rcd.fgs_num_cks = 1;
			host_id_rcd.cks[0].type_fgs = DCC_CK_SRVR_ID;
			if (!db_add_rcd(dcc_emsg, &host_id_rcd))
				db_broken(__LINE__,__FILE__,
					  "adding server-ID claim: %s",
					  dcc_emsg);
		}

		gettimeofday(&db_time, 0);
		misc_timer = db_time.tv_sec + MISC_CK_SECS;
	}

	/* note when hash expansion finishes and collect a zombie */
	ck_dbclean(WNOHANG);

	if (db_failed_line) {
		snprintf(reason_buf, sizeof(reason_buf),
			 "database broken at line %d in %s "DCC_VERSION,
			 db_failed_line, db_failed_file);
		reason = reason_buf;
		mode = "PR";
	} else {
		if (need_del_dbclean != 0
		    && DB_IS_TIME(del_dbclean_next, DEL_DBCLEAN_SECS)) {
			reason = need_del_dbclean;
			mode = "P";
		} else if ((hash_free = db_hash_len - db_hash_used)
			   < MIN_HASH_ENTRIES
			   || hash_free < db_hash_len/20) {
			/* try to expand the hash table when there are only
			 * a few free slots
			 * or the load factor rises above .95 */
			if (hash_free < MIN_HASH_ENTRIES)
				snprintf(reason_buf, sizeof(reason_buf),
					 "%d free hash entries",
					 hash_free);
			else
				snprintf(reason_buf, sizeof(reason_buf),
					 "%d free hash entries among %d total",
					 hash_free, db_hash_len);
			reason = reason_buf;
			mode = "P";

		} else if (DB_IS_TIME(dbclean_cron_sub, 3*24*60*60)) {
			/* clean the database again after being forced so a
			 * missing cron job does not let it bloat */
			reason = "work around broken cron job";
			mode = 0;
		} else {
			reason = 0;
			mode = 0;
		}
	}

	if (reason) {
		if (!DB_IS_TIME(dbclean_limit, dbclean_limit_secs)) {
			if (next_flods_ck > dbclean_limit)
				next_flods_ck = dbclean_limit;
		} else if (dbclean_pid < 0) {
			run_dbclean(mode, reason);
			next_flods_ck = db_time.tv_sec + 1;
		} else {
			next_flods_ck = db_time.tv_sec + 1;
		}
	} else {
		flods_ck(0);
	}

	secs = next_flods_ck - db_time.tv_sec;
	return secs >= 0 ? secs : 0;
}



static void NRATTRIB
recv_job(void)
{
	fd_set rfds, *prfds, wfds, *pwfds;
#	define PFD_SET(_fd,_fds) {p##_fds = &_fds; FD_SET(_fd,p##_fds);}
	int max_fd, nfds;
	IFLOD_INFO *ifp;
	OFLOD_INFO *ofp;
	struct timeval delay;
	time_t secs;
	u_char was_too_busy, db_has_failed;
	SRVR_SOC *sp;
	QUEUE *q;
	int bad_select;
	int fd, i;

	bad_select = 3;
	was_too_busy = 1;
	delay.tv_sec = flods_ck_secs;
	delay.tv_usec = 0;
	db_has_failed = 0;
	for (;;) {
		if (stopint != 0) {
			if (flods_ck_secs > SHUTDOWN_DELAY)
				flods_ck_secs = SHUTDOWN_DELAY;
			if (flods_off < 100000) {
				flods_off = 100000;
				iflods_stop("server stopping", 0);
				oflods_stop(0);
				db_unlock();
				db_unload(0, 0);
			}
			if (iflods.active == 0 && oflods.active == 0) {
				if (stopint < 0)
					db_quit(0, "gracefully stopping");
				db_quit(stopint | 128,
					"exiting with signal %d", stopint);
			}
		}
		if (db_has_failed != db_failed_line) {
			db_has_failed = db_failed_line;
			if (db_failed_line) {
				db_unlock();
				db_unload(0, 0);
				++flod_db_sick;
				++flods_off;
				oflods_stop(1);
				iflods_stop("database corrupt", 1);
			}
		}

		FD_ZERO(&rfds);
		prfds = 0;
		FD_ZERO(&wfds);
		pwfds = 0;
		max_fd = -1;
		if (was_too_busy) {
			if (delay.tv_sec != 0
			    || delay.tv_usec > DCC_USECS/4) {
				delay.tv_sec = 0;
				delay.tv_usec = DCC_USECS/4;
			}
		} else {
			for (sp = srvr_socs; sp; sp = sp->fwd) {
				fd = sp->udp;
				if (fd < 0)
					continue;
				PFD_SET(fd, rfds);
				if (max_fd <= fd)
					max_fd = fd;

				/* accept new incoming flood connections
				 * if flooding is on
				 * and we don't already have too many */
				fd = sp->listen;
				if (fd >= 0
				    && iflods.active < DCCD_MAX_FLOODS) {
					PFD_SET(fd, rfds);
					if (max_fd < fd)
					    max_fd = fd;
				}
			}
			/* pump output floods */
			for (ofp = oflods.infos, i = 0;
			     i < oflods.active;
			     ++ofp) {
				if (ofp->s < 0)
					continue;
				++i;
				if (ofp->flags & OFLOD_FG_CONNECTED) {
					PFD_SET(ofp->s, rfds);
					if (!(ofp->flags & OFLOD_FG_EAGAIN)
					    && (ofp->obuf_len != 0
						|| (ofp->flags
						    & OFLOD_FG_TOO_BUSY)))
					    PFD_SET(ofp->s, wfds);
				} else {
					PFD_SET(ofp->s, wfds);
				}
				if (max_fd < ofp->s)
					max_fd = ofp->s;
			}
		}

		/* process incoming floods if we are not too busy
		 * or have asked them to stop */
		if (!was_too_busy || flods_off) {
			for (ifp = iflods.infos, i = 0;
			     i < iflods.active;
			     ++ifp) {
				if (ifp->s < 0)
					continue;
				++i;
				if (ifp->flags & IFLOD_FG_CONNECTED) {
					PFD_SET(ifp->s, rfds);
				} else {
					PFD_SET(ifp->s, wfds);
				}
				if (max_fd < ifp->s)
					max_fd = ifp->s;
			}
		}

		/* delay only until it is time to answer the oldest
		 * anonymous request */
		nfds = select(max_fd+1, prfds, pwfds, 0, &delay);
		if (nfds < 0) {
			if (errno != EINTR) {
				if (--bad_select < 0)
					bad_stop("give up after select(): %s",
						 ERROR_STR());
				else
					dcc_error_msg("select(): %s",
						      ERROR_STR());
			}
			/* ignore EINTR but recompute timers */
			FD_ZERO(&rfds);
			FD_ZERO(&wfds);
		} else {
			bad_select = 3;
		}
		gettimeofday(&db_time, 0);
		wake_time = db_time;

		for (sp = srvr_socs; sp; sp = sp->fwd) {
			/* queue a new anonymous request
			 * or answer a new authenticated request */
			fd = sp->udp;
			if (fd >= 0 && FD_ISSET(fd, &rfds)) {
				--nfds;
				while (new_job(sp))
					continue;
			}

			/* start a new incoming flood */
			fd = sp->listen;
			if (fd >= 0 && FD_ISSET(fd, &rfds)) {
				--nfds;
				iflod_start(sp);
			}
		}

		/* accept new flood data or start new SOCKS floods */
		for (ifp = iflods.infos, i = 0;
		     nfds > 0 && i < iflods.active;
		     ++ifp) {
			if (ifp->s < 0)
				continue;
			++i;
			if (FD_ISSET(ifp->s, &rfds)
			    || FD_ISSET(ifp->s, &wfds)) {
				--nfds;
				iflod_read(ifp);
			}
		}
		gettimeofday(&db_time, 0);
		if (tv_diff2us(&db_time, &wake_time) > DCC_USECS)
			too_busy = 1;

		/* pump output flood data and receive confirmations */
		for (ofp = oflods.infos, i = 0;
		     nfds > 0 && i < oflods.active;
		     ++ofp) {
			if (ofp->s < 0)
				continue;
			++i;
			if (FD_ISSET(ofp->s, &rfds)) {
				--nfds;
				oflod_read(ofp);
				if (ofp->s < 0)
					continue;
			}
			if (FD_ISSET(ofp->s, &wfds)) {
				--nfds;
				oflod_write(ofp);
			}
		}

		/* process anonymous jobs when their times arrive */
		for (;;) {
			q = queue_anon_head;
			if (!q) {
				delay.tv_sec = flods_ck_secs;
				delay.tv_usec = 0;
				break;
			}

			/* decide whether this job's time has come
			 * while defending against time jumps */
			tvs_diff(&delay, &q->answer, &db_time);
			if ((delay.tv_sec > 0
			     && delay.tv_sec < DCC_MAX_RTT_SECS)
			    || (delay.tv_sec == 0
				&& delay.tv_usec >= 1000))
				break;  /* not yet time for next job */

			queue_anon_head = q->later;
			if (queue_anon_head)
				queue_anon_head->earlier = 0;
			--queue_anon_cur;
			do_work(q);
			free_q(q);
		}

		if (!was_too_busy) {
			/* check configuration changes etc. */
			secs = ck_changes();
			if (delay.tv_sec >= secs) {
				delay.tv_sec = secs;
				delay.tv_usec = 0;
			}
		}

		if (too_busy && stopint == 0) {
			too_busy = 0;
			was_too_busy = 1;
		} else {
			was_too_busy = 0;
		}
	}
}



static void
add_queue(QUEUE *q)
{
	QUEUE *qnext, **qp;

	TMSG3(QUERY, "received %s from %d at %s",
	      dcc_hdr_op2str(0,0, &q->pkt.hdr),
	      (DCC_CLNT_ID)ntohl(q->pkt.hdr.sender), Q_CIP(q));
	if (!ck_clnt_id(q)) {
		free_q(q);
		return;
	}

	/* immediately process requests from authenticated clients */
	if (q->delay_us == 0) {
		do_work(q);
		free_q(q);
		return;
	}

	/* don't let the queue of anonymous clients get too large */
	if (queue_anon_cur >= queue_max) {
		anon_msg("drop excess queued %s request from %s",
			 dcc_hdr_op2str(0,0, &q->pkt.hdr),
			 Q_CIP(q));
		free_q(q);
		return;
	}
	if (anon_off && q->clnt_id == DCC_ID_ANON) {
		anon_msg("drop anomymous %s request from %s",
			 dcc_hdr_op2str(0,0, &q->pkt.hdr),
			 Q_CIP(q));
		free_q(q);
		return;
	}

	if (q->delay_us >= DCC_MAX_RTT) {
		TMSG1(BL, "drop excessively delayed request from %s", Q_CIP(q));
		free_q(q);
		return;
	}
	tvs_add_usec(&q->answer, q->delay_us);


	/* add the new job to the queue */
	++queue_anon_cur;
	qp = &queue_anon_head;
	for (;;) {
		qnext = *qp;
		if (!qnext) {
			*qp = q;
			break;
		}
		if (qnext->answer.tv_sec > q->answer.tv_sec
		    || (qnext->answer.tv_sec == q->answer.tv_sec
			&& qnext->answer.tv_usec > q->answer.tv_usec)) {
			q->later = qnext;
			qnext->earlier = q;
			*qp = q;
			break;
		}
		q->earlier = qnext;
		qp = &qnext->later;
	}
}



/* get a new job in a datagram */
static u_char				/* 1=call again */
new_job(SRVR_SOC *sp)
{
	QUEUE *q;
	static struct iovec iov = {0, sizeof(q->pkt)};
	static struct msghdr msg;
	int i, j;

	/* Find a free queue entry for the job.
	 * Because we don't check for incoming jobs unless we think the
	 * queue is not full, there must always be a free entry or
	 * permission to make more entries. */
	q = queue_free;
	if (q) {
		queue_free = q->later;
	} else {
		i = 16;
		q = dcc_malloc(i * sizeof(*q));
		if (!q)
			dcc_logbad(EX_UNAVAILABLE,
				   "malloc(%d queue entries) failed", i);
		queue_max_cur += i;
		/* put all but the last new queue entry on the free list */
		while (--i > 0) {
			q->later = queue_free;
			queue_free = q;
			++q;
		}
	}

	memset(q, 0, sizeof(*q));
	q->sp = sp;
	iov.iov_base = (char *)&q->pkt;
	msg.msg_name = (void *)&q->clnt_su;
	msg.msg_namelen = sizeof(q->clnt_su);
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	i = recvmsg(sp->udp, &msg, 0);
	if (i < 0) {
		/* ignore some results of ICMP unreachables for UDP
		 * retransmissions seen on some platforms */
		if (DCC_BLOCK_ERROR()) {
			;
		} else if (DCC_CONNECT_ERRORS()) {
			TMSG2(QUERY, "recvmsg(%s): %s",
			      dcc_su2str_err(&sp->su), ERROR_STR());
		} else {
			dcc_error_msg("recvmsg(%s): %s",
				      dcc_su2str_err(&sp->su), ERROR_STR());
		}
		free_q(q);
		return 0;
	}
	if (q->clnt_su.sa.sa_family != sp->su.sa.sa_family
	    && !dcc_ipv4sutoipv6(&q->clnt_su, &q->clnt_su)) {
		dcc_error_msg("recvmsg address family %d instead of %d",
			      q->clnt_su.sa.sa_family,
			      sp->su.sa.sa_family);
		free_q(q);
		return 1;
	}

	q->pkt_len = i;
	if (i < ISZ(DCC_HDR)) {
		if (ck_id(q, DCC_ID_ANON)) {
			anon_msg("short packet of %d bytes from %s",
				 i, Q_CIP(q));
			discard_error(q, "short packet of %d bytes", i);
		}
		free_q(q);
		return 1;
	}
	j = ntohs(q->pkt.hdr.len);
	if (j != i) {
		if (ck_id(q, DCC_ID_ANON)) {
			anon_msg("header length %d instead of %d from %s",
				 j, i, Q_CIP(q));
			discard_error(q, "header length %d instead of %d",
				      j, i);
		}
		free_q(q);
		return 1;
	}

	if ((q->pkt.hdr.pkt_vers > DCC_PKT_VERSION_MAX
	     || q->pkt.hdr.pkt_vers < DCC_PKT_VERSION_MIN)
	    && q->pkt.hdr.op != DCC_OP_NOP) {
		if (ck_id(q, DCC_ID_ANON)) {
			anon_msg("unrecognized protocol version #%d"
				 " for %s from %s",
				 q->pkt.hdr.pkt_vers,
				 dcc_hdr_op2str(0,0, &q->pkt.hdr),
				 Q_CIP(q));
			discard_error(q, "unrecognized protocol version #%d"
				      " for %s",
				      q->pkt.hdr.pkt_vers,
				      dcc_hdr_op2str(0,0, &q->pkt.hdr));
		}
		free_q(q);
		return 1;
	}

	q->answer = wake_time;

	switch ((DCC_OPS)q->pkt.hdr.op) {
	case DCC_OP_NOP:
		do_nop(q);
		return 1;

	case DCC_OP_REPORT:
		if (db_parms.flags & DB_PARM_FG_GREY)
			break;		/* not valid for greylist servers */
		add_queue(q);
		return 1;

	case DCC_OP_QUERY:
		add_queue(q);
		return 1;

	case DCC_OP_ADMN:
		do_admn(q);
		free_q(q);
		return 1;

	case DCC_OP_DELETE:
		do_delete(q);
		free_q(q);
		return 1;

	case DCC_OP_GREY_REPORT:
	case DCC_OP_GREY_QUERY:
	case DCC_OP_GREY_WHITE:
		if (!(db_parms.flags & DB_PARM_FG_GREY))
			break;		/* valid only for greylist servers */
		do_grey(q);
		free_q(q);
		return 1;

	case DCC_OP_GREY_SPAM:
		if (!(db_parms.flags & DB_PARM_FG_GREY))
			break;		/* valid only for greylist servers */
		do_grey_spam(q);
		free_q(q);
		return 1;

	case DCC_OP_INVALID:
	case DCC_OP_ANSWER:
	case DCC_OP_OK:
	case DCC_OP_ERROR:
		break;
	}

	anon_msg("invalid op %d from %s", q->pkt.hdr.op, Q_CIP(q));
	free_q(q);
	return 1;
}



void
free_q(QUEUE *q)
{
	if (q->rl)
		--q->rl->ref_cnt;
	q->later = queue_free;
	queue_free = q;
}



u_char
dccd_db_open(u_char lock_mode)
{
	DCC_CK_TYPES type;
	struct tm tm;
	struct timeval sn;
	DCC_TGTS tgts;
	int cur_hour, i, j;

	if (!db_open(dcc_emsg, 0, 0, lock_mode | db_mode))
		return 0;

	if (grey_on) {
		/* for greylisting, ignore the args an silently impose our
		 * notion of which checksums to keep and flooding thresholds */
		db_parms.nokeep_cks = def_nokeep_cks();
		if (grey_weak_ip)
			DB_RESET_NOKEEP(db_parms.nokeep_cks, DCC_CK_IP);

		for (type = DCC_CK_TYPE_FIRST;
		     type <= DCC_CK_TYPE_LAST;
		     ++type) {
			/* use dbclean time values for server-ID declarations */
			if (type == DCC_CK_SRVR_ID) {
				flod_tholds[type] = 1;
				continue;
			}
			db_parms.ex_secs[type].clean_thold = 1;

			if (DB_TEST_NOKEEP(db_parms.nokeep_cks, type))
				flod_tholds[type] = DCC_TGTS_INVALID;
			else
				flod_tholds[type] = 1;

			if (DCC_CK_IS_GREY_TRIPLE(grey_on,type)
			    || type == DCC_CK_IP) {
				db_parms.ex_secs[type].all = grey_window;
				db_parms.ex_secs[type].spam = grey_white;
			} else if (type == DCC_CK_BODY
				   || DCC_CK_IS_GREY_MSG(grey_on,type)) {
				db_parms.ex_secs[type].all = grey_window;
				db_parms.ex_secs[type].spam = grey_window;
			} else {
				db_parms.ex_secs[type].all = 1;
				db_parms.ex_secs[type].spam = 1;
			}
		}

		summarize_delay_secs = grey_embargo - flods_ck_secs*2;
		if (summarize_delay_secs < 1)
			summarize_delay_secs = 1;

	} else {
		/* impose our notion of which normal checksums to keep */
		DB_SET_NOKEEP(set_new_nokeep_cks, DCC_CK_FLOD_PATH);
		DB_SET_NOKEEP(set_new_nokeep_cks, DCC_CK_INVALID);
		DB_SET_NOKEEP(set_new_nokeep_cks, DCC_CK_REP_TOTAL);
		DB_SET_NOKEEP(set_new_nokeep_cks, DCC_CK_REP_BULK);
		db_parms.nokeep_cks = ((def_nokeep_cks()
					& ~reset_new_nokeep_cks)
				       | set_new_nokeep_cks);

		for (type = DCC_CK_TYPE_FIRST;
		     type <= DCC_CK_TYPE_LAST;
		     ++type) {
			if (type == DCC_CK_SRVR_ID) {
				flod_tholds[type] = 1;
				continue;
			}

			tgts = flod_tholds[type];
			if (type == DCC_CK_REP_TOTAL)
				tgts = DCC_TGTS_INVALID;
			if (DB_TEST_NOKEEP(db_parms.nokeep_cks, type))
				tgts = DCC_TGTS_INVALID;
			else if (DCC_CK_IS_REP(0, type))
				tgts = DCC_TGTS_INVALID;
			else if (tgts == 0)
				tgts = DEF_FLOD_THOLDS(grey_on, type);
			else if (tgts < MIN_BULK_THRESHOLD)
				tgts = MIN_BULK_THRESHOLD;
			flod_tholds[type] = tgts;
		}

		summarize_delay_secs = MAX_SUMMARIZE_DELAY_SECS;
		for (i = 0; i < DIM(db_parms.ex_secs); ++i) {
			j = db_parms.ex_secs[i].all;
			if (j != 0
			    && j < summarize_delay_secs)
				j = summarize_delay_secs;
		}
	}

	memcpy(db_parms.flod_tholds, flod_tholds, sizeof(db_parms.flod_tholds));

	/* Compute the quarter hour when the database was last cleaned
	 * or 3 minutes past a random quarter hour beteen midnight and 06:00 */
	dcc_ts2timeval(&sn, db_parms.sn);
	dcc_localtime(sn.tv_sec, &tm);
	if (!(db_parms.flags & DB_PARM_FG_SELF_CLEAN)) {
		i = tm.tm_hour*60 + tm.tm_min + DBCLEAN_LIMIT_SECS;
	} else if (tm.tm_hour < 6) {
		i = tm.tm_hour*60 + tm.tm_min;
	} else {
		j = (u_int)((sn.tv_sec + db_time.tv_sec) * my_srvr_id) % (6*4);
		i = (j/4)*60 + (j%4)*15 + 3;
	}
	dcc_localtime(db_time.tv_sec, &tm);
	cur_hour = tm.tm_hour;
	tm.tm_hour = i / 60;
	tm.tm_min = i % 60;
	tm.tm_sec = 0;
	dbclean_cron_sub = mktime(&tm);

	/* If we instead of cron asked for the last cleaning, make a note
	 * to clean the database during the graveyard shift.
	 * Otherwise the database will bloat while the cron job is broken. */
	if (!(db_parms.flags & DB_PARM_FG_SELF_CLEAN)) {
		dbclean_cron_sub += 2*24*60*60;
	} else if (!(db_parms.flags & DB_PARM_FG_SELF_CLEAN2)) {
		dbclean_cron_sub += 24*60*60;
	} else if (cur_hour >= 6) {	/* wait until tomorrow if past 06:00 */
		dbclean_cron_sub += 24*60*60;
	}
	if (dbclean_cron_sub < sn.tv_sec + 24*60*60)
		dbclean_cron_sub += 24*60*60;

	return db_flush_parms(dcc_emsg);
}



void
close_srvr_socs(void)
{
	SRVR_SOC *sp;

	for (sp = srvr_socs; sp; sp = sp->fwd) {
		if (sp->udp >= 0) {
			close(sp->udp);
			sp->udp = -1;
		}
		if (sp->listen >= 0) {
			close(sp->listen);
			sp->listen = -1;
		}
	}
}



/* clean shut down */
static void NRATTRIB
db_quit(int exitcode, const char *p, ...)
{
	va_list args;

	va_start(args, p);
	if (exitcode)
		dcc_verror_msg(p, args);
	else
		dcc_vtrace_msg(p, args);
	va_end(args);

	/* db_close() can take a long time, so close some things early.
	 * Do not call close_srvr_socs() but keep the UDP sockets open
	 * to prevent another server from starting until we have flushed
	 * our buffers to prevent problems on some UNIX systems that lack
	 * inter-process coherent mmap(),
	 * except on BSD/OS where close() takes forever. */
#ifdef __bsdi
	close_srvr_socs();
#endif
	stop_children();
	db_close(1);

	if (exitcode)
		dcc_error_msg("stopped");
	else
		dcc_trace_msg("stopped");
	exit(exitcode);
}



/* watch for fatal signals */
static void
sigterm(int sig)
{
	stopint = sig;
	too_busy = 0;
	next_flods_ck = 0;
	(void)signal(sig, SIG_DFL);	/* catch it only once */
}



/* SIGHUP hurries checking the configuration files */
static void
sighup(int sig UATTRIB)
{
	next_flods_ck = 0;
}



/* emergency shutdown but close the database cleanly */
void
bad_stop(const char *pat, ...)
{
	va_list args;

	if (stopint)
		return;

	va_start(args, pat);
	dcc_verror_msg(pat, args);
	va_end(args);

	stopint = -1;
	next_flods_ck = 0;
}



static void
stop_children(void)
{
	if (resolve_hosts_pid > 0) {
		kill(resolve_hosts_pid, SIGKILL);
		resolve_hosts_pid = -1;
	}

	if (dbclean_pid > 0) {
		kill(dbclean_pid, SIGKILL);
		dbclean_pid = -1;
	}
}
