/* Distributed Checksum Clearinghouse
 *
 * ask about a batch of checksums
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.102 $Revision$
 */

#include "dcc_ck.h"
#include "dcc_heap_debug.h"
#include "dcc_xhdr.h"

#define SET_ALL_THOLDS	(DCC_CK_TYPE_LAST+1)
#define ALL_CKSUMS_STR	"ALL"
#define IS_ALL_CKSUM(t)	(!DCC_CK_IS_REP(0, t))
#define SET_CMN_THOLDS	(SET_ALL_THOLDS+1)
#define CMN_CKSUMS_STR	"CMN"
#define IS_CMN_CKSUM(t)	DCC_CK_IS_BODY(t)

static DCC_TGTS dcc_tholds_spam[DCC_DIM_CKS];
static DCC_TGTS dcc_tholds_log[DCC_DIM_CKS];
static u_char dcc_honor_nospam[DCC_DIM_CKS];

static u_char trim_grey_ip_addr;	/* remove IP address from grey triple */
static struct in6_addr grey_ip_mask;

static void honor_cnt(u_int *, DCC_CK_TYPES, DCC_TGTS);



#ifdef DCC_PKT_VERSION5
/* figure old server's target count before our latest report */
static DCC_TGTS				/* return corrected current count */
save_p_tgts(DCC_GOT_SUM *g,		/* put previous count in g->tgts */
	    DCC_OPS op,
	    const DCC_TGTS local_tgts,	/* real local target count */
	    const DCC_TGTS gross_tgts,	/* local count adjusted by blacklist */
	    DCC_TGTS c_tgts)		/* what the old DCC server said */
{
	DCC_CK_TYPES type = g->type;

	if (op == DCC_OP_QUERY) {
		/* if we switched servers and converted a report
		 * to a query, then guess the total that the
		 * server would have produced for a report
		 * instead of the query we sent.
		 *
		 * Assume the server is not running with -K.
		 * If the server's current value is 0 for a body checksum
		 * then assume the report we sent to the other server has not
		 * been flooded.
		 * Assume other checksums will always be zero/unknown. */
		if (DB_GLOBAL_NOKEEP(0, type))
			return 0;

		/* Assume the current value is really the previous value
		 * because flooding has not happened */
		g->tgts = c_tgts;

		if (c_tgts < DCC_TGTS_TOO_MANY
		    && DCC_CK_IS_BODY(type)) {
			c_tgts += local_tgts;
			if (c_tgts > DCC_TGTS_TOO_MANY)
				c_tgts = DCC_TGTS_TOO_MANY;
		}
		return c_tgts;

	} else if (c_tgts >= gross_tgts
		   && gross_tgts < DCC_TGTS_TOO_MANY) {
		/* if possible infer server's value before our report */
		if (c_tgts >= DCC_TGTS_TOO_MANY)
			g->tgts = c_tgts;
		else
			g->tgts = c_tgts - gross_tgts;
	}

	return c_tgts;
}



#endif /* DCC_PKT_VERSION5 */
int					/* 1=ok, 0=no answer, -1=fatal */
ask_dcc(DCC_EMSG emsg,
	DCC_CLNT_CTXT *ctxt,
	u_char clnt_fgs,		/* DCC_CLNT_FG_* */
	DCC_HEADER_BUF *header,		/* put results here */
	DCC_GOT_CKS *cks,		/*	and here */
	ASK_ST *ask_stp,		/*	and here */
	u_char spam,			/* spam==0 && local_tgts==0 --> query */
	DCC_TGTS local_tgts)		/* report these targets to DCC server */
{
	union {
	    DCC_HDR	hdr;
	    DCC_REPORT	r;
	} rpt;
	ASK_RESP resp;
	DCC_OPS op;
	DCC_CK *ck;
	DCC_GOT_SUM *g;
	DCC_TGTS gross_tgts;
	DCC_TGTS c_tgts;		/* server's current, total count */
	DCC_CKS_WTGTS hdr_tgts;		/* values for X-DCC header */
	DCC_CK_TYPES type;
	DCC_SRVR_ID srvr_id;
	int pkt_len, recv_len, exp_len;
	int num_cks, ck_num, i;

	memset(hdr_tgts, 0, sizeof(hdr_tgts));

	/* prepare a report for the nearest DCC server */
	if (local_tgts == 0 && !spam) {
		/* because of greylisting, we can have a target count of 0
		 * but need to report spam discovered by a DNSBL */
		op = DCC_OP_QUERY;
		gross_tgts = 0;
		rpt.r.tgts = 0;
	} else {
		op = DCC_OP_REPORT;
		if (local_tgts == DCC_TGTS_TOO_MANY
		    || local_tgts == 0) {
			spam = 1;
			local_tgts = 1;
		}
		if (spam) {
			*ask_stp |= (ASK_ST_CLNT_ISSPAM | ASK_ST_LOGIT);
			gross_tgts = DCC_TGTS_TOO_MANY;
			rpt.r.tgts = htonl(local_tgts | DCC_TGTS_SPAM);
		} else {
			gross_tgts = local_tgts;
			rpt.r.tgts = htonl(local_tgts);
		}
	}

	ck = rpt.r.cks;
	num_cks = 0;
	for (g = cks->sums; g <= &cks->sums[DCC_CK_TYPE_LAST]; ++g) {
		/* never tell the DCC server about some headers */
		if (!g->rpt2srvr)
			continue;
		ck->len = sizeof(*ck);
		ck->type = g->type;
		memcpy(ck->sum, g->sum, sizeof(ck->sum));
		++ck;
		++num_cks;
	}
	if (num_cks == 0) {
		/* pretend we always have at least a basic body checksum
		 * guess the DCC would have answered 0 */
		dcc_header_init(header, "", 0);
		honor_cnt(ask_stp, DCC_CK_BODY, local_tgts);
		dcc_add_header_ck(header, DCC_CK_BODY, gross_tgts);
		return 1;
	}

	/* send the report and see what the DCC has to say */
	pkt_len = (sizeof(rpt.r) - sizeof(rpt.r.cks)
		   + num_cks * sizeof(rpt.r.cks[0]));
	i = dcc_clnt_op(emsg, ctxt, clnt_fgs, 0, &srvr_id, 0,
			&rpt.hdr, pkt_len, op, &resp.hdr, sizeof(resp));

	/* try a query to different server if the first failed
	 * but a second was found */
	if (!i && srvr_id != DCC_ID_INVALID) {
		if (dcc_clnt_debug) {
			if (emsg && *emsg != '\0') {
				dcc_trace_msg("retry with different server"
					      " after: %s", emsg);
				*emsg = '\0';
			} else {
				dcc_trace_msg("retry with different server");
			}
		}
		op = DCC_OP_QUERY;
		i = dcc_clnt_op(emsg, ctxt, clnt_fgs | DCC_CLNT_FG_RETRY, 0,
				&srvr_id, 0,
				&rpt.hdr, pkt_len,
				op, &resp.hdr, sizeof(resp));
	}
	if (!i) {
		*ask_stp |= ASK_ST_LOGIT;
		header->buf[0] = '\0';
		return 0;
	}

	/* forget about it if the DCC server responded too strangely */
	recv_len = ntohs(resp.hdr.len);
#ifdef DCC_PKT_VERSION5
	if (resp.hdr.pkt_vers <= DCC_PKT_VERSION5)
		exp_len = (sizeof(resp.a5) - sizeof(resp.a5.b)
		   + num_cks*sizeof(DCC_TGTS));
	else
#endif
	exp_len = (sizeof(resp.a) - sizeof(resp.a.b)
		   + num_cks*sizeof(resp.a.b[0]));
	if (recv_len != exp_len) {
		dcc_pemsg(EX_UNAVAILABLE, emsg,
			  "DCC %s: answered with %d instead of %d bytes",
			  dcc_srvr_nm(), recv_len, exp_len);
		*ask_stp |= ASK_ST_LOGIT;
		header->buf[0] = '\0';
		return -1;
	}

	/* check the server's response to see if we have spam */
	ck_num = 0;
	for (g = cks->sums; g <= &cks->sums[DCC_CK_TYPE_LAST]; ++g) {
		if (!g->rpt2srvr) {
			/* pretend we always have a basic body checksum */
			if (g == &cks->sums[DCC_CK_BODY])
				honor_cnt(ask_stp, DCC_CK_BODY, local_tgts);
			continue;
		}
		type = g->type;		/* g->type is valid only if rpt2srvr */

#ifdef DCC_PKT_VERSION5
		if (resp.hdr.pkt_vers <= DCC_PKT_VERSION5) {
			c_tgts = save_p_tgts(g, op,
					     local_tgts, gross_tgts,
					     ntohl(resp.a5.b[ck_num]));
		} else {
#endif /* DCC_PKT_VERSION5 */
			/* server's total before our report */
			g->tgts = ntohl(resp.a.b[ck_num].p);
			/* new total */
			c_tgts = ntohl(resp.a.b[ck_num].c);
#ifdef DCC_PKT_VERSION5
		}
#endif
		++ck_num;

		hdr_tgts[type] = c_tgts;

		/* notice DCC server's whitelist */
		if (dcc_honor_nospam[type]) {
			if (c_tgts == DCC_TGTS_OK) {
				*ask_stp |= ASK_ST_SRVR_NOTSPAM;

			} else if (c_tgts == DCC_TGTS_OK2) {
				/* if server says it is half ok,
				 * look for two halves */
				if (*ask_stp & ASK_ST_SRVR_OK2) {
					*ask_stp |= ASK_ST_SRVR_NOTSPAM;
				} else {
					*ask_stp |= ASK_ST_SRVR_OK2;
				}
			}
		}

		honor_cnt(ask_stp, type, c_tgts);
	}

	/* honor server whitelist */
	if (*ask_stp & ASK_ST_SRVR_NOTSPAM)
		*ask_stp &= ~ASK_ST_SRVR_ISSPAM;

	/* generate the header line now that we have checked all of
	 * the counts against their thresholds and so know if we
	 * must add "bulk".  Add the header even if checking is turned off
	 * so that we won't reject affected messages */
	dcc_header_init(header, dcc_clnt_hostname, srvr_id);
	if (*ask_stp & (ASK_ST_CLNT_ISSPAM | ASK_ST_SRVR_ISSPAM)) {
		dcc_add_header(header, DCC_XHDR_BULK);
	} else if (*ask_stp & ASK_ST_REP_ISSPAM) {
		dcc_add_header(header, DCC_XHDR_BULK_REP);
		hdr_tgts[DCC_CK_BODY] = DCC_TGTS_TOO_MANY;
		hdr_tgts[DCC_CK_FUZ1] = DCC_TGTS_TOO_MANY;
		hdr_tgts[DCC_CK_FUZ2] = DCC_TGTS_TOO_MANY;
	}

	for (g = cks->sums; g <= &cks->sums[DCC_CK_TYPE_LAST]; ++g) {
		if (!g->rpt2srvr) {
			/* pretend we always have a body checksum */
			if (g == &cks->sums[DCC_CK_BODY])
				dcc_add_header_ck(header, DCC_CK_BODY,
						  hdr_tgts[DCC_CK_BODY]);
			continue;
		}
		/* Add interesing counts to the header.
		 * Body checksums are always interestig if we have them.
		 * Pretend we always have a basic body checksum. */
		type = g->type;
		if (DCC_CK_IS_BODY(type)) {
			dcc_add_header_ck(header, type, hdr_tgts[type]);
			continue;
		}
		if (hdr_tgts[type] != 0)
			dcc_add_header_ck(header, type, hdr_tgts[type]);
	}

	return 1;
}



/* check message's checksums in whiteclnt for dccproc or dccsight */
u_char					/* 1=ok 0=something to complain about */
unthr_ask_white(DCC_EMSG emsg,
		ASK_ST *ask_stp,
		const char *white_nm,
		DCC_GOT_CKS *cks,
		DCC_CKS_WTGTS wtgts)
{
	DCC_WHITE_LISTING listing;
	int retval;

	/* fake whiteclnt if not specified */
	if (!white_nm) {
		cmn_wf.info_flags |= (DCC_WHITE_FG_DNSBL_ON
				      | DCC_WHITE_FG_XFLTR_ON);
		return 1;
	}

	/* don't filter if something is wrong with it */
	if (!dcc_new_white_nm(emsg, &cmn_wf, white_nm)) {
		*ask_stp |= ASK_ST_WLIST_NOTSPAM | ASK_ST_LOGIT;
		return 0;
	}

	retval = 1;
	switch (dcc_white_cks(emsg, &cmn_wf, cks, wtgts, &listing)) {
	case DCC_WHITE_OK:
	case DCC_WHITE_NOFILE:
		break;
	case DCC_WHITE_SILENT:
		*ask_stp |= ASK_ST_LOGIT;
		break;
	case DCC_WHITE_COMPLAIN:
	case DCC_WHITE_CONTINUE:
		retval = 0;
		*ask_stp |= ASK_ST_LOGIT;
		break;
	}

	switch (listing) {
	case DCC_WHITE_LISTED:
		/* do not send whitelisted checksums to DCC server */
		*ask_stp |= ASK_ST_WLIST_NOTSPAM;
		break;
	case DCC_WHITE_USE_DCC:
	case DCC_WHITE_UNLISTED:
		break;
	case DCC_WHITE_BLACK:
		*ask_stp |= (ASK_ST_WLIST_ISSPAM
			     | ASK_ST_CLNT_ISSPAM | ASK_ST_LOGIT);
		break;
	}
	if (cmn_wf.info_flags & DCC_WHITE_FG_LOG_ALL)
		*ask_stp |= ASK_ST_LOGIT;
	return retval;
}



/* ask the DCC for dccproc or dccsight but not dccifd or dccm */
u_char					/* 1=ok 0=something to complain about */
unthr_ask_dcc(DCC_EMSG emsg,
	      DCC_CLNT_CTXT *ctxt,
	      DCC_HEADER_BUF *header,	/* put header here */
	      ASK_ST *ask_stp,		/* put state bites here */
	      DCC_GOT_CKS *cks,		/* these checksums */
	      u_char spam,		/* spam==0 && local_tgts==0 --> query */
	      DCC_TGTS local_tgts)	/* number of addressees */
{
	/* if allowed by whitelisting, report our checksums to the DCC
	 * and return with that result including setting logging */
	if (!(*ask_stp & ASK_ST_WLIST_NOTSPAM))
		return (0 < ask_dcc(emsg, ctxt, DCC_CLNT_FG_NONE,
				    header, cks, ask_stp, spam,
				    local_tgts));

	/* else honor log threshold for local counts and white-/blacklists */
	if (spam)
		*ask_stp |= (ASK_ST_CLNT_ISSPAM | ASK_ST_LOGIT);
	dcc_honor_log_cnts(ask_stp, cks, local_tgts);
	return 1;
}



/* parse -g for dccm and dccproc */
void
dcc_parse_honor(const char *arg0)
{
	const char *arg;
	DCC_CK_TYPES type, t2;
	int i;

	arg = arg0;
	if (!CSTRCMP(arg, "not_") || !CSTRCMP(arg, "not-")) {
		arg += STRZ("not_");
		i = 0;
	} else if (!CSTRCMP(arg, "no_") || !CSTRCMP(arg, "no-")) {
		arg += STRZ("no_");
		i = 0;
	} else {
		i = 1;
	}

	type = dcc_str2type(arg);
	if (type == DCC_CK_INVALID
	    || type >= DIM(dcc_honor_nospam)) {
		if (!strcasecmp(arg, CMN_CKSUMS_STR)) {
			type = SET_CMN_THOLDS;
		} else if (!strcasecmp(arg, ALL_CKSUMS_STR)) {
			type = SET_ALL_THOLDS;
		} else {
			dcc_error_msg("unrecognized checksum type in \"-g %s\"",
				      arg0);
			return;
		}
	}
	for (t2 = DCC_CK_TYPE_FIRST; t2 <= DCC_CK_TYPE_LAST; ++t2) {
		if (t2 == type
		    || (type == SET_ALL_THOLDS && IS_ALL_CKSUM(type))
		    || (type == SET_CMN_THOLDS && IS_CMN_CKSUM(t2)))
			dcc_honor_nospam[t2] = i;
	}
}



void
dcc_clear_tholds(void)
{
	DCC_TGTS *tgtsp;

	memset(dcc_honor_nospam, 0, sizeof(dcc_honor_nospam));
	dcc_honor_nospam[DCC_CK_IP] = 1;
	dcc_honor_nospam[DCC_CK_ENV_FROM] = 1;
	dcc_honor_nospam[DCC_CK_FROM] = 1;

	for (tgtsp = dcc_tholds_spam; tgtsp <= LAST(dcc_tholds_spam); ++tgtsp)
		*tgtsp = DCC_TGTS_INVALID;

	for (tgtsp = dcc_tholds_log; tgtsp <= LAST(dcc_tholds_log); ++tgtsp)
		*tgtsp = DCC_TGTS_INVALID;
}



static void
set_tholds(DCC_TGTS tholds[DCC_DIM_CKS],
	   DCC_CK_TYPES type,		/* specified type, DCC_CK_INVALID=all,
					 * SET_CMN_THOLDS=default set */
	   DCC_TGTS tgts)
{
	DCC_CK_TYPES t2;

	for (t2 = DCC_CK_TYPE_FIRST; t2 <= DCC_CK_TYPE_LAST; ++t2) {
		if (t2 == type
		    || (type == SET_ALL_THOLDS && IS_ALL_CKSUM(t2))
		    || (type == SET_CMN_THOLDS && IS_CMN_CKSUM(t2)))
			tholds[t2] = tgts;
	}
}



/* parse type,[log-thold,]rej-thold */
u_char					/* 1=need a log directory */
dcc_parse_tholds(char a,
		 const char *arg)   /* optarg */
{
	DCC_CK_TYPES type;
	DCC_TGTS log_tgts, spam_tgts;
	char *duparg, *thold_spam, *thold_log;
	u_char log_tgts_set, spam_tgts_set;

	duparg = dcc_strdup(arg);
	thold_log = strchr(duparg, ',');
	if (!thold_log) {
		dcc_error_msg("missing comma in \"-%c %s\"", a, arg);
		dcc_free(duparg);
		return 0;
	}
	*thold_log++ = '\0';

	/* if there is only one threshold, take it as the spam threshold */
	thold_spam = strchr(thold_log, ',');
	if (!thold_spam) {
		thold_spam = thold_log;
		thold_log = 0;
	} else {
		*thold_spam++ = '\0';
	}

	if (!strcasecmp(duparg, ALL_CKSUMS_STR)) {
		type = SET_ALL_THOLDS;
	} else if (!strcasecmp(duparg, CMN_CKSUMS_STR)) {
		type = SET_CMN_THOLDS;
	} else {
		type = dcc_str2type(duparg);
		if (type == DCC_CK_INVALID
		    || type > DCC_CK_GREY3) {
			dcc_error_msg("unrecognized checksum type"
				      " \"%s\" in \"-%c %s\"",
				      duparg, a, arg);
			dcc_free(duparg);
			return 0;
		}
	}

	log_tgts = DCC_TGTS_INVALID;
	log_tgts_set = 0;
	if (thold_log && *thold_log != '\0') {
		log_tgts = dcc_str2cnt(thold_log);
		if (log_tgts == DCC_TGTS_INVALID
		    && strcasecmp(thold_log, "never"))
			dcc_error_msg("unrecognized count \"%s\" in \"-%c %s\"",
				      thold_log, a, arg);
		else
			log_tgts_set = 1;
	}


	spam_tgts = DCC_TGTS_INVALID;
	spam_tgts_set = 0;
	if (!thold_spam || *thold_spam == '\0') {
		if (!thold_log || *thold_log == '\0')
			dcc_error_msg("no thresholds in \"-%c %s\"", a, arg);
	} else {
		spam_tgts = dcc_str2cnt(thold_spam);
		if (spam_tgts == DCC_TGTS_INVALID
		    && strcasecmp(thold_spam, "never"))
			dcc_error_msg("unrecognized count \"%s\" in \"-%c %s\"",
				      thold_spam, a, arg);
		else
			spam_tgts_set = 1;
	}

	if (log_tgts_set)
		set_tholds(dcc_tholds_log, type, log_tgts);
	if (spam_tgts_set)
		set_tholds(dcc_tholds_spam, type, spam_tgts);

	dcc_free(duparg);
	return log_tgts_set;
}



static void
honor_cnt(ASK_ST *ask_stp,		/* previous flag bits */
	  DCC_CK_TYPES type,		/* which kind of checksum */
	  DCC_TGTS type_tgts)		/* total count for the checksum */
{
	if (type >= DIM(dcc_honor_nospam))
		return;

	/* reject and log spam */
	if (dcc_tholds_spam[type] != DCC_TGTS_INVALID
	    && dcc_tholds_spam[type] <= type_tgts
	    && type_tgts <= DCC_TGTS_TOO_MANY) {
		*ask_stp |= (ASK_ST_SRVR_ISSPAM | ASK_ST_LOGIT);
		return;
	}

	/* log bulky messages */
	if (dcc_tholds_log[type] != DCC_TGTS_INVALID
	    && type_tgts >= dcc_tholds_log[type])
		*ask_stp |= ASK_ST_LOGIT;
}



/* honor log threshold for local counts and white-/blacklists */
void
dcc_honor_log_cnts(ASK_ST *ask_stp,	/* previous flag bits */
		   const DCC_GOT_CKS *cks,  /* these server counts */
		   DCC_TGTS tgts)
{
	const DCC_GOT_SUM *g;
	DCC_CK_TYPES type;

	if (*ask_stp & ASK_ST_LOGIT)
		return;

	if (tgts == DCC_TGTS_TOO_MANY) {
		*ask_stp |= ASK_ST_LOGIT;
		return;
	}

	/* pretend we always have a body checksum */
	if (dcc_tholds_log[DCC_CK_BODY] != DCC_TGTS_INVALID
	    && dcc_tholds_log[DCC_CK_BODY] <= tgts) {
		*ask_stp |= ASK_ST_LOGIT;
		return;
	}

	for (g = cks->sums; g <= LAST(cks->sums); ++g) {
		type = g->type;
		if (type == DCC_CK_INVALID
		    || type == DCC_CK_ENV_TO)
			continue;
		if (dcc_tholds_log[type] == DCC_TGTS_INVALID)
			continue;
		if (dcc_tholds_log[type] <= tgts) {
			*ask_stp |= ASK_ST_LOGIT;
			return;
		}
	}
}



#define LOG_ASK_ST_BLEN	    160
#define LOG_ASK_ST_OFF	    "(off)"
#define LOG_ASK_ST_OVF	    " ...\n\n"
static int
log_ask_st_sub(char *buf, int blen,
	       const char *s, int slen,
	       u_char off)
{
	int dlen, tlen;

	if (blen >= LOG_ASK_ST_BLEN)
		return LOG_ASK_ST_BLEN;

	dlen = LOG_ASK_ST_BLEN - blen;
	tlen = STRZ(LOG_ASK_ST_OVF)+2+slen;
	if (off)
		tlen += STRZ(LOG_ASK_ST_OFF);
	if (dlen <= tlen) {
		memcpy(&buf[blen], LOG_ASK_ST_OVF, STRZ(LOG_ASK_ST_OVF));
		blen += STRZ(LOG_ASK_ST_OVF);
		if (blen < LOG_ASK_ST_BLEN)
			memset(&buf[blen], ' ', LOG_ASK_ST_BLEN-blen);
		return LOG_ASK_ST_BLEN;
	}

	if (blen > 0 && buf[blen-1] != '\n') {
		buf[blen++] = ' ';
		buf[blen++] = ' ';
	}
	memcpy(buf+blen, s, slen);
	blen += slen;
	if (off) {
		memcpy(buf+blen, LOG_ASK_ST_OFF, STRZ(LOG_ASK_ST_OFF));
		blen += STRZ(LOG_ASK_ST_OFF);
	}
	return blen;
}



void
log_ask_st(LOG_WRITE_FNC fnc, void *cp, ASK_ST ask_st, FLTR_SWS sws,
	   const char *ftype, int ftype_len,
#ifdef USE_XFLTR
	   const char *xfltr_hdr, int xfltr_hdr_len,
#endif
	   const DCC_HEADER_BUF *hdr)
{
	char buf[LOG_ASK_ST_BLEN+3];
	int blen;
#define S(bit,off,str) if (ask_st & bit) blen = log_ask_st_sub(buf, blen,   \
							str, STRZ(str),	    \
							(off) != 0);

	blen = 0;
	if (sws & FLTR_SW_MTA_FIRST) {
		S(ASK_ST_MTA_ISSPAM,	0,  "MTA"DCC_XHDR_ISSPAM);
		S(ASK_ST_MTA_NOTSPAM,	0,  "MTA"DCC_XHDR_ISOK);
	}
	S(ASK_ST_WLIST_ISSPAM,	0,
	  "wlist"DCC_XHDR_ISSPAM);
	S(ASK_ST_WLIST_NOTSPAM,	0,
	  "wlist"DCC_XHDR_ISOK);
	S(ASK_ST_SRVR_ISSPAM,	(sws & FLTR_SW_DCC_OFF),
	  "DCC"DCC_XHDR_ISSPAM);
	S(ASK_ST_SRVR_NOTSPAM,	(sws & FLTR_SW_DCC_OFF),
	  "DCC"DCC_XHDR_ISOK);
	S(ASK_ST_REP_ISSPAM,	!(sws & FLTR_SW_REP_ON),
	  "Rep"DCC_XHDR_ISSPAM);
	S(ASK_ST_DNSBL_ISSPAM,	!(sws & FLTR_SW_DNSBL_ON),
	  "DNSBL"DCC_XHDR_ISSPAM);
#ifdef USE_XFLTR
	S(ASK_ST_XFLTR_ISSPAM,	!(sws & FLTR_SW_XFLTR_ON),
	  "XFLTR"DCC_XHDR_ISSPAM);
#endif
	if (!(sws & FLTR_SW_MTA_FIRST)) {
		S(ASK_ST_MTA_ISSPAM,	0,  "MTA"DCC_XHDR_ISSPAM);
		S(ASK_ST_MTA_NOTSPAM,	0,  "MTA"DCC_XHDR_ISOK);
	}
	blen = log_ask_st_sub(buf, blen, dcc_progname, dcc_progname_len, 0);
	blen = log_ask_st_sub(buf, blen, ftype, ftype_len, 0);
	blen = log_ask_st_sub(buf, blen, "\n\n", 2, 0);
	fnc(cp, buf, blen);

#ifdef USE_XFLTR
	if (xfltr_hdr_len)
		dcc_write_header(fnc, cp, xfltr_hdr, xfltr_hdr_len, 0);
#endif
	if (hdr->used != 0)
		dcc_write_header(fnc, cp, hdr->buf, hdr->used, 0);
#undef S
}



/* parse -G options for DCC clients */
u_char					/* 0=bad */
dcc_parse_client_grey(char *arg)
{
	int bits;
	char *p;

	while (*arg != '\0') {
		if (dcc_ck_word_comma(&arg, "on")) {
			grey_on = 1;
			continue;
		}
		if (dcc_ck_word_comma(&arg, "off")) {
			grey_on = 0;
			continue;
		}
		if (dcc_ck_word_comma(&arg, "query")) {
			grey_query_only = 1;
			continue;
		}
		if (dcc_ck_word_comma(&arg, "noIP")) {
			grey_on = 1;
			trim_grey_ip_addr = 1;
			memset(&grey_ip_mask, 0, sizeof(grey_ip_mask));
			continue;
		}
		if (!CSTRCMP(arg, "IPmask/")) {
			bits = 0;
			for (p = arg+STRZ("IPmask/");
			     *p >= '0' && *p <= '9';
			     ++p)
				bits = bits*10 + *p - '0';
			if (bits > 0 && bits < 128
			    && (*p == '\0' || *p == ',')) {
				arg = p;
				if (*p == ',')
					++arg;
				grey_on = 1;
				trim_grey_ip_addr = 1;
				/* assume giant blocks are really IPv4 */
				if (bits <= 32)
					bits += 128-32;
				dcc_bits2mask(&grey_ip_mask, bits);
				continue;
			}
		}
		return 0;
	}
	return 1;
}



/* sanity check the DCC server's answer */
u_char
dcc_ck_grey_answer(DCC_EMSG emsg, const ASK_RESP *resp)
{
	int recv_len;

	recv_len = ntohs(resp->hdr.len);
	if (resp->hdr.op != DCC_OP_ANSWER) {
		dcc_pemsg(EX_UNAVAILABLE, emsg, "DCC %s: %s %*s",
			  dcc_srvr_nm(),
			  dcc_hdr_op2str(0, 0, &resp->hdr),
			  (resp->hdr.op == DCC_OP_ERROR
			   ? (recv_len - (ISZ(resp->error)
					  - ISZ(resp->error.msg)))
			   : 0),
			  resp->error.msg);
		return 0;
	}

	if (recv_len != sizeof(DCC_GREY_ANSWER)) {
		dcc_pemsg(EX_UNAVAILABLE, emsg,
			  "greylist server %s answered with %d instead of"
			  " %d bytes",
			  dcc_srvr_nm(), recv_len, ISZ(DCC_GREY_ANSWER));
		return 0;
	}

	return 1;
}



ASK_GREY_RESULT
ask_grey(DCC_EMSG emsg,
	 DCC_CLNT_CTXT *ctxt,
	 DCC_OPS op,			/* DCC_OP_GREY_{REPORT,QUERY,WHITE} */
	 DCC_SUM msg_sum,		/* put msg+sender+target cksum here */
	 DCC_SUM triple_sum,		/* put greylist triple checksum here */
	 const DCC_GOT_CKS *cks,
	 const DCC_SUM env_to_sum,
	 DCC_TGTS *pembargo_num,
	 DCC_TGTS *pearly_tgts,		/* ++ report to DCC even if embargoed */
	 DCC_TGTS *plate_tgts)		/* ++ don't report to DCC */
{
	MD5_CTX ctx;
	DCC_REPORT rpt;
	ASK_RESP resp;
	DCC_CK *ck;
	DCC_CK_TYPES type;
	const DCC_GOT_SUM *g;
	DCC_TGTS result_tgts;
	int num_cks;

	if (cks->sums[DCC_CK_IP].type != DCC_CK_IP) {
		dcc_pemsg(EX_UNAVAILABLE, emsg,
			  "IP address not available for greylisting");
		memset(triple_sum, 0, sizeof(*triple_sum));
		memset(msg_sum, 0, sizeof(*msg_sum));
		return ASK_GREY_FAIL;
	}
	if (cks->sums[DCC_CK_ENV_FROM].type != DCC_CK_ENV_FROM) {
		dcc_pemsg(EX_UNAVAILABLE, emsg,
			  "env_From not available for greylisting");
		memset(triple_sum, 0, sizeof(*triple_sum));
		memset(msg_sum, 0, sizeof(*msg_sum));
		return ASK_GREY_FAIL;
	}

	/* Check the common checksums for whitelisting at the greylist server.
	 * This assumes DCC_CK_GREY_TRIPLE > DCC_CK_GREY_MSG > other types */
	ck = rpt.cks;
	num_cks = 0;
	for (type = 0, g = cks->sums;
	     type <= DCC_CK_TYPE_LAST;
	     ++type, ++g) {
		/* greylisting needs a body checksum, even if
		 * it is the fake checksum for a missing body */
		if (!g->rpt2srvr && type != DCC_CK_BODY)
			continue;
		ck->type = type;
		ck->len = sizeof(*ck);
		memcpy(ck->sum, g->sum, sizeof(ck->sum));
		++ck;
		++num_cks;
	}

	/* include in the request the grey message checksum as the checksum
	 * of the body, the env_From sender, and env_To target checksums */
	MD5Init(&ctx);
	MD5Update(&ctx, cks->sums[DCC_CK_BODY].sum, sizeof(DCC_SUM));
	MD5Update(&ctx, cks->sums[DCC_CK_ENV_FROM].sum, sizeof(DCC_SUM));
	MD5Update(&ctx, env_to_sum, sizeof(DCC_SUM));
	MD5Final(msg_sum, &ctx);
	ck->type = DCC_CK_GREY_MSG;
	ck->len = sizeof(*ck);
	memcpy(ck->sum, msg_sum, sizeof(ck->sum));
	++ck;
	++num_cks;

	/* include the triple checksum of the sender, the sender's IP
	 * address, and the target */
	MD5Init(&ctx);
	if (trim_grey_ip_addr) {
		struct in6_addr addr;
		DCC_SUM sum;
		int wno;

		for (wno = 0; wno < 4; ++wno) {
			addr.s6_addr32[wno] = (cks->ip_addr.s6_addr32[wno]
					       & grey_ip_mask.s6_addr32[wno]);
		}
		dcc_ck_ipv6(sum, &addr);
		MD5Update(&ctx, sum, sizeof(DCC_SUM));
	} else {
		MD5Update(&ctx, cks->sums[DCC_CK_IP].sum, sizeof(DCC_SUM));
	}
	MD5Update(&ctx, cks->sums[DCC_CK_ENV_FROM].sum, sizeof(DCC_SUM));
	MD5Update(&ctx, env_to_sum, sizeof(DCC_SUM));
	MD5Final(triple_sum, &ctx);
	ck->type = DCC_CK_GREY3;
	ck->len = sizeof(*ck);
	memcpy(ck->sum, triple_sum, sizeof(ck->sum));
	++num_cks;

	if (!dcc_clnt_op(emsg, ctxt, DCC_CLNT_FG_GREY, 0, 0, 0,
			 &rpt.hdr, (sizeof(rpt) - sizeof(rpt.cks)
				    + num_cks*sizeof(rpt.cks[0])),
			 op, &resp.hdr, sizeof(resp))) {
		return ASK_GREY_FAIL;
	}

	if (!dcc_ck_grey_answer(emsg, &resp))
		return ASK_GREY_FAIL;

	/* see what the greylist server had to say */
	result_tgts = ntohl(resp.g.triple);
	switch (result_tgts) {
	case DCC_TGTS_OK:		/* embargo ended just now */
		/* if we have previously included this target in a count of
		 * targets sent to the DCC, then do not include it now */
		if (resp.g.msg != 0 && plate_tgts)
			++*plate_tgts;
		if (pembargo_num)
			*pembargo_num = 0;
		return ASK_GREY_EMBARGO_END;

	case DCC_TGTS_TOO_MANY:		/* no current embargo */
		if (pembargo_num)
			*pembargo_num = 0;
		return ((resp.g.msg != 0)
			? ASK_GREY_EMBARGO_END
			: ASK_GREY_PASS);

	case DCC_TGTS_GREY_WHITE:	/* whitelisted for greylisting */
		if (pembargo_num)
			*pembargo_num = 0;
		return ASK_GREY_WHITE;

	default:			/* embargoed */
		/* if this is a brand new embargo,
		 * then count this target in the DCC report */
		if (resp.g.msg == 0 && pearly_tgts)
			++*pearly_tgts;
		if (pembargo_num)
			*pembargo_num = result_tgts+1;
		return ASK_GREY_EMBARGO;
	}
}



/* tell the grey list to restore the embargo on a triple */
u_char
dcc_grey_spam(DCC_EMSG emsg,
	      DCC_CLNT_CTXT *ctxt,
	      const DCC_GOT_CKS *cks,
	      const DCC_SUM msg_sum,
	      const DCC_SUM triple_sum)
{
	DCC_GREY_SPAM gs;
	union {
	    DCC_HDR	hdr;
	    DCC_ERROR	error;
	} resp;

	memset(&gs, 0, sizeof(gs));

	if (cks->sums[DCC_CK_IP].type != DCC_CK_IP) {
		dcc_pemsg(EX_SOFTWARE, emsg,
			  "missing IP checksum for dcc_grey_spam()");
		return 0;
	}

	gs.ip.type = DCC_CK_IP;
	gs.ip.len = sizeof(gs.ip);
	memcpy(&gs.ip.sum, &cks->sums[DCC_CK_IP].sum, sizeof(DCC_SUM));

	gs.triple.type = DCC_CK_GREY3;
	gs.triple.len = sizeof(gs.triple);
	memcpy(&gs.triple.sum, triple_sum, sizeof(DCC_SUM));

	gs.msg.type = DCC_CK_GREY_MSG;
	gs.msg.len = sizeof(gs.msg);
	memcpy(&gs.msg.sum, msg_sum, sizeof(DCC_SUM));

	return dcc_clnt_op(emsg, ctxt, DCC_CLNT_FG_GREY,
			   0, 0, 0, &gs.hdr, sizeof(gs),
			   DCC_OP_GREY_SPAM, &resp.hdr, sizeof(resp));
}
