/* Distributed Checksum Clearinghouse
 *
 * helper processes for DNS blacklists and external filters
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.24 $Revision$
 */

#include "helper.h"
#include "dcc_heap_debug.h"
#include <signal.h>			/* for Linux and SunOS */
#include <sys/wait.h>

u_char helpers_threaded;

#ifdef HAVE_HELPERS

static int helper_terminate_write_fd = -1;
static int helper_terminate_read_fd = -1;

/* have a single handle to kill both external filter and DNSBL helpers,
 * count their failures together. */
static int helper_sn;
static int helper_gen;
static int helper_failures;


/* add to the argv list for the helper processes */
void
helper_save_arg(u_char xfltr, const char *value)
{
	HELPER_PARENT *hp;
	char const **new_arg;
	const char *flag;
	int i;

	if (xfltr) {
		hp = &xfltr_helper_parent;
		flag = "-X";
	} else {
		hp = &dnsbl_helper_parent;
		flag = "-B";
	}

	if (hp->free_args <= 1) {
		/* reserve space for the argv[0] and the null terminator */
		hp->free_args += 5;
		i = (hp->argc + 2*hp->free_args) * sizeof(char *);
		new_arg = dcc_malloc(i);
		memset(new_arg, 0, i);
		if (hp->argv) {
			for (i = 0; i < hp->argc; ++i)
				new_arg[i] = hp->argv[i];
			dcc_free(hp->argv);
		} else {
			++hp->argc;
		}
		hp->argv = new_arg;
	}

	hp->argv[hp->argc] = flag;
	hp->argv[++hp->argc] = value;
	hp->argv[++hp->argc] = 0;
	--hp->free_args;
}



static void
helper_init(HELPER_PARENT *hp, const char *type_str,
	    const char *progpath, int max_jobs, int max_threads, int req_len)
{
	hp->soc = INVALID_SOCKET;
	hp->type_str = type_str;
	hp->max_helpers = max_jobs;
	hp->progpath = progpath;
	hp->max_threads = max_threads;
	hp->req_len = req_len;

	if (max_jobs != 0) {
		hp->pids = dcc_malloc(sizeof(pid_t) * hp->max_helpers);
		memset(hp->pids, 0, sizeof(pid_t) * hp->max_helpers);
	}
}



void
helpers_init(int max_jobs, const char *progpath)
{
	if (!helpers_threaded)
		helpers_threaded = helper_cnt_lock_init();

	helper_init(&xfltr_helper_parent, "external filter",
		    progpath, max_jobs,
#ifdef USE_XFLTR
		    xfltr_threads,
#else
		    0,
#endif
		    sizeof(XFLTR_REQ));

	helper_init(&dnsbl_helper_parent, "DNSBL",
		    progpath, max_jobs, 1, sizeof(DNSBL_REQ));
}



static u_char
reap_helpers_type(HELPER_PARENT *hp, pid_t pid, int status)
{
	int i;

	for (i = 0; ; ++i) {
		if (i >= hp->max_helpers)
			return 0;
		if (hp->pids[i] == pid)
			break;
	}

	hp->pids[i] = 0;
	hp->total_helpers -= hp->max_threads;
	hp->idle_helpers -= hp->max_threads;

#if defined(WIFEXITED) && defined(WEXITSTATUS)
	if (WIFEXITED(status) && WEXITSTATUS(status) == 0)
		return 1;
#endif
#if defined(WTERMSIG) && defined(WIFSIGNALED)
	if (WIFSIGNALED(status)) {
		dcc_error_msg("%s helper process %d quit with signal %d",
			      hp->type_str, (int)pid, WTERMSIG(status));
	} else {
#endif
		dcc_error_msg("%s helper process %d quit with %d",
			      hp->type_str, (int)pid, status);
	}
	return 1;
}



void
reap_helpers(u_char locked)
{
	int status;
	pid_t pid;

	if (!locked)
		helper_cnt_lock();

	while (dnsbl_helper_parent.total_helpers > 0
	       || xfltr_helper_parent.total_helpers > 0) {
		pid = waitpid(0, &status, WNOHANG);
		if (0 >= pid)
			break;

		if (reap_helpers_type(&dnsbl_helper_parent, pid, status))
			continue;
		reap_helpers_type(&xfltr_helper_parent, pid, status);
	}

	if (!locked)
		helper_cnt_unlock();
}



/* must be called with the counter mutex */
static void
terminate_helpers(void)
{
	if (helper_terminate_write_fd >= 0) {
		close(helper_terminate_write_fd);
		helper_terminate_write_fd = -1;
	}
	if (helper_terminate_read_fd >= 0) {
		close(helper_terminate_read_fd);
		helper_terminate_read_fd = -1;
	}
	if (dnsbl_helper_parent.soc != INVALID_SOCKET) {
		closesocket(dnsbl_helper_parent.soc);
		dnsbl_helper_parent.soc = INVALID_SOCKET;
	}
#ifdef USE_XFLTR
	if (xfltr_helper_parent.soc != INVALID_SOCKET) {
		closesocket(xfltr_helper_parent.soc);
		xfltr_helper_parent.soc = INVALID_SOCKET;
	}
#endif
	reap_helpers(1);

	++helper_gen;

	helper_failures = 0;
}



static void
help_finish(HELPER_PARENT *hp, int gen, u_char ok, int counted)
{
	helper_cnt_lock();

	/* forget it if the children have been restarted */
	if (gen == helper_gen) {
		hp->idle_helpers += counted;

		if (!ok) {
			if (++helper_failures >= 5) {
				if (hp->debug)
				    dcc_trace_msg("restart helper processes");
				terminate_helpers();
			} else {
				reap_helpers(1);
			}
		}
	}

	helper_cnt_unlock();
}



static u_char
helper_soc_open(DCC_EMSG emsg, DCC_CLNT_CTXT *ctxt)
{
	if (ctxt->soc == INVALID_SOCKET) {
		dcc_ctxts_lock();
		if (!dcc_info_lock(emsg)) {
			dcc_ctxts_unlock();
			return 0;
		}
		if (!dcc_clnt_soc_reopen(emsg, ctxt)) {
			dcc_ctxts_unlock();
			return 0;
		}
		if (!dcc_info_unlock(emsg)) {
			dcc_ctxts_unlock();
			return 0;
		}
		dcc_ctxts_unlock();
	}

	return 1;
}



static u_char
helper_soc_connect(DCC_EMSG emsg, DCC_CLNT_CTXT *ctxt, const DCC_SOCKU *su)
{
#ifdef linux
	/* since at least some versions of Linux refuse to reconnect,
	 * just disconnect */
	su = 0;
#endif

	dcc_ctxts_lock();
	if (!dcc_info_lock(emsg)) {
		dcc_ctxts_unlock();
		return 0;
	}
	if (!dcc_clnt_connect(emsg, ctxt, su)) {
		dcc_ctxts_unlock();
		return 0;
	}
	if (!dcc_info_unlock(emsg)) {
		dcc_ctxts_unlock();
		return 0;
	}
	dcc_ctxts_unlock();

	return 1;
}



/* open one of the helper sockets
 *	must be called with the counter mutex */
static u_char
open_helper(DCC_EMSG emsg, HELPER_PARENT *hp, DCC_CLNT_CTXT *ctxt, void *lp)
{
	struct in6_addr ipv6_loopback;
	struct in_addr ipv4_loopback;
	DCC_SOCKLEN_T soc_len;
	static int rcvbuf = 32*1024;
	static u_char rcvbuf_set = 0;

	if (!hp->progpath)
		return 1;

	/* We want to create a new socket with the same choice of
	 * IPv4 or IPv6 as the DCC client context's socket.  To do that,
	 * we must ensure that the context's socket is healthy. */
	if (!helper_soc_open(emsg, ctxt)) {
		thr_error_msg(lp, "%s helper soc_open(): %s",
			      hp->type_str, emsg);
		return 0;
	}

	if (ctxt->flags & DCC_CTXT_USING_IPV4) {
		ipv4_loopback.s_addr = ntohl(0x7f000001);
		dcc_mk_su(&hp->su, AF_INET, &ipv4_loopback, 0);
	} else {
		memset(&ipv6_loopback, 0, sizeof(ipv6_loopback));
		ipv6_loopback.s6_addr32[3] = ntohl(1);
		dcc_mk_su(&hp->su, AF_INET6, &ipv6_loopback, 0);
	}
	clean_stdio();
	if (!dcc_udp_bind(emsg, &hp->soc, &hp->su, 0)) {
		thr_error_msg(lp, "%s helper bind(%s): %s",
			      hp->type_str, dcc_su2str_err(&hp->su), emsg);
		terminate_helpers();
		return 0;
	}
	soc_len = sizeof(hp->su);
	if (0 > getsockname(hp->soc, &hp->su.sa, &soc_len)) {
		thr_error_msg(lp, "%s helper getsockname(%d, %s): %s",
			      hp->type_str, hp->soc, dcc_su2str_err(&hp->su),
			      ERROR_STR());
		terminate_helpers();
		return 0;
	}
	for (;;) {
		if (!setsockopt(hp->soc, SOL_SOCKET, SO_RCVBUF,
				&rcvbuf, sizeof(rcvbuf)))
			break;
		if (rcvbuf_set || rcvbuf <= 4096) {
			thr_error_msg(lp, "%s setsockopt(%s,SO_RCVBUF=%d): %s",
				      hp->type_str, dcc_su2str_err(&hp->su),
				      rcvbuf, ERROR_STR());
			break;
		}
		rcvbuf -= 4096;
	}
	rcvbuf_set = 1;
	return 1;
}



/* Open the pipe used to terminate the helper processes
 *	must be called with the counter mutex */
static u_char
ready_helpers(void *lp)
{
	int fds[2];

	if (helper_terminate_write_fd >= 0
	    && helper_terminate_read_fd >= 0)
		return 1;

	terminate_helpers();

	/* give the helper child processes an FD that will go dead
	 * if the parent dies or otherwise closes the other end */
	clean_stdio();
	if (0 > pipe(fds)) {
		thr_error_msg(lp, "parent helper pipe(): %s", ERROR_STR());
		terminate_helpers();
		return 0;
	}
	helper_terminate_write_fd = fds[1];
	helper_terminate_read_fd = fds[0];

	if (-1 == fcntl(helper_terminate_write_fd, F_SETFL,
			fcntl(helper_terminate_write_fd,
			      F_GETFL, 0) | O_NONBLOCK)) {
		thr_error_msg(lp, "helper fcntl(O_NONBLOCK): %s", ERROR_STR());
		terminate_helpers();
		return 0;
	}
	if (0 > fcntl(helper_terminate_write_fd, F_SETFD, FD_CLOEXEC)) {
		thr_error_msg(lp, "helper fcntl(FD_CLOEXEC): %s", ERROR_STR());
		terminate_helpers();
		return 0;
	}

	return 1;
}



/* Start a new helper process.
 *	The counter mutex must be locked */
static u_char
new_helper(DCC_EMSG emsg, HELPER_PARENT *hp, DCC_CLNT_CTXT *ctxt, void *lp)
{
	pid_t pid;
	char arg_buf[sizeof("set:")+sizeof(HELPER_PAT)+8+8+8];
	char trace_buf[200];
	char *bufp;
	SOCKET soc;
	int pid_inx, buf_len, i, j;

	/* open the socket(s) if necessary */
	if (hp->soc == INVALID_SOCKET) {
		if (!ready_helpers(lp))
			return 0;
		if (!open_helper(emsg, hp, ctxt, lp))
			return 0;
	}

	reap_helpers(1);
	pid_inx = 0;
	for (;;) {
		if (pid_inx >= hp->max_helpers)
			dcc_logbad(EX_SOFTWARE, "no free %s pids[] entry",
				   hp->type_str);
		if (!hp->pids[pid_inx])
			break;
		++pid_inx;
	}

	fflush(stdout);
	fflush(stderr);
	pid = fork();
	if (pid < 0) {
		thr_error_msg(lp, "%s helper fork(): %s",
			      hp->type_str, ERROR_STR());
		return 0;
	}

	if (pid != 0) {
		/* this is the parent */
		hp->pids[pid_inx] = pid;
		hp->total_helpers += hp->max_threads;
		hp->idle_helpers += hp->max_threads-1;
		gettimeofday(&hp->started, 0);
		return 1;
	}

	/* This is the child
	 * exec() to get rid of the other threads, file descriptors,
	 * and so forth */

	dcc_rel_priv();			/* no fun or games */
	close(STDIN_FILENO);		/* prevent funny business */
	close(STDOUT_FILENO);		/*	with controlling TTYs */
	close(STDERR_FILENO);
	clean_stdio();

	/* reset FD_CLOEXEC without affecting parent */
	soc = dup(hp->soc);
	if (soc == INVALID_SOCKET) {
		thr_error_msg(lp, "%s helper soc dup(%d): %s",
			      hp->type_str, hp->soc, ERROR_STR());
		return 0;
	}
	snprintf(arg_buf, sizeof(arg_buf), "set:"HELPER_PAT,
		 soc, helper_terminate_read_fd);
	helper_save_arg(hp == &xfltr_helper_parent, arg_buf);
	hp->argv[0] = hp->progpath;
	buf_len = sizeof(trace_buf);
	bufp = trace_buf;
	for (i = 0; i < hp->argc && buf_len > 2; ++i) {
		j = snprintf(bufp, buf_len, " %s", hp->argv[i]);
		buf_len -= j;
		bufp += j;
	}
	if (hp->debug >= 4)
		dcc_trace_msg("about to start%s", trace_buf);

	execv(hp->argv[0], (char * const *)hp->argv);
	/* This process should continue at helper_child() */

	dcc_logbad(EX_UNAVAILABLE, "exec(%s): %s",
		   trace_buf, ERROR_STR());
}



/* a helping child needs to send its results to its parent */
static void
helper_respond(const HELPER_CHILD *hp, const DCC_SOCKU *parent_su,
	       const HELPER_REQ_HDR *req,
	       const HELPER_RESP_HDR *resp, int resp_len)
{
	struct timeval now;
	int i;

	/* do not answer if it is too late */
	gettimeofday(&now, 0);
	if (tv_diff2us(&now, &req->start) > req->total_usecs) {
		if (hp->parent->debug > 1)
			dcc_trace_msg("%s %s helper: too late to answer",
				      req->id, hp->parent->type_str);
		return;
	}

	i = sendto(hp->soc, resp, resp_len, 0,
		   &parent_su->sa, DCC_SU_LEN(parent_su));
	if (i != resp_len) {
		if (i < 0)
			dcc_error_msg("%s %s helper sendto(%s): %s",
				      req->id,
				      hp->parent->type_str,
				      dcc_su2str_err(parent_su),
				      ERROR_STR());
		else
			dcc_error_msg("%s %s helper sendto(%s)=%d",
				      req->id,
				      hp->parent->type_str,
				      dcc_su2str_err(parent_su),
				      i);
	}
}



static void NRATTRIB
helper_thread_exit(HELPER_CHILD *hp, const char *reason, u_char error)
{
	u_char num;

	helper_cnt_lock();
	num = hp->parent->cur_threads--;
	helper_cnt_unlock();

	if (error) {
		if (num <= 1) {
			dcc_error_msg("%s helper process on %s %s",
				      hp->parent->type_str,
				      dcc_su2str_err(&hp->su), reason);
		} else {
			dcc_error_msg("%s helper thread %d on %s %s",
				      hp->parent->type_str, num,
				      dcc_su2str_err(&hp->su), reason);
		}

	} else if (hp->parent->debug > 1) {
		if (num <= 1) {
			dcc_trace_msg("%s helper process on %s %s",
				      hp->parent->type_str,
				      dcc_su2str_err(&hp->su), reason);
		} else if (hp->parent->debug > 2) {
			dcc_trace_msg("%s helper thread %d on %s %s",
				      hp->parent->type_str, num,
				      dcc_su2str_err(&hp->su), reason);
		}
	}

#ifdef USE_XFLTR
	if (hp->parent == &xfltr_helper_parent)
		xfltr_old_thread(num <= 1);
	else
#endif
		exit(0);
}



/* helper threads loop here until they die */
void * NRATTRIB
helper_thread(void *hp0)
{
	HELPER_CHILD *hp = hp0;
	HELPER_PARENT *hparent = hp->parent;
	HELPER_CHILD *hp2;
	struct timeval delay;
	union {
	    HELPER_REQ_HDR  hdr;
	    DNSBL_REQ	    dnsbl;
	    XFLTR_REQ	    xfltr;
	} req;
	union {
	    HELPER_RESP_HDR  hdr;
	    DNSBL_RESP	    dnsbl;
	    XFLTR_RESP	    xfltr;
	} resp;
	fd_set rfds;
	DCC_SOCKU parent_su;
	DCC_SOCKLEN_T su_len;
	int req_len;
	struct timeval now;
	time_t last_work = time(0);
	int i;

	if (hparent->debug > 1) {
		if (hparent->cur_threads == 1) {
			dcc_trace_msg("%s helper process starting on %s",
				      hparent->type_str,
				      dcc_su2str_err(&hp->su));
		} else if (hparent->debug > 2) {
			dcc_trace_msg("%s helper thread %d starting on %s",
				      hparent->type_str,
				      hparent->cur_threads,
				      dcc_su2str_err(&hp->su));
		}
	}
#ifdef USE_XFLTR
	if (hparent == &xfltr_helper_parent
	    && !xfltr_new_thread(hparent->cur_threads == 1))
		helper_thread_exit(hp, "start failed", 1);
#endif

	hp2 = 0;
	FD_ZERO(&rfds);
	for (;;) {
		helper_cnt_lock();
		++hparent->idle_helpers;
		helper_cnt_unlock();

		helper_recv_lock();
		FD_SET(helper_terminate_read_fd, &rfds);
		if (hp2) {
			/* give new new thread a chance to run first */
			hp2 = 0;
			FD_CLR(hp->soc, &rfds);
			delay.tv_sec = 0;
			delay.tv_usec = DCC_USECS/30;	/* <0 new threads/sec */
		} else {
			FD_SET(hp->soc, &rfds);
			delay.tv_sec = 60;
			delay.tv_usec = 0;
		}
		i = select(max(hp->soc, helper_terminate_read_fd)+1,
			   &rfds, 0, 0, &delay);
		gettimeofday(&now, 0);
		helper_cnt_lock();
		--hparent->idle_helpers;

		if (i < 0) {
			helper_cnt_unlock();
			helper_recv_unlock();
			if (DCC_SELECT_NERROR())
				continue;
			dcc_logbad(EX_OSERR, "%s helper select(): %s",
				   hparent->type_str, ERROR_STR());
		}
		if (i == 0) {
			helper_cnt_unlock();
			helper_recv_unlock();
			if (now.tv_sec >= last_work+5*60
			    || (now.tv_sec >= last_work+60
				&& hparent->cur_threads > 2))
				helper_thread_exit(hp, "idle, exiting", 0);
			continue;
		}

		if (FD_ISSET(helper_terminate_read_fd, &rfds)) {
			helper_cnt_unlock();
			helper_recv_unlock();
			helper_thread_exit(hp, "shutdown", 0);
		}
#ifdef USE_XFLTR
		if (hparent->idle_helpers == 0
		    && hparent->cur_threads < hparent->max_threads) {
			hp2 = dcc_malloc(sizeof(*hp2));
			memset(hp2, 0, sizeof(*hp2));
			hp2->parent = &xfltr_helper_parent;
			hp2->soc = hp->soc;
			hp2->su = hp->su;
			if (xfltr_thread_create(hparent->type_str, hp2)) {
				++hparent->cur_threads;
				helper_cnt_unlock();
				helper_recv_unlock();
				continue;
			}
		}
#endif
		su_len = sizeof(parent_su);
		req_len = recvfrom(hp->soc, &req, ISZ(req), 0,
				   &parent_su.sa, &su_len);
		helper_cnt_unlock();
		helper_recv_unlock();

		/* we might get stray packets because we cannot connect to
		 * a single port */
		if (!DCC_SU_EQ(&hp->su, &hp->su)) {
			if (hparent->debug)
				dcc_trace_msg("%s request from"
					      " %s instead of %s",
					      hparent->type_str,
					      dcc_su2str_err(&parent_su),
					      dcc_su2str_err(&hp->su));
			continue;
		}
		if (req_len != hparent->req_len) {
			if (req_len < 0) {
				if (!DCC_BLOCK_ERROR())
					dcc_logbad(EX_IOERR,
						   "%s recvfrom(parent): %s",
						   hparent->type_str,
						   ERROR_STR());
				continue;
			}
			if (hparent->debug)
				dcc_trace_msg("%s recvfrom(parent %s)=%d"
					      " instead of %d",
					      hparent->type_str,
					      dcc_su2str_err(&parent_su),
					      req_len, hparent->req_len);
			continue;
		}
		if (req.hdr.magic != HELPER_MAGIC_REQ) {
			if (hparent->debug)
				dcc_trace_msg("%s recvfrom(parent %s)"
					      " magic=%#08x",
					      hparent->type_str,
					      dcc_su2str_err(&parent_su),
					      req.hdr.magic);
			continue;
		}

		last_work = now.tv_sec;

		/* forget it if it is already too late to answer,
		 * perhaps because a previous helper died */
		if (tv_diff2us(&now, &req.hdr.start) > req.hdr.total_usecs) {
			if (hparent->debug > 1)
				dcc_trace_msg("%s %s: too late to consider",
					      req.hdr.id,
					      hparent->type_str);
			continue;
		}

		memset(&resp, 0, sizeof(resp));
		resp.hdr.magic = HELPER_MAGIC_RESP;
		resp.hdr.sn = req.hdr.sn;

#ifdef USE_XFLTR
		if (hparent == &xfltr_helper_parent) {
			if (xfltr_work(&req.xfltr, &resp.xfltr))
				helper_respond(hp, &parent_su, &req.hdr,
					       &resp.hdr, sizeof(resp.xfltr));
			continue;
		}
#endif
		if (dnsbl_work(&req.dnsbl, &resp.dnsbl))
			helper_respond(hp, &parent_su, &req.hdr,
				       &resp.hdr, sizeof(resp.dnsbl));
	}
}



/* helper processes start here via fork()/exec() in the parent  */
void NRATTRIB
helper_child(u_char xfltr, SOCKET soc, int fd)
{
	HELPER_CHILD *hp;
	DCC_SOCKLEN_T soc_len;

	if (!helper_cnt_lock_init()
	    || !helper_recv_lock_init())
		dcc_logbad(EX_SOFTWARE, "no threads for helpers");

	signal(SIGHUP, SIG_DFL);
	signal(SIGINT, SIG_DFL);
	signal(SIGTERM, SIG_DFL);
	signal(SIGUSR1, SIG_DFL);
	signal(SIGUSR2, SIG_DFL);

	helpers_init(0, 0);
	hp = dcc_malloc(sizeof(*hp));
	memset(hp, 0, sizeof(*hp));
	hp->parent = xfltr ? &xfltr_helper_parent : &dnsbl_helper_parent;
	hp->parent->cur_threads = 1;

	/* Ensure FDs are small enough for select() */
	if (fd < (int)FD_SETSIZE) {
		helper_terminate_read_fd = fd;
	} else {
		helper_terminate_read_fd = dup(fd);
		if (helper_terminate_read_fd < 0)
			dcc_logbad(EX_IOERR, "helper dup pipe(%d): %s",
				   fd, ERROR_STR());
		close(fd);
	}
	if (soc < (int)FD_SETSIZE) {
		hp->soc = soc;
	} else {
		hp->soc = dup(soc);
		if (hp->soc < 0)
			dcc_logbad(EX_IOERR, "helper dup soc(%d): %s",
				   soc, ERROR_STR());
		close(soc);
	}
	soc_len = sizeof(hp->su);
	if (0 > getsockname(hp->soc, &hp->su.sa, &soc_len))
		dcc_logbad(EX_IOERR, "helper getsockname(%d): %s",
			   hp->soc, ERROR_STR());
	helper_thread(hp);
}



/* ask a helper process to do some filtering */
u_char					/* 1=got an answer */
ask_helper(DCC_CLNT_CTXT *ctxt, u_char xfltr, void *log_ctxt,
	   const struct timeval *start, time_t total_usecs,
	   HELPER_REQ_HDR *req, int req_len,    /* request sent to helper */
	   HELPER_RESP_HDR *resp, int resp_len) /* put answer here */
{
	DCC_EMSG emsg;
	HELPER_PARENT *hp;
	DCC_SOCKU send_su;
	DCC_SOCKLEN_T su_len;
	DCC_SOCKU recv_su;
	char sustr[DCC_SU2STR_SIZE];
	int counted;
	int gen;
	struct timeval now;
	time_t usecs;
	int i;

	emsg[0] = '\0';

	hp = xfltr ? &xfltr_helper_parent : &dnsbl_helper_parent;

	req->magic = HELPER_MAGIC_REQ;
	req->start = *start;
	req->total_usecs = total_usecs;

	helper_cnt_lock();
	req->sn = ++helper_sn;
	counted = 1;
	if (hp->idle_helpers > 0) {
		/* avoid taking the last idle helper because there are
		 * usually fewer truly idle helpers than we think because
		 * we don't wait for them to finish */
		if (hp->idle_helpers < 2
		    && hp->total_helpers < hp->max_helpers
		    && tv_diff2us(&req->start, &hp->started) > 10*1000)
			new_helper(emsg, hp, ctxt, log_ctxt);
		else
			--hp->idle_helpers;
	} else if (hp->total_helpers >= hp->max_helpers) {
		counted = 0;
		if (hp->debug > 0)
			thr_trace_msg(log_ctxt, "%s no idle helpers", req->id);
	} else {
		if (!new_helper(emsg, hp, ctxt, log_ctxt)) {
			helper_cnt_unlock();
			return 0;
		}
	}
	/* snapshot the address in case another thread restarts the children */
	send_su = hp->su;
	gen = helper_gen;
	helper_cnt_unlock();

	/* Use sendto() if the socket is not already conencted.
	 * If it is already connected, then on many systems other than Linux,
	 * it is possible and presumably cheap to reconnected it, so do so. */
	if (ctxt->conn_su.sa.sa_family != AF_UNSPEC
	    && memcmp(&ctxt->conn_su, &send_su, sizeof(ctxt->conn_su))
	    && !helper_soc_connect(emsg, ctxt, &send_su)) {
		thr_trace_msg(log_ctxt, "soc_connect(): %s", emsg);
		help_finish(hp, gen, 0, counted);
		return 0;
	}
	if (ctxt->conn_su.sa.sa_family == AF_UNSPEC) {
		i = sendto(ctxt->soc, req, req_len, 0,
			   &send_su.sa, DCC_SU_LEN(&send_su));
	} else {
		i = send(ctxt->soc, req, req_len, 0);
	}
	if (i != req_len) {
		if (hp->debug) {
			if (i < 0)
				thr_trace_msg(log_ctxt, "%s sendto(%s): %s",
					      req->id,
					      dcc_su2str(sustr, sizeof(sustr),
							&send_su),
					      ERROR_STR());
			else
				thr_trace_msg(log_ctxt, "%s sendto(%s)=%d",
					      req->id,
					      dcc_su2str(sustr, sizeof(sustr),
							&send_su),
					      i);
		}
		help_finish(hp, gen, 0, counted);
		return 0;
	}

	for (;;) {
		gettimeofday(&now, 0);
		usecs = total_usecs - tv_diff2us(&now, &req->start);
		if (usecs < 0)
			usecs = 0;
		i = dcc_select_poll(0, ctxt->soc, 1, usecs);
		if (i < 0) {
			thr_error_msg(log_ctxt, "%s select_poll: %s",
				      req->id, ERROR_STR());
			help_finish(hp, gen, 0, counted);
			return 0;
		}
		if (i == 0) {
			if (hp->debug)
				thr_trace_msg(log_ctxt,
					      "%s no %s helper answer",
					      req->id, hp->type_str);
			help_finish(hp, gen, 0, counted);
			return 0;
		}

		su_len = sizeof(recv_su);
		i = recvfrom(ctxt->soc, resp, resp_len,
			     0, &recv_su.sa, &su_len);
		/* because we are using UDP, we might get stray packets */
		if (i != resp_len) {
			if (i < 0) {
				thr_trace_msg(log_ctxt,
					      "%s recvfrom(%s): %s",
					      req->id, hp->type_str,
					      ERROR_STR());
				if (DCC_BLOCK_ERROR())
					continue;
				help_finish(hp, gen, 0, counted);
				return 0;
			}
			if (hp->debug > 1)
				thr_trace_msg(log_ctxt,
					      "%s recvfrom(%s %s)=%d",
					      req->id, hp->type_str,
					      dcc_su2str_err(&recv_su), i);
			continue;
		}
		if (!DCC_SU_EQ(&send_su, &recv_su)) {
			if (hp->debug != 0)
				thr_trace_msg(log_ctxt, "%s recvfrom(%s %s)"
					      " instead of %s",
					      req->id, hp->type_str,
					      dcc_su2str_err(&recv_su),
					      dcc_su2str_err(&send_su));
			continue;
		}
		if (resp->magic != HELPER_MAGIC_RESP
		    || resp->sn != req->sn) {
			if (hp->debug >1 )
				thr_trace_msg(log_ctxt, "%s recvfrom(%s %s)"
					      " magic=%#08x sn=%d",
					      req->id, hp->type_str,
					      dcc_su2str_err(&recv_su),
					      resp->magic, resp->sn);
			continue;
		}

		help_finish(hp, gen, 1, counted);
		return 1;
	}
}
#endif /* HAVE_HELPERS */
