/* Distributed Checksum Clearinghouse
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.22 $Revision$
 */

#include "dcc_defs.h"


/* Get a "word", while honoring backslashes
 *	on exit the remainder of the line is trimmed of leading whitespace */
const char *				/* 0 or remainder of line */
dcc_parse_word(DCC_EMSG emsg,
	       char *tgt,		/* copy word to here if not null */
	       int tgt_len,		/* includes trailing '\0' */
	       const char *line,	/* line of words */
	       const char *fieldname,
	       const char *fnm, int lno)
{
	DCC_FNM_LNO_BUF fnm_buf;
	const char *p;
	char c;

	if (!tgt_len && tgt)
		dcc_logbad(EX_SOFTWARE, "bad tgt_len for dcc_get_word(%s)%s",
			   fieldname, fnm_lno(fnm_buf, fnm, lno));

	if (!line) {
		if (tgt)
			*tgt = '\0';
		if (fieldname)
			dcc_pemsg(EX_USAGE, emsg, "%s missing%s",
				  fieldname, fnm_lno(fnm_buf, fnm, lno));
		return 0;
	}

	line = line+strspn(line, DCC_WHITESPACE);   /* skip leading blanks */

	p = line;
	do {
		c = *p;

		if (c != '\0') {
			++p;
			if (c == '\\' && *p != '\0') {
				/* recognize and convert escape sequences to
				 * their real equivalents */
				if ((c = *p++) == 'n') {
					c = '\n';
				} else if (c == 'r') {
					c = '\r';
				} else if (c == 't') {
					c = '\t';
				} else if (c == 'b') {
					c = '\b';
				} else if (c >= '0' && c <= '7') {
					c -= '0';
					if (*p >= '0' && *p <= '7') {
					    c = (c<<3)+(*p++ - '0');
					    if (*p >= '0' && *p <= '7')
						c = (c<<3)+(*p++ - '0');
					}
				}
			} else if (strchr(DCC_WHITESPACE, c)) {
				/* Stop on the first whitespace or delimiter */
				c = '\0';
				/* Skip trailing whitespace */
				p += strspn(p, DCC_WHITESPACE);
			}
		}

		if (tgt) {
			if (!tgt_len) {
				if (fieldname)
					dcc_pemsg(EX_USAGE, emsg,
						  "%s \"%.*s...\" too long%s",
						  fieldname,
						  min(16, (int)(p-line)), line,
						  fnm_lno(fnm_buf, fnm, lno));
				return 0;
			}
			*tgt++ = c;
			--tgt_len;
		}
	} while (c != '\0');

	return p;
}



/* get a password */
const char *				/* 0 or remainder of line */
parse_passwd(DCC_EMSG emsg,
	     DCC_PASSWD passwd,		/* copy password here */
	     const char *line,		/* line of words */
	     const char *fieldname,
	     const char *fnm, int lno)
{
	char buf[sizeof(DCC_PASSWD)+1];
	const char *result;

	memset(buf, 0, sizeof(buf));
	result = dcc_parse_word(emsg, buf, sizeof(buf),
				line, fieldname, fnm, lno);
	if (result)
		memcpy(passwd, buf, sizeof(DCC_PASSWD));
	else
		memset(passwd, 0, sizeof(DCC_PASSWD));
	return result;
}



/* see if a string starts with a word possibly followed by a comma */
u_char					/* 1=yes */
dcc_ck_word_comma(char **parg, const char *word)
{
	u_int word_len = strlen(word);
	char *arg = *parg;

	if (strncasecmp(arg, word, word_len))
		return 0;
	arg += word_len;
	if (*arg == '\0') {
		*parg = arg;
		return 1;
	}
	if (*arg == ',') {
		*parg = arg+1;
		return 1;
	}
	return 0;
}
