/* Distributed Checksum Clearinghouse
 *
 * common server definitions
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.62 $Revision$
 */

#ifndef SRVR_DEFS_H
#define SRVR_DEFS_H

#include "dcc_clnt.h"
#include "dcc_ids.h"
#include "dcc_heap_debug.h"
#include "db.h"


typedef struct {
    struct timeval reset;
    u_int   nops;			/* DCC_OP_NOP packets */
    u_int   reports;			/* DCC_OP_REPORT or DCC_OP_REPORT_OLD */
    u_int   report_retrans;		/* duplicate reports */
    u_int   report_reject;		/* reports ignored by -Q */
    u_int   report10;			/* reports of >10 targets */
    u_int   report100;			/* reports of >100 targets */
    u_int   report1000;			/* reports of >1000 targets */
    u_int   reportmany;			/* reports of spam */
    u_int   queries;			/* DCC_OP_QUERY */
    u_int   resp10;			/* responses of >10 targets */
    u_int   resp100;			/*   "       " >100 targets */
    u_int   resp1000;			/*   "       " >1000 targets */
    u_int   respmany;			/*   "       " spam */
    u_int   respwhite;			/* white-listed responses */
    u_int   unknown_ids;		/* packets with unknown IDs */
    u_int   bad_passwd;			/* packets with bad passwords */
    u_int   send_errors;		/* error responses sent */
    u_int   admin;			/* DCC_OP_ADMN */
    u_int   rl;				/* responses rate-limited */
    u_int   anon_rl;			/* anonymous responses rate-limited */
    u_int   adds;			/* reports added */
    u_int   iflod_total;		/* total reports received */
    u_int   iflod_accepted;		/* timely and properly signed */
    u_int   iflod_stale;
    u_int   iflod_dup;
    u_int   iflod_ok2;			/* locally whitelisted */
    u_int   iflod_not_deleted;		/* delete commands ignored */
    u_int   norep;
    u_int   rep1;
    u_int   rep10;
    u_int   rep20;
    u_int   rep30;
    u_int   rep60;
} DCCD_STATS;



typedef struct {
    time_t	ok;			/* assumed ok until then */
#    define	 LAST_ERROR_OK_SECS (60*60*2)
#    define	 LAST_ERROR_ACT_SECS (FLOD_RETRY_SECS*3)
    time_t	rep_report;		/* resume error messages after this */
#    define	 LAST_ERROR_SECS    (60*60*24)
    int		old_errno;
    enum FLOD_ERR_OP {
	FLOD_ERR_OK = 0,
	FLOD_ERR_SAME,
	FLOD_ERR_NO_LINK,
	FLOD_ERR_GET_HOST,
	FLOD_ERR_CONNECT,
	FLOD_ERR_CONNECT2,
	FLOD_ERR_IO,
	FLOD_ERR_DUP,
	FLOD_ERR_LOCAL_OFF,
	FLOD_ERR_REMOTE_OFF,
	FLOD_ERR_ID,
	FLOD_ERR_SIGN,
	FLOD_ERR_BAD_VERS,
	FLOD_ERR_REP,
	FLOD_ERR_BAD_DATA,
	FLOD_ERR_PEER_FAIL,
	FLOD_ERR_KEEPALIVE
    } op;
} LAST_ERROR;


/* memory mapped file of flooding information
 *  it is memory mapped so that dblist can report the state of flooding
 *  and so dbclean can see when flooding has stopped */

#define GREY_FLOD_NM	"grey_flod"
#define DCCD_FLOD_NM	"flod"
extern DCC_PATH flod_mmap_path, flod_path;

#ifndef DCCD_MAX_FLOODS
#define DCCD_MAX_FLOODS	32
#endif

typedef struct {
    char	rem_hostname[MAXHOSTNAMELEN];
    char	rem_portname[MAXPORTNAMELEN+1];
    u_int16_t	rem_port;
    DCC_SOCKU	rem_su;			/* address for hostname */
    int		host_error;		/* for failure to resolve hostname */
    DB_PTR	confirm_pos;		/* confirmed sent to here */
    int		oflod_index;
    DCC_SRVR_ID	rem_id;
    u_int16_t	flags;
#    define	 FLOD_MMAP_FG_REWINDING	    0x0001  /* answering rewind */
#    define	 FLOD_MMAP_FG_NEED_REWIND   0x0002  /* database purged */
#    define	 FLOD_MMAP_FG_PASSIVE	    0x0004  /* ignore hostname */
#    define	 FLOD_MMAP_FG_SOCKS	    0x0008  /* use Rgethostbyname() */
#    define	 FLOD_MMAP_FG_PASSWD_NEXT   0x0010  /* use second password */
#    define	 FLOD_MMAP_FG_FFWD_IN	    0x0020  /* want fastforward */
#    define	 FLOD_MMAP_FG_IPv4	    0x0040  /* override IPv6 choice */
#    define	 FLOD_MMAP_FG_IPv6	    0x0080  /* override IPv6 choice */
#    define	 FLOD_MMAP_FG_FLOD_REP_OFF  0x0100  /* off in flod file */
#    define	 FLOD_MMAP_FG_PEER_REP_OFF  0x0200  /* turned off by peer */
#    define	 FLOD_MMAP_FG_LEAF	    0x0400  /* path length restricted */
#    define	 FLOD_MMAP_FG_MAPPED	    0x0800  /* server-IDs translated */
#    define	 FLOD_MMAP_FG_IN_CONN	    0x1000  /* input connected */
#    define	 FLOD_MMAP_FG_OUT_CONN	    0x1000  /* input connected */
    struct {
	time_t	    cnts_cleared;
	time_t	    in_conn_changed;
	time_t	    out_conn_changed;
	int	    out_total_conn;	/* seconds connected */
	int	    out_reports;
	int	    in_total_conn;	/* seconds connected */
	int	    total;		/* total reports received */
	int	    accepted;
	int	    stale;		/* too old or in the future */
	int	    dup;		/* already received */
	int	    ok2;		/* whitelisted */
	int	    not_deleted;	/* delete commands ignored */
    } cnts;
    LAST_ERROR	o_err;
    LAST_ERROR	i_err;
} FLOD_MMAP;

typedef struct {
    char	magic[32];
#    define	 FLOD_MMAP_MAGIC	"DCC flod map version 10"
    char	pad[32-sizeof(DB_PTR)];
    DB_PTR	delay_pos;		/* delay flooding newer records this */
    DB_SN	sn;			/* ensure match with database */
    FLOD_MMAP	mmaps[DCCD_MAX_FLOODS];
    DCCD_STATS	dccd_stats;
} FLOD_MMAPS;
extern FLOD_MMAPS *flod_mmaps;


static inline void
dcc_secs2ts(DCC_TS ts, time_t secs)
{
	u_int64_t t;

	t = ((u_int64_t)secs) << DCC_TS_SEC_LSHIFT;
	ts[0] = t>>40; ts[1] = t>>32;
	ts[2] = t>>24; ts[3] = t>>16; ts[4] = t>>8; ts[5] = t;
}

static inline void
dcc_timeval2ts(DCC_TS ts, const struct timeval *tv, int delta_secs)
{
	u_int64_t t;

	t = ((u_int64_t)tv->tv_sec+delta_secs) << DCC_TS_SEC_LSHIFT;
	t += tv->tv_usec >> DCC_TS_USEC_RSHIFT;
	ts[0] = t>>40; ts[1] = t>>32;
	ts[2] = t>>24; ts[3] = t>>16; ts[4] = t>>8; ts[5] = t;
}

static inline void
dcc_ts2timeval(struct timeval *tv, const DCC_TS ts)
{
	u_int64_t t = ((((u_int64_t)ts[0])<<40)
		       + (((u_int64_t)ts[1])<<32)
		       + (((u_int64_t)ts[2])<<24)
		       + (((u_int64_t)ts[3])<<16)
		       + (((u_int64_t)ts[4])<<8)
		       + ts[5]);
	tv->tv_sec = t >> DCC_TS_SEC_LSHIFT;
	tv->tv_usec = ((t &  DCC_TS_USEC_MASK) << DCC_TS_USEC_RSHIFT);
}

#define DCC_TS_NEWER_TS(ts1,ts2) (memcmp(ts1, ts2, sizeof(DCC_TS)) > 0)
#define DCC_TS_OLDER_TS(ts1,ts2) (memcmp(ts1, ts2, sizeof(DCC_TS)) < 0)

#define DB_TYPE2STR(t) dcc_type2str_err(t,0,1,grey_on)

/* not thread safe */
extern DCC_PATH db_path_buf;
#define DB_NM2PATH_ERR(nm) fnm2path_err(db_path_buf, nm)


extern void flod_mmap_path_set(void);
extern u_char flod_mmap_sync(DCC_EMSG, u_char);
extern u_char flod_unmap(DCC_EMSG, const DCCD_STATS *);
extern u_char flod_mmap(DCC_EMSG, const DB_SN, const DCCD_STATS *,
			u_char, u_char);
extern const char *flod_stats_printf(char *, int, int, int, int, int);
extern const char *flod_mmap_fg(char *, int, const char *, const FLOD_MMAP *);
extern int flod_running(const char *);

extern int read_db(DCC_EMSG, void *, u_int, int, off_t, const char *);
extern u_char read_db_hdr(DCC_EMSG, DB_HDR *, int fd, const char *);
extern void read_rcd_invalidate(void);
extern int read_rcd(DCC_EMSG, DB_RCD *, int, off_t, const char *);
#ifdef DB_VERSION2_STR
extern int read_old_rcd(DCC_EMSG, DB_OLD_RCD *, int, const char *);
#endif

extern char *dcc_ts2str(char *, u_int, const DCC_TS);
extern const char *dcc_ts2str_err(const DCC_TS);
extern char *dcc_srvr_id2str(char *, u_int, DCC_SRVR_ID);


#endif /* SRVR_DEFS_H */
