/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * dc_com.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: dc_com.c,v 2.1 2003/03/14 13:59:23 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <glib.h>

#include "dc_com.h"

char last_cmd[5120];			/* contains the first string of the last send_dc_line call */
									/* mainly used for debug features.                         */
time_t last_cmd_time;

/****************************************************/
/* write the given strings on the socket            */
/* a space will be written between each string      */
/* a | will be added at the end                     */
/***********************************************************/
/* to avoid network problem, we use send instead of write, */
/* cf. get_a_dc_line comment to understand why             */
/***********************************************************/
void send_dc_line(int sck,...)
{
	va_list ap;
	char *t;
	int have=0;
	GString *str;
	str=g_string_new(NULL);

	if(sck==-1)		/* sometimes, after a network error, it is possible to come here with an invalid socket descriptor */
		return;

	va_start(ap,sck);
	t=va_arg(ap,char *);

	strcpy(last_cmd,t);

	while(t!=NULL)
	{
		if(have!=0)
		{	/* add a space between fields */
			str=g_string_append_c(str,' ');
		}
		have=1;

		str=g_string_append(str,t);

		t=va_arg(ap,char *);
	}

	if(have==1)
	{
		str=g_string_append_c(str,'|');
	}

	if(str->len)
	{
#ifndef MSG_NOSIGNAL
	  void (*old_handler)(int);
	  old_handler = signal (SIGPIPE, SIG_IGN);
#endif
	  if(send(sck,str->str,str->len,
#ifdef MSG_NOSIGNAL
			MSG_NOSIGNAL
#else
			0
#endif
			)!=str->len)
		{
			/* abort network operation on this socket */
			/* this will either generated a hub_disconnection message (main thread) */
			/* or leave xfer threads on network error. */
			shutdown(sck,2);
		}
#ifndef MSG_NOSIGNAL
		signal (SIGPIPE, old_handler);
#endif

		last_cmd_time=time(NULL);
	}

#if 0
	disp_msg(DEBUG_MSG,"send_dc_line","Sent",str->str,NULL);
#endif

	g_string_free(str,TRUE);
	va_end(ap);
}

/**************************************/
/* get a DC line (always ends by a |) */
/************************************************************************/
/* theory: DC client used send (not write) to send data over network    */
/*         so there is (invisible) message boundary inside network data */
/*         Thus, if you use recv to obtain data, we are sure to obtain  */
/*         a line ending by a |. However, we can receive more than one  */
/*         line at the same time.                                       */
/* real: to avoid buffer overflow and to be sure to have exactly one    */
/*       line, we read byte by byte network data. It is less efficient  */
/*       but it is safer and who cares about efficency to get 50 bytes. */
/*******************************************************************************/
/* input: sck= socket to read from                                             */
/* output: NULL or a GString (must be freed when no more used) ending with a | */
/*******************************************************************************/
GString *get_a_dc_line(int sck)
{
	GString *s;
	int n;
	char c;
	fd_set inp;
#ifndef DC_INPUT_NO_TIMEOUT
	struct timeval tv;
#endif
	int retval;


	if(sck==-1)		/* sometimes, after a network error, it is possible to come here with an invalid socket descriptor */
		return NULL;

	s=g_string_sized_new(512);

	do
	{
		FD_ZERO(&inp);
		FD_SET(sck,&inp);
#ifndef DC_INPUT_NO_TIMEOUT
		tv.tv_sec=30;
		tv.tv_usec=0;
		retval=select(sck+1,&inp,NULL,NULL,&tv);
#else
		retval=select(sck+1,&inp,NULL,NULL,NULL);
#endif

		if((retval==-1)&&(errno==EINTR))
			continue;
		if((retval==-1)||(!FD_ISSET(sck,&inp)))
		{
			perror("get_a_dc_line - select");
			printf("%s\n",s->str);
			g_string_free(s,TRUE);
			return NULL;
		}
	
		n=read(sck,&c,1);
		if(n!=1)
		{
			printf("%s\n",s->str);
			g_string_free(s,TRUE);
			return NULL;
		}

		s=g_string_append_c(s,c);
	}while(c!='|');

#if 0
	disp_msg(DEBUG_MSG,"get_a_dc_line","Received",s->str,NULL);
#endif
	return s;
}

