#! /usr/bin/perl

# $Id: simple_myinfo_handler.pl,v 1.4 2003/07/05 13:22:21 blusseau Exp $

# Used db keys: MIN_SHARE, HUB_MASTER, HUB_DIR, LANGUAGE. All four keys are strings.
#
# HUB_MASTER shall contain the nick of the hub master, probably you. It can contain any string, but many messages
# will then be lost.
#
# HUB_DIR specifyes the perl conf directory, this is not the bin-dir, but the dir where you keep all the conf for
# your perl scripts. The default value of HUB_DIR is /etc/dchub/perl
#
# The key LANGUAGE defines which language that will be used in messages (ie locale), ex: fr_FR, hu_HU, etc..
#
# MIN_SHARE contains the minimal shared size in bytes.
# the MIN_SHARE requirement is not used against hub OP or Master, only with standard user.
#
# To add key to the database, use "-dbadd KEYNAME str xxx" where KEYNAME is the key's name and xxx is the wanted value.

# Files used: ${HUB_DIR}/rules ${HUB_DIR}/ipmask ${HUB_DIR}/prefixchk

# ${HUB_DIR}/rules:
#	This file contain text message that is display when a user log on the hub.
#
# ${HUB_DIR}/ipmask:
#	This file contain ip like addresses (one by line) that are use to test if the user
#	is allow to enter the hub. You can use IP addresses made of 1,2,3 or 4 bytes like: 196.23.
#
# ${HUB_DIR}/prefixchk:
#	This file contain regular expressions (one by line) that are use to test if the nick of the user is valid.
#	For example, to allow nick that begin with [HUT] to log into the hub, put a string like this: ^\[HUT\]
#	in the prefixchk file. Be carefull regular expressions are case sensitive.

# If you change one of this files reload the perl interpreter with the -perlclear command.

use strict;
our $HUB_DIR;
our $RULES;
our @IP_MASK;
our @PREFIXCHK;

add_hook ("myinfo",\&simple_myinfo_handler);

BEGIN {
	my $default_HUB_DIR='/etc/dchub/perl';
	$HUB_DIR=dchub::db_get("HUB_DIR");
	if (not defined $HUB_DIR) {
		$HUB_DIR=$default_HUB_DIR if (-d "$default_HUB_DIR");
	}
	if (defined $HUB_DIR) {
		# Load rules text in RULES if the file exist
		if (open(RULESFILE, "<$HUB_DIR/rules")) {
			$RULES='';
			while (<RULESFILE>) {
				$RULES.=$_;
			}
			close(RULESFILE);
			$RULES=~s/\n/\r\n/g;
		}
		# Load all the ipmask in IPMASK if the file exist
		@IP_MASK=();
		if (open(IPMASKFILE, "<$HUB_DIR/ipmask")) {
			$/ = "\n";			# Read line by line
			while (<IPMASKFILE>) {
				chomp;
				s/^\s+//;	# delete begining space
				s/^#.*//;	# delete all comment
				push @IP_MASK,$_ if ($_ ne '');
			}
		}
		# Load all the prefixchk in PREFIXCHK if file exist
		@PREFIXCHK=();
		if (open(PREFIXCHKFILE, "<$HUB_DIR/prefixchk")) {
			$/ = "\n";			# Read line by line
			while (<PREFIXCHKFILE>) {
				chomp;
				s/^\s+//;	# delete begining space
				s/^#.*//;	# delete all comment
				push @PREFIXCHK,$_ if ($_ ne '');
			}
		}
	}
	$|=1; # AutoFlush
}

sub simple_myinfo_handler {

	my $evt_array = $_[0];

	# print "myinfo handler\n";
	# print "event   = '".$evt_array->{"event"}."'\n";
	# print "nickname= '".$evt_array->{"nickname"}."'\n";
	# print "argc    = '".$evt_array->{"argc"}."'\n";

	#Declaration of hub variables
	my $hubmaster=dchub::db_get("HUB_MASTER");
	my $language=dchub::db_get("LANGUAGE");
	
	#Declaration of txt variables. This will make it easy to translate. Default is eng.
	my ($disconnected_because_wrong_prefix);
	if((not defined($language)) or ($language eq "en_US")) {
		#Nickcheck strings.
		$disconnected_because_wrong_prefix=" disconnected (Invalid prefix)";		
	} elsif($language eq "fr_FR") {
		#Nickcheck strings.
		$disconnected_because_wrong_prefix=" dconnect (prefix invalide)";
	} elsif($language eq "sv_SE") {
		#Nickcheck strings.
		$disconnected_because_wrong_prefix=" bortkopplad (Ogiltigt prefix)";
	}
		
	# Declaration of useful variables
	my $nickname=$evt_array->{"nickname"};
	my $duration=dchub::nick_duration($nickname);
	my $ip=dchub::nick_ip($nickname);
	my ($type_cnx,$share_size,$email,$descr,$user_flag,$privilege)=split(/\$/,dchub::nickinfo($nickname));
	my $index=0;
		
	#Declaration of misc variables
	my $str="";		# A temporary string variable.
	my $char=""; 	# A temporary string variable used only for single chars.
	
	# Eric, why is this "if" needed? /Daniel Marmader
	if ($evt_array->{"argc"} != 1) {
		dchub::disconnect($nickname);
	} else {
		# Alert users when Master or OPerator logs on.
		if($privilege != 0 && $duration < 4) {
			if (defined $hubmaster) {
				dchub::send_pchat_msg("Hub-Log",$hubmaster,$nickname." IP:".$ip._(" logged on.")."|");
			}
			dchub::send_to_all_users("<Hub-Security> ".$nickname._(" logged on.")."|");
		}

		################################################
		#Welcome message.
		################################################
		if($duration < 4) {
			#Welcome
			$str = _("Welcome ");
			$str.= $nickname;
			$str.= _(". This is the greatest hub known to mankind.");
			dchub::send_pchat_msg("Hub",$nickname,"$str|");
			#Print rules.
			if ($RULES) {
				dchub::send_pchat_msg("Hub",$nickname,$RULES."|");
			}
		}
		
		################################################
		#IP check. Reads from file. If file does not exist, any IP will be accepted.
		################################################
		my $ip_ok="yes";
		if (@IP_MASK) {
			$ip_ok="no";
			my @bytes_ip=split(/\./,$ip);
			foreach (@IP_MASK) {	# Test all ip in IP_MASK
				my @mask=split(/\./); # mask: array of ip byte
				my $number_of_bytes=$#mask+1; # Number of bytes of the mask
				my $index=0;
				foreach (@mask) {
					# print "$_ <-> $bytes_ip[$index]\n";
					last if ($bytes_ip[$index] != $_); # last if byte don't match
					$index++;
				}
				if ($index == $number_of_bytes) { # if all match -> ip_ok=yes
					$ip_ok="yes";
					last;
				}
			}
		}
		if( $ip_ok eq "no" && $privilege == 0 ) {
			dchub::send_to_named_user($nickname,"<Hub-Security> "._("You can't connect to this hub with your IP")."|");
			if (defined $hubmaster) {
				dchub::send_pchat_msg("Hub-Log",$hubmaster,$nickname." IP:".$ip._(" disconnected (Invalid IP)")."|");
			}
			dchub::kick($nickname);
			return;
		}
		
		################################################
		# Nick check.
		################################################
		my $nick_ok="yes";
		if (@PREFIXCHK) {
			$nick_ok="no";
			foreach my $prefix (@PREFIXCHK) {
				if ($nickname=~/$prefix/) {
					$nick_ok="yes";
					last;
				}
			}
		}
		if ($nick_ok eq "no" && $privilege == 0 ) {
			dchub::send_to_named_user($nickname,"<Hub-Security> "._("You can't connect to this hub with this nick (invalid nick prefix).")."|");
			if (defined $hubmaster) {
				dchub::send_pchat_msg("Hub-Log",$hubmaster,$nickname." IP:".$ip._(" disconnected (invalid nick prefix)")."|");
			}
			dchub::kick($nickname);
			return;
		}
			
		################################################
		# Minimun share check.
		################################################
		my $min_share=dchub::db_get("MIN_SHARE");
		if(defined($min_share)) {
			if ($share_size < $min_share) {
				$str=_("You must share more than ");
				if($min_share<1024) {
					$str.="$min_share ";
				} elsif ($min_share<(1024*1000)) {
					$str.=round($min_share/1024)." K";
				} elsif ($min_share<(1024*1000*1000)) {
					$str.=round($min_share/(1024*1000))." M";
				} else {
					$str.=round($min_share/(1024*1000*1000))." G";
				}
   				$str.=_("bytes. Add some files and come back later.");

				if ($privilege == 0) {
					dchub::send_to_named_user($nickname,"<Hub-Security> $str|");
					dchub::disconnect($nickname);
					return;
				} else {
					if ($duration < 4) {
						dchub::send_to_named_user($nickname,"<Hub-Security> $str ."._("... hopefully, you are an OPerator.")."|");
					}
				}
			}
		}
	}
}
print "Perl: simple_myinfo_handler "._("module loaded").".\n";
