/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * bin_xf_io.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: bin_xf_io.c,v 2.3 2003/02/25 14:33:32 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <glib.h>

#include "network.h"
#include "bin_xf_io.h"
#include "cnx_if_detect.h"
#include "ged_if.h"
#include "gvar.h"
#include "chunk.h"

#define EXPECTED_DATA_LEN 700

#if 0
static void bin_xfio_append_new_cmd_to_glist_incoming(BIN_XF_IO *bxfio, CHUNK_CORE *cc);
#endif

static GByteArray *bin_xfio_peek_first_chunk_of_glist_outgoing(BIN_XF_IO *bxfio);
static GByteArray *bin_xfio_take_first_chunk_of_glist_outgoing(BIN_XF_IO *bxfio);

/****************************************************/
/* array of BIN_XF_IO *                             */
/* for speed, this array is sorted on the socket fd */
/****************************************************/
static GPtrArray *bin_xf_io_array=NULL;

/*******************/
/* global counters */
/*******************/
static guint64 gl_up_count=0;			/* uploaded bytes */
static guint64 gl_down_count=0;		/* download bytes */

static int comp_bin_xf_io(const void *a, const void *b)
{
	return (*((BIN_XF_IO**)a))->sock_fd - (*((BIN_XF_IO**)b))->sock_fd;
}

/* ------------------------------------------------------------------------------------ */
/* --------------------- XF_IO creation/destruction functions ------------------------- */
/* ------------------------------------------------------------------------------------ */
/*********************************************************************************************/
/* create a new BIN_XF_IO. The BIN_XF_IO entry is automatically registered in the IO handler */
/*********************************************************************************************/
/* if remote_addr ==NULL, the function computes it */
/***************************************************/
BIN_XF_IO *create_bin_xfio(int sock_fd, struct in_addr *remote_addr)
{
	BIN_XF_IO *nw;

	nw=malloc(sizeof(BIN_XF_IO));
	if(nw==NULL)
	{
		fprintf(stderr,"create_xfio: out of memory.\n");
		return nw;
	}

	/* initialize const values */
	nw->sock_fd=sock_fd;
	nw->start_time=time(NULL);
	if(remote_addr!=NULL)
		nw->user_ip=*remote_addr;
	else
	{
		get_remote_if_ip(sock_fd,&(nw->user_ip));
	}

	nw->if_type=identify_if_of_cnx(sock_fd);

	/* initialize variable values */
	nw->incoming_chunks=NULL;
	nw->in_partial=NULL;
	nw->outgoing_chunks=NULL;
	nw->closed=CNX_OPENED;

	/* initialize counters */
	nw->in_data=0;
	nw->out_data=0;

	g_ptr_array_add(bin_xf_io_array,nw);

	/* sort to speed up future scan */
	qsort(bin_xf_io_array->pdata,bin_xf_io_array->len,sizeof(void*), comp_bin_xf_io);
	return nw;
}

static void free_a_chunk_core(gpointer data, gpointer user_data)
{
	chunk_free((CHUNK_CORE *)data);
}

static void free_a_gbyte_array(gpointer data, gpointer user_data)
{
	g_byte_array_free((GByteArray *)data,TRUE);
}

/*********************************************************************************************/
/* delete a BIN_XF_IO. The BIN_XF_IO entry is automatically unregistered from the IO handler */
/* and the socket is closed.                                                                 */
/*********************************************************************************************/
void delete_bin_xfio(BIN_XF_IO *to_del)
{
	if(g_ptr_array_remove(bin_xf_io_array,to_del)==FALSE)
	{
		fprintf(stderr,"delete_bin_xfio: error, BIN_XF_IO not found, ignored\n");
		return;
	}

	close(to_del->sock_fd);

	/* flush incoming commands */
	g_list_foreach(to_del->incoming_chunks,free_a_chunk_core,NULL);
	g_list_free(to_del->incoming_chunks);

	if(to_del->in_partial)
		g_byte_array_free(to_del->in_partial,TRUE);

	/* flush outgoing commands */
	g_list_foreach(to_del->outgoing_chunks,free_a_gbyte_array,NULL);
	g_list_free(to_del->outgoing_chunks);

	free(to_del);
}

/* ------------------------------------------------------------------------------------ */
/* -------------------------- BIN_XF_IO private functions ----------------------------- */
/* ------------------------------------------------------------------------------------ */
/***************************************************************/
/* read the BIN_XF_IO socket, append it to the partial content */
/* and split it by row into the incoming_commands              */
/***************************************************************/
static void bin_xfio_process_incoming_cnx_data(BIN_XF_IO *bxfio)
{
	char buf[8192];
	int ret;
	CHUNK_CORE *inc;
	gboolean too_small;
	gboolean is_invalid;

	ret=recv(bxfio->sock_fd, buf,sizeof(buf)-1,MSG_NOSIGNAL);
	if(ret==-1)
	{
		if((errno!=EAGAIN)&&(errno!=EINTR))
		{
			perror("bin_xfio_process_incoming_cnx_data");
			bxfio->closed=CNX_CLOSED;
		}
		return;
	}

	if(ret==0)
	{
		/* connection closed */
		bxfio->closed=CNX_CLOSED;
		return;
	}

	if(bxfio->in_partial==NULL)
		bxfio->in_partial=g_byte_array_new();

	g_byte_array_append(bxfio->in_partial,buf,ret);

	bxfio->in_data += ret;
	gl_down_count += ret;

	/* decode incoming chunk */
	inc=chunk_convert_mem_to_chunk(bxfio->in_partial,&too_small,&is_invalid);
	while(inc!=NULL)
	{
		(bxfio->incoming_chunks)=g_list_append(bxfio->incoming_chunks,inc);

		/* perhaps there is more than one incoming chunk */
		inc=chunk_convert_mem_to_chunk(bxfio->in_partial,&too_small,&is_invalid);
	}

	if((too_small==FALSE)&&(is_invalid==TRUE))
	{
		unsigned int hip;

		hip=ntohl(bxfio->user_ip.s_addr);
		printf("invalid crc: %hhu.%hhu.%hhu.%hhu %lu %Lu/%Lu\n",
							(unsigned char)(hip>>24)&0xff,
							(unsigned char)(hip>>16)&0xff,
							(unsigned char)(hip>>8)&0xff,
							(unsigned char)hip&0xff,
							(long unsigned int)(gl_cur_time - bxfio->start_time),
							bxfio->out_data,
							bxfio->in_data
							);
		shutdown(bxfio->sock_fd, SHUT_RD);
		bxfio->closed=CNX_CLOSED;
	}

	/* protection against incoming enormous single instruction */
	if((bxfio->in_partial!=NULL)&&(bxfio->in_partial->len>MAX_BIN_SIZE))
	{
		unsigned int hip;

		hip=ntohl(bxfio->user_ip.s_addr);
		printf("loflood: %hhu.%hhu.%hhu.%hhu %lu %Lu/%Lu\n",
							(unsigned char)(hip>>24)&0xff,
							(unsigned char)(hip>>16)&0xff,
							(unsigned char)(hip>>8)&0xff,
							(unsigned char)hip&0xff,
							(long unsigned int)(gl_cur_time - bxfio->start_time),
							bxfio->out_data,
							bxfio->in_data
							);
		shutdown(bxfio->sock_fd, SHUT_RD);
		bxfio->closed=CNX_CLOSED;
	}
	return;
}

/******************************************/
/* send outgoing_commands into the socket */
/******************************************/
static void bin_xfio_process_outgoing_cnx_data(BIN_XF_IO *bxfio)
{
	GByteArray *gba;
	
	int err;
#ifdef WIN32
		int errlen = sizeof(err);
#else
	#ifdef __APPLE__
		int errlen = sizeof(err);
	#else
		socklen_t errlen = sizeof(err);
	#endif
#endif
	int sres;

	/* pack several outgoing messages into a biggest block */
	gba=bin_xfio_take_first_chunk_of_glist_outgoing(bxfio);
	if(gba==NULL)
		return;			/* having nothing to send is not a fatal error */

	while(gba->len < EXPECTED_DATA_LEN)
	{
		GByteArray *peeked_gba;

		peeked_gba = bin_xfio_peek_first_chunk_of_glist_outgoing(bxfio);
		if (peeked_gba == NULL)
			break;
		if ((gba->len + peeked_gba->len) < EXPECTED_DATA_LEN)
		{
			peeked_gba=bin_xfio_take_first_chunk_of_glist_outgoing(bxfio);
			g_byte_array_append(gba,peeked_gba->data,peeked_gba->len);
			g_byte_array_free(peeked_gba,TRUE);
		}
		else
			break; /* The buffer is too small, data will be send next time */
	}

	/* send the message in one part, it should not be broken into serveral parts */
	if (getsockopt(bxfio->sock_fd , SOL_SOCKET, SO_ERROR , &err , &errlen) != 0 ||
		err == ETIMEDOUT || err == ECONNREFUSED || err == EHOSTDOWN ||
		err == EHOSTUNREACH)
	{
		/* connexion error -> shutdown the socket */
		shutdown(bxfio->sock_fd,SHUT_RDWR);
		bxfio->closed=CNX_CLOSED;
	}
	else
	{
		retry_send:
		sres=send(bxfio->sock_fd,gba->data,gba->len,MSG_NOSIGNAL); /* send data using local buffer */
		if (sres!=gba->len)
		{
			if(sres==-1)
			{
				switch(errno)
				{
					case EAGAIN:
					case EINTR:	/* non fatal errors */
						goto retry_send;

					default:	/* other errors are fatal -> shutdown the socket */
						shutdown(bxfio->sock_fd,SHUT_RDWR);
						bxfio->closed=CNX_CLOSED;
						break;
				}
			}
		}

		/* update counters */
		bxfio->out_data += gba->len;
		gl_up_count += gba->len;
	}

	g_byte_array_free(gba,TRUE);
	return;
}

	
/* ------------------------------------------------------------------------------------ */
/* --------------------------- XF_IO incoming functions ------------------------------- */
/* ------------------------------------------------------------------------------------ */

/*************************************************************/
/* same as peek_first_string_of_glist except the CHUNK_CORE  */
/* is also removed from the GList. You must free it yourself */
/* when you don't need it anymore.                           */
/*************************************************************/
/* output: the pointer or NULL */ 
/*******************************/
CHUNK_CORE *bin_xfio_take_first_chunk_of_glist_incoming(BIN_XF_IO *bxfio)
{
	CHUNK_CORE *st;

	if(bxfio->incoming_chunks==NULL)
		st=NULL;
	else
	{
		st=bxfio->incoming_chunks->data;
		bxfio->incoming_chunks=g_list_remove(bxfio->incoming_chunks,st);
	}
	return st;
}

#if 0
/**********************************************************/
/* append the given CHUNK_CORE to the given incoming list */
/**********************************************************/
static void bin_xfio_append_new_cmd_to_glist_incoming(BIN_XF_IO *bxfio, CHUNK_CORE *cc)
{
	bxfio->incoming_chunks=g_list_append(bxfio->incoming_chunks,cc);
}
#endif

/* ------------------------------------------------------------------------------------ */
/* --------------------------- XF_IO outgoing functions ------------------------------- */
/* ------------------------------------------------------------------------------------ */
/*****************************************************************/
/* return the pointer of the first GByteArray of the given GList */
/*****************************************************************/
/* output: the pointer or NULL. the pointer and the CHUNK_CORE */
/*         content should not be modified                      */
/***************************************************************/
static GByteArray *bin_xfio_peek_first_chunk_of_glist_outgoing(BIN_XF_IO *bxfio)
{
	GByteArray *gba;
	
	if(bxfio->outgoing_chunks==NULL)
		gba=NULL;
	else
		gba=bxfio->outgoing_chunks->data;
	return gba;
}

/*************************************************************/
/* same as peek_first_string_of_glist except the CHUNK_CORE  */
/* is also removed from the GList. You must free it yourself */
/* when you don't need it anymore.                           */
/*************************************************************/
/* output: the pointer or NULL */
/*******************************/
static GByteArray *bin_xfio_take_first_chunk_of_glist_outgoing(BIN_XF_IO *bxfio)
{
	GByteArray *gba;
	
	if(bxfio->outgoing_chunks==NULL)
		gba=NULL;
	else
	{
		gba=bxfio->outgoing_chunks->data;
		bxfio->outgoing_chunks=g_list_remove(bxfio->outgoing_chunks,gba);
	}
	return gba;
}

/**************************************************************/
/* append a (copy of the given) GByteArray to the given glist */
/*****************************************************************************/
/* if do_copy==TRUE, a copy is created and used                              */
/* if do_copy==FALSE, the given GByteArray is used and must not be destroyed */
/*****************************************************************************/
void bin_xfio_append_new_gba_to_glist_outgoing(BIN_XF_IO *bxfio, GByteArray *gba, int do_copy)
{
	if(do_copy==FALSE)
	{
		bxfio->outgoing_chunks=g_list_append(bxfio->outgoing_chunks,gba);
	}
	else
	{
		GByteArray *cpy;

		cpy=g_byte_array_new();
		g_byte_array_append(cpy,gba->data,gba->len);
		bxfio->outgoing_chunks=g_list_append(bxfio->outgoing_chunks,cpy);
	}
}

/* ------------------------------------------------------------------------------------ */
/* ----------------------------- XF_IO misc functions --------------------------------- */
/* ------------------------------------------------------------------------------------ */
/************************/
/* return I/O statistic */
/************************/
void bin_xfio_get_io_stats(guint64 *incoming_bytes, guint64 *outgoing_bytes)
{
	*incoming_bytes=gl_down_count;
	*outgoing_bytes=gl_up_count;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------- GED handler for cluster connections ----------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************/
/* function called at end */
/**************************/
static int bin_xf_io_ged_exit(const GED_CALLBACKS *ged)
{
	return 0;
}

/****************************/
/* add FD event to wait for */
/****************************/
static void bin_xf_io_ged_add_fd(const GED_CALLBACKS *ged, GArray *struct_pollfd_array)
{
	/* for each XF_IO, add an entry in the pollfd_array */
	/* if the outgoing command is empty, the event if POLLIN */
	/* else, it is POLLIN|POLLOUT */
	int i;
	BIN_XF_IO *bxfio;
	struct pollfd nw;

#ifdef DEBUG
	fprintf(stderr,"bin_xf_io_ged_add_fd: xf_io len: %d\n",bin_xf_io_array->len);
#endif
	/* for speed, we must ALWAYS add an entry for each fd to monitor */
	for(i=0;i<bin_xf_io_array->len;i++)
	{
		bxfio=g_ptr_array_index(bin_xf_io_array,i);

		nw.fd=bxfio->sock_fd;

		switch(bxfio->closed)
		{
			case CNX_OPENED:
									nw.events=POLLIN;
									if(bxfio->outgoing_chunks)
										nw.events|=POLLOUT;
									break;

			case CNX_CLOSED:	continue;		/* this entry will disappear at the next loop, don't poll it */

			case CNX_WILL_DIE:
									if(bxfio->outgoing_chunks)
										nw.events=POLLOUT;
									else
									{
										bxfio->closed=CNX_CLOSED;		/* the queue is finally empty */
										shutdown(bxfio->sock_fd,SHUT_RD);	/* just to be sure nothing will come accidentally */
										nw.events=POLLIN;		/* this entry will disappear at the next loop */
									}
									break;
		}

		nw.revents=0;
		
#ifdef DEBUG
		fprintf(stderr,"xfio [%d]=%d (status: %d, outq: %p) (events=%d)\n",i,xfio->sock_fd,xfio->closed,xfio->outgoing_commands,nw.events);
#endif
		g_array_append_val(struct_pollfd_array,nw);
	}
}

/*******************************/
/* check for the arrived event */
/*******************************/
static void bin_xf_io_ged_scan_fd(const GED_CALLBACKS *ged, GArray *struct_pollfd_array, int low_range, int high_range)
{
	/* foreach ours fd, handle reception of data and splitting into incoming_commands */
	/* and sending of outgoing_commands */
	BIN_XF_IO *bxfio;
	struct pollfd *nw;

	/* both arrays are sorted on sock_fd */
	int xfio_minus;
	int i;

#ifdef DEBUG
	/* debug code */
	fprintf(stderr,"xf_io_ged_scan_fd: xf_io len: %d  range=[%d:%d[ => %d\n",xf_io_array->len,low_range,high_range,high_range-low_range);
	for(i=low_range;i<high_range;i++)
	{
		fprintf(stderr,"pollfd [%d]=%d (revents=%d)\n",i,(g_array_index(struct_pollfd_array,struct pollfd,i)).fd,(g_array_index(struct_pollfd_array,struct pollfd,i)).revents);
	}
	for(i=0;i<xf_io_array->len;i++)
	{
		fprintf(stderr,"xfio [%d]=%d\n",i,((XF_IO*)g_ptr_array_index(xf_io_array,i))->sock_fd);
	}
#endif

	xfio_minus=0;
	i=low_range;
	while(i<high_range)
	{
		int idx;

		nw=&(g_array_index(struct_pollfd_array,struct pollfd,i));
		
		/* now, we must find the xfio with the same sock_fd */
		/* it is pretty easy, its index (idx) is >=xfio_minus, <xf_io_array->len and sock_fd[idx] <=nw->sock_fd */
		idx=xfio_minus;
		while(idx<bin_xf_io_array->len)
		{
			bxfio=g_ptr_array_index(bin_xf_io_array,idx);
			if(bxfio->sock_fd==nw->fd)
			{
				/* we have found it */
				xfio_minus=idx+1;		/* the next time, start after this idx */
				if(nw->revents&POLLIN)
				{
					/* process incoming data */
					bin_xfio_process_incoming_cnx_data(bxfio);
				}

				if(nw->revents&POLLOUT)
				{
					/* process outgoing data */
					bin_xfio_process_outgoing_cnx_data(bxfio);
				}
				break;
			}

			if(bxfio->sock_fd>nw->fd)
			{	/* the fd does not exists, we are already above what we search for */
				xfio_minus=i;		/* the next time, start here */
				break;
			}
			idx++;
		}
		if(idx==bin_xf_io_array->len)	/* end of array reached */
			break;						/* all other fd will be unavailable */
		
		i++;
	}
}

static GED_CALLBACKS bin_xf_io_ged=
					{
						"BIN_XF_IO ged",
						bin_xf_io_ged_exit,
						bin_xf_io_ged_add_fd,
						bin_xf_io_ged_scan_fd,
						NULL,
						NULL,
						NULL,
						NULL,
					};

/************************************************************/
/* function initializing the handler of all the connections */
/************************************************************/
GED_CALLBACKS *bin_xf_io_handler_init(void)
{
	bin_xf_io_array=g_ptr_array_new();
   return &bin_xf_io_ged;
}

