/*
 * DCHub Functions for scripts
 *
 * dchub::send_to_named_user( string nickname, string dcline)
 *
 *   * send the given dcline to the user named nickname.
 *
 * dchub::send_to_all_users( string dcline)
 *
 *   * send the given dcline to all connected users.
 *
 *
 * b) dedicated chat functions
 * ------------------------
 *
 * dchub::send_gchat_msg( string sender_nickname, string message)
 *
 *   * send a message on the global chat as user "sender_nickname". This function
 *     builds the appropriate DC message from the given information.
 *
 * dchub::send_pchat_msg( string sender_nickname, string dest_nickname, string message)
 *
 *   * send a private message as user "sender_nickname" to user "dest_nickname".
 *     This function builds the appropriate DC message from the given information.
 *
 *
 * c) misc functions
 * --------------
 *
 * dchub::disconnect(string nickname)
 *
 *   * disconnect the user having the given nickname.
 *
 * dchub::kick(string nickname)
 *
 *   * same as disconnect but a kick message is displayed on the global chat.
 *
 * dchub::forcemove(string nickname, string destination_address)
 *
 *   * redirect the given user to the destination address
 *
 * dchub::redirect(string nickname)
 *
 *   * redirect the given user to "REDIR_ADDR".
 *
 * dchub::redirectto(string nickname, string ip)
 *
 *   * redirect the given user to the given address (ip).
 *
 * dchub::userlist()
 *
 *   * retrieve the list of hub users.
 *
 * dchub::oplist()
 *
 *   * retrieve the list of hub operators.
 *
 * dchub::lstmultihubs()
 *
 *   (N/A) retrieve the list of hubs linked with this hub.
 *
 * dchub::nickinfo(string nickname)
 *
 *  * retrieve information about the given nickname.
 *   The result string contains the following field and uses $ as field separator:
 *        + connection type
 *        + shared size
 *        + user e-mail
 *        + user description
 *        + user flag (bit 0: always set)
 *                    (bit 1: =0: user here, =1: user away)
 *                    (bit 2: =0: client is normal, =1, client is server ==online for more than 2 days)
 *                    (bit 3: =0: normal upload, =1: fast upload == upload speed greater than 100KB/s)
 *        + op flag  (==0: normal user, ==1: operator, ==2: hub master)
 *        + client version
 *
 * dchub::nick_duration(string nickname)
 *
 *   * return the number of seconds ellapsed since the beginning of this connexion
 *     This function is mainly useful when some events are periodic and when you don't want to
 *     reply to all of them (like the $MyINFO, only the first one should be useful at least to
 *     display "ok" messages).
 *
 * dchub::nick_type(string nickname)
 *
 *   * return the type of "account" used by the nickname. 
 *     This function returns "M" for hub master, "O" for operator, "N" for normal
 *     registered user and "U" for unregistered user.
 *
 * d) hub configuration
 * -----------------
 *
 * dchub::get_max_users()
 *
 *   * get the max number of users of the hub.
 *
 * dchub::get_txtdescription()
 *
 *   * get the hub description (for hub registration on DChublist)
 *
 * dchub::get_hubip()
 *
 *   * get the hub address (for hub registration on DChublist)
 *
 * e) database access
 * ---------------
 *
 * dchub::db_get(string keyname)
 *
 *   * get the value associated to the given key. The returned value is always a string
 *
 * dchub::db_set(string keyname, keyvalue)
 *
 *   * set the value associated to the given key. The key value can be an integer, a float or a string.
 *
 */
/*
$Id: emb_python.c,v 2.3 2003/02/04 06:17:48 eric Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


#ifdef WITH_PYTHON

#include "emb_python.h"
#include "global_user_if.h"

#ifndef METH_NOARGS
#define METH_NOARGS METH_VARARGS
#endif

static gchar *emb_python = NULL;

int init_python_interpreter( void )
{
   Py_Initialize();
   return Py_IsInitialized();
}

void exit_python_interpreter( void )
{
   Py_Finalize();
   printf( "Python Interpreter exiting.\n" );
}

void init_python( void )
{
   gchar *path;

   if ( ( python_script_dir == NULL ) || ( python_main_script == NULL ) )
   {
      fprintf( stderr, "Python scripting disabled.\n" );
      return;
   }

   emb_python = g_strconcat( python_script_dir, "/", python_main_script, NULL );

   if ( !( init_python_interpreter() ) )
   {
      fprintf( stderr, "Could not initialize python interpreter.\n" );
      g_free( emb_python );
      return;
   }

   printf("Python Interpreter loaded.\n");

   initdchub();

   path = g_strconcat( "sys.path.append('", python_script_dir, "')", NULL );

   PyRun_SimpleString("import sys");
   PyRun_SimpleString( path );
   g_free( path );
}

void send_evt_to_pscript( const char *evt_name, const char *evt_emitter, int nb_args, ...)
{
   int i;
   va_list ap;
   PyObject *PyArg, *pName, *pTuple, *pModule, *pDict, *pFunc;

   if ( emb_python == NULL )
      return;

   if ( NULL == ( PyArg = PyDict_New() ) )
      return;

   /* Construct script name */

   pName = PyString_FromString( python_main_script );

   /* Import the module */

   pModule = PyImport_Import( pName );

   if ( NULL != pModule )
   {
      pDict = PyModule_GetDict( pModule );

      /* Get the main function in the script */

      pFunc = PyDict_GetItemString( pDict, "dchub_python_handler" );

      if ( pFunc && PyCallable_Check( pFunc ) )
      {
         pTuple = PyTuple_New( 1 );

         /* Build the default entries in the argument list */

         PyDict_SetItem( PyArg,
                         PyString_FromString( "event" ),
                         PyString_FromString( evt_name )
                       );
         PyDict_SetItem( PyArg,
                         PyString_FromString( "nickname" ),
                         PyString_FromString( evt_emitter )
                       );
         PyDict_SetItem( PyArg,
                         PyString_FromString( "argc" ),
                         PyInt_FromLong( nb_args )
                       );

         /* Optional args */

         va_start( ap, nb_args );

         for ( i = 0; i < nb_args; i++ )
         {
            char *arg;
            char tmp_num[10];

            arg = va_arg( ap, char* );
            sprintf( tmp_num, "%u", i );
            PyDict_SetItem( PyArg,
                            PyString_FromString( tmp_num ),
                            PyString_FromString( arg )
                          );
         }
         /* Build the actual arg that is passed */
         PyTuple_SetItem( pTuple, 0, PyArg );
         /* Call the script and pass the arg */
         PyObject_CallObject( pFunc, pTuple );
         Py_DECREF( pTuple );
      }
      else
      {
         /* Function in script was not found */
         PyErr_Print();
         fprintf( stdout, "Cannot find function \"%s\"\n",
                  "dchub_python_handler"
                );
      }
      Py_DECREF( pModule );
   }
   else
   {
      /* Module name was not found */
      PyErr_Print();
      fprintf( stderr, "Failed to load \"%s\"\n", emb_python );
      return;
   }
   /* All done */
   Py_DECREF( pName );
}

/*
 * Dchub scripting functions below
 *
 */

static PyObject * dchub_send_to_named_user( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *arg1, *arg2;

      if ( 2 != PyTuple_Size( args ) )
      {
         printf( "dchub.send_to_named_user(): Expected 2 arguments.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );
      arg2 = PyTuple_GetItem( args, 1 );

      if ( !( PyString_Check( arg1 ) ) || !( PyString_Check( arg2 ) ) )
      {
         printf( "dchub.send_to_named_user(): Both arguments must be strings.\n" );
         return PyInt_FromLong(0L);
      }

      GLUS_SEND_TO_A_NICK_char_ptr( PyString_AsString( arg1 ), PyString_AsString( arg2 ) );
      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_send_to_all_users( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *arg1;

      if ( 1 != PyTuple_Size( args ) )
      {
         printf( "dchub.send_to_all_users(): Expected 1 argument.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );

      if ( !( PyString_Check( arg1 ) ) )
      {
         printf( "dchub.send_to_all_users(): Argument must be of type string.\n ");
         return PyInt_FromLong(0L);
      }

      GLUS_SEND_TO_EVERYONE_char_ptr( PyString_AsString( arg1 ) );
      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_send_gchat_msg( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      gchar *msg;
      PyObject *arg1, *arg2;

      if ( 2 != PyTuple_Size( args ) )
      {
         printf( "dchub.send_gchat_msg(): Expected 2 arguments.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );
      arg2 = PyTuple_GetItem( args, 1 );

      if ( !( PyString_Check( arg1 ) ) || !( PyString_Check( arg2 ) ) )
      {
         printf( "dchub.send_gchat_msg(): Both arguments must be strings.\n" );
         return PyInt_FromLong(0L);
      }

      msg = g_strconcat( "<",
                         PyString_AsString( arg1 ),
                         "> ",
                         PyString_AsString( arg2 ),
                         "|",
                         NULL
                       );
      GLUS_SEND_TO_EVERYONE_char_ptr( msg );
      g_free( msg );
      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_send_pchat_msg( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      gchar *msg;
      PyObject *arg1, *arg2, *arg3;

      if ( 3 != PyTuple_Size( args ) )
      {
         printf( "dchub.send_pchat_msg(): Expected 3 arguments.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );
      arg2 = PyTuple_GetItem( args, 1 );
      arg3 = PyTuple_GetItem( args, 2 );

      if ( !( PyString_Check( arg1 ) )
           ||
           !( PyString_Check( arg2 ) )
           ||
           !( PyString_Check( arg3 ) )
         )
      {
         printf( "dchub.send_pchat_msg(): All 3 arguments must be of type string.\n" );
         return PyInt_FromLong(0L);
      }

      msg = g_strconcat( "$To: ",
                         PyString_AsString( arg2 ),
                         " From: ",
                         PyString_AsString( arg1 ),
                         " $<",
                         PyString_AsString( arg1 ),
                         "> ",
                         PyString_AsString( arg3 ),
                         "|",
                         NULL
                       );
      GLUS_SEND_TO_A_NICK_char_ptr( PyString_AsString( arg2 ), msg );
      g_free( msg );
      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_disconnect( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *arg1;

      if ( 1 != PyTuple_Size( args ) )
      {
         printf( "dchub.disconnect(): Expected 1 argument.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );

      if ( !( PyString_Check( arg1 ) ) )
      {
         printf( "dchub.disconnect(): Argument must be of type string.\n" );
         return PyInt_FromLong(0L);
      }

      glus_disconnect_named_user( PyString_AsString( arg1 ) );

      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_kick( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *arg1;

      if ( 1 != PyTuple_Size( args ) )
      {
         printf( "dchub.kick(): Expected 1 argument.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );

      if ( !( PyString_Check( arg1 ) ) )
      {
         printf( "dchub.kick(): Argument must be of type string.\n" );
         return PyInt_FromLong(0L);
      }

      glus_kick_named_user( NULL, PyString_AsString( arg1 ) );
      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_forcemove( PyObject *self, PyObject *args )
{
   return PyInt_FromLong(42L);
}

static PyObject * dchub_redirect( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      GString *rd_msg;
      PyObject *arg1;
      char *rd;

      if ( 1 != PyTuple_Size( args ) )
      {
         printf( "dchub.redirect(): Expected 1 argument.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );

      if ( !( PyString_Check( arg1 ) ) )
      {
         printf( "dchub.redirect(): Argument must be of type string.\n" );
         return PyInt_FromLong(0L);
      }

      G_LOCK( conf_file );
      rd = e_db_str_get( conf_file, "REDIR_ADDR" );
      G_UNLOCK( conf_file );

      if ( ( rd != NULL ) && ( 0 != strlen( rd ) ) )
      {
         rd_msg = g_string_new( "" );
         g_string_sprintf( rd_msg, "<Hub-Security> You are being redirected to %s.|", rd );
         g_string_sprintfa( rd_msg, "$ForceMove %s|", rd );
         GLUS_SEND_TO_A_NICK( PyString_AsString( arg1 ), rd_msg);	 /* don't free rd_msg */
      }

      glus_disconnect_named_user( PyString_AsString( arg1 ) );

      if ( rd != NULL )
         free( rd );

      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_redirectto( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *arg1, *arg2;
      GString *rd_msg;

      if ( 2 != PyTuple_Size( args ) )
      {
         printf( "dchub.redirectto(): Expected 2 arguments.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );
      arg2 = PyTuple_GetItem( args, 1 );

      if ( !( PyString_Check( arg1 ) ) || !( PyString_Check( arg2 ) ) )
      {
         printf( "dchub.redirectto(): Both arguments must be of type string.\n" );
         return PyInt_FromLong(0L);
      }

      rd_msg = g_string_new( "" );
      g_string_sprintf( rd_msg,
                        "<Hub-Security> You are being redirected to %s.|",
                        PyString_AsString( arg2 )
                      );
      g_string_sprintfa( rd_msg, "$ForceMove %s|", PyString_AsString( arg2 ) );
      GLUS_SEND_TO_A_NICK( PyString_AsString( arg1 ), rd_msg );		/* don't free rd_msg */

      glus_disconnect_named_user( PyString_AsString( arg1 ) );

      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_userlist( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *string;
      GString *str;

      if ( 0 != PyTuple_Size( args ) )
      {
         printf( "dchub.userlist(): No arguments expected.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

		str=txt_glus_get_userlist();

      if ( NULL == str )
      {
         Py_INCREF( Py_None );
         return Py_None;
      }

      string = PyString_FromString( strdup( str -> str ) );
      g_string_free( str, TRUE );
      return string;
   }
   Py_INCREF( Py_None );
   return Py_None;
}

static PyObject * dchub_oplist( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *string;
      GString *str;

      if ( 0 != PyTuple_Size( args ) )
      {
         printf( "dchub.oplist(): No arguments expected.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

		str=txt_glus_get_oplist();

      if ( NULL == str )
      {
         Py_INCREF( Py_None );
         return Py_None;
      }

      string = PyString_FromString( strdup( str -> str ) );
      g_string_free( str, TRUE );
      return string;
   }
   Py_INCREF( Py_None );
   return Py_None;
}

static PyObject * dchub_lstmultihubs( PyObject *self, PyObject *args )
{
   return PyInt_FromLong(42L);
}

static PyObject * dchub_nickinfo( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *arg1, *string;
      GString *str;

      if ( 1 != PyTuple_Size( args ) )
      {
         printf( "dchub.nickinfo(): Expected 1 argument.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

      arg1 = PyTuple_GetItem( args, 0 );

      if ( !( PyString_Check( arg1 ) ) )
      {
         printf( "dchub.nickinfo(): Argument must be of type string.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

		str=txt_glus_get_user_info(PyString_AsString( arg1 ) );

      if ( NULL == str )
      {
         Py_INCREF( Py_None );
         return Py_None;
      }

      string = PyString_FromString( strdup( str -> str ) );
      g_string_free( str, TRUE );
      return string;
   }
   Py_INCREF( Py_None );
   return Py_None;
}

static PyObject * dchub_nick_duration( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *arg1;
		GLUS_USER_INFO *gui;

      if ( 1 != PyTuple_Size( args ) )
      {
         printf( "dchub.nick_duration(): Expected 1 argument.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

      arg1 = PyTuple_GetItem( args, 0 );

      if ( !( PyString_Check( arg1 ) ) )
      {
         printf( "dchub.nick_duration(): Argument must be of type string.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

		gui=glus_get_user_info(PyString_AsString( arg1 ));
		if(gui==NULL)
      {
         Py_INCREF( Py_None );
         return Py_None;
      }
      else
		{
      	time_t duration=gl_cur_time-gui->cnx_start_time;
			glus_free_user_info(gui);
         return PyInt_FromLong(duration);
		}
   }
   return PyInt_FromLong(0L);
}

static PyObject * dchub_nick_type( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
//      int op_type;
//      int ext_type;
	char	type[2];
	PyObject *arg1;

	if ( 1 != PyTuple_Size( args ) )
	{
		printf( "dchub.nick_type(): Expected 1 argument.\n" );
		Py_INCREF( Py_None );
		return Py_None;
	}

	arg1 = PyTuple_GetItem( args, 0 );

	if ( !( PyString_Check( arg1 ) ) )
	{
		printf( "dchub.nick_type(): Argument must be of type string.\n" );
		Py_INCREF( Py_None );
		return Py_None;
	}

	type[1] = '\0';
	if (get_account_type(PyString_AsString( arg1 ), type) == 0)
		return PyString_FromString( "U" );
	return PyString_FromString(type);
   }
   Py_INCREF( Py_None );
   return Py_None;
}

static PyObject * dchub_get_max_users( PyObject *self, PyObject *args )
{
   int nmb = -1;

   if ( PyTuple_Check( args ) )
   {
      if ( 0 != PyTuple_Size( args ) )
         printf( "dchub.get_max_users(): No arguments expected.\n" );
      else
      {
         G_LOCK( conf_file );
         if ( !( e_db_int_get( conf_file, "MAXUSER", &nmb ) ) )
            nmb = 100;
         G_UNLOCK( conf_file );
      }
   }
   return PyInt_FromLong( nmb );
}

static PyObject * dchub_get_txtdescription( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      char *hubdesc;
      PyObject *desc;

      if ( 0 != PyTuple_Size( args ) )
      {
         printf( "dchub.get_txtdescription(): No arguments expected.\n" );
         return Py_None;
      }

      G_LOCK( conf_file );
      hubdesc = e_db_str_get( conf_file, "HUBDESC" );
      G_UNLOCK( conf_file );

      if ( NULL == hubdesc )
      {
         Py_INCREF( Py_None );
         return Py_None;
      }

      desc = PyString_FromString( strdup( hubdesc ) );
      free( hubdesc );
      return desc;
   }
   Py_INCREF( Py_None );
   return Py_None;
}

static PyObject * dchub_get_hubip( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      char *hubdesc;
      PyObject *desc;

      if ( 0 != PyTuple_Size( args ) )
      {
         printf( "dchub.get_hubip(): No arguments expected.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

      G_LOCK( conf_file );
      hubdesc = e_db_str_get( conf_file, "HUBADDR" );
      G_UNLOCK( conf_file );

      if ( NULL == hubdesc )
      {
         Py_INCREF( Py_None );
         return Py_None;
      }

      desc = PyString_FromString( strdup( hubdesc ) );
      free( hubdesc );
      return desc;
   }
   Py_INCREF( Py_None );
   return Py_None;
}

static PyObject * dchub_db_get( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      char *str;
      PyObject *arg1, *result;

      if ( 1 != PyTuple_Size( args ) )
      {
         printf( "dchub.db_get(): Expected 1 argument.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

      arg1 = PyTuple_GetItem( args, 0 );

      if ( !( PyString_Check( arg1 ) ) )
      {
         printf( "dchub.db_get(): Argument must be of type string.\n" );
         Py_INCREF( Py_None );
         return Py_None;
      }

      G_LOCK( conf_file );
      str = e_db_str_get( conf_file, PyString_AsString( arg1 ) );
      G_UNLOCK( conf_file );

      if ( NULL == str )
      {
         Py_INCREF( Py_None );
         return Py_None;
      }

      result = PyString_FromString( strdup( str ) );
      free( str );
      return result;
   }
   Py_INCREF( Py_None );
   return Py_None;
}

static PyObject * dchub_db_set( PyObject *self, PyObject *args )
{
   if ( PyTuple_Check( args ) )
   {
      PyObject *arg1, *arg2;

      if ( 2 != PyTuple_Size( args ) )
      {
         printf( "dchub.db_set(): Expected 2 arguments.\n" );
         return PyInt_FromLong(0L);
      }

      arg1 = PyTuple_GetItem( args, 0 );
      arg2 = PyTuple_GetItem( args, 1 );

      if ( !( PyString_Check( arg1 ) ) )
      {
         printf( "dchub.db_set(): Argument 1 must be of type string.\n" );
         return PyInt_FromLong(0L);
      }

      G_LOCK( conf_file );
      e_db_str_set( conf_file, PyString_AsString( arg1 ), PyString_AsString( arg2 ) );
      G_UNLOCK( conf_file );
      return PyInt_FromLong(1L);
   }
   return PyInt_FromLong(0L);
}

static PyMethodDef dchub_methods[] = {

   { "send_to_named_user",
     dchub_send_to_named_user,
     METH_VARARGS,
     "Send the given dcline to the user named nickname."
   },
   { "send_to_all_users",
     dchub_send_to_all_users,
     METH_VARARGS,
     "Send the given dcline to all connected users."
   },

   /* Dedicated Chat scripts */

   { "send_gchat_msg",
     dchub_send_gchat_msg,
     METH_VARARGS,
     "Send a message on the global chat as user \"sender_nickname\". "
     "This function builds the appropriate DC message from the given "
     "information."
   },
   { "send_pchat_msg",
     dchub_send_pchat_msg,
     METH_VARARGS,
     "Send a private message as user \"sender_nickname\" to user "
     "\"dest_nickname\". This function builds the appropriate DC "
     "message from the given information."
   },

   /* Misc functions */

   { "disconnect",
     dchub_disconnect,
     METH_VARARGS,
     "Disconnect the user having the given nickname."
   },
   { "kick",
      dchub_kick,
      METH_VARARGS,
     "Same as disconnect but a kick message is displayed on the "
     "global chat."
   },
   { "forcemove",
     dchub_forcemove,
     METH_VARARGS,
     "Redirect the given user to the destination address."
   },
   { "redirect",
     dchub_redirect,
     METH_VARARGS,
     "Redirect the given user to \"REDIR_ADDR\"."
   },
   { "redirectto",
     dchub_redirectto,
     METH_VARARGS,
     "Redirect the given user to the given address (ip)."
   },
   { "userlist",
     dchub_userlist,
     METH_NOARGS,
     "Retrieve the list of hub users."
   },
   { "oplist",
     dchub_oplist,
     METH_VARARGS,
     "Retrieve the list of hub operators."
   },
   { "lstmultihubs",
     dchub_lstmultihubs,
     METH_VARARGS,
     "(N/A) retrieve the list of hubs linked with this hub."
   },
   { "nickinfo",
     dchub_nickinfo,
     METH_VARARGS,
     "retrieve information about the given nickname."
     "The result string contains the following field and "
     "uses $ as field separator:\n"
     "   + connection type\n"
     "   + shared size\n"
     "   + user e-mail\n"
     "   + user description\n"
     "   + user flag (bit 0: always set)\n"
     "               (bit 1: =0: user here, =1: user away)\n"
     "               (bit 2: =0: client is normal, =1, client is server ==online for more than 2 days)\n"
     "               (bit 3: =0: normal upload, =1: fast upload == upload speed greater than 100KB/s)\n"
     "   + op flag  (==0: normal user, ==1: operator, ==2: hub master)\n"
     "   + client version"
   },
   { "nick_duration",
     dchub_nick_duration,
     METH_VARARGS,
     "return the number of seconds ellapsed since the beginning of this connection."
     "This function is mainly useful when some events are periodic and when you don't "
     "want to reply to all of them (like the $MyINFO, only the first one should be "
     "useful at least to display \"ok\" messages.)"
   },
   { "nick_type",
     dchub_nick_type,
     METH_VARARGS,
     "Return the type of \"account\" used by the nickname."
     "This function returns \"M\" for hub master, \"O\" for operator, "
     "\"N\" for normal registered user and \"U\" for unregistered user."
   },
   { "get_max_users",
     dchub_get_max_users,
     METH_VARARGS,
     "Get the max number of users of the hub."
   },
   { "get_txtdescription",
     dchub_get_txtdescription,
     METH_VARARGS,
     "Get the hub description (for hub registration on DChublist)"
   },
   { "get_hubip",
     dchub_get_hubip,
     METH_VARARGS,
     "Get the hub address (for hub registration on DChublist)"
   },
   { "db_get",
     dchub_db_get,
     METH_VARARGS,
     "get the value associated to the given key. The returned value "
     "is always a string."
   },
   { "db_set",
     dchub_db_set,
     METH_VARARGS,
     "set the value associated to the given key. The key value can be an "
     "integer, a float or a string."
   },
   { NULL,
     NULL,
     0,
     NULL
   }
};

void initdchub( void )
{
   PyImport_AddModule( "dchub" );
   Py_InitModule( "dchub", dchub_methods );
}
#endif

