/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * ged.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: ged.c,v 2.3 2003/02/15 15:12:12 eric Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <glib.h>

#include "config.h"
#include "ged.h"
#include "ged_if.h"

typedef struct
{
	GED_CALLBACKS *ged_handler;

	int priority;		/* lowest value means higher priority */

	/* for the add_fd & scan_fd functions */
	/* technically, low_range is the size of the pollfd_array before the add_fd call */
	/* and high_range is the size after the call */
	/* thus, this ged_handler has register entry [low_range:high_range[ of the pollfd_array at the last add_fd call */
	/* To be faster, scan_fd call should only scan these entries */
	int low_range;
	int high_range;
	
	int to_delete;	/* ==0, entry is active, !=0, entry is to delete */
	void *extra_info;			/* reserved */
} EMBEDDED_GED;

static GArray *emb_ged_array=NULL;

static int order_embedded_ged(const void *a, const void *b)
{
	return ((EMBEDDED_GED*)a)->priority - ((EMBEDDED_GED*)b)->priority;
}

/**********************************************************************/
/* smallest priority value means the function has an highest priority */
/**********************************************************************/
void ged_register(GED_INIT fnc, int priority)
{
	GED_CALLBACKS *nw;

	if(emb_ged_array==NULL)
	{
		emb_ged_array=g_array_new(FALSE,FALSE,sizeof(EMBEDDED_GED));
	}

	nw=(*fnc)();		/* call the initialization function of this handler */
	if(nw==NULL)
	{
		fprintf(stderr,"ged_register: initialization failed, NULL returned\n");
	}
	else
	{
		EMBEDDED_GED eg;

		eg.ged_handler=nw;
		eg.priority=priority;
		eg.low_range=0;
		eg.high_range=0;
		eg.to_delete=0;
		eg.extra_info=NULL;

		emb_ged_array=g_array_append_val(emb_ged_array,eg);
		
		/* do priority sorting */
		qsort(emb_ged_array->data,emb_ged_array->len,sizeof(EMBEDDED_GED),order_embedded_ged);
	}
}

/****************************/
/* unregister a GED handler */
/*********************************************************************/
/* NOTE: an unregister handler still receives calls as long as its   */
/* exit function has not been called. it is just tagged for deletion */
/* until then.                                                       */
/*********************************************************************/
void ged_unregister(GED_CALLBACKS *nw)
{
	int i;
	EMBEDDED_GED *eg;

	if(emb_ged_array==NULL)
		return;

	for(i=0;i<emb_ged_array->len;i++)
	{
		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if(eg->ged_handler==nw)
		{
			eg->to_delete=1;
		}
	}

	fprintf(stderr,"ged_unregister: unknown GED handler\n");
}

/***************************************************************************/
/* function called before the poll call to add the FD to poll to the array */
/***************************************************************************/
void ged_pre_poll(GArray *pollfd_struct)
{
	int i;

	if(emb_ged_array==NULL)
		return;

	for(i=0;i<emb_ged_array->len;i++)
	{
		EMBEDDED_GED *eg;

		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if(eg->ged_handler->ged_add_fd!=NULL)
		{
			eg->low_range=pollfd_struct->len;
			(eg->ged_handler->ged_add_fd)(eg->ged_handler,pollfd_struct);
			eg->high_range=pollfd_struct->len;
		}
	}
}

/************************************************************************************************************/
/* function called after the poll call to scan the array and perform tasks according to the occuring events */
/************************************************************************************************************/
void ged_post_poll(GArray *pollfd_struct)
{
	int i;

	if(emb_ged_array==NULL)
		return;

	for(i=0;i<emb_ged_array->len;i++)
	{
		EMBEDDED_GED *eg;

		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if((eg->low_range!=eg->high_range)&&(eg->ged_handler->ged_scan_fd!=NULL))
		{
			/* FIXME: possible optimization: check if a revents is not null in the [low_range:high_range[  */
			/* interval. If all are null, no need to call the function */
			(eg->ged_handler->ged_scan_fd)(eg->ged_handler,pollfd_struct, eg->low_range,eg->high_range);
		}
	}
}

/***********************************************/
/* function called to start GED periodic tasks */
/***********************************************/
void ged_periodic(void)
{
	int i;
	static time_t last_call;
	time_t cur_time;

	cur_time=time(NULL);

	if((cur_time-last_call)<10)		/* check only every 10 seconds */
		return;

	last_call=cur_time;

	if(emb_ged_array==NULL)
		return;

	for(i=0;i<emb_ged_array->len;i++)
	{
		EMBEDDED_GED *eg;

		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if(eg->ged_handler->ged_periodic!=NULL)
		{
			(eg->ged_handler->ged_periodic)(eg->ged_handler);
		}
	}
}

/***********************************************/
/* function called to start GED end loop tasks */
/***********************************************/
void ged_always(void)
{
	int i;

	if(emb_ged_array==NULL)
		return;

	/* process always1 */
	for(i=0;i<emb_ged_array->len;i++)
	{
		EMBEDDED_GED *eg;

		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if(eg->ged_handler->ged_always1!=NULL)
		{
			(eg->ged_handler->ged_always1)(eg->ged_handler);
		}
	}

	/* process always2 */
	for(i=0;i<emb_ged_array->len;i++)
	{
		EMBEDDED_GED *eg;

		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if(eg->ged_handler->ged_always2!=NULL)
		{
			(eg->ged_handler->ged_always2)(eg->ged_handler);
		}
	}

	/* process always3 */
	for(i=0;i<emb_ged_array->len;i++)
	{
		EMBEDDED_GED *eg;

		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if(eg->ged_handler->ged_always3!=NULL)
		{
			(eg->ged_handler->ged_always3)(eg->ged_handler);
		}
	}

	/* now, we must purge all deleted entries */
	i=emb_ged_array->len-1;
	while(i>=0)
	{
		EMBEDDED_GED *eg;
		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if(eg->to_delete)
		{
			if(eg->ged_handler->ged_exit!=NULL)
				(eg->ged_handler->ged_exit)(eg->ged_handler);
			g_array_remove_index_fast(emb_ged_array,i);
		}
		else
			i--;
	}
}

/**********************************************/
/* function called to call GED exit functions */
/**********************************************/
void ged_exit(void)
{
	int i;
	EMBEDDED_GED *eg;

	for(i=0;i<emb_ged_array->len;i++)
	{
		eg=&(g_array_index(emb_ged_array,EMBEDDED_GED,i));

		if(eg->ged_handler->ged_exit!=NULL)
		{
			(eg->ged_handler->ged_exit)(eg->ged_handler);
		}
	}
}

