/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * toolkit.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: toolkit.c,v 1.2 2003/12/28 08:12:38 uid68112 Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>

#include "toolkit.h"

/*************************************************************************/
/* this function splits the string (in_string) using the given separator */
/* Each part of the splitted string is appended to chunk. The address of */
/* the appended string is appended to the given tbl.                     */
/* if *tbl or *chunk ==NULL, it is created.                              */
/* multiple consecutive separators are ignored (like strtok).            */
/*************************************************************************/
void split_string_into_array(GPtrArray **tbl, GStringChunk **chunk, const char *in_string, const char *separators)
{
	char *dp;
	char *ptr;
	char *cur;

	if(*tbl==NULL)
		*tbl=g_ptr_array_new();
	if(*chunk==NULL)
		*chunk=g_string_chunk_new(64);

	/* duplicate the initial string */
	dp=strdup(in_string);
	if(dp==NULL)
		return;

	ptr=dp;

	/* and split the string */
	do
	{
		int quoted=0;

		cur=ptr;
		if(*cur=='"')
		{	/* the block is enclosed between quotes */
			cur++;
			ptr=strchr(cur,'"');
			quoted=1;
		}
		else
		{
			ptr=strchr(cur,' ');
		}

		if(ptr!=NULL)
		{
			*ptr++='\0';
		}
		
		/* the string must have a size not null or be quoted (an quoted empty string is valid) */
		if((strlen(cur))||quoted)
		{
			gchar *added;

			added=g_string_chunk_insert((*chunk),cur);
			g_ptr_array_add((*tbl),added);
		}
	}while(ptr!=NULL);
		
	free(dp);
}

/************************************************************************************************/
/* search inside the struct_pollfd_array [low_range:high_range[ for a fd matching the given one */
/************************************************************************************************/
/* NOTE: the returned pointer points inside the Garray*, it is not a copy  */
/*       no test is made to verify if low_range and high_range are indices */
/*       valid in struct_pollfd_array (value must been clamped before)     */
/***************************************************************************/
struct pollfd *pollfd_array_find_fd(GArray *struct_pollfd_array, int fd, int low_range, int high_range)
{
	struct pollfd *pfd;
	int j;

	for(j=low_range;j<high_range;j++)
	{
		pfd=&(g_array_index(struct_pollfd_array, struct pollfd, j));
		if(pfd->fd==fd)
			return pfd;
	}

	return NULL;
}

/***************************************************/
/* compute the size of a NULL terminated ptr array */
/***************************************************/
int size_of_null_array(void **array)
{
	int nb=0;

	while(*array!=NULL)
	{
		nb++;
		array++;
	}
	return nb;
}

/*********************************************************************/
/* return a static const string with the same value as the given one */
/*********************************************************************/
static const char *lst_cnx_type[]=
									{
										"56Kbps",
										"33.6Kbps",
										"28.8Kbps",
										"Satellite",		/* according to DC, passive mode is required here */
										"ISDN",
										"DSL",
										"Cable",
										"LAN(T1)",
										"LAN(T3)",
										NULL
									};


const char *fnd_static_cnx_type(char *type)
{
	int i;

	i=0;
	while(lst_cnx_type[i]!=NULL)
	{
		if(!strcmp(lst_cnx_type[i],type))
			return lst_cnx_type[i];
		i++;
	}

	return "???";
}

const char *fnd_static_cnx_type_from_str_len(char *type, int type_length)
{
	int i;

	i=0;
	while(lst_cnx_type[i]!=NULL)
	{
		if( (!strncmp(lst_cnx_type[i],type,type_length)) && (lst_cnx_type[i][type_length]=='\0') )
			return lst_cnx_type[i];
		i++;
	}

	return "???";
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------- memory access macro ------------------------------ */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*****************************************************************************/
/* some CPU accepts to read word not aligned on a word boundary, other don't */
/*****************************************************************************/
#ifdef NO_UNALIGNED_ACCESS
/* bad luck, the CPU is very stupid */
guint16 get_unaligned_guint16(guint8 *addr)
{
	guint16 a,b;

	/* the &0xFF is not required because we work with unsigned values */
	a=((guint16)addr[0]);
	b=((guint16)addr[1]);
#if G_BYTE_ORDER == G_BIG_ENDIAN
	return (a<<8)|b;
#elif G_BYTE_ORDER == G_LITTLE_ENDIAN
	return (b<<8)|a;
#else
#error unsupported byte order
#endif
}

guint32 get_unaligned_guint32(guint8 *addr)
{
	guint32 a,b;

	/* the &0xFFFF is not required because we work with unsigned values */
	a=get_unaligned_guint16(addr);
	b=get_unaligned_guint16(addr+2);

#if G_BYTE_ORDER == G_BIG_ENDIAN
	return (a<<16)|b;
#elif G_BYTE_ORDER == G_LITTLE_ENDIAN
	return (b<<16)|a;
#else
#error unsupported byte order
#endif
}

guint64 get_unaligned_guint64(guint8 *addr)
{
	guint64 a,b;

	/* the &0xFFFF is not required because we work with unsigned values */
	a=get_unaligned_guint32(addr);
	b=get_unaligned_guint32(addr+4);

#if G_BYTE_ORDER == G_BIG_ENDIAN
	return (a<<32)|b;
#elif G_BYTE_ORDER == G_LITTLE_ENDIAN
	return (b<<32)|a;
#else
#error unsupported byte order
#endif
}

#endif




