/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * Logger.cc
 *   This class allows logging to either the syslog or to
 *   a file or both.
 *
 * $Id: Logger.cc,v 1.8 2003/04/24 04:44:56 benoit Exp $
 *
 * Copyright (c) 2000 Remi Lefebvre <remi@debian.org>
 *                and Luca Filipozzi <lfilipoz@debian.org>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#include <iostream>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <time.h>
#include <unistd.h>

#include "Logger.h"
#include "Exception.h"

Logger *Logger::instance_ = NULL;

Logger *Logger::Instance()
{
    if (instance_ == NULL)
    {
        instance_ = new Logger();
    }
    return instance_;
}

Logger::~Logger ()
{
    closeSyslog();
    closeFilelog();
}

int Logger::openSyslog(int facility, int syslogPriority)
{
    if (facility >= LOG_FAC(LOG_KERN) && facility <= LOG_FAC(LOG_LOCAL7))
    {
        if (syslogPriority >= LOG_CRIT && syslogPriority <= LOG_DEBUG)
        {
            openlog((const char *)ident_, LOG_PID, facility);
            syslogPriority_  = syslogPriority;
            isSyslogEnabled_ = true;
            return 0;
        }
    }
    isSyslogEnabled_ = false;
    return -1;
}

void Logger::closeSyslog()
{
    if (isSyslogEnabled_)
    {
        closelog();
        isSyslogEnabled_ = false;
    }
}

int Logger::openFilelog(char* fileName, int filelogPriority)
{
    if (fileName != NULL && strlen(fileName) != 0)
    {
        if (filelogPriority >= LOG_CRIT && filelogPriority <= LOG_DEBUG)
        {
            file_ = fopen(fileName, "a");
            if (file_ == NULL)
            {
                isFilelogEnabled_ = false;
                return -1;
            }
            filelogPriority_ = filelogPriority;
            isFilelogEnabled_ = true;
            return 0;
        }
    }
    isFilelogEnabled_ = false;
    return -1;
}

void Logger::closeFilelog()
{
    if (isFilelogEnabled_)
    {
        fclose(file_);
        isFilelogEnabled_ = false;
        file_ = NULL;
    }
}

int Logger::setSyslogPriority(int syslogPriority)
{
    if (syslogPriority >= LOG_CRIT && syslogPriority <= LOG_DEBUG)
    {
        syslogPriority_ = syslogPriority;
        return 0;
    }
    return -1;
}

int Logger::setFilelogPriority(int filelogPriority)
{
    if (filelogPriority >= LOG_CRIT && filelogPriority <= LOG_DEBUG)
    {
        filelogPriority_ = filelogPriority;
        return 0;
    }
    return -1;
}

int Logger::setIdent(char* ident)
{
    if (ident_)
    {
        free(ident_);
        ident_ = NULL;
    }

    if (ident != NULL)
    {
        ident_ = strdup(ident);
        return 0;
    }

    ident_ = strdup("logger");
    return -1;
}

void Logger::debug(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    char buf[1024];
    vsnprintf(buf, sizeof(buf), fmt, args);
    log(LOG_DEBUG, buf);

    va_end(args);
}

void Logger::info(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    char buf[1024];
    vsnprintf(buf, sizeof(buf), fmt, args);
    log(LOG_INFO, buf);

    va_end(args);
}

void Logger::notice(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    char buf[1024];
    vsnprintf(buf, sizeof(buf), fmt, args);
    log(LOG_NOTICE, buf);

    va_end(args);
}

void Logger::warning(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    char buf[1024];
    vsnprintf(buf, sizeof(buf), fmt, args);
    log(LOG_WARNING, buf);

    va_end(args);
}

void Logger::error(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    char buf[1024];
    vsnprintf(buf, sizeof(buf), fmt, args);
    log(LOG_ERR, buf);

    va_end(args);
}

void Logger::crit(const char* fmt, ...)
{
    va_list args;
    va_start(args, fmt);

    char buf[1024];
    vsnprintf(buf, sizeof(buf), fmt, args);
    log(LOG_CRIT, buf);

    va_end(args);
}

Logger::Logger()
{
    ident_              = NULL;
    isSyslogEnabled_    = false;
    isFilelogEnabled_   = false;
    syslogPriority_     = LOG_DEBUG;
    filelogPriority_    = LOG_DEBUG;
    file_               = NULL;
    instance_           = NULL;
}

void Logger::log(int priority, const char* string)
{
    if (isSyslogEnabled_ && priority <= syslogPriority_)
    {
        syslog(priority, string);
    }

    if (isFilelogEnabled_ && priority <= filelogPriority_)
    {
        time_t t;
        struct tm *tm;
        char   buf[MAXHOSTNAMELEN];

        time(&t);
        tm = localtime(&t);
        strftime(buf, MAXHOSTNAMELEN, "%b %d %H:%M:%S", tm);
        fprintf(file_, "%s ", buf);

        gethostname(buf, MAXHOSTNAMELEN); // note: reusing buf
        fprintf(file_, "%s %s[%d]: ", buf, ident_, getpid());

        fprintf(file_, "%s\n", string);

        fflush(file_);
    }
}

