package com.tildemh.debbug;

import java.io.IOException;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.util.Date;
import java.util.LinkedList;
import java.util.Vector;

/**
 * Java representation of a Debian Bug Tracking System report 
 *
 * <p>This is released under the terms of the GNU Lesser General public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: Bug.java,v 1.52 2004/06/29 12:04:46 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class Bug implements Serializable, BugListener{

	private BTS bts = BTS.getInstance();
	/** Version of bug class.  */
	private int version = 4;
	/** Are we allowed to write to cache?  */
	private boolean noWrite = false;		

	// Bug details
		/** Bug # */
		private int number = -1;						// BTS record #
		/** Bug Severity */
		private Severity severity = Severity.UNKNOWN;				// Severity of the bug
		private String title = "Bug data not yet downloaded";
		private String packageName = "";
		private String submitter = "";
		private Object data = null; 		// app attached data e.g. Hashtable

		private Vector listings = new Vector(); // of ListingStubs
		
		private volatile BugComment[] commentsArray= new BugComment[0];
		
		private long lastSync = 0;							// Date when record last checked.
		private long  created = 0;							// Date when the bug report was created
		private String forwarded = "";					// Address where the report has been forwarded to.
		private String done = "";						// Name/email of person who closed the bug report, if closed.
		private Status status = Status.INVALID;

		private int[] merged = new int[0];
	
	// tags
		private boolean tagPatch = false;
		private boolean tagSecurity = false;
		private boolean tagUpstream = false;
		private boolean tagPotato = false;
		private boolean tagWoody = false;
		private boolean tagSarge = false;
		private boolean tagSid = false;
		private boolean tagExperimental = false;
		private boolean tagFixed = false;
		private boolean tagFixedInExperimental = false;
		private boolean tagPending = false;
		private boolean tagHelp = false;
		private boolean tagUnreproducible = false;
		 
		private boolean tagD_I = false;
		private boolean tagConfirmed = false;
		private boolean tagIPV6 = false;
		private boolean tagLFS = false;
		private boolean tagWontfix = false;
		private boolean tagMoreinfo = false;
		private boolean tagFixedUpstream = false;
		private Vector tagOthers = new Vector();

	/** Creates a new bug report object */
	public static Bug makeBug( Integer bugNumber ){
		Bug b = new Bug();
		b.setNumber( bugNumber.intValue() );
		return b;
	}
		
	public static Bug makeBug(){
		return new Bug();
	}
	private Bug(){
		addListener(this);
		Cache.getInstance().store(this);
	}


	/**
	 * Sets the last updated field to the current date
	 */
	public void setUpdated() {
		lastSync = (new Date()).getTime();
	}

	public void setLastUpdated( long date ){
		lastSync = date;
	}

	public long getLastUpdated(){
		return lastSync;
	}

	public void setNumber(int number){
		this.number = number;
	}
	public int getNumber(){
		return number;
	}
	public void setSeverity(Severity severity){
		if (severity.equals( this.severity )){ return; };
		Severity old = this.severity;
		this.severity = severity;
		if (!noWrite)
			Cache.getInstance().store(this);
		for (int i = listeners.size(); i > 0; i--)
			((BugListener) listeners.elementAt(i-1)).bugSeverityChanged( this, old );
	}
	public Severity getSeverity(){
		return severity;
	}
	public int getCommentCount(){
		return commentsArray.length;
	}
	public LinkedList getComments(){
		 LinkedList comments = new LinkedList();
                for (int i = 0; i< commentsArray.length; i++){
                        comments.addLast( commentsArray[i] );
                }

		return comments;
	}
	public void setComments( LinkedList comments ){
		if (comments == null) throw new NullPointerException();
		Object[] objs = comments.toArray();
		commentsArray = new BugComment[ objs.length ];
		for(int i = 0; i < objs.length; i++){
				commentsArray[i] = (BugComment) objs[i];
		}
	}

	public String getTitle(){
		return title;
	}
	public void setTitle(String title){
		this.title = title;
	}
	public String getForwarded(){
		return forwarded;
	}
	public void setForwarded(String forwarded){
		this.forwarded = forwarded;
	}
	public String getPackageName(){
		return packageName;
	}
	public void setPackageName(String packageName){
		this.packageName = packageName;
	}
	public String getDone(){
		return done;
	}
	public void setDone(String done){
		this.done = done;
	}
	
	public LinkedList getTags(){
		LinkedList tags = new LinkedList();
		if (tagPatch) tags.add("Patch");
		if (tagSecurity) tags.add("Security");
		if (tagUpstream) tags.add("Upstream");
		if (tagPotato) tags.add("Potato");
		if (tagWoody) tags.add("Woody");
		if (tagSarge) tags.add("Sarge");
		if (tagSid) tags.add("Sid");
		if (tagExperimental) tags.add("Experimental");
		if (tagFixed) tags.add("Fixed in NMU");
		if (tagFixedInExperimental) tags.add("Fixed in experimental");
		if (tagPending) tags.add("Pending");
		if (tagHelp) tags.add("Help" );
		if (tagUnreproducible) tags.add("Unreproducible" );
		if (tagD_I) tags.add("D_I");
		if (tagConfirmed) tags.add("Confirmed");
		if (tagIPV6) tags.add("IPV6");
		if (tagLFS) tags.add("LFS");
		if (tagWontfix) tags.add("Wontfix");
		if (tagMoreinfo) tags.add("Moreinfo");
		if (tagFixedUpstream) tags.add("Fixed Upstream");
		for( int i = 0; i < tagOthers.size(); i++){
			tags.add( tagOthers.get(i) );
		}
		return tags;
	}
	public void parseTags(String tags){
		while (tags.indexOf(",") > 0){
			String tag = tags.substring(0, tags.indexOf(","));
			parseTag(tag);
			if (tags.indexOf(",") > 0)
				tags = tags.substring( tags.indexOf(", ")+2 );
			else
				tags = "";
		}
		if (tags.length() > 0){
			parseTag(tags);
		}
	}
	
		 
	public void parseTag(String tag){
			if (tag.equals("patch")){
				tagPatch = true;
			}else if (tag.equals("upstream")){
				tagUpstream = true;
			}else if (tag.equals("security")){
				tagSecurity = true;
			}else if (tag.equals("potato")){
				tagPotato = true;
			}else if (tag.equals("woody")){
				tagWoody = true;
			}else if (tag.equals("sarge")){
				tagSarge = true;
			}else if (tag.equals("sid")){
				tagSid = true;
			}else if (tag.equals("experimental")){
				tagExperimental = true;
			}else if (tag.equals("fixed")){
				tagFixed = true;
			}else if (tag.equals("fixed-in-experimental")){
				tagFixedInExperimental = true;
			}else if (tag.equals("fixed-upstream")){
				tagFixedUpstream = true;
			}else if (tag.equals("pending")){
				tagPending = true;
			}else if (tag.equals("help")){
				tagHelp = true;
			}else if (tag.equals("unreproducible")){
				tagUnreproducible = true;
			}else if (tag.equals("d-i")){
				tagD_I = true;
			}else if (tag.equals("confirmed")){
				tagConfirmed = true;
			}else if (tag.equals("ipv6")){
				tagIPV6 = true;
			}else if (tag.equals("lfs")){
				tagLFS = true;
			}else if (tag.equals("wontfix")){
				tagWontfix = true;
			}else if (tag.equals("moreinfo")){
				tagMoreinfo = true;
			}else{
				if (! tagOthers.contains( tag ) ){
					tagOthers.add( tag );
				}
			}
	}

	public void setAge(int age){
		//		Calendar cal = Calendar.getInstance();
		//		cal.roll( Calendar.DATE, 0-age);
		//		created = cal.getTime();
		//		TODO?
	}
	/**
	 * WARNING: THIS DOES NOT WORK.
	 * @return the age of the bug report, in days.
	 */
	public int getAge(){
		//		Date now = new Date();
		return -1; //now.getDate() - created.getDate();
	}
	public long getCreated(){
		return created;
	}
	public void setCreated(long created){
		this.created = created;
	}
	public boolean getIsForwarded(){
		return !forwarded.equals("");
	}
	public boolean getIsDone(){
		return !done.equals("");
	}
	public void setStatus(Status status){
		//System.out.println("Bug - status changed");
		Status oldStatus = this.status;
		this.status = status;
		if (!noWrite)
			Cache.getInstance().store(this);
		for (int i = listeners.size(); i > 0; i--)
			((BugListener) listeners.elementAt(i-1)).bugStatusChanged( this, oldStatus );
	}
	public Status getStatus(){
		return status;
	}

	// get/set tags
		public boolean getTagPatch(){ return tagPatch; }					public void setTagPatch( boolean setting ){ tagPatch = setting; }
		public boolean getTagSecurity(){ return tagSecurity; }				public void setTagSecurity( boolean setting ){ tagSecurity = setting; }
		public boolean getTagUpstream(){ return tagUpstream; }				public void setTagUpstream( boolean setting ){ tagUpstream = setting; }
		public boolean getTagPotato(){ return tagPotato; }					public void setTagPotato( boolean setting ){ tagPotato = setting; }
		public boolean getTagWoody(){ return tagWoody; }					public void setTagWoody( boolean setting ){ tagWoody = setting; }
		public boolean getTagSarge(){ return tagSarge; }					public void setTagSarge( boolean setting ){ tagSarge = setting; }
		public boolean getTagSid(){ return tagSid; }						public void setTagSid( boolean setting ){ tagSid = setting; }
		public boolean getTagExperimental(){ return tagExperimental; }		public void setTagExperimental( boolean setting ){ tagExperimental = setting; }
		public boolean getTagFixed(){ return tagFixed; }					public void setTagFixed( boolean setting ){ tagFixed = setting; }
		public boolean getTagFixedInExperimental(){ return tagFixedInExperimental; } public void setTagFixedInExperimental( boolean setting ){ tagFixedInExperimental = setting; }
		public boolean getTagPending(){ return tagPending; }				public void setTagPending( boolean setting ){ tagPending = setting; }
		public boolean getTagHelp(){ return tagHelp; }						public void setTagHelp( boolean setting ){ tagHelp = setting; }
		public boolean getTagUnreproducible(){ return tagUnreproducible; }	public void setTagUnreproducible( boolean setting ){ tagUnreproducible = setting; }
		
		public boolean getTagD_I(){ return tagD_I; }						public void setTagD_I( boolean setting ){ tagD_I = setting; }
		public boolean getTagConfirmed(){ return tagConfirmed; }			public void setTagConfirmed( boolean setting ){ tagConfirmed = setting; }
		public boolean getTagIPV6(){ return tagIPV6; }						public void setTagIPV6( boolean setting ){ tagIPV6 = setting; }
		public boolean getTagLFS(){ return tagLFS; }						public void setTagLFS( boolean setting ){ tagLFS = setting; }
		public boolean getTagWontfix(){ return tagWontfix; }				public void setTagWontfix( boolean setting ){ tagWontfix = setting; }
		public boolean getTagMoreinfo(){ return tagMoreinfo; }		public void setTagMoreinfo( boolean setting ){ tagMoreinfo = setting; }
		public boolean getTagFixedUpstream(){ return tagFixedUpstream; }		public void setTagFixedUpstream( boolean setting ){ tagFixedUpstream = setting; }

	/**
	 * Returns a Vector of strings of tags not known by debbuggtk
	 */
	public Vector getTagOthers(){ return tagOthers; }
	public void setTagOthers( Vector setting ){
		tagOthers = setting;
	}
	public void addTagOthers( String tag ){
		if (!tagOthers.contains(tag))
			tagOthers.add( tag );
	}

	public int[] getMerged(){
		return merged;
	}
	public void setMerged(int[] merged){
		this.merged = merged;
	}
	public void addMerged(int merge){
		int[] newArr = new int[ merged.length + 1 ];
		for (int i = 0; i < merged.length; i++)
			newArr[i] = merged[i];
		newArr[ merged.length ] = merge;
		merged = newArr;
	}

	public String getSubmitter(){
		return submitter;
	}
	public void setSubmitter(String submitter){
		this.submitter = submitter;
	}
	public int hashcode(){
		return number;
	}
	public String toString(){
		return "bug #"+number;
	}

	public void update() throws MalformedURLException, IOException, BugNotFound{
		update(false);
	}
	public void updateVerbose() throws MalformedURLException, IOException, BugNotFound{
		update(true);
	}
	private void update( boolean verbose) throws MalformedURLException, IOException, BugNotFound{
		if (Thread.currentThread() == BTS.GUITHREAD ){
			throw new RuntimeException("Trying to update in main thread");
		}
	
		//System.out.println("bug.update()");
		if (WWWBugParser.checkModified(this))
			WWWBugParser.downloadBug( ""+ number, this, verbose );
		for (int i = listeners.size(); i > 0; i--)
			((BugListener) listeners.elementAt(i-1)).bugUpdated( this );

		//System.out.println("bug.update() - end");
		// TODO
	}

	public Object getData(){
		return data;
	}

	public void setData(Object data){
		this.data = data;
	}
	private boolean complete = false;   // Whether the list is complete or not (false if sync not completed successfully)

	public void setComplete( boolean complete ){
		//System.out.println("bug.setComplete("+complete);
		this.complete = complete;
	}
	public boolean getComplete(){
		return complete;
	}

	/** List of objects interested in events from this listing */
	private transient Vector listeners = new Vector();

	/**
	 * Give us a way to locate a specific listener in a Vector.
	* @param list The Vector of listeners to search.
	* @param listener The object that is to be located in the Vector.
	* @return Returns the index of the listener in the Vector, or -1 if
	*                 the listener is not contained in the Vector.
	 */
	protected static int findListener(Vector list, Object listener) {
		if (null == list || null == listener)
			return -1;
		return list.indexOf(listener);
	}
	/**
	 * Register an object to receive notification of events on this object
	 */
	public synchronized void addListener( BugListener listener ){
		//System.out.println("***Listing: Adding listener " + this);
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(listeners, listener);
		if (i == -1) {
			if (null == listeners)
				listeners = new Vector();
			listeners.addElement(listener);
		}
	}
	
	/**
	 * Unregister an object that was receiving event notification.
	 * 
	 * @param listener The object that is to no longer receive
	 */
	public synchronized void removeListener( BugListener listener) {
		//System.out.println("***Listing: Removing listener " + this);
		int i = findListener(listeners, listener);
		//System.out.println("***Listing: Removing listener " +i);
		if (i > -1)
			listeners.remove(i);
	}

	/**
	 * if <code>setting</code> is true, the bug will not attempt to write itself
	 * to disk (used during construction of the report.
	 */
	public void setNoWrite( boolean setting ){
		noWrite = setting;
	}

	/*
	 * BugListener Interface implementation
	 */
	public void bugStatusChanged( Bug bug, Status oldStatus ){
		for (int i = 0; i < listings.size(); i++){
			Listing l = bts.getListingIfCached( (ListingStub) listings.get(i) );
			if (l != null) l.bugStatusChanged( bug, oldStatus );
		}
	}
	public void bugSeverityChanged( Bug bug, Severity oldSeverity ){
		for (int i = 0; i < listings.size(); i++){
			Listing l = bts.getListingIfCached( (ListingStub) listings.get(i) );
			if (l != null) l.bugSeverityChanged( bug, oldSeverity );
		}
	}
	public void bugUpdated( Bug bug ){ // ignore 
	}
	public boolean bugException( Bug bug, Exception e ){ //ignore
		return false;
	}

	/**
	 * Sets listings
	 */
	public void setListings(Vector listings){
		// TODO: check for new listings and removed listings - If a listing has
		// been removed, we need to tell the listing about it.
		this.listings = listings;
	}
	public Vector getListings(){
		return listings;
	}
	public void addListing( ListingStub stub ){
		if (listings.contains(stub)) return;
		listings.add( stub );
		Cache.getInstance().store(this);
	}
	public void removeListing( ListingStub stub ){
		if (!listings.contains(stub)) return;
		listings.remove( stub );
		Cache.getInstance().store(this);
	}


	/**
	 * Called when a bug is being retrieved, If verbose progress notifications
	 * have been requested
	 */
	public void retrievingBug( Bug bug ){}
	/**
	 * Called when downloading of a bug report has completed
	 */
	public void bugDownloaded( Bug bug ){}

	/**
	 * this is called by wwwbugparser only. It will be removed at some point in
	 * the future
	 */
	public void eventRetrievingBug(){
		for (int i = listeners.size(); i > 0; i--)
			((BugListener) listeners.elementAt(i-1)).retrievingBug( this );
	}
	/**
	 * this is called by wwwbugparser only. It will be removed at some point in
	 * the future
	 */
	public void eventDownloadedBug(){
		for (int i = listeners.size(); i > 0; i--)
			((BugListener) listeners.elementAt(i-1)).bugDownloaded( this );
	}

}
