package com.tildemh.debbuggtk;

import java.io.*;
import java.util.*;

/**
 * A class for storing strings to be sent to control&at;bugs.debian.org.
 * Includes storage to temporary file in case anything goes wrong.
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: ControlQueue.java,v 1.4 2003/12/28 18:51:35 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class ControlQueue{

	/**
	 * An interface for object wanting to be informed when the control queue
	 * becomes empty or has items added to it,
	 */
	public interface Listener{
		/**
		 * Called whenever the control queue is changed.
		 * @param nonEmpty <code>true</code> if there are items waiting in the
		 * controlQueue.
		 */
		public void controlQueueChanged( boolean nonEmpty );
	}
	/** List of objects interested in events from this listing */
	private transient Vector listeners = new Vector();

	/**
	 * Give us a way to locate a specific listener in a Vector.
	* @param list The Vector of listeners to search.
	* @param listener The object that is to be located in the Vector.
	* @return Returns the index of the listener in the Vector, or -1 if
	*                 the listener is not contained in the Vector.
	 */
	protected static int findListener(Vector list, Object listener) {
		if (null == list || null == listener)
			return -1;
		return list.indexOf(listener);
	}
	/**
	 * Register an object to receive notification of events on this object
	 */
	public synchronized void addListener( Listener listener ){
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(listeners, listener);
		if (listeners == null) 
			listeners = new Vector();
		if (i == -1) {
			listeners.addElement(listener);
		}
	}
	
	/**
	 * Unregister an object that was receiving event notification.
	 * 
	 * @param listener The object that is to no longer receive
	 */
	public synchronized void removeListener( Listener listener) {
		int i = findListener(listeners, listener);
		if (i > -1)
			listeners.remove(i);
	}

		
	private static ControlQueue instance = null;

	private StringBuffer messages = new StringBuffer();

	private ControlQueue(){
		loadBackup();
	}

	public static ControlQueue getInstance(){
		if (instance == null){
			instance = new ControlQueue();
		}
		return instance;
	}

	/**
	 * Adds an item to the queue
	 */
	public synchronized void addItem( String item ){
		messages.append(item);
		createBackup();
		for (int i = listeners.size(); i > 0; i--)
			((Listener) listeners.elementAt(i-1)).controlQueueChanged(true);	}

	/**
	 * Generates the email and sends it using the gnome mail handler
	 */
	public synchronized void generateMail(){
		if (messages.length() <=0)
			return;
		org.gnu.gnome.Url.show("mailto:control@bugs.debian.org"
						+ "?subject=Bug updates by debbuggtk"
						+ "&X-debbuggtk-Version="+DebbugGtk.VERSION
						+ "&body="+messages.toString()
						);
		messages = new StringBuffer();
		removeBackup();
		System.out.println("Message sent, queue emptied");
		for (int i = listeners.size(); i > 0; i--)
			((Listener) listeners.elementAt(i-1)).controlQueueChanged(false);	
	}

	/**
	 * Returns the status of the queue
	 */
	public synchronized boolean isEmpty(){
		return (messages.length() <= 0);
	}

	private final String BACKUP_FILE = System.getProperty("user.home") + "/.debbug-java/" + "controlqueue.backup";

	/**
	 * Creates a backup of the data
	 */
	private synchronized void createBackup(){
		try{
			PrintWriter out = new PrintWriter( new BufferedWriter( new FileWriter( new File(BACKUP_FILE) ) ) );
			out.println(messages.toString());
			out.close();
			System.out.println("Backup of unsent control messages saved in "+BACKUP_FILE);
		}catch(IOException e){
			e.printStackTrace();
			// TODO ?
		}
	}
	/**
	 * Deletes the backup of control messages (after sending them to the
	 * server)
	 */
	private synchronized void removeBackup(){
		File f = new File(BACKUP_FILE);
		if (f.exists()){
			f.delete();
		}
	}
	private synchronized void loadBackup(){
		File f = new File(BACKUP_FILE);
		if (!f.exists())
			return;
		try{
			BufferedReader in = new BufferedReader( new FileReader( f ) );
			messages = new StringBuffer();
			String curLine = in.readLine();
			while (curLine != null){
				messages.append(curLine);
				curLine = in.readLine();
			}
			System.out.println("Restored control queue items from backup");
		}catch(IOException e){
			e.printStackTrace();
			// TODO ?
		}
	}
}
