# Copyright 2024 The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Template tag library to render artifacts."""

from django import template

from debusine.artifacts.models import ArtifactCategory
from debusine.db.models import Artifact

register = template.Library()

ARTIFACT_CATEGORY_ICON_NAMES = {
    ArtifactCategory.AUTOPKGTEST: "file-earmark-medical",
    ArtifactCategory.BINARY_PACKAGE: "file-earmark-binary",
    ArtifactCategory.BINARY_PACKAGES: "file-earmark-binary",
    ArtifactCategory.BLHC: "file-earmark-medical",
    ArtifactCategory.LINTIAN: "file-earmark-medical",
    ArtifactCategory.PACKAGE_BUILD_LOG: "journal-text",
    ArtifactCategory.SIGNING_INPUT: "unlock",
    ArtifactCategory.SIGNING_KEY: "filetype-key",
    ArtifactCategory.SIGNING_OUTPUT: "lock",
    ArtifactCategory.SOURCE_PACKAGE: "file-earmark-text",
    ArtifactCategory.SYSTEM_IMAGE: "boxes",
    ArtifactCategory.SYSTEM_TARBALL: "boxes",
    ArtifactCategory.UPLOAD: "file-arrow-up",
    ArtifactCategory.WORK_REQUEST_DEBUG_LOGS: "journal-x",
}

ARTIFACT_CATEGORY_SHORT_NAMES = {
    ArtifactCategory.AUTOPKGTEST: "autopkgtest",
    ArtifactCategory.BINARY_PACKAGE: "binary package",
    ArtifactCategory.BINARY_PACKAGES: "binary packages",
    ArtifactCategory.BLHC: "blhc report",
    ArtifactCategory.LINTIAN: "lintian report",
    ArtifactCategory.PACKAGE_BUILD_LOG: "build log",
    ArtifactCategory.SIGNING_INPUT: "signing input",
    ArtifactCategory.SIGNING_KEY: "signing key",
    ArtifactCategory.SIGNING_OUTPUT: "signing output",
    ArtifactCategory.SOURCE_PACKAGE: "source package",
    ArtifactCategory.SYSTEM_IMAGE: "system image",
    ArtifactCategory.SYSTEM_TARBALL: "system tar",
    ArtifactCategory.UPLOAD: "package upload",
    ArtifactCategory.WORK_REQUEST_DEBUG_LOGS: "debug log",
}


@register.filter
def artifact_icon_name(artifact: Artifact | str) -> str:
    """Return the Bootstrap icon name for this artifact's category."""
    if isinstance(artifact, Artifact):
        category = artifact.category
    else:
        category = artifact
    return ARTIFACT_CATEGORY_ICON_NAMES.get(ArtifactCategory(category), "file")


@register.filter
def artifact_category_label(artifact: Artifact | str) -> str:
    """Return a short label to use to represent an artifact category."""
    if isinstance(artifact, Artifact):
        category = artifact.category
    else:
        category = artifact
    return ARTIFACT_CATEGORY_SHORT_NAMES.get(
        ArtifactCategory(category), "artifact"
    )
