/**
 * Set of integers data type. This implementation uses fd_set from
 * sys/select.h and assumes it's implemented as a bitmask.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */

#include <limits.h>
#include <stdio.h>
#include <string.h>
#include "set.h"
#include "util.h"


/** The empty set. */
const Set empty_set;


/** Prints the specified set. */
void
print_set( const Set* set)
{
	Set tmp = *set;
	int i;
	putchar( '[');
	i = remove_first_element( &tmp);
	while( i != EMPTY_SET) {
		printf( "%d", i);
		i = remove_first_element( &tmp);
		if( i != EMPTY_SET)
			putchar( ' ');
	}
	putchar( ']');
}


/** Reads a set from the specified file. */
void
read_set( Set* set, FILE* file)
{
	int element;
	assume_near( getc( file) == '[', file, "expected [");
	clear_set( set);
	while( fscanf( file, "%d ", &element) == 1)
		add_element( set, element);
	assume_near( getc( file) == ']', file, "expected ]");
}


/** Adds all the elements in the second set to the first (the union).
 * */
void
add_set( Set* left, const Set* right)
{
	long* l = (long*)left;
	const long* r = (const long*)right;
	const long* end = (const long*)(left+1);
	while( l < end)
		*l++ |= *r++;
}


/** Removes and returns the first element in this set.
 * If this set is empty*/
int
remove_first_element( Set* set)
{
	const long* p;
	const long* end = (const long*)(set+1);
	int i = 0;
	for( p = (const long*)set; p < end; p++) {
		if( *p) {
			i += ffs( *p) - 1;
			remove_element( set, i);
			return i;
		}
		i += CHAR_BIT * sizeof *p;
	}
	return EMPTY_SET;
}


/** Returns true if the specified set intersects this set. */
bool
intersects( const Set* left, const Set* right)
{
	long* l = (long*)left;
	const long* r = (const long*)right;
	const long* end = (const long*)(left+1);
	while( l < end)
		if( *l++ & *r++)
			return true;
	return false;
}


/** Returns the first element contained in both sets, or EMPTY_SET if
 * the sets are disjoint. */
int
first_intersection( const Set* left, const Set* right)
{
	const long* l = (const long*)left;
	const long* r = (const long*)right;
	const long* end = (const long*)(left+1);
	int i = 0;
	while( l < end) {
		if( *l & *r)
			return i + ffs( *l & *r) - 1;
		i += CHAR_BIT * sizeof *l;
		l++;
		r++;
	}
	return EMPTY_SET;
}
