# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Francois.Ingelrest@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import os, playlist, tools, traceback

from format          import asf, flac, mp3, mp4, mpc, ogg
from os.path         import splitext
from tools.log       import logger
from track.fileTrack import FileTrack


# Supported formats with associated modules
mFormats = {'.flac': flac, '.m4a': mp4, '.mp3': mp3, '.mp4': mp4, '.mpc': mpc,'.ogg': ogg, '.wma': asf}


def isSupported(file):
    """ Return True if the given file is a supported format """
    file = file.lower()
    return file[-4:] in mFormats or file[-5:] in mFormats


def getSupportedFormats():
    """ Return a list of all formats from which tags can be extracted """
    return ['*' + ext for ext in mFormats]


def getTrackFromFile(file):
    """
        Return a Track object, based on the tags of the given file
        The 'file' parameter must be a real file (not a playlist or a directory)
    """
    file2 = file.lower()

    try:
        try:    return mFormats[file2[-4:]].getTrack(file)
        except: return mFormats[file2[-5:]].getTrack(file)
    except:
        logger.error('Unable to extract information from %s\n\n%s' % (file, traceback.format_exc()))
        return FileTrack(file)


def getTracksFromFiles(files):
    """ Same as getTrackFromFile(), but works on a list of files instead of a single one """
    return [getTrackFromFile(file) for file in files]


def getTracks(filenames):
    """ Same as getTracksFromFiles(), but works for any kind of filenames (files, playlists, directories) """
    allTracks = []

    # Directories
    for directory in [filename for filename in filenames if os.path.isdir(filename)]:
        files, playlists = [], []
        for (filename, path) in tools.listDir(directory):
            if os.path.isfile(path):
                if isSupported(filename):            files.append(path)
                elif playlist.isSupported(filename): playlists.append(path)

        allTracks.extend(sorted(getTracksFromFiles(files)))

        for pl in playlists:
            allTracks.extend(getTracksFromFiles(playlist.load(pl)))

    # Files
    tracks = getTracksFromFiles([filename for filename in filenames if os.path.isfile(filename) and isSupported(filename)])
    allTracks.extend(sorted(tracks))

    # Playlists
    for pl in [filename for filename in filenames if os.path.isfile(filename) and playlist.isSupported(filename)]:
        allTracks.extend(getTracksFromFiles(playlist.load(pl)))

    return allTracks
