#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Francois.Ingelrest@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gettext, gobject, gtk, locale, optparse

from tools import consts, loadGladeFile, log, prefs


def realStartup():
    """
        Perform all the initialization stuff which is not mandatory to display the window
        This function should be called within the GTK main loop, once the window has been displayed
    """
    import atexit, dbus.mainloop.glib, gui.about, modules, webbrowser

    def onDelete(win, event):
        """ Use our own quit sequence, that will itself destroy the window """
        window.hide()
        modules.postQuitMsg()
        return True

    def onResize(win, rect):
        """ Save the new size of the window """
        prefs.set(__name__, 'win-width',  rect.width)
        prefs.set(__name__, 'win-height', rect.height)

    def onToggleExplorer(checkMenuItem):
        """ Show/hide the current explorer """
        prefs.set(__name__, 'show-explorer', checkMenuItem.get_active())
        (width, height) = window.get_size()
        if checkMenuItem.get_active():
            paned.get_child1().show()
            window.resize(width + paned.get_position(), height)
        else:
            paned.get_child1().hide()
            window.resize(width - paned.get_position(), height)

    def onToggleTrackInfo(checkMenuItem):
        """ Show/hide the 'track information' panel """
        prefs.set(__name__, 'show-trackinfo', checkMenuItem.get_active())
        if checkMenuItem.get_active(): wTree.get_widget('box-trkinfo').show()
        else:                          wTree.get_widget('box-trkinfo').hide()

    def atExit():
        """ Final function, called just before exiting the Python interpreter """
        prefs.save()
        log.logger.info('Stopped')

    # D-Bus
    dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)

    # Make sure to perform a few actions before exiting the Python interpreter
    atexit.register(atExit)

    # GTK handlers
    window.connect('delete-event', onDelete)
    window.connect('size-allocate', onResize)
    paned.connect('size-allocate', lambda win, rect: prefs.set(__name__, 'paned-pos', paned.get_position()))
    wTree.get_widget('menu-view-explorer').connect('toggled', onToggleExplorer)
    wTree.get_widget('menu-view-trackinfo').connect('toggled', onToggleTrackInfo)
    wTree.get_widget('menu-quit').connect('activate', lambda item: onDelete(window, None))
    wTree.get_widget('menu-about').connect('activate', lambda item: gui.about.show(window))
    wTree.get_widget('menu-help').connect('activate', lambda item: webbrowser.open(consts.urlHelp))
    wTree.get_widget('menu-preferences').connect('activate', lambda item: modules.showPreferences())

    # Let's go
    modules.postMsg(consts.MSG_EVT_APP_STARTED)


# --== Entry point ==--


log.logger.info('Started')

# Localization
locale.setlocale(locale.LC_ALL, '')
gettext.textdomain(consts.appNameShort)
gettext.bindtextdomain(consts.appNameShort, consts.dirLocale)
gtk.glade.textdomain(consts.appNameShort)
gtk.glade.bindtextdomain(consts.appNameShort, consts.dirLocale)

# Command line
parser = optparse.OptionParser(usage='Usage: %prog [FILE(s)]')
prefs.setCmdLine(parser.parse_args())

# PyGTK initialization
gobject.threads_init()
gtk.window_set_default_icon_list(gtk.gdk.pixbuf_new_from_file(consts.fileImgIcon16),
                                 gtk.gdk.pixbuf_new_from_file(consts.fileImgIcon24),
                                 gtk.gdk.pixbuf_new_from_file(consts.fileImgIcon32),
                                 gtk.gdk.pixbuf_new_from_file(consts.fileImgIcon48),
                                 gtk.gdk.pixbuf_new_from_file(consts.fileImgIcon64),
                                 gtk.gdk.pixbuf_new_from_file(consts.fileImgIcon128))

# Create the GUI
wTree  = loadGladeFile('MainWindow.glade')
paned  = wTree.get_widget('pan-main')
window = wTree.get_widget('win-main')
prefs.setWidgetsTree(wTree)

# RGBA support
try:
    colormap = window.get_screen().get_rgba_colormap()
    if colormap:
        gtk.widget_set_default_colormap(colormap)
except:
    log.logger.info('No RGBA support (requires PyGTK 2.10+)')

# Restore sizes
window.resize(prefs.get(__name__, 'win-width', 750), prefs.get(__name__, 'win-height', 470))
paned.set_position(prefs.get(__name__, 'paned-pos', 300))

# Show the window
window.show_all()

# Restore the state of the explorer
showExplorer = prefs.get(__name__, 'show-explorer', True)
wTree.get_widget('menu-view-explorer').set_active(showExplorer)
if not showExplorer:
    paned.get_child1().hide()

# Restore the state of the 'track information' panel
showTrackInfo = prefs.get(__name__, 'show-trackinfo', True)
wTree.get_widget('menu-view-trackinfo').set_active(showTrackInfo)
if not showTrackInfo:
    wTree.get_widget('box-trkinfo').hide()

# Initialization done, let's continue the show
gobject.idle_add(realStartup)
gtk.main()
