/*
 * A Decibel Demo
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <QtCore/QStringList>
#include <QtCore/QDebug>
#include <QtCore/QCoreApplication>
#include <QtDBus/QDBusConnection>

#include <Decibel/AccountManager>
#include <Decibel/DBusNames>

// Add an existing account to the accountmanager.
// This demo can not register new accounts with servers!
//
// Usage: $0 (key=TYPE:value)+
//
//
// TYPE is either 'i' (integer) or 's' (string)
//
// Required key/value pairs:
//    decibel_protocol=s:<PROTOCOL>
// plus all the data required by the preferred connection manager
// for the protocol used.

int main(int argc, char ** argv)
{
    QCoreApplication app(argc, argv);
    Q_ASSERT(argc >= 2);

    if (!QDBusConnection::sessionBus().isConnected())
    {
        qDebug("Cannot connect to the D-BUS session bus.\n"
               "To start it, run:\n"
               "\teval `dbus-launch --auto-syntax`\n");
        return 1;
    }

    Decibel::registerTypes();

    // Evaluate parameters and construct QVariantMap:
    QVariantMap data;
    for (int i = 1; i < argc; ++i)
    {
        QStringList keyValue(QString(argv[i]).split("="));

        // instead of proper error handling:
        Q_ASSERT(keyValue.size() == 2);
        Q_ASSERT(!keyValue[0].isEmpty());
        Q_ASSERT(keyValue[1].size() >= 2);
        Q_ASSERT(keyValue[1][1] == ':');

        QString key(keyValue[0]);
        QChar type = keyValue[1][0].toLower();
        QString value_string(keyValue[1].mid(2));

        QVariant value;
        if (type == QChar('i')) { value = QVariant(value_string.toInt()); }
        else if (type == QChar('s')) { value = QVariant(value_string); }
        else
        {
            qDebug() << "Unsupported Type" << type << "found.";
            return 2;
        }

        data.insert(key, value);
    }

    // connect to the Decibel daemon:
    de::basyskom::Decibel::AccountManager
    account_mgr(Decibel::daemon_service, Decibel::daemon_accountmanager_path,
                QDBusConnection::sessionBus(), &app);

    QDBusReply<uint> reply = account_mgr.addAccount(data);
    if(!reply.isValid())
    {
            QDBusError error = reply.error();
            qDebug() << "DBus Error:" << error.type() << ":" << error.message();
            return 3;
    }

    if (reply.value() > 0)
    { qDebug() << "Success: Account Handle is:" << reply.value(); }
    else
    { qDebug() << "Error: Error code was:" << reply.value(); }

    return 0;
}
