/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DECIBEL_DAEMON_CONTACTCONNECTOR_H
#define _DECIBEL_DAEMON_CONTACTCONNECTOR_H

#include <QtCore/QVariantMap>

#include <QtTapioca/ContactBase>

/**
 * @brief A API for connectors to contact data management systems.
 *
 *
 * The ContactConnector is a abstract base class defining an interface
 * used by the ContactManager to interact with the users preferred PIM system.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class ContactConnector
{
public:
    /** @brief Destructor. */
    virtual ~ContactConnector() { }

    /**
     * @brief Check whether a contact of the given ID exists.
     * @param contact_id A unique ID for a contact.
     * @returns True if the contact exists and false otherwise.
     *
     * Check whether a contact of the given ID exists.
     */
    virtual bool gotContact(const uint & contact_id) const = 0;

    /**
     * @brief Get contact information from the pim system.
     * @param contact_id A unique ID for a contact.
     * @returns The contact's data (is empty if the contact_id was unknown).
     *
     * Get contact information from the pim system. Returns a empty
     * QVariantMap if the contact was unknown to the PIM system.
     */
    virtual QVariantMap contact(const uint & contact_id) const = 0;

    /**
     * @brief Update presence information on a contact.
     * @param contact_id The unique ID of the contact to update.
     * @param presence_state The new presence state.
     * @param message The new presence message.
     *
     * This method updates the presence information of a contact in the
     * PIM system the connector talks to.
     */
    virtual void setPresence(const uint & contact_id,
                             const QtTapioca::ContactBase::Presence presence_state,
                             const QString & message) = 0;

    /**
     * @brief Find the contact that owns the URI.
     * @param protocol The protocol the URI is using.
     * @param uri The uri used.
     * @return The contact ID that own given URI. 0 if the URI was not found.
     *
     * This method returns the ID of the contact owning the given URI.
     */
    virtual unsigned int findURI(const QString & protocol,
                                 const QString & uri) const = 0;

    /**
     * @brief Adds a contact to the PIM DB the connector is associated with.
     * @param contact_infos The infos to be stored in the DB.
     * @return The ID of the new contact. A ID of 0 indicates an error.
     *
     * This method adds a contact to the DB the connector is associated with.
     * It fetches its information from the IM network and adds it to the database.
     */
    virtual unsigned int addContact(const QVariantMap & contact_infos) = 0;

protected:
private:
};

#endif
