/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "contactmanager.h"
#include "contactmanageradaptor.h"

#include "accountmanager.h"
#include "contactconnector.h"
#include "simplisticcontactconnector.h"

#include <QtCore/QHash>
#include <QtCore/QCoreApplication>
#include <QtCore/QPointer>
#include <QtCore/QDebug>

#include <QtTapioca/Connection>
#include <QtTapioca/Contact>
#include <QtTapioca/ContactList>
#include <QtTapioca/Handle>

/**
 * @brief Private data class for the ContactManager.
 * @author Tobias Hunger <info@basyskom.de>
 */
class ContactManagerPrivate : public QObject
{
public:
    /** @brief Constructor. */
    ContactManagerPrivate(AccountManager * account_mgr) :
        accountManager(account_mgr),
        m_connector(new SimplisticContactConnector())
    {
        Q_ASSERT(0 != accountManager);
        Q_ASSERT(0 != m_connector);
    }

    /** @brief Destructor. */
    ~ContactManagerPrivate()
    {
        delete m_connector;
    }

    /**
     * @brief Register a new contact found on a connection.
     * @param protocol The protocol of the connection.
     * @param contact A pointer to the contact.
     * @param connection A pointer to the connection.
     *
     * Register a contact found on one of the connections in the mappings
     * to/from the external ID.
     */
    void registerContact(const QString & protocol,
                         QtTapioca::Contact * contact,
                         const QtTapioca::Connection * connection)
    {
        Q_ASSERT(0 != connection);
        Q_ASSERT(0 != contact);
        Q_ASSERT(!protocol.isEmpty());
        QtTapioca::Handle * handle = contact->handle();
        Q_ASSERT(0 != handle);

        // mark the handle as belonging to the given connection:
        m_connectionlist[connection].append(contact);

        // Update mapping between handles:
        uint external_id = m_connector->findURI(protocol, contact->uri());
        if (external_id == 0)
        {
            // FIXME: Add data to PIM?
        }
        else
        {
            Q_ASSERT(0 != external_id);
            // FIXME: Do I need to update presence, etc. info here?
            m_telepathy2external.insert(contact, external_id);
            m_external2telepathy.insert(external_id, contact);
        }
        Q_ASSERT(m_telepathy2external.size() == m_external2telepathy.size());
    }

    /**
     * @brief Deregister a contact.
     * @param connection A pointer to a connection.
     *
     * Deregister all contacts found on the given connection from the
     * mappings between internal and external contact IDs.
     */
    void deregisterContacts(const QtTapioca::Connection * connection)
    {
        Q_ASSERT(0 != connection);
        if (!m_connectionlist.contains(connection)) { return; }

        QtTapioca::Contact * internal_contact = 0;
        foreach (internal_contact, m_connectionlist[connection])
        {
            Q_ASSERT(m_telepathy2external.contains(internal_contact));
            uint external_id = m_telepathy2external[internal_contact];
            Q_ASSERT(m_external2telepathy.contains(external_id));
            Q_ASSERT(m_external2telepathy[external_id] == internal_contact);

            m_external2telepathy.remove(external_id);
            m_telepathy2external.remove(internal_contact);
        }
        Q_ASSERT(m_telepathy2external.size() == m_external2telepathy.size());
        m_connectionlist.remove(connection);
    }

    /**
     * @brief Update the presence of a contact.
     * @param internal_contact internal (telepathy) contact handle.
     * @param presence The presence state.
     * @param message The presence message.
     *
     * Update the presence state and message of a contact.
     */
    void setPresence(QtTapioca::Contact * internal_contact,
                     const QtTapioca::ContactBase::Presence presence,
                     const QString message)
    {
        Q_ASSERT(0 != internal_contact);
        Q_ASSERT(presence >= QtTapioca::ContactBase::Offline &&
                 presence <= QtTapioca::ContactBase::Busy);

        uint external_id = mapToExternal(internal_contact);
        if (0 == external_id) { return; }
        m_connector->setPresence(external_id, presence, message);
    }

    /**
     * @brief Check whether a contact ID is defined.
     * @param external_id contact ID from the external PIM system.
     * @return true if the contact is defined and false otherwise.
     *
     * Check whether a contact ID is defined.
     */
    bool gotContact(const uint external_id)
    { return m_connector->gotContact(external_id); }

    /**
     * @brief Map a Contact to its external ID.
     * @param contact A pointer to the Contact.
     * @return 0 if the contact is not known in the external PIM system and
     * the contact's ID otherwise.
     *
     * Map a Contact to its external ID.
     */
    uint mapToExternal(QtTapioca::Contact * contact)
    {
        if (!m_telepathy2external.contains(contact)) { return 0; }
        else
        {
            Q_ASSERT(m_telepathy2external[contact] != 0);
            return m_telepathy2external[contact];
        }
    }

    /**
     * @brief Map a external contact ID to the internal Contact data.
     * @param external_id The contact's ID in the external PIM system.
     * @return A pointer to the internal Contact structure. This pointer is 0
     * if the contact is not know to telepathy.
     *
     * Map a external contact ID to the internal Contact data.
     */
    QtTapioca::Contact * mapFromExternal(const uint & external_id)
    {
        if (!m_external2telepathy.contains(external_id)) { return 0; }
        else
        {
            Q_ASSERT(m_external2telepathy[external_id] != 0);
            return m_external2telepathy[external_id];
        }
    }

    /**
     * @brief Check whether a contact can be reached using a given account.
     * @param external_id The contact's ID in the external PIM system.
     * @param account_handle A handle to an account.
     * @return true if the contact can be reached from the account and false
     * otherwise
     *
     * Check whether a contact can be reached using a given account. This
     * does not include a check whether the contact is online. It is about
     * whether there is any chance to ever see the contact from the account.
     */
    bool isReachable(const uint external_id, const int account_handle)
    {
        // FIXME: Implement this.
        return true;
    }

    /** @brief A pointer to the AccountManager. */
    QPointer<AccountManager> accountManager;
    /** @brief A pointer to the D-Bus Adaptor of the ContactManager. */
    QPointer<ContactManagerAdaptor> adaptor;

private:
    /** @brief A pointer to the ContactConnector used. */
    SimplisticContactConnector * m_connector;

    /**
     * @brief A mapping from the Tapioca Contact structure to the external ID
     * of the contact.
    */
    QHash<QtTapioca::Contact *, uint> m_telepathy2external;
    /**
     * @brief A mapping from the external ID of a contact to its Tapioca contact
     * structure.
     */
    QHash<uint, QtTapioca::Contact *> m_external2telepathy;
    /**
     * @brief A mapping between Connections and their Contacts.
     */
    QHash<const QtTapioca::Connection *,
          QList<QtTapioca::Contact *> > m_connectionlist;
};

// **************************************************************************

ContactManager::ContactManager(AccountManager * account_mgr,
                               QObject * parent) :
        QObject(parent),
        d(new ContactManagerPrivate(account_mgr))
{
    Q_ASSERT(0 != d);
    d->adaptor = new ContactManagerAdaptor(this);
    Q_ASSERT(0 != d->adaptor);
}

ContactManager::~ContactManager()
{
    delete d;
}

Decibel::ChannelInfo
ContactManager::contactContactUsingAccount(const uint contact_id,
                                           const int account_handle,
                                           const int type,
                                           const bool suppress_handler)
{
    // Make sure data is valid:
    // * handles:
    if (!d->gotContact(contact_id))
    {
        qWarning() << tr("Contact %1 is invalid.",
                         "1: contact_id").arg(contact_id);
        return Decibel::ChannelInfo();
    }

    if (!d->accountManager->gotAccount(account_handle))
    {
        qWarning() << tr("Account %1 is invalid.",
                         "1: account_handle").arg(account_handle);
        return Decibel::ChannelInfo();
    }

    // * type:
    if (type < int(QtTapioca::Channel::Text) ||
        type > int(QtTapioca::Channel::Video))
    {
        qWarning() << tr("Invalid type of connection requested: %1.",
                         "1: type").arg(type);
        return Decibel::ChannelInfo();
    }

    // check whether contact is reachable from the account:
    if (!d->isReachable(contact_id, account_handle))
    {
        qWarning() << tr("contact %1 can not be reached from account %2.",
                         "1: contact_id, 2: account_handle").
                      arg(contact_id).
                      arg(account_handle);
        return Decibel::ChannelInfo();
    }

    // Bring up account (if not already up):
    if (d->accountManager->presence(account_handle) ==
         QtTapioca::ContactBase::Offline)
    {
        Q_ASSERT(0 == d->accountManager->connectionOf(account_handle));
        d->accountManager->setPresence(account_handle,
                                       QtTapioca::ContactBase::Available);
    }

    // Wait for connection to get established...
    QtTapioca::Connection * connection(0);
    qDebug(qPrintable(tr("Looping for connection...")));
    while (0 == connection)
    {
        QCoreApplication::instance()->processEvents();
        connection = d->accountManager->connectionOf(account_handle);
    }
    qDebug(qPrintable(tr("Connection found.")));

    // We have a connection from this point on:
    Q_ASSERT(0 != connection);

    // Check whether contact is online:
    QtTapioca::Contact * contact = d->mapFromExternal(contact_id);
    if (0 == contact)
    {
        qWarning() << tr("Can not open a channel to contact %1: Not online.",
                         "1: contact_id").arg(contact_id);
        return Decibel::ChannelInfo();
    }
    qDebug(qPrintable(tr("Contact is online.")));

    QtTapioca::Channel * channel =
        connection->createChannel(QtTapioca::Channel::Type(type), contact,
                                  suppress_handler);
    Q_ASSERT(channel != 0);

    return Decibel::ChannelInfo(connection, channel);
}


void ContactManager::onConnectionOpened(QtTapioca::Connection * connection)
{
    Q_ASSERT(0 != connection);
    Q_ASSERT(connection->status() == QtTapioca::Connection::Connected);

    const QtTapioca::ContactList * contact_list = connection->contactList();
    Q_ASSERT(0 != contact_list);

    const QString protocol(connection->protocol());
    Q_ASSERT(!protocol.isEmpty());

    QList<QtTapioca::Contact *> known_contacts = contact_list->knownContacts();
    QtTapioca::Contact * internal_contact = 0;
    foreach (internal_contact, known_contacts)
    {
        Q_ASSERT(0 != internal_contact);

        QtTapioca::Handle * internal_handle = internal_contact->handle();
        Q_ASSERT(0 != internal_handle);
        // FIXME: Handle non-contact handles properly:
        switch (internal_handle->type())
        {
            case QtTapioca::Handle::Contact:
                d->registerContact(protocol, internal_contact, connection);
                connect(internal_contact,
                        SIGNAL(presenceUpdated(QtTapioca::ContactBase *,
                                               QtTapioca::ContactBase::Presence,
                                               const QString &)),
                        this,
                        SLOT(onPresenceUpdated(QtTapioca::ContactBase *,
                                               QtTapioca::ContactBase::Presence,
                                               const QString &)));
                break;
            case QtTapioca::Handle::Room:
            case QtTapioca::Handle::List:
            case QtTapioca::Handle::None:
                // FIXME: Handle other types of handles...
                break;
            default:
                qWarning() << "Unknown Handle type encountered.";
        }
    }
}

void ContactManager::onConnectionClosed(QtTapioca::Connection * connection)
{
    Q_ASSERT(0 != connection);
    Q_ASSERT(connection->status() == QtTapioca::Connection::Disconnected);

    d->deregisterContacts(connection);
}

void ContactManager::onPresenceUpdated(QtTapioca::ContactBase * internal_contactbase,
                                       QtTapioca::ContactBase::Presence presence,
                                       const QString & message)
{
    Q_ASSERT(0 != internal_contactbase);
    QtTapioca::Contact * internal_contact =
        dynamic_cast<QtTapioca::Contact *>(internal_contactbase);
    if (0 == internal_contact) { return; }

    const QtTapioca::Handle * handle = internal_contact->handle();
    Q_ASSERT(0 != handle);
    switch(handle->type())
    {
        case QtTapioca::Handle::Contact:
            d->setPresence(internal_contact, presence, message);
            break;
        case QtTapioca::Handle::Room:
        case QtTapioca::Handle::List:
        case QtTapioca::Handle::None:
            // FIXME: Handle other types of handles...
            break;
        default:
            qWarning() << "Unknown Handle type encountered.";
    }
}
