/*
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "decibelchannelhandler.h"

#include <Decibel/ContactManager>
#include <Decibel/DBusNames>

#include <QtCore/QDebug>

#include <QtTapioca/StreamChannel>
#include <QtTapioca/ChannelTarget>

class DecibelChannelHandlerPrivate
{
public:
    DecibelChannelHandlerPrivate() :
        last_state(QtTapioca::Stream::Paused)
    { }
    ~DecibelChannelHandlerPrivate() { }

    QList<QtTapioca::StreamChannel *> streamChannels;
    QtTapioca::Stream::StreamState last_state;
private:
};

// ---------------------------------------------------------------

DecibelChannelHandler::DecibelChannelHandler(QObject * parent) :
    Decibel::ChannelHandler(parent),
    d(new DecibelChannelHandlerPrivate())
{ }

DecibelChannelHandler::~DecibelChannelHandler()
{ delete d; }

bool DecibelChannelHandler::handleChannel(QtTapioca::Connection * connection,
                                          QtTapioca::Channel * channel)
{
    Q_ASSERT(connection != 0);
    Q_ASSERT(channel != 0);

    QtTapioca::StreamChannel * stream_channel =
        dynamic_cast<QtTapioca::StreamChannel *>(channel);
    if (0 == stream_channel)
    { return false; }

    connect(stream_channel, SIGNAL(closed()), this, SLOT(doChannelClosed()));
    connect(stream_channel, SIGNAL(newStream(QtTapioca::Stream *)),
            this, SLOT(doNumberOfStreamsChanged()));
    connect(stream_channel, SIGNAL(streamRemoved(QtTapioca::Stream *)),
            this, SLOT(doNumberOfStreamsChanged()));

    d->streamChannels.append(stream_channel);

    QtTapioca::Stream * stream;
    foreach (stream, stream_channel->streams())
    {
        connect(stream, SIGNAL(stateChanged(QtTapioca::Stream *, QtTapioca::Stream::StreamState)),
                this, SLOT(doNumberOfStreamsChanged()));
        connect(stream, SIGNAL(streamError(QtTapioca::Stream *, int, const QString &)),
                this, SLOT(doStreamError(QtTapioca::Stream *, int, const QString &)));
    }

    doNumberOfStreamsChanged();

    return true;
}

void DecibelChannelHandler::hangUp()
{
    QtTapioca::StreamChannel * stream_channel;
    foreach (stream_channel, d->streamChannels)
    { stream_channel->close(); }
}

void DecibelChannelHandler::doNumberOfStreamsChanged()
{
    QtTapioca::StreamChannel * stream_channel(0);
    QtTapioca::Stream * stream(0);

    QtTapioca::Stream::StreamState state = QtTapioca::Stream::Paused;
    QString number;

    foreach (stream_channel, d->streamChannels)
    {
        foreach (stream, stream_channel->streams())
        {
            switch(stream->state())
            {
            case QtTapioca::Stream::Paused:
                break;
            case QtTapioca::Stream::Connecting:
                if (state == QtTapioca::Stream::Paused)
                {
                    state = QtTapioca::Stream::Connecting;
                    number = stream_channel->target()->uri();
                }
                break;
            case QtTapioca::Stream::Playing:
                if (state == QtTapioca::Stream::Paused ||
                    state == QtTapioca::Stream::Connecting)
                {
                    state = QtTapioca::Stream::Playing;
                    number = stream_channel->target()->uri();
                }
                break;
            default:
                // Should never get here!
                Q_ASSERT(false);
            }
        }
    }
    if (state != d->last_state)
    {
        d->last_state = state;
        switch(state)
        {
        case QtTapioca::Stream::Paused:
            emit disconnected();
            break;
        case QtTapioca::Stream::Connecting:
            emit ringing(number);
            break;
        case QtTapioca::Stream::Playing:
            emit connected(number);
            break;
        default:
            // Should never get here!
            Q_ASSERT(false);
        }
    }
}

void DecibelChannelHandler::doStreamError(QtTapioca::Stream * stream,
                                          int code, const QString & message)
{ qWarning() << "STREAM ERROR:" << stream << code << ":" << message; }

void DecibelChannelHandler::doChannelClosed()
{
    QtTapioca::StreamChannel * stream_channel = dynamic_cast<QtTapioca::StreamChannel *>(sender());
    if (0 == stream_channel) { return; }

    d->streamChannels.removeAll(stream_channel);
    doNumberOfStreamsChanged();
}

#include "decibelchannelhandler.moc"
